﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nn/fssystem/fs_BucketTreeBuilder.h>
#include <nn/fssystem/fs_BucketTreeUtility.h>
#include "testFs_Unit_BucketTreeUtil.h"

namespace {

typedef nn::util::ConstBytePtr Pointer;
typedef nn::fssystem::BucketTree BucketTree;
typedef nn::fssystem::BucketTree::NodeHeader NodeHeader;
typedef nn::fssystem::BucketTreeBuilder BucketTreeBuilder;
typedef nn::fssystem::detail::SafeValue ValueProxy;

/**
 * @brief   BucketTreeBuilder のテストクラスです。
 */
class BucketTreeBuilderTest : public nn::fssystem::BucketTreeTest
{
public:
    class DataChecker;
};

typedef BucketTreeBuilderTest BucketTreeBuilderDeathTest;

/**
 * @brief   BucketTreeBuilder で作成したデータをチェックするクラスです。
 */
class BucketTreeBuilderTest::DataChecker
{
public:
    // コンストラクタです。
    DataChecker(
        size_t nodeSize,
        size_t entrySize,
        int entryCount,
        int64_t endOffset,
        const void* pNodeL1,
        const void* pEntrySet,
        const void* pEntryData
    ) NN_NOEXCEPT
        : m_NodeUnitSize(nodeSize)
        , m_EntryUnitSize(entrySize)
        , m_TotalEntryCount(entryCount)
        , m_EntryCount(GetEntryCount(nodeSize, entrySize))
        , m_OffsetCount(GetOffsetCount(nodeSize))
        , m_NodeL2Count(GetNodeL2Count(nodeSize, entrySize, entryCount))
        , m_EntrySetCount(GetEntrySetCount(nodeSize, entrySize, entryCount))
        , m_EndOffset(endOffset)
        , m_pNodeL1(pNodeL1)
        , m_pEntry(pEntrySet)
        , m_pEntrySet(pEntrySet)
        , m_pEntryData(pEntryData)
        , m_MemZero(new char[nodeSize])
    {
        NN_ABORT_UNLESS_NOT_NULL(m_MemZero.get());
        std::memset(m_MemZero.get(), 0, nodeSize);
    }

    // データをチェックします。
    void Verify() NN_NOEXCEPT
    {
        if( m_NodeL2Count == 0 )
        {
            VerifyL1();
        }
        else
        {
            VerifyL2();
        }
    }

private:
    // L1 のみのデータをチェックします。
    void VerifyL1() NN_NOEXCEPT
    {
        auto pOffset = m_pNodeL1;

        // L1 ノードヘッダのチェック
        {
            auto headerL1 = *m_pNodeL1.Get<NodeHeader>();

            EXPECT_EQ(0, headerL1.index);
            EXPECT_EQ(m_EntrySetCount, headerL1.count);
            EXPECT_EQ(m_EndOffset, headerL1.offset);

            pOffset.Advance(sizeof(NodeHeader));
        }

        NodeHeader header = { 0, 0, -1 };

        for( int i = 0; i < m_TotalEntryCount; ++i )
        {
            VerifyEntry(&header, pOffset, i);

            if( i % m_EntryCount == 0 )
            {
                pOffset.Advance(sizeof(int64_t));
            }
        }

        // 末尾の空き領域をチェック
        if( (m_TotalEntryCount % m_EntryCount) != 0 )
        {
            m_pNodeL1.Advance(m_NodeUnitSize);
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(
                pOffset.Get(), m_MemZero.get(), pOffset.Distance(m_pNodeL1.Get()));

            m_pEntrySet.Advance(m_NodeUnitSize);
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(
                m_pEntry.Get(), m_MemZero.get(), m_pEntry.Distance(m_pEntrySet.Get()));
        }
    }

    // L2 を含むデータをチェックします。
    void VerifyL2() NN_NOEXCEPT
    {
        auto pOffsetL1 = m_pNodeL1;

        // L1 ノードヘッダのチェック
        {
            auto headerL1 = *m_pNodeL1.Get<NodeHeader>();

            EXPECT_EQ(0, headerL1.index);
            EXPECT_EQ(m_NodeL2Count, headerL1.count);
            EXPECT_EQ(m_EndOffset, headerL1.offset);

            pOffsetL1.Advance(sizeof(NodeHeader));
        }

        const auto L2OnL1EntryCount = (m_OffsetCount - m_NodeL2Count) * m_EntryCount;
        NN_SDK_ASSERT_LESS(L2OnL1EntryCount, m_TotalEntryCount);

        NodeHeader header = { 0, 0, -1 };

        // L1 ノード内の L2 部分をチェック
        {
            auto pOffset = pOffsetL1;
            pOffset.Advance(m_NodeL2Count * sizeof(int64_t));

            for( int i = 0; i < L2OnL1EntryCount; ++i )
            {
                VerifyEntry(&header, pOffset, i);

                if( i % m_EntryCount == 0 )
                {
                    pOffset.Advance(sizeof(int64_t));
                }
            }
        }

        auto pNodeL2 = m_pNodeL1;
        pNodeL2.Advance(m_NodeUnitSize);

        int nodeIndex = 0;
        auto pOffsetL2 = pNodeL2;
        NodeHeader headerL2 = { 0, 0, -1 };

        for( int i = L2OnL1EntryCount; i < m_TotalEntryCount; ++i )
        {
            if( (i % m_EntryCount == 0) && (nodeIndex % m_OffsetCount == 0) )
            {
                if( 0 <= headerL2.offset )
                {
                    pNodeL2.Advance(m_NodeUnitSize);

                    // １つ前の L2 ノードのオフセットをチェック
                    const auto nextOffset = ValueProxy::GetInt64(m_pEntryData.Get());
                    const auto headerOffset = headerL2.offset;
                    EXPECT_EQ(nextOffset, headerOffset);

                    NN_SDK_ASSERT_EQUAL(pOffsetL2.Get(), pNodeL2.Get());
                }

                std::memcpy(&headerL2, pNodeL2.Get<NodeHeader>(), sizeof(NodeHeader));

                // L2 ノードのヘッダチェック
                {
                    const auto index = nodeIndex / m_OffsetCount;
                    EXPECT_EQ(index, headerL2.index);

                    const auto entrySetCount =
                        m_EntrySetCount - (m_OffsetCount - m_NodeL2Count);
                    const auto count = m_OffsetCount <= (entrySetCount - nodeIndex)
                                            ? m_OffsetCount
                                            : (entrySetCount % m_OffsetCount);
                    EXPECT_EQ(count, headerL2.count);
                }

                pOffsetL2.Advance(sizeof(NodeHeader));

                // オフセットが一致することを確認
                const auto nodeOffset = ValueProxy::GetInt64(pOffsetL1.Get());
                const auto dataOffset = ValueProxy::GetInt64(m_pEntryData.Get());
                EXPECT_EQ(dataOffset, nodeOffset);

                pOffsetL1.Advance(sizeof(int64_t));
            }

            VerifyEntry(&header, pOffsetL2, i);

            if( i % m_EntryCount == 0 )
            {
                pOffsetL2.Advance(sizeof(int64_t));

                ++nodeIndex;
            }
        }

        // 末尾の空き領域をチェック
        if( (m_TotalEntryCount % m_EntryCount) != 0 )
        {
            pNodeL2.Advance(m_NodeUnitSize);
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(
                pOffsetL2.Get(), m_MemZero.get(), pOffsetL2.Distance(pNodeL2.Get()));

            m_pEntrySet.Advance(m_NodeUnitSize);
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(
                m_pEntry.Get(), m_MemZero.get(), m_pEntry.Distance(m_pEntrySet.Get()));
        }
    }

    // エントリデータをチェックします。
    void VerifyEntry(NodeHeader* pHeader, Pointer pOffset, int entryIndex) NN_NOEXCEPT
    {
        if( entryIndex % m_EntryCount == 0 )
        {
            if( 0 <= pHeader->offset )
            {
                m_pEntrySet.Advance(m_NodeUnitSize);

                // 予約領域が０埋めされていることを確認
                const auto zeroSize = m_pEntry.Distance(m_pEntrySet.Get());
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(m_pEntry.Get(), m_MemZero.get(), zeroSize);

                // １つ前のエントリセットのオフセットをチェック
                const auto nextOffset = ValueProxy::GetInt64(m_pEntryData.Get());
                const auto headerOffset = pHeader->offset;
                EXPECT_EQ(nextOffset, headerOffset);

                m_pEntry.Advance(zeroSize);
            }

            std::memcpy(pHeader, m_pEntrySet.Get<NodeHeader>(), sizeof(NodeHeader));

            // エントリセットのヘッダチェック
            {
                const auto index = entryIndex / m_EntryCount;
                EXPECT_EQ(index, pHeader->index);

                const auto count = m_EntryCount <= (m_TotalEntryCount - entryIndex)
                                        ? m_EntryCount
                                        : (m_TotalEntryCount % m_EntryCount);
                EXPECT_EQ(count, pHeader->count);
            }

            m_pEntry.Advance(sizeof(NodeHeader));

            // オフセットが一致することを確認
            const auto nodeOffset = ValueProxy::GetInt64(pOffset.Get());
            const auto dataOffset = ValueProxy::GetInt64(m_pEntryData.Get());
            EXPECT_EQ(dataOffset, nodeOffset);
            const auto entryOffset = ValueProxy::GetInt64(m_pEntry.Get());
            EXPECT_EQ(dataOffset, entryOffset);
        }

        // 書き込んだデータと元データが一致する
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(m_pEntry.Get(), m_pEntryData.Get(), m_EntryUnitSize);

        m_pEntry.Advance(m_EntryUnitSize);
        m_pEntryData.Advance(m_EntryUnitSize);
    }

private:
    const size_t m_NodeUnitSize;
    const size_t m_EntryUnitSize;
    const int m_TotalEntryCount;
    const int m_EntryCount;
    const int m_OffsetCount;
    const int m_NodeL2Count;
    const int m_EntrySetCount;
    const int64_t m_EndOffset;
    Pointer m_pNodeL1;
    Pointer m_pEntry;
    Pointer m_pEntrySet;
    Pointer m_pEntryData;
    std::unique_ptr<char[]> m_MemZero;
};

/**
 * @brief   BucketTreeBuilder のテストを行うクラスです。
 */
template< typename T >
class TestCase : public BucketTreeData<T>
{
public:
    // コンストラクタです。
    TestCase(size_t nodeSize, uint32_t entryCount) NN_NOEXCEPT
        : BaseType(nodeSize, entryCount)
    {
    }

    // テストを実行します。
    void Run() NN_NOEXCEPT
    {
        // データチェック
        BucketTreeBuilderTest::DataChecker(
            this->m_NodeSize,
            this->m_EntrySize,
            this->m_EntryCount,
            this->m_EndOffset,
            this->m_NodeStorage.GetBuffer(),
            this->m_EntryStorage.GetBuffer(),
            this->m_Entry.get()
        ).Verify();
    }

private:
    typedef BucketTreeData<T> BaseType;
};

}

#if !defined(NN_SDK_BUILD_RELEASE)
/**
 * @brief   事前検証のデステストをします。
 */
TEST_F(BucketTreeBuilderDeathTest, Precondition)
{
    nn::fs::SubStorage storage;

    // pAllocator 不正
    EXPECT_DEATH_IF_SUPPORTED(
        BucketTreeBuilder().Initialize(
            nullptr,
            storage,
            storage,
            storage,
            BucketTree::NodeSizeMin,
            128,
            8
        ),
        ""
    );

    // entrySize の境界値外
    EXPECT_DEATH_IF_SUPPORTED(
        BucketTreeBuilder().Initialize(
            nullptr,
            storage,
            storage,
            storage,
            BucketTree::NodeSizeMin,
            sizeof(int64_t) - 1,
            8
        ),
        ""
    );
    EXPECT_DEATH_IF_SUPPORTED(
        BucketTreeBuilder().Initialize(
            nullptr,
            storage,
            storage,
            storage,
            BucketTree::NodeSizeMin,
            BucketTree::NodeSizeMin - sizeof(NodeHeader) + 1,
            8
        ),
        ""
    );

    // nodeSize の境界値外
    EXPECT_DEATH_IF_SUPPORTED(
        BucketTreeBuilder().Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            storage,
            storage,
            storage,
            BucketTree::NodeSizeMin - 1,
            128,
            8
        ),
        ""
    );
    EXPECT_DEATH_IF_SUPPORTED(
        BucketTreeBuilder().Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            storage,
            storage,
            storage,
            BucketTree::NodeSizeMax + 1,
            128,
            8
        ),
        ""
    );

    // nodeSize が２の累乗以外
    EXPECT_DEATH_IF_SUPPORTED(
        BucketTreeBuilder().Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            storage,
            storage,
            storage,
            BucketTree::NodeSizeMin + 1,
            128,
            8
        ),
        ""
    );

    // entryCount の境界値外
    EXPECT_DEATH_IF_SUPPORTED(
        BucketTreeBuilder().Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            storage,
            storage,
            storage,
            BucketTree::NodeSizeMin,
            128,
            -1
        ),
        ""
    );

    const auto headerSize = BucketTreeBuilder::QueryHeaderStorageSize();
    nnt::fs::util::SafeMemoryStorage headerStorage(headerSize);

    // endOffset の境界値外
    EXPECT_DEATH_IF_SUPPORTED(BucketTreeBuilder().Commit(-1), "");
    // 未初期化
    EXPECT_DEATH_IF_SUPPORTED(BucketTreeBuilder().Commit(0), "");

    // 再初期化に失敗
    {
        BucketTreeBuilder builder;
        NNT_EXPECT_RESULT_SUCCESS(
            builder.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                nn::fs::SubStorage(&headerStorage, 0, headerSize),
                storage, // NOTE: 今はアクセスしないのでこれでOK
                storage, //       同上
                BucketTree::NodeSizeMin,
                sizeof(int64_t),
                8
            )
        );

        EXPECT_DEATH_IF_SUPPORTED(
            builder.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                nn::fs::SubStorage(&headerStorage, 0, headerSize),
                storage,
                storage,
                BucketTree::NodeSizeMin,
                sizeof(int64_t),
                8
            ),
            ""
        );
    }

    // Write() のテストのために初期化
    BucketTreeBuilder builder;
    NNT_EXPECT_RESULT_SUCCESS(
        builder.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&headerStorage, 0, headerSize),
            storage, // NOTE: 今はアクセスしないのでこれでOK
            storage, //       同上
            BucketTree::NodeSizeMin,
            sizeof(int64_t),
            8
        )
    );

    struct JustSize
    {
        char data[sizeof(int64_t)];
    };
    JustSize just[10] = {};

    struct OverSize
    {
        char data[sizeof(int64_t) + 4];
    };
    OverSize over[1] = {};

    // エントリサイズ不正
    EXPECT_DEATH_IF_SUPPORTED(builder.Write(over[0]), "");
    EXPECT_DEATH_IF_SUPPORTED(builder.Write(over, 1), "");

    // pEntry 不正
    JustSize* const ptr = nullptr;
    EXPECT_DEATH_IF_SUPPORTED(builder.Write(ptr, 1), "");

    // entryCount 不正
    EXPECT_DEATH_IF_SUPPORTED(builder.Write(just, 10), "");
} // NOLINT(impl/function_size)
#endif

/**
 * @brief   事前検証のテストをします。
 */
TEST_F(BucketTreeBuilderTest, Precondition)
{
    const auto headerSize = BucketTreeBuilder::QueryHeaderStorageSize();

    nn::fs::SubStorage storage;
    nnt::fs::util::SafeMemoryStorage headerStorage(headerSize);
    nn::fs::SubStorage headerSubStorage(&headerStorage, 0, headerSize);

    // entrySize の境界値内
    NNT_EXPECT_RESULT_SUCCESS(
        BucketTreeBuilder().Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            headerSubStorage,
            storage, // NOTE: 今はアクセスしないのでこれでOK
            storage, //       同上
            BucketTree::NodeSizeMin,
            sizeof(int64_t),
            8
        )
    );
    NNT_EXPECT_RESULT_SUCCESS(
        BucketTreeBuilder().Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            headerSubStorage,
            storage,
            storage,
            BucketTree::NodeSizeMin,
            BucketTree::NodeSizeMin - sizeof(NodeHeader),
            8
        )
    );

    // nodeSize の境界値内
    NNT_EXPECT_RESULT_SUCCESS(
        BucketTreeBuilder().Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            headerSubStorage,
            storage,
            storage,
            BucketTree::NodeSizeMin,
            128,
            8
        )
    );
    NNT_EXPECT_RESULT_SUCCESS(
        BucketTreeBuilder().Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            headerSubStorage,
            storage,
            storage,
            BucketTree::NodeSizeMax,
            128,
            8
        )
    );

    // nodeSize が２の累乗
    for( size_t nodeSize = BucketTree::NodeSizeMin;
         nodeSize <= BucketTree::NodeSizeMax;
         nodeSize <<= 1 )
    {
        NNT_EXPECT_RESULT_SUCCESS(
            BucketTreeBuilder().Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                headerSubStorage,
                storage,
                storage,
                nodeSize,
                128,
                8
            )
        );
    }

    // entryCount の境界値内
    NNT_EXPECT_RESULT_SUCCESS(
        BucketTreeBuilder().Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            headerSubStorage,
            storage,
            storage,
            BucketTree::NodeSizeMin,
            128,
            0
        )
    );

    // Write() のテストのために初期化
    BucketTreeBuilder builder;
    NNT_EXPECT_RESULT_SUCCESS(
        builder.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            headerSubStorage,
            storage,
            storage,
            BucketTree::NodeSizeMin,
            sizeof(int64_t),
            8
        )
    );

    struct JustSize
    {
        char data[sizeof(int64_t)];
    };
    JustSize* const ptr = nullptr;

    // pEntry == nullptr && entryCount == 0
    NNT_EXPECT_RESULT_SUCCESS(builder.Write(ptr, 0));
} // NOLINT(impl/function_size)

/**
 * @brief   初期化・終了処理をテストします。
 */
TEST_F(BucketTreeBuilderTest, Initialize)
{
    const auto headerSize = BucketTreeBuilder::QueryHeaderStorageSize();

    nn::fs::SubStorage storage;
    BucketTreeBuilder builder;

    // 初期化前に呼び出しても問題ない
    builder.Finalize();

    // ストレージ不足で初期化に失敗することを確認
    {
        nnt::fs::util::SafeMemoryStorage headerStorage(headerSize - 4);
        nn::fs::SubStorage headerSubStorage(&headerStorage, 0, headerSize - 4);

        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultOutOfRange,
            builder.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                headerSubStorage,
                storage, // NOTE: 今はアクセスしないのでこれでOK
                storage, //       同上
                BucketTree::NodeSizeMin,
                16,
                1
            )
        );
    }

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::SafeMemoryStorage headerStorage(headerSize);
    nn::fs::SubStorage headerSubStorage(&headerStorage, 0, headerSize);

    // ランダムな引数でテスト
    for( size_t nodeSize = BucketTree::NodeSizeMin;
         nodeSize <= BucketTree::NodeSizeMax;
         nodeSize <<= 1 )
    {
        for( int i = 0; i < 500; ++i )
        {
            const auto entrySize = std::uniform_int_distribution<size_t>(16, 256)(mt);
            const auto entryCount = std::uniform_int_distribution<>(10, 10000)(mt);

            NNT_EXPECT_RESULT_SUCCESS(
                builder.Initialize(
                    nnt::fs::util::GetTestLibraryAllocator(),
                    headerSubStorage,
                    storage,
                    storage,
                    nodeSize,
                    entrySize,
                    entryCount
                )
            );

            builder.Finalize();
        }
    }

    // 再度呼び出しても問題ない
    builder.Finalize();
}

/**
 * @brief   エントリ書き出しで意図したリザルトが返ってくるかをテストします。
 */
TEST_F(BucketTreeBuilderTest, WriteResult)
{
    typedef IndirectStorageData StorageData;
    typedef std::unique_ptr<StorageData[]> DataArray;

    const auto headerSize = BucketTreeBuilder::QueryHeaderStorageSize();
    const auto nodeSize = BucketTree::NodeSizeMin;
    const auto entrySize = sizeof(IndirectStorageData);
    const int entryCount = 2;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    nnt::fs::util::SafeMemoryStorage smallStorage(4);
    nn::fs::SubStorage smallSubStorage(&smallStorage, 0, 4);

    nnt::fs::util::SafeMemoryStorage headerStorage(headerSize);
    nn::fs::SubStorage headerSubStorage(&headerStorage, 0, headerSize);

    const auto nodeStorageSize =
        BucketTreeBuilder::QueryNodeStorageSize(nodeSize, entrySize, entryCount);
    nnt::fs::util::SafeMemoryStorage nodeStorage(nodeStorageSize);
    nn::fs::SubStorage nodeSubStorage(&nodeStorage, 0, nodeStorageSize);

    const auto entryStorageSize =
        BucketTreeBuilder::QueryEntryStorageSize(nodeSize, entrySize, entryCount);
    nnt::fs::util::SafeMemoryStorage entryStorage(entryStorageSize);
    nn::fs::SubStorage entrySubStorage(&entryStorage, 0, entryStorageSize);

    DataArray entry = StorageData::Create(entryCount);
    const int64_t endOffset = entry[1].GetOffset() + BucketTreeEntryData::GetOffsetStride();

    // ノードストレージが小さくて書き出せない
    {
        BucketTreeBuilder builder;
        NNT_EXPECT_RESULT_SUCCESS(
            builder.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                headerSubStorage,
                smallSubStorage,
                entrySubStorage,
                nodeSize,
                entrySize,
                entryCount
            )
        );

        NNT_EXPECT_RESULT_SUCCESS(builder.Write(entry[0]));
        NNT_EXPECT_RESULT_SUCCESS(builder.Write(entry[1]));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, builder.Commit(endOffset));
    }

    // エントリストレージが小さくて書き出せない
    {
        BucketTreeBuilder builder;
        NNT_EXPECT_RESULT_SUCCESS(
            builder.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                headerSubStorage,
                nodeSubStorage,
                smallSubStorage,
                nodeSize,
                entrySize,
                entryCount
            )
        );

        NNT_EXPECT_RESULT_SUCCESS(builder.Write(entry[0]));
        NNT_EXPECT_RESULT_SUCCESS(builder.Write(entry[1]));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, builder.Commit(endOffset));
    }

    BucketTreeBuilder builder;
    NNT_EXPECT_RESULT_SUCCESS(
        builder.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            headerSubStorage,
            nodeSubStorage,
            entrySubStorage,
            nodeSize,
            entrySize,
            entryCount
        )
    );

    NNT_EXPECT_RESULT_SUCCESS(builder.Write(entry[0]));

    // オフセットが同じデータは書き出せない
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, builder.Write(entry[0]));

    // オフセットが同じデータはコミットできない
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, builder.Commit(entry[0].GetOffset()));

    // データが書き出し終わっていない
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, builder.Commit(entry[1].GetOffset()));

    // オフセットが小さいデータも書き出せない
    const StorageData dummy = {};
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, builder.Write(dummy));

    NNT_EXPECT_RESULT_SUCCESS(builder.Write(entry[1]));

    // 書き出すエントリが多すぎる
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, builder.Write(entry[1]));

    NNT_EXPECT_RESULT_SUCCESS(builder.Commit(endOffset));

    // 再コミットできない
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidOffset, builder.Commit(endOffset));
}

/**
 * @brief   エントリ書き出しで出力されたデータが意図した通りかをテストします。
 */
TEST_F(BucketTreeBuilderTest, VerifyWriteData)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 50; ++i )
    {
        TestCase<IndirectStorageData>(
            BucketTree::NodeSizeMin << std::uniform_int_distribution<>(0, 6)(mt),
            std::uniform_int_distribution<uint32_t>(1, 50000)(mt)
        ).Run();
    }

    for( int i = 0; i < 50; ++i )
    {
        TestCase<AesCtrCounterExtendedStorageData>(
            BucketTree::NodeSizeMin << std::uniform_int_distribution<>(0, 6)(mt),
            std::uniform_int_distribution<uint32_t>(1, 50000)(mt)
        ).Run();
    }

    for( int i = 0; i < 20; ++i )
    {
        TestCase<UnknownStorageData<32>>(
            BucketTree::NodeSizeMin << std::uniform_int_distribution<>(0, 6)(mt),
            std::uniform_int_distribution<uint32_t>(1, 50000)(mt)
        ).Run();
    }

    for( int i = 0; i < 20; ++i )
    {
        TestCase<UnknownStorageData<50>>(
            BucketTree::NodeSizeMin << std::uniform_int_distribution<>(0, 6)(mt),
            std::uniform_int_distribution<uint32_t>(1, 50000)(mt)
        ).Run();
    }

    for( int i = 0; i < 20; ++i )
    {
        TestCase<UnknownStorageData<63>>(
            BucketTree::NodeSizeMin << std::uniform_int_distribution<>(0, 6)(mt),
            std::uniform_int_distribution<uint32_t>(1, 50000)(mt)
        ).Run();
    }
}

/**
 * @brief   ノードに収まるエントリ数とオフセット数の境界チェック
 */
TEST_F(BucketTreeBuilderTest, EntryAndOffsetCountBoundary)
{
    const auto NodeSize = 1024;
    const auto EntrySize = sizeof(IndirectStorageData); // 20

    const auto entryCountPerNode = GetEntryCount(NodeSize, EntrySize);
    const auto offsetCountPerNode = GetOffsetCount(NodeSize);

    // ノードにエントリがちょうど収まる数
    {
        const auto entryCount = entryCountPerNode;

        const auto nodeStorageSize =
            BucketTreeBuilder::QueryNodeStorageSize(NodeSize, EntrySize, entryCount);
        EXPECT_EQ(NodeSize, nodeStorageSize);

        const auto entryStorageSize =
            BucketTreeBuilder::QueryEntryStorageSize(NodeSize, EntrySize, entryCount);
        EXPECT_EQ(NodeSize, entryStorageSize);

        TestCase<IndirectStorageData>(NodeSize, entryCount).Run();
    }

    // ノードからエントリがはみ出る数
    {
        const auto entryCount = entryCountPerNode + 1;

        const auto nodeStorageSize =
            BucketTreeBuilder::QueryNodeStorageSize(NodeSize, EntrySize, entryCount);
        EXPECT_EQ(NodeSize, nodeStorageSize);

        const auto entryStorageSize =
            BucketTreeBuilder::QueryEntryStorageSize(NodeSize, EntrySize, entryCount);
        EXPECT_EQ(NodeSize * 2, entryStorageSize);

        TestCase<IndirectStorageData>(NodeSize, entryCount).Run();
    }

    // オフセットが L1 ノードにちょうど収まる数
    {
        const auto entryCount = entryCountPerNode * offsetCountPerNode;

        const auto nodeStorageSize =
            BucketTreeBuilder::QueryNodeStorageSize(NodeSize, EntrySize, entryCount);
        EXPECT_EQ(NodeSize, nodeStorageSize);

        const auto entryStorageSize =
            BucketTreeBuilder::QueryEntryStorageSize(NodeSize, EntrySize, entryCount);
        EXPECT_EQ(NodeSize * offsetCountPerNode, entryStorageSize);

        TestCase<IndirectStorageData>(NodeSize, entryCount).Run();
    }

    // オフセットが L2 ノードを使用し始める数
    {
        const auto entryCount = entryCountPerNode * offsetCountPerNode + 1;

        const auto nodeStorageSize =
            BucketTreeBuilder::QueryNodeStorageSize(NodeSize, EntrySize, entryCount);
        EXPECT_EQ(NodeSize * 2, nodeStorageSize);

        const auto entryStorageSize =
            BucketTreeBuilder::QueryEntryStorageSize(NodeSize, EntrySize, entryCount);
        EXPECT_EQ(NodeSize * (offsetCountPerNode + 1), entryStorageSize);

        TestCase<IndirectStorageData>(NodeSize, entryCount).Run();
    }

    // オフセットが L2 ノードを使用しつつ、ノードにエントリがちょうど収まる数
    {
        const auto entryCount =
            entryCountPerNode * offsetCountPerNode + entryCountPerNode;

        const auto nodeStorageSize =
            BucketTreeBuilder::QueryNodeStorageSize(NodeSize, EntrySize, entryCount);
        EXPECT_EQ(NodeSize * 2, nodeStorageSize);

        const auto entryStorageSize =
            BucketTreeBuilder::QueryEntryStorageSize(NodeSize, EntrySize, entryCount);
        EXPECT_EQ(NodeSize * (offsetCountPerNode + 1), entryStorageSize);

        TestCase<IndirectStorageData>(NodeSize, entryCount).Run();
    }

    // オフセットが 1 つ目の L2 ノードを全て使用しつつ、最後のエントリノードが一部余る
    {
        const auto entryCount = entryCountPerNode * (offsetCountPerNode * 2 - 1) - 3;

        const auto nodeStorageSize =
            BucketTreeBuilder::QueryNodeStorageSize(NodeSize, EntrySize, entryCount);
        EXPECT_EQ(NodeSize * 2, nodeStorageSize);

        const auto entryStorageSize =
            BucketTreeBuilder::QueryEntryStorageSize(NodeSize, EntrySize, entryCount);
        EXPECT_EQ(NodeSize * (offsetCountPerNode * 2 - 1), entryStorageSize);

        TestCase<IndirectStorageData>(NodeSize, entryCount).Run();
    }

    // オフセットが 1 つ目の L2 ノードを全て使用しつつ、エントリノードはすべて埋まる
    {
        const auto entryCount = entryCountPerNode * (offsetCountPerNode * 2 - 1);

        const auto nodeStorageSize =
            BucketTreeBuilder::QueryNodeStorageSize(NodeSize, EntrySize, entryCount);
        EXPECT_EQ(NodeSize * 2, nodeStorageSize);

        const auto entryStorageSize =
            BucketTreeBuilder::QueryEntryStorageSize(NodeSize, EntrySize, entryCount);
        EXPECT_EQ(NodeSize * (offsetCountPerNode * 2 - 1), entryStorageSize);

        TestCase<IndirectStorageData>(NodeSize, entryCount).Run();
    }

    // エントリ数が上限いっぱい
    {
        const auto entryCount = entryCountPerNode * offsetCountPerNode * offsetCountPerNode;

        const auto nodeStorageSize =
            BucketTreeBuilder::QueryNodeStorageSize(NodeSize, EntrySize, entryCount);
        EXPECT_EQ(NodeSize * (1 + offsetCountPerNode), nodeStorageSize);

        const auto entryStorageSize =
            BucketTreeBuilder::QueryEntryStorageSize(NodeSize, EntrySize, entryCount);
        EXPECT_EQ(NodeSize * offsetCountPerNode * offsetCountPerNode, entryStorageSize);

        TestCase<IndirectStorageData>(NodeSize, entryCount).Run();
    }
} // NOLINT(impl/function_size)
