﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/detail/fs_IFileDataCache.h>
#include <nn/fs/fsa/fs_IFile.h>

#include <nn/os/os_Event.h>
#include <nn/os/os_Thread.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>

#include "shim/fs_PathBasedFileDataCache.h"

using namespace nn;
using namespace fs;

namespace
{
    os::TlsSlot g_tlsSlot;

    void TlsDestructFunction(uintptr_t value) NN_NOEXCEPT
    {
        NN_UNUSED(value);
    }

    struct ThreadContext
    {
        fs::detail::PathBasedFileDataCache* pCache;
        fsa::IFile* pFile;
        size_t offset;
        size_t unitSize;
        os::Event* pEvent;
    };

    class TestBaseFile : public fs::fsa::IFile, public fs::detail::Newable
    {
    public:
        TestBaseFile(void* pBuffer, size_t bufferSize) NN_NOEXCEPT
            : m_pBuffer(pBuffer)
            , m_BufferSize(bufferSize)
            , m_FileSize(static_cast<int64_t>(bufferSize))
        {
        }

        virtual Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(option);

            NN_ABORT_UNLESS(m_BufferSize == static_cast<size_t>(m_FileSize));

            int64_t restSize = std::max(static_cast<int64_t>(m_FileSize) - offset, static_cast<int64_t>(0));
            size_t readSize = std::min(size, static_cast<size_t>(restSize));
            if (restSize == 0)
            {
                *outValue = 0;
                NN_RESULT_SUCCESS;
            }

            std::memcpy(buffer, reinterpret_cast<char*>(m_pBuffer) + offset, readSize);

            *outValue = readSize;
            NN_RESULT_SUCCESS;
        }

        virtual Result DoWrite(int64_t offset, const void* buffer, size_t size, const fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(option);

            NN_ABORT_UNLESS(m_BufferSize == static_cast<size_t>(m_FileSize));

            if (size == 0)
            {
                NN_RESULT_SUCCESS;
            }

            std::memcpy(reinterpret_cast<char*>(m_pBuffer) + offset, buffer, size);

            NN_RESULT_SUCCESS;
        }

        virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_SUCCESS;
        }

        virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(size);
            NN_RESULT_SUCCESS;
        }

        virtual Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(outValue);
            NN_RESULT_SUCCESS;
        }

        virtual Result DoOperateRange(void* outBuffer, size_t outBufferSize, fs::OperationId operationId, int64_t offset, int64_t size, const void* inBuffer, size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(outBuffer);
            NN_UNUSED(outBufferSize);
            NN_UNUSED(operationId);
            NN_UNUSED(offset);
            NN_UNUSED(size);
            NN_UNUSED(inBuffer);
            NN_UNUSED(inBufferSize);
            NN_RESULT_SUCCESS;
        }

    private:
        void* m_pBuffer;
        size_t m_BufferSize;
        int64_t m_FileSize;

    };

    const size_t BufferSize = 128 * 1024 * 1024;

    class PathBasedFileDataCacheUnitTest : public ::testing::Test
    {
    public:
        PathBasedFileDataCacheUnitTest() NN_NOEXCEPT : m_FileBuffer(nnt::fs::util::AllocateBuffer(16)), m_CacheBuffer(nnt::fs::util::AllocateBuffer(16))
        {
        }

    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            m_FileBuffer = nnt::fs::util::AllocateBuffer(BufferSize);
            m_CacheBuffer = nnt::fs::util::AllocateBuffer(BufferSize);
            ASSERT_TRUE(m_FileBuffer != nullptr && m_CacheBuffer != nullptr);
            nnt::fs::util::FillBufferWith32BitCount(m_FileBuffer.get(), BufferSize, 0);
            nnt::fs::util::FillBufferWith32BitCount(m_CacheBuffer.get(), BufferSize, 0);

            m_pFile.reset(new TestBaseFile(m_FileBuffer.get(), BufferSize));

            std::unique_ptr<fs::detail::PathBasedFileDataCacheEntry> pEntry(new fs::detail::PathBasedFileDataCacheEntry());
            {
                std::memset(&(pEntry->hash), 0x0, fs::detail::FilePathHashSize);
                pEntry->hashIndex = 0;
                pEntry->pFileSystemAccessor = nullptr;
                auto path = fs::detail::MakeUnique<char[]>(16);
                std::strncpy(path.get(), "/test", 16);
                pEntry->path = std::move(path);
                pEntry->pathLength = 5;
                pEntry->pBuffer = m_CacheBuffer.get();
                pEntry->fileSize = BufferSize;
            }

            m_Cache.Add(std::move(pEntry));
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        fs::detail::PathBasedFileDataCache* GetCache() NN_NOEXCEPT
        {
            return &m_Cache;
        }

        TestBaseFile* GetFile() NN_NOEXCEPT
        {
            return m_pFile.get();
        }

    private:
        fs::detail::PathBasedFileDataCache m_Cache;
        std::unique_ptr<char, decltype(&nnt::fs::util::DeleterBuffer)> m_FileBuffer;
        std::unique_ptr<char, decltype(&nnt::fs::util::DeleterBuffer)> m_CacheBuffer;
        std::unique_ptr<TestBaseFile> m_pFile;
    };

    typedef PathBasedFileDataCacheUnitTest PathBasedFileDataCacheUnitTestRaceCondition;

    const int ThreadCount = 2;
    const int ThreadStackSize = 4 * 1024 * 1024;

    os::ThreadType g_Thread[ThreadCount];
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ThreadCount][ThreadStackSize];

    void CheckActualRead(void* arg)
    {
        auto threadContext = reinterpret_cast<ThreadContext*>(arg);
        auto readBuffer = nnt::fs::util::AllocateBuffer(threadContext->unitSize);
        int64_t offset = threadContext->offset;
        int cacheReadCount = 0;
        int actualReadCount = 0;
        while (!threadContext->pEvent->TryWait())
        {
            fs::detail::FileDataCacheAccessResult cacheAccessResult;
            size_t readSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(threadContext->pCache->Read(threadContext->pFile, OpenMode_Read, nullptr, {}, 0, &readSize, offset, readBuffer.get(), threadContext->unitSize, ReadOption::MakeValue(0), &cacheAccessResult));

            if (!cacheAccessResult.IsFileDataCacheUsed())
            {
                actualReadCount++;
            }
            else
            {
                cacheReadCount++;
            }

            offset += readSize;
            if (offset >= BufferSize)
            {
                offset = 0;
            }
        }
        // NN_LOG("cacheReadCount: %d, actualReadCount: %d\n", cacheReadCount, actualReadCount);
        EXPECT_GT(cacheReadCount, 0);
        EXPECT_GT(actualReadCount, 0);
        return;
    }

    void WriteAccess(void* arg)
    {
        auto threadContext = reinterpret_cast<ThreadContext*>(arg);
        auto writeBuffer = nnt::fs::util::AllocateBuffer(threadContext->unitSize);
        NNT_ASSERT_RESULT_SUCCESS(threadContext->pCache->Write(threadContext->pFile, OpenMode_Write, nullptr, {}, 0, threadContext->offset, writeBuffer.get(), threadContext->unitSize, WriteOption::MakeValue(fs::WriteOptionFlag::WriteOptionFlag_Flush)));
        threadContext->pEvent->Signal();
        return;
    }

    void ReadAccess(void* arg)
    {
        auto threadContext = reinterpret_cast<ThreadContext*>(arg);
        auto readBuffer = nnt::fs::util::AllocateBuffer(threadContext->unitSize);
        fs::detail::FileDataCacheAccessResult cacheAccessResult;
        size_t readSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(threadContext->pCache->Read(threadContext->pFile, OpenMode_Read, nullptr, {}, 0, &readSize, threadContext->offset, readBuffer.get(), threadContext->unitSize, ReadOption::MakeValue(0), &cacheAccessResult));
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount(readBuffer.get(), threadContext->unitSize, 0));
        EXPECT_TRUE(cacheAccessResult.IsFileDataCacheUsed());
        threadContext->pEvent->Signal();
        return;
    }

    void CheckWriteAccessDeferred(void* arg)
    {
        auto threadContext = reinterpret_cast<ThreadContext*>(arg);
        auto writeBuffer = nnt::fs::util::AllocateBuffer(threadContext->unitSize);
        NNT_ASSERT_RESULT_SUCCESS(threadContext->pCache->Write(threadContext->pFile, OpenMode_Write, nullptr, {}, 0, threadContext->offset, writeBuffer.get(), threadContext->unitSize, WriteOption::MakeValue(fs::WriteOptionFlag::WriteOptionFlag_Flush)));
        EXPECT_TRUE(threadContext->pEvent->TryWait());

        nnt::fs::util::FillBufferWith32BitCount(writeBuffer.get(), threadContext->unitSize, threadContext->offset);
        NNT_ASSERT_RESULT_SUCCESS(threadContext->pCache->Write(threadContext->pFile, OpenMode_Write, nullptr, {}, 0, threadContext->offset, writeBuffer.get(), threadContext->unitSize, WriteOption::MakeValue(fs::WriteOptionFlag::WriteOptionFlag_Flush)));
        return;
    }
}

TEST_F(PathBasedFileDataCacheUnitTestRaceCondition, ReadWhileWrite)
{
    // キャッシュ Write 中の Read アクセスがキャッシュを経由せずにできること
    // 0: Write, 1: Read

    os::Event writeEnd(os::EventClearMode_AutoClear);
    ThreadContext writeContext = { GetCache(), GetFile(), 0, BufferSize, &writeEnd };
    ThreadContext readContext = { GetCache(), GetFile(), 0, BufferSize / (1024 * 1024), &writeEnd };

    const int LoopCount = 16;

    for (int i = 0; i < LoopCount; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(os::CreateThread(&g_Thread[0], WriteAccess, &writeContext, g_ThreadStack[0], ThreadStackSize, os::DefaultThreadPriority, 0));
        NNT_ASSERT_RESULT_SUCCESS(os::CreateThread(&g_Thread[1], CheckActualRead, &readContext, g_ThreadStack[1], ThreadStackSize, os::DefaultThreadPriority, 1));

        os::StartThread(&g_Thread[1]);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        os::StartThread(&g_Thread[0]);

        os::DestroyThread(&g_Thread[0]);
        os::DestroyThread(&g_Thread[1]);
    }
}

TEST_F(PathBasedFileDataCacheUnitTestRaceCondition, WriteWhileRead)
{
    // キャッシュ Read 中のキャッシュへの Write アクセスが待たされること
    // 0: Read, 1: Write

    os::Event readEnd(os::EventClearMode_AutoClear);
    ThreadContext readContext = { GetCache(), GetFile(), 0, BufferSize, &readEnd };
    ThreadContext writeContext = { GetCache(), GetFile(), BufferSize / 2, BufferSize / 2 , &readEnd };

    const int LoopCount = 16;

    for (int i = 0; i < LoopCount; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(os::CreateThread(&g_Thread[0], ReadAccess, &readContext, g_ThreadStack[0], ThreadStackSize, os::DefaultThreadPriority, 0));
        NNT_ASSERT_RESULT_SUCCESS(os::CreateThread(&g_Thread[1], CheckWriteAccessDeferred, &writeContext, g_ThreadStack[1], ThreadStackSize, os::DefaultThreadPriority, 1));

        os::StartThread(&g_Thread[0]);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
        os::StartThread(&g_Thread[1]);

        os::DestroyThread(&g_Thread[0]);
        os::DestroyThread(&g_Thread[1]);
    }
}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    fs::SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();

    NN_ABORT_UNLESS_RESULT_SUCCESS(os::AllocateTlsSlot(&g_tlsSlot,TlsDestructFunction));

    auto ret = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
