﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <chrono>
#include <thread>
#include <cstring>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Windows.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/nnt_Argument.h>

using namespace nnt::fs::util;

namespace nn { namespace fs { namespace detail {

namespace
{

    void OpenWithFileShareDeleteForSeconds(const String& path, int seconds)
    {
        const int pathBufferLength = 1024;
        wchar_t pathW[pathBufferLength];
        mbstowcs_s(NULL, pathW, pathBufferLength, path.c_str(), pathBufferLength - 1);

        auto handle = ::CreateFile( pathW, GENERIC_READ, FILE_SHARE_DELETE, NULL, OPEN_EXISTING, FILE_FLAG_BACKUP_SEMANTICS, NULL);
        ASSERT_NE(INVALID_HANDLE_VALUE, handle);

        auto thread = std::thread([seconds, handle]{
            Sleep(seconds * 1000);
            CloseHandle(handle);
        });

        thread.detach();
    }

    class HostFileSystemTest : public ::testing::Test
    {
    protected:
        String m_RootPath;
        bool m_IsSetUpDone;
        HostFileSystemTest() : m_IsSetUpDone(false)
        {
        }

        virtual void SetUp()
        {
            const int RetryNumMax = 10;
            int retryNum = 0;

            MountHostRoot();
            String tempPath = String(nnt::fs::util::GetHostTemporaryPath());
            m_RootPath = tempPath + "/" + nnt::fs::util::CreateUniquePath("SIGLO-FS-HostFileSystem");
            auto result = CreateDirectory(m_RootPath.c_str());
            while (ResultPathAlreadyExists::Includes(result) && retryNum != RetryNumMax)
            {
                retryNum++;
                m_RootPath = tempPath + "/" + nnt::fs::util::CreateUniquePath("SIGLO-FS-HostFileSystem");
                result = CreateDirectory(m_RootPath.c_str());
            }
            ASSERT_TRUE(retryNum != RetryNumMax);
            m_IsSetUpDone = true;
        }
        virtual void TearDown()
        {
            if (m_IsSetUpDone)
            {
                NNT_EXPECT_RESULT_SUCCESS(DeleteDirectoryRecursively(m_RootPath.c_str()));
            }
            UnmountHostRoot();
        }
    };

    typedef HostFileSystemTest HostFileSystemDeathTest;
}

TEST_F(HostFileSystemTest, CreateDeleteSharedFile)
{
    auto filePath = m_RootPath + "\\file";
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(filePath.c_str(), 1024));

    OpenWithFileShareDeleteForSeconds(filePath, 2);

    NNT_EXPECT_RESULT_SUCCESS(DeleteFile(filePath.c_str()));
    NNT_EXPECT_RESULT_FAILURE(fs::ResultPathNotFound, DeleteFile(filePath.c_str()));
}

TEST_F(HostFileSystemTest, CreateDeleteSharedDirectory)
{
    auto directoryPath = m_RootPath + "\\dir";
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));

    OpenWithFileShareDeleteForSeconds(directoryPath, 2);

    NNT_EXPECT_RESULT_SUCCESS(DeleteDirectory(directoryPath.c_str()));
    NNT_EXPECT_RESULT_FAILURE(fs::ResultPathNotFound, DeleteDirectory(directoryPath.c_str()));
}


TEST_F(HostFileSystemTest, CreateDeleteSharedDirectoryRecursively)
{
    auto directoryPath = m_RootPath + "\\dir";
    auto filePath      = m_RootPath + "\\dir\\file";
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(directoryPath.c_str()));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(filePath.c_str(), 1024));

    OpenWithFileShareDeleteForSeconds(filePath, 2);

    NNT_EXPECT_RESULT_SUCCESS(DeleteDirectoryRecursively(directoryPath.c_str()));
    NNT_EXPECT_RESULT_FAILURE(fs::ResultPathNotFound, DeleteDirectory(directoryPath.c_str()));
    NNT_EXPECT_RESULT_FAILURE(fs::ResultPathNotFound, DeleteFile(filePath.c_str()));
}

TEST_F(HostFileSystemTest, QueryRange)
{
    const auto filePath = m_RootPath + "\\file";
    const auto fileSize = 1024;
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(filePath.c_str(), fileSize));

    nn::fs::FileHandle file;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&file, filePath.c_str(), nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        CloseFile(file);
        NNT_ASSERT_RESULT_SUCCESS(DeleteFile(filePath.c_str()));
    };

    nn::fs::QueryRangeInfo info;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryRange(&info, file, 0, fileSize));
    EXPECT_EQ(0, info.aesCtrKeyTypeFlag);
    EXPECT_EQ(0, info.speedEmulationTypeFlag);
}

TEST_F(HostFileSystemDeathTest, QueryRange)
{
    const auto filePath = m_RootPath + "\\file";
    const auto fileSize = 1024;
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(filePath.c_str(), fileSize));

    nn::fs::FileHandle file;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&file, filePath.c_str(), nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        CloseFile(file);
        NNT_ASSERT_RESULT_SUCCESS(DeleteFile(filePath.c_str()));
    };

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultNullptrArgument,
        nn::fs::QueryRange(nullptr, file, 0, fileSize));
}

TEST_F(HostFileSystemTest, ReadWriteLargeHeavy)
{
    const auto filePath = m_RootPath + "\\file";

    static const size_t AccessSize = 1024;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(
        filePath.c_str(),
        nnt::fs::util::LargeOffsetMax + static_cast<int64_t>(AccessSize)));
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteFile(filePath.c_str()));
    };

    {
        nn::fs::FileHandle file;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(
            &file,
            filePath.c_str(),
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        NNT_FS_ASSERT_NO_FATAL_FAILURE(nnt::fs::util::TestFileHandleAccessWithLargeOffset(file, AccessSize));
    }
    {
        NNT_FS_ASSERT_NO_FATAL_FAILURE(nnt::fs::util::TestFileHandleAccessNotAllowAppend(AccessSize,
            [&filePath](std::function<void(nn::fs::FileHandle)> fileAccessor) NN_NOEXCEPT
            {
                nn::fs::FileHandle file;
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(
                    &file,
                    filePath.c_str(),
                    static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)));
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseFile(file);
                };
                NNT_FS_ASSERT_NO_FATAL_FAILURE(fileAccessor(file));
            }
        ));
    }
    {
        nn::fs::FileHandle file;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(
            &file,
            filePath.c_str(),
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend)));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        NNT_FS_ASSERT_NO_FATAL_FAILURE(nnt::fs::util::TestFileHandleAccessAllowAppend(file, AccessSize));
    }
}

}}}


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    ::testing::InitGoogleTest(&argc, argv);
    nn::fs::SetEnabledAutoAbort(false);
    int result = RUN_ALL_TESTS();

    nnt::Exit(result);
}
