﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_BitTypes.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>

#include <nn/fssystem/dbm/fs_FileSystemTemplate.h>
#include <nn/fssystem/dbm/fs_FileSystemTemplate.impl.h>

#include <nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.h>
#include <nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.impl.h>

#include <nn/fssystem/dbm/fs_DirectoryObjectTemplate.h>
#include <nn/fssystem/dbm/fs_DirectoryObjectTemplate.impl.h>
#include <nn/fssystem/dbm/fs_FileObjectTemplate.h>
#include <nn/fssystem/dbm/fs_FileObjectTemplate.impl.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

namespace nnt { namespace fs {
    nn::fssystem::IBufferManager* GetBufferManager() NN_NOEXCEPT;
}}

//! ファイルシステム
typedef nn::fssystem::dbm::FileSystemObjectTemplate<
            nn::fssystem::dbm::DirectoryName,
            nn::fssystem::dbm::FileName
        > SimpleFileSystemObject;

//! ディレクトリオブジェクト
typedef nn::fssystem::dbm::DirectoryObjectTemplate<
            nn::fssystem::dbm::DirectoryName,
            nn::fssystem::dbm::FileName
        > SimpleDirectoryObject;

//! ファイルオブジェクト
typedef nn::fssystem::dbm::FileObjectTemplate<
            nn::fssystem::dbm::DirectoryName,
            nn::fssystem::dbm::FileName
        > SimpleFileObject;

//! ファイルシステムとそれに使う管理領域、アロケーションテーブル、ストレージを保持するクラスです。
class TestStorage
{
public:
    //! コンストラクタです。
    explicit TestStorage(
                 uint32_t countAllocationBlocks,
                 uint32_t allocationTablePadding,
                 uint32_t sizeBlock
             ) NN_NOEXCEPT
    : m_StorageControlArea(sizeof(nn::fssystem::dbm::FileSystemControlArea::ControlArea)),
      m_StorageData(
          SimpleFileSystemObject::QueryAllocationTableStorageSize(
              countAllocationBlocks + allocationTablePadding
          )
          + sizeBlock * (countAllocationBlocks + allocationTablePadding)
      )
    {
        nn::fs::SubStorage storageControlArea(
            &m_StorageControlArea, 0, sizeof(nn::fssystem::dbm::FileSystemControlArea::ControlArea)
        );
        nn::fs::SubStorage storageData(
            &m_StorageData, 0, sizeof(nn::fssystem::dbm::FileSystemControlArea::ControlArea)
        );

        int64_t offset = 0;

        // 管理領域の初期化用パラメーターを計算します。
        // アロケーションテーブルの開始オフセット
        const int64_t offsetAllocation = offset;
        const int64_t sizeAllocation = SimpleFileSystemObject::QueryAllocationTableStorageSize(
                                           countAllocationBlocks + allocationTablePadding
                                       );
        offset += sizeAllocation;
        nn::fs::SubStorage storage(&m_StorageData, offsetAllocation, sizeAllocation);

        // 実データ領域の開始オフセット
        const int64_t offsetBody = offset;
        const int64_t sizeBody = sizeBlock * countAllocationBlocks;
        offset += sizeBody;
        nn::fs::SubStorage bodyStorage(&m_StorageData, offsetBody, sizeBody);

        nn::Result result;

        // 管理領域をマウントします。
        m_ControlArea.Initialize(storageControlArea);

        // 管理領域内の各データを初期化します。
        // (コンストラクタでは NNT_ASSERT_RESULT_SUCCESS は使えません)
        result = m_ControlArea.WriteBlockSize(sizeBlock);
        NN_ASSERT(result.IsSuccess());
        result = m_ControlArea.WriteAllocationTableInfo(offsetAllocation, countAllocationBlocks);
        NN_ASSERT(result.IsSuccess());
        result = m_ControlArea.WriteDataBodyInfo(offsetBody, countAllocationBlocks);
        NN_ASSERT(result.IsSuccess());

        // アローケーションテーブル(可変長 DB)を生成します。
        result = m_AllocationTable.Format(storage, countAllocationBlocks);
        NN_ASSERT(result.IsSuccess());
        m_AllocationTable.Initialize(storage, countAllocationBlocks);

        // ディレクトリエントリー用領域を可変長 DB 上に確保します。
        uint32_t indexDirectoryEntry;
        {
            const uint32_t sizeDirectoryEntry
                = SimpleFileSystemObject::QueryDirectoryEntryStorageSize(0);
            const uint32_t blockCountDirectoryEntry
                = (sizeDirectoryEntry + sizeBlock - 1) / sizeBlock;
            result = m_AllocationTable.Allocate(&indexDirectoryEntry, blockCountDirectoryEntry);
            NN_ASSERT(result.IsSuccess());
            // 割り当てた領域の先頭インデックスを管理領域に設定
            result = m_ControlArea.WriteDirectoryEntryInfo(indexDirectoryEntry);
            NN_ASSERT(result.IsSuccess());
        }

        // ファイルエントリー用領域を可変長 DB 上に確保します。
        uint32_t indexFileEntry;
        {
            const uint32_t sizeFileEntry = SimpleFileSystemObject::QueryFileEntryStorageSize(0);
            const uint32_t blockCountFileEntry = (sizeFileEntry + sizeBlock - 1) / sizeBlock;
            result = m_AllocationTable.Allocate(&indexFileEntry, blockCountFileEntry);
            NN_ASSERT(result.IsSuccess());
            // 割り当てた領域の先頭インデックスを管理領域に設定
            result = m_ControlArea.WriteFileEntryInfo(indexFileEntry);
            NN_ASSERT(result.IsSuccess());
            m_AllocationTable.Finalize();
        }

        // 管理領域に書き込んだ値を使用して各オブジェクトをフォーマットします。
        m_AllocationTable.Initialize(storage, countAllocationBlocks);
        result = m_StorageDirectoryEntry.InitializeBuffered(
                     &m_AllocationTable,
                     indexDirectoryEntry,
                     sizeBlock,
                     bodyStorage,
                     nnt::fs::GetBufferManager()
                 );
        NN_ASSERT(result.IsSuccess());
        result = m_StorageFileEntry.InitializeBuffered(
                     &m_AllocationTable,
                     indexFileEntry,
                     sizeBlock,
                     bodyStorage,
                     nnt::fs::GetBufferManager()
                 );
        NN_ASSERT(result.IsSuccess());
        result = m_FileSystem.Format(
                    &m_StorageDirectoryEntry,
                    &m_StorageFileEntry
                 );
        NN_ASSERT(result.IsSuccess());
        m_StorageFileEntry.Finalize();
        m_StorageDirectoryEntry.Finalize();
        m_AllocationTable.Finalize();

        // ファイルシステムをマウントします。
        Mount();
    }

    //! デストラクタです。
    ~TestStorage() NN_NOEXCEPT
    {
        // 管理領域をアンマウントします。
        m_ControlArea.Finalize();
    }

    //! ファイルシステムをマウントします。
    void Mount() NN_NOEXCEPT
    {
        // 管理領域から各データを取得します。
        int64_t sizeBlock;
        NNT_ASSERT_RESULT_SUCCESS(m_ControlArea.ReadBlockSize(&sizeBlock));

        int64_t offsetAllocation;
        uint32_t countAllocationBlocks;
        NNT_ASSERT_RESULT_SUCCESS(
            m_ControlArea.ReadAllocationTableInfo(&offsetAllocation, &countAllocationBlocks)
        );

        int64_t offsetBody;
        uint32_t countAllocationBlocksBody;
        NNT_ASSERT_RESULT_SUCCESS(
            m_ControlArea.ReadDataBodyInfo(&offsetBody, &countAllocationBlocksBody)
        );

        uint32_t indexDirectoryEntry;
        NNT_ASSERT_RESULT_SUCCESS(m_ControlArea.ReadDirectoryEntryInfo(&indexDirectoryEntry));

        uint32_t indexFileEntry;
        NNT_ASSERT_RESULT_SUCCESS(m_ControlArea.ReadFileEntryInfo(&indexFileEntry));

        // 実データ領域のサイズを取得します。
        const int64_t sizeBody = sizeBlock * countAllocationBlocksBody;

        nn::fs::SubStorage storage(&m_StorageData, offsetAllocation, offsetBody - offsetAllocation);
        nn::fs::SubStorage storageBody(&m_StorageData, offsetBody, sizeBody);

        // ファイルシステムをマウントします。
        m_AllocationTable.Initialize(storage, countAllocationBlocks);
        NNT_ASSERT_RESULT_SUCCESS(
            m_StorageDirectoryEntry.InitializeBuffered(
                &m_AllocationTable,
                indexDirectoryEntry,
                static_cast<uint32_t>(sizeBlock),
                storageBody,
                nnt::fs::GetBufferManager()
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(
            m_StorageFileEntry.InitializeBuffered(
                &m_AllocationTable,
                indexFileEntry,
                static_cast<uint32_t>(sizeBlock),
                storageBody,
                nnt::fs::GetBufferManager()
            )
        );
        m_FileSystem.Initialize(
            &m_AllocationTable,
            static_cast<uint32_t>(sizeBlock),
            &m_StorageDirectoryEntry,
            &m_StorageFileEntry
        );
    }

    //! アンマウントします。
    void Unmount() NN_NOEXCEPT
    {
        m_StorageControlArea.CheckValid();
        m_StorageData.CheckValid();

        m_FileSystem.Finalize();
        m_StorageFileEntry.Finalize();
        m_StorageDirectoryEntry.Finalize();
        m_AllocationTable.Finalize();
    }

public:
    //! 管理領域
    nn::fssystem::dbm::FileSystemControlArea& GetControlArea() NN_NOEXCEPT
    {
        return m_ControlArea;
    }

    //! ファイルシステム
    SimpleFileSystemObject& GetFileSystem() NN_NOEXCEPT
    {
        return m_FileSystem;
    }

    //! 実データ用ストレージ
    nnt::fs::util::SafeMemoryStorage& GetStorageData() NN_NOEXCEPT
    {
        return m_StorageData;
    }

private:
    //! 管理領域
    nn::fssystem::dbm::FileSystemControlArea m_ControlArea;
    //! ファイルシステム
    SimpleFileSystemObject m_FileSystem;
    //! ファイルアロケーションテーブル
    nn::fssystem::dbm::AllocationTable m_AllocationTable;
    //! 管理領域用ストレージ
    nnt::fs::util::SafeMemoryStorage m_StorageControlArea;
    //! 実データ用ストレージ
    nnt::fs::util::SafeMemoryStorage m_StorageData;
    //! ディレクトリエントリ用ストレージ
    nn::fssystem::dbm::BufferedAllocationTableStorage m_StorageDirectoryEntry;
    //! ファイルエントリ用ストレージ
    nn::fssystem::dbm::BufferedAllocationTableStorage m_StorageFileEntry;
};

//! ディレクトリを開こうとして失敗するテストをします。
TEST(FileSystemTest, TestOpenDirectoryFailure)
{
    TestStorage testStorage(32, 32, 512);
    SimpleFileSystemObject& fileSystem = testStorage.GetFileSystem();

    SimpleDirectoryObject directory;

    // ルートディレクトリはデフォルトで生成されている必要があります。
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenDirectory(&directory, "/"));

    // 存在しないディレクトリを指定します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotFound,
        fileSystem.OpenDirectory(&directory, "/dir")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotFound,
        fileSystem.OpenDirectory(&directory, "/dir1/")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotFound,
        fileSystem.OpenDirectory(&directory, "/dir1/dir2")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotFound,
        fileSystem.OpenDirectory(&directory, "/dir1/dir2/dir3/")
    );

    // 長いエントリー名です。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        fileSystem.OpenDirectory(&directory, "/longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglongd")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        fileSystem.OpenDirectory(&directory, "/dir1/longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglongd/dir2")
    );

    testStorage.Unmount();
}

//! ファイルを開こうとして失敗するテストをします。
TEST(FileSystemTest, TestOpenFileFailure)
{
    TestStorage testStorage(32, 32, 512);
    SimpleFileSystemObject& fileSystem = testStorage.GetFileSystem();

    SimpleFileObject file;

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotFound,
        fileSystem.OpenFile(&file, "/dir1/file1")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryNotFound,
        fileSystem.OpenFile(&file, "/dir1/dir2/file1")
    );

    // 存在しないファイルを指定します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultFileNotFound,
        fileSystem.OpenFile(&file, "/file")
    );

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        fileSystem.OpenFile(&file, "/longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglongd/file")
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        fileSystem.OpenFile(
            &file,
            "/longlonglonglonglonglonglonglonglonglonglonglonglonglonglonglongd/longlonglonglonglonglonglonglonglonglonglonglonglonglonglongfile"
        )
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        fileSystem.OpenFile(&file, "/longlonglonglonglonglonglonglonglonglonglonglonglonglonglongfilen")
    );

    testStorage.Unmount();
}

//! ファイルサイズを変更してイテレーションするテストをします。
TEST(FileSystemTest, TestIterateResizedFile)
{
    TestStorage testStorage(32, 32, 512);
    SimpleFileSystemObject& fileSystem = testStorage.GetFileSystem();

    SimpleFileObject file;

    // ファイルを作成します。
    nn::fssystem::dbm::FileOptionalInfo fileOption;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/hoge.txt", fileOption));

    // ファイサイズを初期化してイテレーションを行います。
    {
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/hoge.txt"));

        // ファイルサイズを変更して、サイズが反映されているかをテストします。
        NNT_ASSERT_RESULT_SUCCESS(file.Resize(100));
        ASSERT_EQ(100, file.GetSize());

        uint32_t index;
        uint32_t nextIndex;
        bool isFinished;

        // 最初のブロックを取得し、イテレーションが続けられることをテストします。
        NNT_ASSERT_RESULT_SUCCESS(file.IterateBegin(&index, &isFinished, 0));
        ASSERT_FALSE(isFinished);

        // ファイルサイズは 1 ブロックに収まるサイズなので最初のブロックでイテレーションは
        // 終了します。
        NNT_ASSERT_RESULT_SUCCESS(file.IterateNext(&nextIndex, &isFinished, index));
        ASSERT_TRUE(isFinished);

        // 戻る方向のイテレーションを行います。
        NNT_ASSERT_RESULT_SUCCESS(file.IterateBegin(&index, &isFinished, 0));
        ASSERT_FALSE(isFinished);

        // 戻るイテレーションは最初のブロックで終了します。
        NNT_ASSERT_RESULT_SUCCESS(file.IteratePrevious(&nextIndex, &isFinished, index));
        ASSERT_TRUE(isFinished);

        // 再度開き、ファイルサイズをテストします。
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/hoge.txt"));
        ASSERT_EQ(100, file.GetSize());
    }

    // ファイサイズを拡大してイテレーションを行います。
    {
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/hoge.txt"));

        // 2 ブロック必要なサイズに拡張します。
        NNT_ASSERT_RESULT_SUCCESS(file.Resize(513));
        ASSERT_EQ(513, file.GetSize());

        uint32_t index;
        uint32_t nextIndex;
        uint32_t nextIndex2;
        bool isFinished;

        // 2 ブロック分だけイテレーションを進められることをテストします。
        NNT_ASSERT_RESULT_SUCCESS(file.IterateBegin(&index, &isFinished, 0));
        ASSERT_FALSE(isFinished);
        NNT_ASSERT_RESULT_SUCCESS(file.IterateNext(&nextIndex, &isFinished, index));
        ASSERT_FALSE(isFinished);
        NNT_ASSERT_RESULT_SUCCESS(file.IterateNext(&nextIndex2, &isFinished, nextIndex));
        ASSERT_TRUE(isFinished);

        // 2 ブロック目まで進み、最初のブロックまで戻ります。
        NNT_ASSERT_RESULT_SUCCESS(file.IterateBegin(&index, &isFinished, 0));
        ASSERT_FALSE(isFinished);
        NNT_ASSERT_RESULT_SUCCESS(file.IterateNext(&nextIndex, &isFinished, index));
        ASSERT_FALSE(isFinished);
        uint32_t previousIndex;
        NNT_ASSERT_RESULT_SUCCESS(file.IteratePrevious(&previousIndex, &isFinished, nextIndex));
        ASSERT_FALSE(isFinished);

        // 最初のブロックに戻ったことをテストします。
        ASSERT_EQ(index, previousIndex);

        // 戻るイテレーションは最初のブロックで終了します。
        uint32_t previousIndex2;
        NNT_ASSERT_RESULT_SUCCESS(
            file.IteratePrevious(&previousIndex2, &isFinished, previousIndex)
        );
        ASSERT_TRUE(isFinished);

        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/hoge.txt"));
        ASSERT_EQ(513, file.GetSize());
    }

    // ファイサイズを縮小してイテレーションを行います。
    {
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/hoge.txt"));

        // 拡張前のファイルサイズに縮小します。
        NNT_ASSERT_RESULT_SUCCESS(file.Resize(100));
        ASSERT_EQ(100, file.GetSize());

        uint32_t index;
        uint32_t nextIndex;
        bool isFinished;

        // 拡張前と同様のイテレーション結果になることをテストします。
        NNT_ASSERT_RESULT_SUCCESS(file.IterateBegin(&index, &isFinished, 0));
        ASSERT_FALSE(isFinished);
        NNT_ASSERT_RESULT_SUCCESS(file.IterateNext(&nextIndex, &isFinished, index));
        ASSERT_TRUE(isFinished);

        NNT_ASSERT_RESULT_SUCCESS(file.IterateBegin(&index, &isFinished, 0));
        ASSERT_FALSE(isFinished);
        NNT_ASSERT_RESULT_SUCCESS(file.IteratePrevious(&nextIndex, &isFinished, index));
        ASSERT_TRUE(isFinished);

        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/hoge.txt"));
        ASSERT_EQ(100, file.GetSize());
    }

    testStorage.Unmount();
}

//! ファイルの作成、削除をテストをします。
TEST(FileSystemTest, TestCreateAndDeleteFiles)
{
    TestStorage testStorage(32, 32, 512);
    SimpleFileSystemObject& fileSystem = testStorage.GetFileSystem();

    // 複数のファイル作成が成功することをテストします。
    nn::fssystem::dbm::FileOptionalInfo fileOption;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/hoge.txt", fileOption));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/foo.txt", fileOption));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/foo2.txt", fileOption));

    // ファイルの削除が成功することをテストします。
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteFile("/hoge.txt"));

    testStorage.Unmount();
}

//! ファイルをランダムにイテレーションするテストをします。
TEST(FileSystemTest, TestIterateFileRandom)
{
    static const uint32_t SizeBlock = 512;
    TestStorage testStorage(32, 32, SizeBlock);
    SimpleFileSystemObject& fileSystem = testStorage.GetFileSystem();

    // あらかじめファイルを作ります。
    nn::fssystem::dbm::FileOptionalInfo fileOption;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/foo.txt", fileOption));

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    // 適当なサイズのファイルをイテレーションします。
    for( int i = 0; i < 10000; ++i )
    {
        SimpleFileObject file;
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/foo.txt"));

        // ランダムにファイルサイズを変更します。
        const uint32_t size = std::uniform_int_distribution<uint32_t>(0, 8191)(mt);
        NNT_ASSERT_RESULT_SUCCESS(file.Resize(size));
        ASSERT_EQ(size, file.GetSize());

        int count = 0;

        // イテレーションをランダムな開始位置で行います。
        const int offset = (size == 0) ?
            0 : std::uniform_int_distribution<int>(0, (size + 511) / 512 - 1)(mt);
        uint32_t index;
        bool isFinished;
        NNT_ASSERT_RESULT_SUCCESS(file.IterateBegin(&index, &isFinished, offset));
        if( size == 0 )
        {
            ASSERT_TRUE(isFinished);
        }
        else
        {
            // イテレーションを行い、ブロック数をカウントします。
            ASSERT_FALSE(isFinished);
            while( NN_STATIC_CONDITION(true) )
            {
                ++count;
                uint32_t nextIndex;
                NNT_ASSERT_RESULT_SUCCESS(file.IterateNext(&nextIndex, &isFinished, index));
                if( isFinished )
                {
                    break;
                }
                index = nextIndex;
            }
        }

        // ファイルサイズ、ブロックサイズからファイルに必要なブロック数を計算し、
        // イテレーション開始オフセットを考慮して実際のカウント数と一致するかテストします。
        ASSERT_EQ(((size + SizeBlock - 1) / SizeBlock) - offset, count);
    }

    testStorage.Unmount();
}

//! ファイルをランダムにサイズ変更するテストをします。
TEST(FileSystemTest, TestResizeFileRandom)
{
    TestStorage testStorage(32, 32, 512);
    SimpleFileSystemObject& fileSystem = testStorage.GetFileSystem();

    // あらかじめファイルを作ります。
    nn::fssystem::dbm::FileOptionalInfo fileOption;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/foo.txt", fileOption));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/foo2.txt", fileOption));

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    // リサイズを繰り返し行います。
    for( int i = 0; i < 10000; ++i )
    {
        // ランダムなサイズでリサイズを行います。
        const uint64_t size = std::uniform_int_distribution<uint64_t>(0, 9999)(mt);
        SimpleFileObject file;
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/foo.txt"));
        NNT_ASSERT_RESULT_SUCCESS(file.Resize(size));
        ASSERT_EQ(size, file.GetSize());

        // 別のファイルのリサイズを行います。
        {
            SimpleFileObject file2;
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file2, "/foo2.txt"));
            const uint32_t size2 = std::uniform_int_distribution<uint32_t>(0, 1023)(mt);
            NNT_ASSERT_RESULT_SUCCESS(file2.Resize(size2));
            ASSERT_EQ(size2, file2.GetSize());
        }

        // 再度ファイルを開いてファイルサイズをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/foo.txt"));
        ASSERT_EQ(size, file.GetSize());
    }

    testStorage.Unmount();
}

//! ディレクトリに含まれるディレクトリを列挙するテストをします。
TEST(FileSystemTest, TestFindDirectory)
{
    TestStorage testStorage(32, 32, 512);
    SimpleFileSystemObject& fileSystem = testStorage.GetFileSystem();

    // ディレクトリを列挙します。

    // ルートディレクトリを開きます。
    SimpleDirectoryObject directory;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenDirectory(&directory, "/"));

    // ディレクトリの作成に成功することをテストします。
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory("/dir1"));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory("/dir2"));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory("/dir3"));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory("/dir1/dir4"));

    // ルートディレクトリから子のファイル、ディレクトリをイテレーションします。
    NNT_ASSERT_RESULT_SUCCESS(directory.FindOpen());

    // 取得したディレクトリ名が作成した子ディレクトリのもので、
    // イテレーションが続くことをテストします。
    nn::fssystem::dbm::DirectoryName directoryName;
    bool isFinished = true;
    NNT_ASSERT_RESULT_SUCCESS(directory.FindNextDirectory(&directoryName, &isFinished));
    ASSERT_EQ(
        0,
        std::memcmp(
            &directoryName,
            "dir3\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",
            sizeof(directoryName)
        )
    );
    ASSERT_FALSE(isFinished);
    isFinished = true;
    NNT_ASSERT_RESULT_SUCCESS(directory.FindNextDirectory(&directoryName, &isFinished));
    ASSERT_EQ(
        0,
        std::memcmp(
            &directoryName,
            "dir2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",
            sizeof(directoryName)
        )
    );
    ASSERT_FALSE(isFinished);
    isFinished = true;
    NNT_ASSERT_RESULT_SUCCESS(directory.FindNextDirectory(&directoryName, &isFinished));
    ASSERT_EQ(
        0,
        std::memcmp(
            &directoryName,
            "dir1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",
            sizeof(directoryName)
        )
    );
    ASSERT_FALSE(isFinished);
    isFinished = false;

    // 作成した子ディレクトリを全て取得してイテレーションが終わることをテストします。
    NNT_ASSERT_RESULT_SUCCESS(directory.FindNextDirectory(&directoryName, &isFinished));
    ASSERT_TRUE(isFinished);

    testStorage.Unmount();
}

//! ディレクトリに含まれるファイルを列挙するテストをします。
TEST(FileSystemTest, TestFindFile)
{
    TestStorage testStorage(32, 32, 512);
    SimpleFileSystemObject& fileSystem = testStorage.GetFileSystem();

    // ファイルを列挙します。

    // あらかじめディレクトリを作ります。
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory("/dir1"));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory("/dir2"));

    // ファイルの作成に成功することをテストします。
    nn::fssystem::dbm::FileOptionalInfo fileOption;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/dir1/file1", fileOption));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/dir1/file2", fileOption));

    // 開いているディレクトリが変わってもファイル作成に成功することをテストします。
    SimpleDirectoryObject directory;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenDirectory(&directory, "/dir1"));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/dir1/file3", fileOption));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/dir2/file4", fileOption));    // dir2 に作成

    // "/dir1" のディレクトリから子のファイル、ディレクトリをイテレーションします。
    NNT_ASSERT_RESULT_SUCCESS(directory.FindOpen());

    // 取得したファイル名が作成した子ファイルのもので、イテレーションが続くことをテストします。
    nn::fssystem::dbm::FileName fileName;
    int64_t size;
    bool isFinished = true;
    NNT_ASSERT_RESULT_SUCCESS(directory.FindNextFile(&size, &fileName, &isFinished));
    ASSERT_EQ(
        0,
        std::memcmp(
            &fileName,
            "file3\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",
            sizeof(fileName)
        )
    );
    ASSERT_EQ(0, size);
    ASSERT_FALSE(isFinished);
    isFinished = true;
    NNT_ASSERT_RESULT_SUCCESS(directory.FindNextFile(&size, &fileName, &isFinished));
    ASSERT_EQ(
        0,
        std::memcmp(
            &fileName,
            "file2\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",
            sizeof(fileName)
        )
    );
    ASSERT_FALSE(isFinished);
    isFinished = true;
    NNT_ASSERT_RESULT_SUCCESS(directory.FindNextFile(&size, &fileName, &isFinished));
    ASSERT_EQ(
        0,
        std::memcmp(
            &fileName,
            "file1\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",
            sizeof(fileName)
        )
    );
    ASSERT_FALSE(isFinished);
    isFinished = true;

    // 作成した子ファイルを全て取得してイテレーションが終わることをテストします。
    NNT_ASSERT_RESULT_SUCCESS(directory.FindNextFile(&size, &fileName, &isFinished));
    ASSERT_TRUE(isFinished);

    testStorage.Unmount();
}

//! ファイルを限界までリサイズする境界値テストを行います。
TEST(FileSystemTest, TestResizeMax)
{
    // FileSystemObject を作成します。
    const uint32_t countBlocks = 32;
    const uint32_t allocationTablePadding = 0;
    const uint32_t blockSize = 512;
    TestStorage testStorage(countBlocks, allocationTablePadding, blockSize);
    SimpleFileSystemObject& fileSystem = testStorage.GetFileSystem();

    // ファイルを作成します。
    nn::fssystem::dbm::FileOptionalInfo fileOption;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/file", fileOption));

    //! ファイルサイズの限界（4G x ブロックサイズ）までリサイズします。
    const int64_t maxSize = blockSize * (static_cast<int64_t>(1) << 30) * 4 - 1;
    SimpleFileObject file;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/file"));
    NNT_ASSERT_RESULT_SUCCESS(file.Resize(maxSize));

    // 限界を超えてリサイズしようとすると失敗します。
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultOutOfResource, file.Resize(maxSize + 1));

    testStorage.Unmount();
}

//! ディレクトリをエントリー限界まで作るテストを行います。
TEST(FileSystemTest, TestCreateDirectoryForLimit)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    static const uint32_t MaxBlockCount = 312;

    static const uint32_t CountEntryDirectory = 50;
    static const uint32_t BlockSize = 512;

    // ディレクトリをエントリーの限界まで作り続けるテストです。
    // テーブルに割り当てるブロック数
    static const uint32_t CountBlocks = 11;

    // 作成できるディレクトリ数の半分
    static const uint32_t CountBlocksDirectoryHalf = CountEntryDirectory / 2;

    // 作成できるディレクトリ数の半分に必要なサイズ
    const uint32_t sizeStorageHalf = SimpleFileSystemObject::QueryDirectoryEntryStorageSize(
                                         CountBlocksDirectoryHalf
                                     );

    // 作成できるディレクトリ数の半分に必要なブロック数
    const uint32_t countBlocksDirectoryEntry = (sizeStorageHalf + BlockSize - 1) / BlockSize;

    // ブロック数が 11 なら、ディレクトリが 50 個作れます。
    // 詳細:
    //   開始時点でディレクトリエントリー用ストレージには 1 ブロック割り当てられていて、
    //   空きブロックが 9 個あるため、最大で 10 ブロック = 5120 バイト割り当て可能です。
    //   （残りの 1 ブロックはファイルエントリー用ストレージに割り当てられています。）
    //   ディレクトリエントリー SimpleFileSystemObject::DirectoryElement
    //   のサイズは 96 バイトなので、ディレクトリエントリーは 5120 / 96 = 53 個作成可能です。
    //   ただしシステムの予約領域で 2 エントリー使用しており、
    //   (SimpleFileSystemObject::DirectoryEntryList::AllocationTableStorageReservedCount)
    //   ルートディレクトリは既に作成されているため、ユーザーが作成できるディレクトリ数は
    //   50 となります。
    TestStorage testStorage(CountBlocks, 0, BlockSize);
    SimpleFileSystemObject& fileSystem = testStorage.GetFileSystem();

    uint32_t indexBlock = 0;
    nnt::fs::util::String strFullPath = "/";
    // 限界の半分までディレクトリを作成
    for( ; indexBlock < CountBlocksDirectoryHalf; ++indexBlock )
    {
        // ランダムな名前を作成します。
        char directoryName[nn::fssystem::dbm::MaxDirectoryNameLength + 1];
        std::sprintf(
            directoryName, "%08X", std::uniform_int_distribution<uint32_t>(0, 0xFFFFFFFF)(mt)
        );
        strFullPath += directoryName;
        strFullPath += "/";

        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory(strFullPath.c_str()));
        SimpleDirectoryObject directory;
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenDirectory(&directory, strFullPath.c_str()));
    }

    // ファイルシステムの空きブロック数をチェックします。
    uint32_t freeBlockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CalcFreeListLength(&freeBlockCount));

    // ディレクトリエントリーに割り当てられたブロックを途中まで使用しているのと
    // カレントディレクトリが作成されているため、空きブロック数は計算値より
    // 1 少なくなります。
    ASSERT_EQ(CountBlocks - countBlocksDirectoryEntry - 1, freeBlockCount);

    for( ; indexBlock < MaxBlockCount; ++indexBlock )
    {
        // ランダムな名前を作成します。
        char directoryName[nn::fssystem::dbm::MaxDirectoryNameLength + 1];
        std::sprintf(
            directoryName, "%08X", std::uniform_int_distribution<uint32_t>(0, 0xFFFFFFFF)(mt)
        );
        strFullPath += directoryName;
        strFullPath += "/";

        if( indexBlock < CountEntryDirectory )
        {
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory(strFullPath.c_str()));
            SimpleDirectoryObject directory;
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenDirectory(&directory, strFullPath.c_str()));
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAllocationTableFull,
                fileSystem.CreateDirectory(strFullPath.c_str())
            );
            break;
        }
    }

    // エントリーいっぱいまで作りました。
    // これ以上作ろうとしても空き不足になります。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        fileSystem.CreateDirectory("/dir")
    );
}

//! ファイルをエントリー限界まで作るテストを行います。
TEST(FileSystemTest, TestCreateFileForLimit)
{
    static const uint32_t MaxBlockCount = 312;

    static const uint32_t CountEntryFile = 14;
    static const uint32_t BlockSize = 512;

    // テーブルに割り当てるブロック数
    static const uint32_t CountBlocks = 4;

    // 作成できるファイル数の半分
    static const uint32_t CountBlocksFileHalf = CountEntryFile / 2;

    // 作成できるファイル数の半分に必要なサイズ
    const uint32_t sizeFileEntry
        = SimpleFileSystemObject::QueryFileEntryStorageSize(CountBlocksFileHalf);

    // 作成できるファイル数の半分に必要なブロック数
    const uint32_t countBlocksFileEntry = (sizeFileEntry + BlockSize - 1) / BlockSize;

    // ブロック数が 4 なら、ディレクトリが 14 個作れます。
    // 詳細:
    //   開始時点でファイルエントリー用ストレージには 1 ブロック割り当てられていて、
    //   空きブロックが 2 個あるため、最大で 3 ブロック = 1536 バイト割り当て可能です。
    //   （残りの 1 ブロックはディレクトリエントリー用ストレージに割り当てられています。）
    //   ファイルエントリー SimpleFileSystemObject::FileElement の
    //   サイズは 96 バイトなのでファイルエントリーは 1536 / 96 = 16 個作成可能です。
    //   ただしシステムの予約領域で 2 エントリー使用しているため、
    //   (SimpleFileSystemObject::FileEntryList::AllocationTableStorageReservedCount)
    //   実際に作成できるファイル数は 14 個となります。
    TestStorage testStorage(CountBlocks, 0, BlockSize);
    SimpleFileSystemObject& fileSystem = testStorage.GetFileSystem();

    nnt::fs::util::String directoryRoot = "/";
    nnt::fs::util::String strFullPath;
    // 限界の半分までファイル作成
    uint32_t indexCreateFile = 0;
    for( ; indexCreateFile < CountBlocksFileHalf; ++indexCreateFile )
    {
        // ファイル名は後でアクセスするために連番にします。
        char fileName[nn::fssystem::dbm::MaxFileNameLength + 1];
        std::sprintf(fileName, "%08X%04X", indexCreateFile, indexCreateFile);
        strFullPath = directoryRoot;
        strFullPath += fileName;

        // ファイル情報は後で内容をチェックするために連番にします。
        nn::fssystem::dbm::FileOptionalInfo fileOption;
        std::memset(&fileOption, indexCreateFile & 0xFF, sizeof(fileOption));

        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(strFullPath.c_str(), fileOption));
        SimpleFileObject file;
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, strFullPath.c_str()));
    }

    // ファイルシステムの空きブロック数をチェックします。
    uint32_t freeBlockCount = 0;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CalcFreeListLength(&freeBlockCount));
    // ファイルエントリーに割り当てられたブロックを途中まで使用しているため
    // 空きブロック数は計算値より 1 少なくなります。
    ASSERT_EQ(CountBlocks - countBlocksFileEntry - 1, freeBlockCount);

    for( ; indexCreateFile < MaxBlockCount; ++indexCreateFile )
    {
        // ファイル名は後でアクセスするために連番にします。
        char fileName[nn::fssystem::dbm::MaxFileNameLength + 1];
        std::sprintf(fileName, "%08X%04X", indexCreateFile, indexCreateFile);
        strFullPath = directoryRoot;
        strFullPath += fileName;

        // ファイル情報は後で内容をチェックするために連番にします。
        nn::fssystem::dbm::FileOptionalInfo fileOption;
        std::memset(&fileOption, indexCreateFile & 0xFF, sizeof(fileOption));

        if( indexCreateFile < CountEntryFile )
        {
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(strFullPath.c_str(), fileOption));
            SimpleFileObject file;
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, strFullPath.c_str()));
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAllocationTableFull,
                fileSystem.CreateFile(strFullPath.c_str(), fileOption)
            );
            break;
        }
    }

    // エントリーいっぱいまで作りました。
    // これ以上作ろうとしても空き不足になります。
    {
        nn::fssystem::dbm::FileOptionalInfo fileOptionTemporary;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultAllocationTableFull,
            fileSystem.CreateFile("/file", fileOptionTemporary)
        );
    }

    // 付加情報の内容をチェックします。
    for( uint32_t indexFileCheck = 0; indexFileCheck < MaxBlockCount; ++indexFileCheck )
    {
        // 連番のファイル名でアクセスします。
        char fileName[nn::fssystem::dbm::MaxFileNameLength + 1];
        std::sprintf(fileName, "%08X%04X", indexFileCheck, indexFileCheck);
        strFullPath = directoryRoot;
        strFullPath += fileName;

        SimpleFileObject file;
        if( indexFileCheck < CountEntryFile )
        {
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, strFullPath.c_str()));
        }
        else
        {
            break;
        }

        nn::fssystem::dbm::FileOptionalInfo fileOption;
        file.GetOptionalInfo(&fileOption);

        // ファイル情報が連番で埋められていることをテストします。
        for( int i = 0; i < sizeof(nn::fssystem::dbm::FileOptionalInfo); ++i )
        {
            ASSERT_EQ((indexFileCheck & 0xFF), fileOption.info[i]);
        }
    }
}

//! パラメタを変えつつファイルシステムのマウントとアンマウントを繰り返します。
TEST(FileSystemTest, TestMountAndUnmount)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    // Init + Mount
    // パラメタをいろいろ変えつつマウントとアンマウントを繰り返します。
    for( int loop = 0; loop < 1000; loop += 50 )
    {
        const uint32_t blockSize = 1 << std::uniform_int_distribution<int>(7, 8)(mt);
        const uint32_t countBlocks = std::uniform_int_distribution<uint32_t>(10, 99)(mt);

        // ブロック数からディレクトリとファイルの個数を決定する
        // テストの内容の関係上、ディレクトリを 2 個以上、ファイルを 1 個以上作れるように
        // ディレクトリに与えられるブロック数は 5 以上、ファイルに与えられるブロック数は 3 以上
        const uint32_t countDirectoryBlock
            = std::uniform_int_distribution<uint32_t>(5, countBlocks - 4)(mt);
        // 作成するディレクトリの数
        uint32_t countEntryDirectory;
        {
            const uint32_t maxEntrySize = blockSize * countDirectoryBlock;
            // 1 エントリーのサイズ は 96
            const uint32_t singleEntrySize = sizeof(SimpleFileSystemObject::DirectoryElement);
            NN_STATIC_ASSERT(96 == singleEntrySize);

            // 予約エントリー数
            // SimpleFileSystemObject::DirectoryEntryList::AllocationTableStorageReservedCount: 2
            // ルートディレクトリのエントリー: 1
            static const auto CountAllocationTableStorageReserved = 2;
            static const auto CountRootDirectory = 1;

            countEntryDirectory
                = maxEntrySize / singleEntrySize
                - CountAllocationTableStorageReserved - CountRootDirectory;
        }

        const uint32_t countFileBlock = countBlocks - countDirectoryBlock;
        // 作成するファイルの数
        uint32_t countEntryFile;
        {
            const uint32_t maxEntrySize = blockSize * countFileBlock;
            // 1 エントリーのサイズ は 96
            const uint32_t singleEntrySize = sizeof(SimpleFileSystemObject::FileElement);
            NN_STATIC_ASSERT(96 == singleEntrySize);

            // 予約エントリー数
            // SimpleFileSystemObject::FileEntryList::AllocationTableStorageReservedCount: 2
            static const auto AllocationTableReservedCount = 2;

            countEntryFile = maxEntrySize / singleEntrySize - AllocationTableReservedCount;
        }

        TestStorage testStorage(countBlocks, 0, blockSize);
        SimpleFileSystemObject& fileSystem = testStorage.GetFileSystem();

        uint32_t curCountBlocks = 0;
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CalcFreeListLength(&curCountBlocks));

        // ファイルとディレクトリで空きブロックを全て使用することをテストします。
        uint32_t countBlocksDirectoryEntry;
        uint32_t countBlocksFileEntry;
        {
            // ディレクトリのテーブルに使用するサイズ
            const uint32_t sizeDirectoryEntryStorage
                = SimpleFileSystemObject::QueryDirectoryEntryStorageSize(countEntryDirectory);
            // ディレクトリのテーブルに使用するブロック数
            countBlocksDirectoryEntry = (sizeDirectoryEntryStorage + blockSize - 1) / blockSize;
            // ファイルのテーブルに使用するサイズ
            const uint32_t sizeFileEntryStorage
                = SimpleFileSystemObject::QueryFileEntryStorageSize(countEntryFile);
            // ファイルのテーブルに使用するブロック数
            countBlocksFileEntry = (sizeFileEntryStorage + blockSize - 1) / blockSize;
            ASSERT_EQ(countBlocks - countBlocksDirectoryEntry - countBlocksFileEntry, 0);
        }

        uint32_t counts = 0;

        // ルートディレクトリを開き、子のディレクトリやファイルが無いことをテストします。
        SimpleDirectoryObject directoryObj;
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenDirectory(&directoryObj, "/"));
        NNT_ASSERT_RESULT_SUCCESS(directoryObj.CountDirectory(&counts));
        ASSERT_EQ(0, counts);
        NNT_ASSERT_RESULT_SUCCESS(directoryObj.CountFile(&counts));
        ASSERT_EQ(0, counts);

        // 子ディレクトリを作成して子ディレクトリの数が増えることをテストします。
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory("/dir1"));
        NNT_ASSERT_RESULT_SUCCESS(directoryObj.CountDirectory(&counts));
        ASSERT_EQ(1, counts);

        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory("/dir2"));
        NNT_ASSERT_RESULT_SUCCESS(directoryObj.CountDirectory(&counts));
        ASSERT_EQ(2, counts);

        // ファイルの作成、ファイル名の変更をテストします。
        nn::fssystem::dbm::FileOptionalInfo foi;
        foi.info[0] = static_cast<nn::Bit8>(loop & 0xFF);
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/dir1/file1", foi));
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.RenameFile("/dir1/file2", "/dir1/file1"));
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.RenameFile("/dir2/file2", "/dir1/file2"));

        // ディレクトリとファイル作成のために拡張されたのを確認
        // ディレクトリとファイル作成に使ったブロック分だけ空きが減っていることをテストします。
        {
            // ディレクトリ作成に使ったブロック数を計算します。
            NNT_ASSERT_RESULT_SUCCESS(directoryObj.CountDirectory(&counts));
            const uint32_t sizeDirectoryEntryStorage
                = SimpleFileSystemObject::QueryDirectoryEntryStorageSize(counts);
            countBlocksDirectoryEntry = (sizeDirectoryEntryStorage + blockSize - 1) / blockSize;

            // ファイル作成に使ったブロック数を計算します。
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenDirectory(&directoryObj, "/dir2"));
            NNT_ASSERT_RESULT_SUCCESS(directoryObj.CountFile(&counts));
            const uint32_t sizeFileEntryStorage
                = SimpleFileSystemObject::QueryFileEntryStorageSize(counts);
            countBlocksFileEntry = (sizeFileEntryStorage + blockSize - 1) / blockSize;

            // 空きブロック数を確認します。
            // 出力用引数が書き換わることをテストするために一度クリアします。
            curCountBlocks = 0;
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.CalcFreeListLength(&curCountBlocks));
            const uint32_t countUseBlocks = countBlocksDirectoryEntry + countBlocksFileEntry;
            ASSERT_EQ(countBlocks - countUseBlocks, curCountBlocks);
        }

        // ファイルとディレクトリを削除します。
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteFile("/dir2/file2"));
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteDirectory("/dir2"));
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteDirectory("/dir1"));

        // 削除しても拡張されたままであることを確認
        curCountBlocks = 0;
        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CalcFreeListLength(&curCountBlocks));
        ASSERT_EQ(countBlocks - countBlocksDirectoryEntry - countBlocksFileEntry, curCountBlocks);

        {
            // ルートディレクトリを開きます。
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenDirectory(&directoryObj, "/"));

            // ディレクトリの個数が 0 であることを確認します。
            NNT_ASSERT_RESULT_SUCCESS(directoryObj.CountDirectory(&counts));
            ASSERT_EQ(0, counts);

            // ファイルの個数が 0 であることを確認します。
            NNT_ASSERT_RESULT_SUCCESS(directoryObj.CountFile(&counts));
            ASSERT_EQ(0, counts);

            // 最大値まで要素を確保します。
            // ディレクトリとファイルの両方で限界まで使う
            nn::fssystem::dbm::PathChar path[16];
            for( int indexDirectory = 0;
                    indexDirectory < static_cast<int>(countEntryDirectory);
                    ++indexDirectory
                )
            {
                std::sprintf(path, "/dir%d.d", indexDirectory);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory(path));
            }
            for( int indexFile = 0; indexFile < static_cast<int>(countEntryFile); ++indexFile )
            {
                std::sprintf(path, "/file%d.dat", indexFile);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile(path, foi));
            }

            // ディレクトリもファイルも作成できない
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAllocationTableFull,
                fileSystem.CreateDirectory("/dirdmy")
            );
            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultAllocationTableFull,
                fileSystem.CreateFile("/filedmy", foi)
            );

            // 作成したファイル、ディレクトリを削除します。
            for( int indexDirectory = 0;
                    indexDirectory < static_cast<int>(countEntryDirectory);
                    ++indexDirectory
                )
            {
                std::sprintf(path, "/dir%d.d", indexDirectory);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteDirectory(path));
            }
            for( int indexFile = 0; indexFile < static_cast<int>(countEntryFile); ++indexFile )
            {
                std::sprintf(path, "/file%d.dat", indexFile);
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteFile(path));
            }

            NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenDirectory(&directoryObj, "/"));

            // ディレクトリの個数が 0 であることを確認します。
            NNT_ASSERT_RESULT_SUCCESS(directoryObj.CountDirectory(&counts));
            ASSERT_EQ(0, counts);

            // ファイルの個数が 0 であることを確認します。
            NNT_ASSERT_RESULT_SUCCESS(directoryObj.CountFile(&counts));
            ASSERT_EQ(0, counts);

            // 限界までファイルを広げていきます
            // 拡張対象のファイルを作成して開きます。
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/.test.arc.", foi));
            SimpleFileObject file;
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/.test.arc."));
            {
                // ファイルシステムの空きブロック数をチェックします。
                // ファイル、ディレクトリの作成可能個数だけ空きブロックがあることをテストします。
                const uint32_t sizeDirectoryEntryStorage
                    = SimpleFileSystemObject::QueryDirectoryEntryStorageSize(countEntryDirectory);
                countBlocksDirectoryEntry = (sizeDirectoryEntryStorage + blockSize - 1) / blockSize;
                const uint32_t sizeFileEntryStorage
                    = SimpleFileSystemObject::QueryFileEntryStorageSize(countEntryFile);
                countBlocksFileEntry = (sizeFileEntryStorage + blockSize - 1) / blockSize;
                uint32_t freeBlock = 0;
                NNT_ASSERT_RESULT_SUCCESS(fileSystem.CalcFreeListLength(&freeBlock));
                ASSERT_EQ(
                    countBlocks - countBlocksDirectoryEntry - countBlocksFileEntry, freeBlock
                );

                const uint32_t maxSize = freeBlock * blockSize;
                int64_t sizeCurrent = 0;
                for( int j = 0; ; ++j )
                {
                    // 拡張サイズをランダムに決定します。
                    const int64_t sizeNew
                        = sizeCurrent
                        + std::uniform_int_distribution<int64_t>(0, blockSize * 2 - 1)(mt);
                    const int64_t oldBlockCount = (sizeCurrent + blockSize - 1) / blockSize;
                    const int64_t newBlockCount = (sizeNew     + blockSize - 1) / blockSize;

                    if( sizeNew > maxSize )
                    {
                        // データいっぱいまで取り続けたときは
                        // ResultAllocationTableFullで終わります
                        NNT_ASSERT_RESULT_FAILURE(
                            nn::fs::ResultAllocationTableFull,
                            file.Resize(sizeNew)
                        );

                        // 拡張したサイズだけ空きブロック数が減っていることをテストします。
                        uint32_t curFreeBlock = 0;
                        NNT_ASSERT_RESULT_SUCCESS(fileSystem.CalcFreeListLength(&curFreeBlock));
                        ASSERT_EQ((maxSize - sizeCurrent) / blockSize, curFreeBlock);
                        break;
                    }
                    NNT_ASSERT_RESULT_SUCCESS(file.Resize(sizeNew));
                    sizeCurrent = sizeNew;

                    // 空き領域の減少量をチェックします。
                    uint32_t curFreeBlock = 0;
                    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CalcFreeListLength(&curFreeBlock));
                    ASSERT_TRUE(freeBlock - curFreeBlock == newBlockCount - oldBlockCount);
                    freeBlock = curFreeBlock;
                }
            }
            NNT_ASSERT_RESULT_SUCCESS(fileSystem.DeleteFile("/.test.arc."));
        }

        testStorage.Unmount();
    }
} // NOLINT(impl/function_size)

//! ファイルサイズを利用しないファイルシステム用の拡張領域への読み書きをテストします。
TEST(FileSystemTest, TestSessionId)
{
    TestStorage testStorage(32, 32, 512);
    SimpleFileSystemObject& fileSystem = testStorage.GetFileSystem();

    // サイズを持たないファイルへのアロケーションテーブルインデックス取得、変更をテストします。
    // サイズを持たないファイルを作成します。
    nn::fssystem::dbm::FileOptionalInfo fileOption;
    std::memset(&fileOption, 0, sizeof(fileOption));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/000.txt", fileOption));
    SimpleFileObject file;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/000.txt"));

    // ファイルのアロケーションテーブルのインデックスの取得に成功することをテストします。
    uint32_t index;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.GetAllocationTableIndex(&index, "/000.txt"));
    ASSERT_EQ(0x80000000, index);

    // ファイルのアロケーションテーブルのインデックスの変更に成功することをテストします。
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.SetAllocationTableIndex(0x1234, "/000.txt"));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.GetAllocationTableIndex(&index, "/000.txt"));
    ASSERT_EQ(0x1234, index);

    // 別のファイルへのアロケーションテーブルインデックス取得、変更をテストします。
    std::memset(&fileOption, 0, sizeof(fileOption));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/999.txt", fileOption));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/999.txt"));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.GetAllocationTableIndex(&index, "/999.txt"));
    ASSERT_EQ(0x80000000, index);
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.SetAllocationTableIndex(0x9876, "/999.txt"));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.GetAllocationTableIndex(&index, "/999.txt"));
    ASSERT_EQ(0x9876, index);

    // 最初に作成したファイルのインデックスが変化していないことをテストします。
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/000.txt"));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.GetAllocationTableIndex(&index, "/000.txt"));
    ASSERT_EQ(0x1234, index);

    int64_t id;
    // サイズを持たないファイルへの識別 ID の取得、変更をテストします。
    // サイズを持たないファイルを作成します。
    std::memset(&fileOption, 0, sizeof(fileOption));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/AAA.txt", fileOption));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/AAA.txt"));

    // ファイルの識別 ID の取得に成功することをテストします。
    // 出力用引数が書き換わることをテストするために一度クリアします。
    id = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.GetSessionId(&id, "/AAA.txt"));
    ASSERT_EQ(0, id);

    // ファイルの識別 ID の変更に成功することをテストします。
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.SetSessionId(0x1234, "/AAA.txt"));
    id = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.GetSessionId(&id, "/AAA.txt"));
    ASSERT_EQ(0x1234, id);

    // 別のファイルへの識別 ID の取得、変更をテストします。
    std::memset(&fileOption, 0, sizeof(fileOption));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/BBB.txt", fileOption));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/BBB.txt"));
    id = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.GetSessionId(&id, "/BBB.txt"));
    ASSERT_EQ(0, id);
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.SetSessionId(0x9876, "/BBB.txt"));
    id = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.GetSessionId(&id, "/BBB.txt"));
    ASSERT_EQ(0x9876, id);

    // また別のファイルへの識別 ID の取得、変更をテストします。
    std::memset(&fileOption, 0, sizeof(fileOption));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/CCC.txt", fileOption));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/CCC.txt"));
    id = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.GetSessionId(&id, "/CCC.txt"));
    ASSERT_EQ(0, id);
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.SetSessionId(12345678901234LL, "/CCC.txt"));
    id = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.GetSessionId(&id, "/CCC.txt"));
    ASSERT_EQ(12345678901234LL, id);

    // 先に変更したファイルの識別 ID が変化していないことをテストします。
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.OpenFile(&file, "/AAA.txt"));
    id = 1;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.GetSessionId(&id, "/AAA.txt"));
    ASSERT_EQ(0x1234, id);
}

//! ファイルシステムをストレージ拡張しディレクトリを限界まで作成します。
TEST(FileSystemTest, TestExpandForCreateDirectory)
{
    // 256 バイトブロックが 4 個あれば、ディレクトリを 7 個作って 64 バイト余ります。
    // ディレクトリエントリー 10 個分のサイズの内 3 個がシステムで使用されます。
    // 256 * 4 = 96 * (7 + 3) + 64
    // ファイルエントリーのために 1 ブロック使われるので、4+1 ブロックを用意しておきます。
    TestStorage testStorage(4 + 1, 5, 256);
    static const int DirectoryCount = 7;
    SimpleDirectoryObject directory;
    char path[8];
    for( int i = 0; i < DirectoryCount; ++i )
    {
        std::sprintf(path, "/dir%02d/", i);
        NNT_ASSERT_RESULT_SUCCESS(testStorage.GetFileSystem().CreateDirectory(path));
        NNT_ASSERT_RESULT_SUCCESS(testStorage.GetFileSystem().OpenDirectory(&directory, path));
    }
    // すでに 7 個作ったので作成できません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        testStorage.GetFileSystem().CreateDirectory("/miss/")
    );

    // アンマウントして拡張します。
    nn::fs::SubStorage storage(
        &testStorage.GetStorageData(), 0, testStorage.GetStorageData().GetSize()
    );
    testStorage.Unmount();
    nn::fssystem::dbm::AllocationTable::Expand(storage, 4 + 1, 8 + 1);
    testStorage.GetControlArea().ExpandAllocationTableInfo(8 + 1);
    testStorage.Mount();

    // 256 バイトブロックが 8 個あれば ディレクトリを 18 個作って 32 バイト余ります。
    static const int NewDirectoryCount = 18;
    for( int i = DirectoryCount; i < NewDirectoryCount; ++i )
    {
        std::sprintf(path, "/dir%02d/", i);
        NNT_ASSERT_RESULT_SUCCESS(testStorage.GetFileSystem().CreateDirectory(path));
        NNT_ASSERT_RESULT_SUCCESS(testStorage.GetFileSystem().OpenDirectory(&directory, path));
    }
    // すでに 21 個作ったので作成できません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        testStorage.GetFileSystem().CreateDirectory("/miss/")
    );

    testStorage.Unmount();
}

//! ファイルシステムをストレージ拡張しファイルを限界まで作成します。
TEST(FileSystemTest, TestExpandForCreateFile)
{
    // 256 バイトブロックが 4 個あれば、ファイルを 8 個作って 64 バイト余ります。
    // ファイルエントリー 10 個分のサイズの内 2 個がシステムで使用されます。
    // 256 * 4 = 96 * (8 + 2) + 64
    // ディレクトリエントリーのために 2 ブロック使われるので、4+2 ブロックを用意しておきます。
    static const int FileCount = 8;
    TestStorage testStorage(4 + 2, 5, 256);
    SimpleFileObject file;
    char path[8];
    const nn::fssystem::dbm::FileOptionalInfo info = {};
    for( int i = 0; i < FileCount; ++i )
    {
        std::sprintf(path, "/file%02d", i);
        NNT_ASSERT_RESULT_SUCCESS(testStorage.GetFileSystem().CreateFile(path, info));
        NNT_ASSERT_RESULT_SUCCESS(testStorage.GetFileSystem().OpenFile(&file, path));
    }
    // すでに 8 個作ったので作成できない
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        testStorage.GetFileSystem().CreateFile("/miss", info)
    );

    // アンマウントして拡張する
    nn::fs::SubStorage storage(
        &testStorage.GetStorageData(), 0, testStorage.GetStorageData().GetSize()
    );
    testStorage.Unmount();
    nn::fssystem::dbm::AllocationTable::Expand(storage, 4 + 2, 8 + 2);
    testStorage.GetControlArea().ExpandAllocationTableInfo(8 + 2);
    testStorage.Mount();

    // 256 バイトブロックが 8 個あればファイルを 19 個作って 32 バイト余ります。
    static const int NewFileCount = 19;
    for( int i = FileCount; i < NewFileCount; ++i )
    {
        std::sprintf(path, "/file%02d", i);
        NNT_ASSERT_RESULT_SUCCESS(testStorage.GetFileSystem().CreateFile(path, info));
        NNT_ASSERT_RESULT_SUCCESS(testStorage.GetFileSystem().OpenFile(&file, path));
    }
    // すでに 22 個作ったので作成できない
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        testStorage.GetFileSystem().CreateFile("/miss", info)
    );

    testStorage.Unmount();
}

//! ディレクトリ名を既に存在するものに変更しようとする失敗をテストします。
TEST(FileSystemTest, TestRenameDirectoryAlreadyExist)
{
    // FileSystemObject を作成します。
    const uint32_t CountBlocks = 32;
    const uint32_t AllocationTablePadding = 0;
    const uint32_t BlockSize = 512;
    TestStorage testStorage(CountBlocks, AllocationTablePadding, BlockSize);
    SimpleFileSystemObject& fileSystem = testStorage.GetFileSystem();

    // ファイルとディレクトリを作成します。
    nn::fssystem::dbm::FileOptionalInfo fileInfo;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/file", fileInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory("/dir1"));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory("/dir2"));

    bool isFile = false;
    bool isSameEntry = false;
    bool isParentEntry = false;

    // 同名のディレクトリが既に存在します。
    isFile = true;
    isSameEntry = true;
    isParentEntry = true;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileSystem.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "/dir2", "/dir1")
    );
    ASSERT_FALSE(isFile);
    ASSERT_FALSE(isSameEntry);
    ASSERT_FALSE(isParentEntry);

    // 同名のファイルが既に存在します。
    isFile = false;
    isSameEntry = true;
    isParentEntry = true;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileSystem.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "/file", "/dir1")
    );
    ASSERT_TRUE(isFile);
    ASSERT_FALSE(isSameEntry);
    ASSERT_FALSE(isParentEntry);

    // 変更前後で名前が同じです。
    isFile = true;
    isSameEntry = false;
    isParentEntry = true;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAlreadyExists,
        fileSystem.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "/dir1", "/dir1")
    );
    ASSERT_FALSE(isFile);
    ASSERT_TRUE(isSameEntry);
    ASSERT_FALSE(isParentEntry);

    testStorage.Unmount();
}

//! ディレクトリ名の変更をテストします。
TEST(FileSystemTest, TestRenameDirectory)
{
    // FileSystemObject を作成します。
    const uint32_t CountBlocks = 32;
    const uint32_t AllocationTablePadding = 0;
    const uint32_t BlockSize = 512;
    TestStorage testStorage(CountBlocks, AllocationTablePadding, BlockSize);
    SimpleFileSystemObject& fileSystem = testStorage.GetFileSystem();

    // ファイルとディレクトリを作成します。
    nn::fssystem::dbm::FileOptionalInfo fileInfo;
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateFile("/file", fileInfo));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory("/dir1"));
    NNT_ASSERT_RESULT_SUCCESS(fileSystem.CreateDirectory("/dir1/dir11"));

    bool isFile = false;
    bool isSameEntry = false;
    bool isParentEntry = false;

    // パスがルートディレクトリから始まっていません。
    isSameEntry = true;
    isParentEntry = true;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultInvalidPathFormat,
        fileSystem.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "rename", "/dir1")
    );
    ASSERT_FALSE(isSameEntry);
    ASSERT_FALSE(isParentEntry);

    // ディレクトリ名が長すぎます。（65 文字以上）
    isSameEntry = true;
    isParentEntry = true;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultTooLongPath,
        fileSystem.RenameDirectory(
            &isFile,
            &isSameEntry,
            &isParentEntry,
            "/12345678901234567890123456789012345678901234567890123456789012345",
            "/dir1"
        )
    );
    ASSERT_FALSE(isSameEntry);
    ASSERT_FALSE(isParentEntry);

    // ルートディレクトリの親ディレクトリは取得できません。
    isSameEntry = true;
    isParentEntry = true;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnobtainable,
        fileSystem.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "/dir1/../../rename", "/dir1")
    );
    ASSERT_FALSE(isSameEntry);
    ASSERT_FALSE(isParentEntry);

    // リネーム元のパスがファイル名のパスです。
    isSameEntry = true;
    isParentEntry = true;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultIncompatiblePath,
        fileSystem.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "/rename", "/file")
    );
    ASSERT_FALSE(isSameEntry);
    ASSERT_FALSE(isParentEntry);

    // ルートディレクトリはリネームできません。
    isSameEntry = true;
    isParentEntry = true;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileSystem.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "/rename", "/")
    );
    ASSERT_FALSE(isSameEntry);
    ASSERT_FALSE(isParentEntry);

    // リネーム元とリネーム先は親子関係にあります。
    isFile = true;
    isSameEntry = true;
    isParentEntry = false;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDirectoryUnrenamable,
        fileSystem.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "/dir1", "/dir1/dir11/")
    );
    ASSERT_FALSE(isFile);
    ASSERT_FALSE(isSameEntry);
    ASSERT_TRUE(isParentEntry);

    // リネームに成功します。
    isSameEntry = true;
    isParentEntry = true;
    NNT_ASSERT_RESULT_SUCCESS(
        fileSystem.RenameDirectory(&isFile, &isSameEntry, &isParentEntry, "/rename", "/dir1")
    );
    ASSERT_FALSE(isSameEntry);
    ASSERT_FALSE(isParentEntry);

    testStorage.Unmount();
}
