﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_BitTypes.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>

#include <nn/fssystem/dbm/fs_DuplexBitmap.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

//! 二重化ビットマップとそのバッファを保持するクラスです。
class DuplexBitmapSetup
{
public:
    /**
    * @brief        二重化ビットマップを作成します。
    *
    * @param[in]    bitCount        二重化ビットマップのビット数
    *
    * @details      二重化ビットマップを作成します。
    *               指定したビット数でメタデータ、オリジナルメタデータのバッファを確保し、
    *               メタデータストレージをフォーマット、マウントします。
    */
    explicit DuplexBitmapSetup(uint32_t bitCount) NN_NOEXCEPT
    : m_Storage(nn::fssystem::dbm::DuplexBitmap::QuerySize(bitCount)),
      m_OriginalStorage(nn::fssystem::dbm::DuplexBitmap::QuerySize(bitCount)),
      m_SubStorage(&m_Storage, 0, nn::fssystem::dbm::DuplexBitmap::QuerySize(bitCount)),
      m_OriginalSubStorage(
            &m_OriginalStorage,
            0,
            nn::fssystem::dbm::DuplexBitmap::QuerySize(bitCount)
        ),
        m_BitCount(bitCount)
    {
        Initialize(bitCount);
    }

    /**
    * @brief        二重化ビットマップを作成します。
    *
    * @param[in]    bitCount        二重化ビットマップのビット数
    * @param[in]    bitMarginCount  二重化ビットマップの拡張用マージンビット数
    *
    * @details      二重化ビットマップを作成します。
    *               指定したビット数でメタデータ、オリジナルメタデータのバッファを確保し、
    *               メタデータストレージをフォーマット、マウントします。
    */
    explicit DuplexBitmapSetup(uint32_t bitCount, uint32_t bitMarginCount) NN_NOEXCEPT
    : m_Storage(nn::fssystem::dbm::DuplexBitmap::QuerySize(bitCount + bitMarginCount)),
      m_OriginalStorage(nn::fssystem::dbm::DuplexBitmap::QuerySize(bitCount + bitMarginCount)),
      m_SubStorage(
            &m_Storage, 0, nn::fssystem::dbm::DuplexBitmap::QuerySize(bitCount + bitMarginCount)
        ),
        m_OriginalSubStorage(
            &m_OriginalStorage,
            0,
            nn::fssystem::dbm::DuplexBitmap::QuerySize(bitCount + bitMarginCount)
        ),
        m_BitCount(bitCount)
    {
        Initialize(bitCount);
    }

    //! デストラクタで二重化ビットマップをアンマウントします。
    ~DuplexBitmapSetup() NN_NOEXCEPT
    {
        m_DuplexBitmap.Finalize();
    }

    //! メタデータのバッファの内容をオリジナルメタデータのものに同期します。
    void Sync() NN_NOEXCEPT
    {
        m_Storage.Copy(m_OriginalStorage);
    }

    //! オリジナルメタデータにランダムな値を書き込みます。
    void FillRandom(std::mt19937* pMt) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pMt);

        nn::Bit8* pBuf = reinterpret_cast<nn::Bit8*>(m_OriginalStorage.GetBuffer());
        const nn::Bit8* pBufEnd = pBuf + m_BitCount / 8;
        nn::Bit8 mask = 0xFF;
        for( ; pBuf < pBufEnd; ++pBuf )
        {
            if( pBuf + 1 == pBufEnd )
            {
                mask >>= 8 - (m_BitCount % 8);
            }
            *pBuf = static_cast<nn::Bit8>(std::uniform_int_distribution<>(0, 0xFF)(*pMt) & mask);
        }
    }

    //! オリジナルメタデータを指定した値で埋め尽くします。
    void FillChar(nn::Bit8 value) NN_NOEXCEPT
    {
        nn::Bit8* pBuf = reinterpret_cast<nn::Bit8*>(m_OriginalStorage.GetBuffer());
        const nn::Bit8* pBufEnd = pBuf + m_OriginalStorage.GetSize();
        nn::Bit8 mask = 0xFF;
        for( ; pBuf < pBufEnd; ++pBuf )
        {
            if( pBuf + 1 == pBufEnd )
            {
                mask >>= 8 - ( m_BitCount % 8 );
            }
            *pBuf = value & mask;
        }
    }

    //! メタデータの指定したビットが有効かどうかを取得します。
    bool GetBits(uint32_t bit) const NN_NOEXCEPT
    {
        const nn::Bit8* pBuf = reinterpret_cast<const nn::Bit8*>(m_Storage.GetBuffer());

        // 4バイト単位でアドレス付けします。
        const nn::Bit8* pInt32 = pBuf + 4 * (bit / 32);

        // uint32_tでアクセスする前に確認します。
        NN_ASSERT(pInt32 + 4 <= pBuf + m_Storage.GetSize());

        // CLZ命令に依存し、最上位ビットから0/1状態を確認する必要があります。
        return (((*reinterpret_cast<const uint32_t*>(pInt32)) << (bit & 31)) & 0x80000000) != 0;
    }

    //! メタデータの指定したビットが有効かどうかを取得します。
    bool GetBitsOriginal(uint32_t bit) const NN_NOEXCEPT
    {
        const nn::Bit8* pBuf = reinterpret_cast<const nn::Bit8*>(m_OriginalStorage.GetBuffer());

        // 4バイト単位でアドレス付けします。
        const nn::Bit8* pInt32 = pBuf + 4 * (bit / 32);

        // uint32_tでアクセスする前に確認します。
        NN_ASSERT(pInt32 + 4 <= pBuf + m_OriginalStorage.GetSize());

        // CLZ命令に依存し、最上位ビットから0/1状態を確認する必要があります。
        return (((*reinterpret_cast<const uint32_t*>(pInt32)) << (bit & 31)) & 0x80000000) != 0;
    }

public:
    nn::fssystem::dbm::DuplexBitmap& GetDuplexBitmap() NN_NOEXCEPT
    {
        return m_DuplexBitmap;
    }

private:
    //! 二重化ビットマップを初期化します。
    void Initialize(uint32_t bitCount) NN_NOEXCEPT
    {
        // 確保したバッファでメタデータストレージをフォーマットします。
        const nn::Result result = nn::fssystem::dbm::DuplexBitmap::Format(
                                      bitCount,
                                      m_SubStorage,
                                      m_OriginalSubStorage
                                  );
        NN_ASSERT(result.IsSuccess());

        // 確保したバッファでメタデータストレージをマウントします。
        m_DuplexBitmap.Initialize(
            bitCount,
            m_SubStorage,
            m_OriginalSubStorage
        );
    }

private:
    nn::fssystem::dbm::DuplexBitmap m_DuplexBitmap;
    nnt::fs::util::SafeMemoryStorage m_Storage;
    nnt::fs::util::SafeMemoryStorage m_OriginalStorage;
    nn::fs::SubStorage m_SubStorage;
    nn::fs::SubStorage m_OriginalSubStorage;
    size_t m_BitCount;
};

//! 二重化ビットマップに必要なデータサイズの取得をテストします。
TEST(DuplexBitmapTest, TestQuerySize)
{
    ASSERT_EQ(0,   nn::fssystem::dbm::DuplexBitmap::QuerySize(0));
    ASSERT_EQ(64,  nn::fssystem::dbm::DuplexBitmap::QuerySize(1));
    ASSERT_EQ(64,  nn::fssystem::dbm::DuplexBitmap::QuerySize(8));
    ASSERT_EQ(64,  nn::fssystem::dbm::DuplexBitmap::QuerySize(33));
    ASSERT_EQ(64,  nn::fssystem::dbm::DuplexBitmap::QuerySize(511));
    ASSERT_EQ(64,  nn::fssystem::dbm::DuplexBitmap::QuerySize(512));
    ASSERT_EQ(128, nn::fssystem::dbm::DuplexBitmap::QuerySize(513));
    ASSERT_EQ(128, nn::fssystem::dbm::DuplexBitmap::QuerySize(1024));
    ASSERT_EQ(192, nn::fssystem::dbm::DuplexBitmap::QuerySize(1025));
}

//! 1 ビットの二重化ビットマップのイテレーションをテストします。
TEST(DuplexBitmapTest, TestIterate1Bit)
{
    // 1 ビットの二重化ビットマップのテストを行います。
    DuplexBitmapSetup bitmapSetup(1);
    nn::fssystem::dbm::DuplexBitmap& bitmap = bitmapSetup.GetDuplexBitmap();

    size_t oneCount = 0;
    size_t zeroCount = 0;

    // オリジナルメタデータに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::DuplexBitmap::Iterator iter;

        // イテレータを初期化します。
        bitmap.IterateOriginalBegin(&iter, 0, 1);

        // 先頭に 0 が 1 個だけあることをチェックします。
        // 出力用引数が書き換わることをテストするために一度クリアします。
        zeroCount = 0;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateOriginalNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(1, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateOriginalNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // メタデータに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::DuplexBitmap::Iterator iter;

        // イテレータを初期化します。
        bitmap.IterateBegin(&iter, 0, 1);

        // 先頭に 0 が 1 個だけあることをチェックします。
        zeroCount = 0;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(1, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
}

//! 1 ビットの二重化ビットマップのビット書き換えをテストします。
TEST(DuplexBitmapTest, TestMarkModified1Bit)
{
    // 1 ビットの二重化ビットマップのテストを行います。
    DuplexBitmapSetup bitmapSetup(1);
    nn::fssystem::dbm::DuplexBitmap& bitmap = bitmapSetup.GetDuplexBitmap();

    size_t oneCount = 0;
    size_t zeroCount = 0;

    // 1 ビットを書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(bitmap.MarkModified(0, 1));
    NNT_ASSERT_RESULT_SUCCESS(bitmap.Flush());

    {
        nn::fssystem::dbm::DuplexBitmap::Iterator iter;

        // イテレータを初期化します。
        bitmap.IterateOriginalBegin(&iter, 0, 1);

        // オリジナルビットマップは先頭に 0 が 1 個だけあることをチェックします。
        // 出力用引数が書き換わることをテストするために一度クリアします。
        zeroCount = 0;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateOriginalNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(1, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateOriginalNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    {
        nn::fssystem::dbm::DuplexBitmap::Iterator iter;

        // イテレータを初期化します。
        bitmap.IterateBegin(&iter, 0, 1);

        // ビットマップは先頭に 1 が 1 個だけあることをチェックします。
        zeroCount = 1;
        oneCount = 0;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(1, oneCount);

        // イテレーションが完了していることをチェックします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
}

//! 1 バイトの二重化ビットマップのイテレーションをテストします。
TEST(DuplexBitmapTest, TestIterate1Byte)
{
    // 1 バイトの二重化ビットマップのテストを行います。
    DuplexBitmapSetup bitmapSetup(8);
    nn::fssystem::dbm::DuplexBitmap& bitmap = bitmapSetup.GetDuplexBitmap();

    size_t zeroCount = 0;
    size_t oneCount = 0;

    // オリジナルメタデータに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::DuplexBitmap::Iterator iter;

        // イテレータを初期化します。
        bitmap.IterateOriginalBegin(&iter, 0, 8);

        // 0 が 1 バイト分連続していることをチェックします。
        // 出力用引数が書き換わることをテストするために一度クリアします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateOriginalNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(8, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateOriginalNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // メタデータに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::DuplexBitmap::Iterator iter;

        // イテレータを初期化します。
        bitmap.IterateBegin(&iter, 0, 8);

        // 0 が 1 バイト分連続していることをチェックします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(8, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateOriginalNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
}

//! 1 バイトの二重化ビットマップのビット書き換えをテストします。
TEST(DuplexBitmapTest, TestMarkModified1Byte)
{
    // 1 バイトの二重化ビットマップのテストを行います。
    DuplexBitmapSetup bitmapSetup(8);
    nn::fssystem::dbm::DuplexBitmap& bitmap = bitmapSetup.GetDuplexBitmap();

    size_t zeroCount = 0;
    size_t oneCount = 0;

    // ビットを書き換えます。
    {
        // 書き換え後のビット
        // org: 00000000
        // cur: 00011000
        NNT_ASSERT_RESULT_SUCCESS(bitmap.MarkModified(3, 2));
        NNT_ASSERT_RESULT_SUCCESS(bitmap.Flush());

        {
            nn::fssystem::dbm::DuplexBitmap::Iterator iter;

            // イテレータを初期化します。
            bitmap.IterateOriginalBegin(&iter, 0, 8);

            // オリジナルビットマップは先頭に 0 が 8 個だけあることをチェックします。
            // 出力用引数が書き換わることをテストするために一度クリアします。
            zeroCount = 1;
            oneCount = 1;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateOriginalNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(8, zeroCount);
            ASSERT_EQ(0, oneCount);

            // イテレーションが完了していることをチェックします。
            zeroCount = 1;
            oneCount = 1;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateOriginalNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(0, zeroCount);
            ASSERT_EQ(0, oneCount);
        }

        {
            nn::fssystem::dbm::DuplexBitmap::Iterator iter;

            // イテレータを初期化します。
            bitmap.IterateBegin(&iter, 0, 8);

            // 0 が 3 個、1 が 2 個連続していることをチェックします。
            zeroCount = 1;
            oneCount = 1;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(3, zeroCount);
            ASSERT_EQ(2, oneCount);

            // 0 が 3 個連続していることをチェックします。
            zeroCount = 1;
            oneCount = 1;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(3, zeroCount);
            ASSERT_EQ(0, oneCount);

            // イテレーションが完了していることをチェックします。
            zeroCount = 1;
            oneCount = 1;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(0, zeroCount);
            ASSERT_EQ(0, oneCount);
        }

        // 途中からイテレーションを行います。
        {
            nn::fssystem::dbm::DuplexBitmap::Iterator iter;

            // イテレータを初期化します。
            bitmap.IterateBegin(&iter, 2, 6);

            // 0 が 1 個、1 が 2 個連続していることをチェックします。
            zeroCount = 0;
            oneCount = 0;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(1, zeroCount);
            ASSERT_EQ(2, oneCount);

            // 0 が 3 個連続していることをチェックします。
            zeroCount = 1;
            oneCount = 1;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(3, zeroCount);
            ASSERT_EQ(0, oneCount);

            // イテレーションが完了していることをチェックします。
            zeroCount = 1;
            oneCount = 1;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(0, zeroCount);
            ASSERT_EQ(0, oneCount);
        }
    }

    // 同一位置のビットを複数回書き換えます。
    {
        // 書き換え後のビット
        // org: 00000000
        // cur: 00011110
        NNT_ASSERT_RESULT_SUCCESS(bitmap.MarkModified(3, 4));
        NNT_ASSERT_RESULT_SUCCESS(bitmap.Flush());

        {
            nn::Bit32 outputBit = 1;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.ReadOriginalBitmap32(&outputBit, 0));
            ASSERT_EQ(0x00000000, outputBit);

            outputBit = 0;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.ReadBitmap32(&outputBit, 0));
            ASSERT_EQ(0x1E000000, outputBit);

            outputBit = 0;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.ReadBitmap32(&outputBit, 3));
            ASSERT_EQ(0xF0000000, outputBit);
        }

        // 書き換え後のビット
        // org: 00000000
        // cur: 01111110
        NNT_ASSERT_RESULT_SUCCESS(bitmap.MarkModified(1, 3));
        NNT_ASSERT_RESULT_SUCCESS(bitmap.Flush());

        {
            nn::fssystem::dbm::DuplexBitmap::Iterator iter;

            // イテレータを初期化します。
            bitmap.IterateBegin(&iter, 0, 8);

            // 0 が 1 個、1 が 6 個連続していることをチェックします。
            zeroCount = 0;
            oneCount = 0;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(1, zeroCount);
            ASSERT_EQ(6, oneCount);

            // 0 が 1 個連続していることをチェックします。
            zeroCount = 0;
            oneCount = 1;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(1, zeroCount);
            ASSERT_EQ(0, oneCount);

            // イテレーションが完了していることをチェックします。
            zeroCount = 1;
            oneCount = 1;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(0, zeroCount);
            ASSERT_EQ(0, oneCount);
        }

        {
            nn::Bit32 outputBit = 1;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.ReadOriginalBitmap32(&outputBit, 0));
            ASSERT_EQ(0x00000000, outputBit);

            outputBit = 0;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.ReadBitmap32(&outputBit, 0));
            ASSERT_EQ(0x7E000000, outputBit);

            outputBit = 0;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.ReadBitmap32(&outputBit, 2));
            ASSERT_EQ(0xF8000000, outputBit);
        }
    }

    // 書き換え後のメタデータをリセットします。
    bitmapSetup.Sync();

    // ビットを書き換えます。
    {
        // 書き換え後のビット
        // org: 00000000
        // cur: 00000001
        NNT_ASSERT_RESULT_SUCCESS(bitmap.MarkModified(7, 1));
        NNT_ASSERT_RESULT_SUCCESS(bitmap.Flush());

        {
            nn::fssystem::dbm::DuplexBitmap::Iterator iter;

            // イテレータを初期化します。
            bitmap.IterateOriginalBegin(&iter, 2, 6);

            // オリジナルビットマップは先頭に 0 が 6 個だけあることをチェックします。
            zeroCount = 1;
            oneCount = 1;
            NNT_ASSERT_RESULT_SUCCESS(
                bitmap.IterateOriginalNext(
                    &zeroCount,
                    &oneCount,
                    &iter
                )
            );
            ASSERT_EQ(6, zeroCount);
            ASSERT_EQ(0, oneCount);

            // イテレーションが完了していることをチェックします。
            zeroCount = 1;
            oneCount = 1;
            NNT_ASSERT_RESULT_SUCCESS(
                bitmap.IterateOriginalNext(
                    &zeroCount,
                    &oneCount,
                    &iter
                )
            );
            ASSERT_EQ(0, zeroCount);
            ASSERT_EQ(0, oneCount);
        }

        {
            nn::fssystem::dbm::DuplexBitmap::Iterator iter;

            // イテレータを初期化します。
            bitmap.IterateBegin(&iter, 2, 6);

            // 先頭に 0 が 5 個、1 が 1 個連続していることをチェックします。
            zeroCount = 0;
            oneCount = 0;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(5, zeroCount);
            ASSERT_EQ(1, oneCount);

            // イテレーションが完了していることをチェックします。
            zeroCount = 1;
            oneCount = 1;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            ASSERT_EQ(0, zeroCount);
            ASSERT_EQ(0, oneCount);
        }

        {
            nn::Bit32 outputBit = 1;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.ReadOriginalBitmap32(&outputBit, 0));
            ASSERT_EQ(0x00000000, outputBit);

            outputBit = 0;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.ReadBitmap32(&outputBit, 0));
            ASSERT_EQ(0x01000000, outputBit);
        }
    }
} // NOLINT(impl/function_size)

//! 1 バイトより大きい中途半端なサイズの二重化ビットマップのイテレーションをテストします。
TEST(DuplexBitmapTest, TestIterateSomeBits)
{
    // 1 バイトよりも上 + 中途半端なサイズの二重化ビットマップのテストを行います。
    DuplexBitmapSetup bitmapSetup(19);
    nn::fssystem::dbm::DuplexBitmap& bitmap = bitmapSetup.GetDuplexBitmap();

    size_t zeroCount = 0;
    size_t oneCount = 0;

    // オリジナルメタデータに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::DuplexBitmap::Iterator iter;

        // イテレータを初期化します。
        bitmap.IterateBegin(&iter,0, 19);

        // 0 が 19 ビット分連続していることをチェックします。
        // 出力用引数が書き換わることをテストするために一度クリアします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateOriginalNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(19, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateOriginalNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // メタデータに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::DuplexBitmap::Iterator iter;

        // イテレータを初期化します。
        bitmap.IterateBegin(&iter,0, 19);

        // 0 が 19 ビット分連続していることをチェックします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(19, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
}

//! 1 バイトより大きい中途半端なサイズの二重化ビットマップの書き換えをテストします。
TEST(DuplexBitmapTest, TestMarkModifiedSomeBits)
{
    // 1 バイトよりも上 + 中途半端なサイズの二重化ビットマップのテストを行います。
    DuplexBitmapSetup bitmapSetup(19);
    nn::fssystem::dbm::DuplexBitmap& bitmap = bitmapSetup.GetDuplexBitmap();

    size_t zeroCount = 0;
    size_t oneCount = 0;

    // ビットを書き換えます。
    //
    // 書き換え後のビット
    // org: 0000000000000000000
    // cur: 0000011111111100000
    NNT_ASSERT_RESULT_SUCCESS(bitmap.MarkModified(5, 9));
    NNT_ASSERT_RESULT_SUCCESS(bitmap.Flush());

    {
        nn::fssystem::dbm::DuplexBitmap::Iterator iter;

        // イテレータを初期化します。
        bitmap.IterateBegin(&iter,0, 19);

        // 0 が 5 個、1 が 9 個連続していることをチェックします。
        // 出力用引数が書き換わることをテストするために一度クリアします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(5, zeroCount);
        ASSERT_EQ(9, oneCount);

        // 0 が 5 個連続していることをチェックします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(5, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // 途中からイテレーションを行います。
    {
        nn::fssystem::dbm::DuplexBitmap::Iterator iter;

        // イテレータを初期化します。
        bitmap.IterateBegin(&iter,6, 10);

        // 1 が 8 個連続していることをチェックします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(8, oneCount);

        // 0 が 2 個連続していることをチェックします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(2, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        zeroCount = 1;
        oneCount = 1;
        NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
}

//! ランダムな範囲のイテレーションをテストします。
TEST(DuplexBitmapTest, TestRandomIterateToRandomFill)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( uint32_t bits = 1; bits <= 4096; ++bits )
    {
        DuplexBitmapSetup bitmapSetup(bits);
        nn::fssystem::dbm::DuplexBitmap& bitmap = bitmapSetup.GetDuplexBitmap();

        // テストデータをランダムにセットします。
        if( std::uniform_int_distribution<>(0, 1)(mt) == 0 )
        {
            bitmapSetup.FillRandom(&mt);
        }
        else
        {
            bitmapSetup.FillChar(
                static_cast<nn::Bit8>(std::uniform_int_distribution<>(0, 255)(mt))
            );
        }
        bitmapSetup.Sync();

        // メタデータとオリジナルメタデータが同じ値を返すことを確認します。
        uint32_t index = std::uniform_int_distribution<uint32_t>(0, bits - 1)(mt);
        uint32_t size = std::uniform_int_distribution<uint32_t>(0, bits - index - 1)(mt);
        if( size == 0 )
        {
            index = bits - 1;
            size = 1;
        }
        nn::fssystem::dbm::DuplexBitmap::Iterator iter;
        nn::fssystem::dbm::DuplexBitmap::Iterator iterOriginal;
        bitmap.IterateBegin(&iter, index, size);
        bitmap.IterateOriginalBegin(&iterOriginal, index, size);
        for( ; ; )
        {
            size_t zeroCount = 0;
            size_t originalZeroCount = 1;
            size_t oneCount = 0;
            size_t originalOneCount = 1;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            NNT_ASSERT_RESULT_SUCCESS(
                bitmap.IterateOriginalNext(
                    &originalZeroCount, &originalOneCount, &iterOriginal
                )
            );
            if( zeroCount == 0 && oneCount == 0 )
            {
                break;
            }
            ASSERT_EQ(zeroCount, originalZeroCount);
            ASSERT_EQ(oneCount, originalOneCount);

            for( ; zeroCount > 0; --zeroCount, ++index )
            {
                ASSERT_FALSE(bitmapSetup.GetBits(index));
            }
            for( ; oneCount > 0; --oneCount, ++index )
            {
                ASSERT_TRUE(bitmapSetup.GetBits(index));
            }
        }

        // テスト高速化のため歯抜けにまわします。
        if( (bits >= 16) && (bits <= 4096 - 16) )
        {
            bits += 3;
            if( (bits >= 48) && (bits <= 4096 - 512) )
            {
                bits += std::uniform_int_distribution<uint32_t>(256, 511)(mt);
            }
        }
    }
}

//! ランダムなビット反転をテストします。
TEST(DuplexBitmapTest, TestRandomModify)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    // ランダムに 3 箇所反転させ、反転した範囲のビットのみが
    // 書き換わっていることをチェックします。
    for( uint32_t bits = 1; bits <= 32768; )
    {
        DuplexBitmapSetup bitmapSetup(bits);
        nn::fssystem::dbm::DuplexBitmap& bitmap = bitmapSetup.GetDuplexBitmap();

        // テストデータをセットします。
        if( std::uniform_int_distribution<>(0, 1)(mt) == 0 )
        {
            bitmapSetup.FillRandom(&mt);
        }
        else
        {
            bitmapSetup.FillChar(
                static_cast<nn::Bit8>(std::uniform_int_distribution<>(0, 1)(mt) * 0xFF)
            );
        }
        bitmapSetup.Sync();

        // ランダムに 3 箇所反転します。
        static const uint32_t Pairs = 3;
        uint32_t bitBegin[Pairs];
        uint32_t bitSize[Pairs];
        for( int i = 0; i < Pairs; ++i )
        {
            uint32_t begin = std::uniform_int_distribution<uint32_t>(0, bits - 1)(mt);
            uint32_t size = std::uniform_int_distribution<uint32_t>(0, bits - begin - 1)(mt);
            if( size == 0 )
            {
                begin = bits - 1;
                size = 1;
            }
            bitBegin[i] = begin;
            bitSize[i] = size;

            NNT_ASSERT_RESULT_SUCCESS(bitmap.MarkModified(begin, size));
            NNT_ASSERT_RESULT_SUCCESS(bitmap.Flush());
        }

        // 反転した範囲のみが書き換わっていることを確認します。
        for( uint32_t b = 0; b < bits; ++b )
        {
            bool isInsideReverse = false;
            for( int i = 0; i < Pairs; ++i )
            {
                if( (bitBegin[i] <= b) && (b < bitBegin[i] + bitSize[i]) )
                {
                    // "b"は領域内の値です。
                    isInsideReverse = true;
                    break;
                }
            }

            // 反転領域内のビットは値が書き換わっていることをチェックします。
            if( bitmapSetup.GetBits(b) != bitmapSetup.GetBitsOriginal(b) )
            {
                // new != orgであれば、反転領域に入っています。
                ASSERT_TRUE(isInsideReverse);
            }
            else
            {
                // new == orgであれば、反転領域外です。
                ASSERT_TRUE(!isInsideReverse);
            }
        }

        ++bits;
        // 歯抜けにまわします。
        if( bits > 64 )
        {
            bits += bits / 2;
        }
    }
}

//! 規則的に書き換えたビットに対するランダムなイテレーションをテストします。
TEST(DuplexBitmapTest, TestRandomIterateToRegularFill)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( uint32_t bits = 1; bits <= 4096; ++bits )
    {
        DuplexBitmapSetup bitmapSetup(bits);
        nn::fssystem::dbm::DuplexBitmap& bitmap = bitmapSetup.GetDuplexBitmap();

        // サイズチェックを行います。
        // 二重化ビットマップに必要なデータサイズの計算が正しいかテストします。
        static const auto CacheCount = 64; // DuplexBitmap::FormatCacheSize
        static const auto CacheSize = 8 * CacheCount;
        ASSERT_EQ(
            (bits + CacheSize - 1) / CacheSize * CacheCount,
            nn::fssystem::dbm::DuplexBitmap::QuerySize(bits)
        );

        // データを初期化します。
        bitmapSetup.FillChar(0);
        bitmapSetup.Sync();

        // 0、1 の個数を数えます。
        // 初期化直後なので全て 0 になっていることをテストします。
        {
            nn::fssystem::dbm::DuplexBitmap::Iterator iter;
            size_t zeros = 0;
            size_t ones = 1;
            bitmap.IterateBegin(&iter, 0, bits);
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeros, &ones, &iter));
            ASSERT_EQ(bits, zeros);
            ASSERT_EQ(0, ones);
        }

        // サイズ分、1bitずつfillしていきます。 (0,bits-1)
        for( uint32_t i = 0; i < bits; ++i )
        {
            NNT_ASSERT_RESULT_SUCCESS(bitmap.MarkModified(i, 1));
            NNT_ASSERT_RESULT_SUCCESS(bitmap.Flush());
            // 初めの 64 ビットと終わりの 64 ビットは 1 ビットごとにチェックします。
            // それ以外はランダムにチェックします。
            if( (i < 64) || (std::uniform_int_distribution<>(0, 62)(mt) == 0) || (i > (bits - 64)) )
            {
                for( uint32_t j = 0; j < bits; ++j )
                {
                    if( i >= j )
                    {
                        // ビットを fill し終えた範囲について、ビットをテストします。
                        ASSERT_TRUE(bitmapSetup.GetBits(j));

                        // テスト高速化のため反転させた範囲の境界付近以外は飛ばします。
                        if( (i >= 64) && (i - 15 > j) )
                        {
                            j += 7;
                        }
                    }
                    else
                    {
                        // ビットを fill していない範囲について、ビットをテストします。
                        ASSERT_TRUE(!bitmapSetup.GetBits(j));

                        // テスト高速化のため反転させた範囲の境界付近以外は飛ばします。
                        if( (i >= 64) && (i + 15 < j) )
                        {
                            j += 7;
                        }
                    }
                }

                // BitFillした結果をイテレーションします。
                {
                    nn::fssystem::dbm::DuplexBitmap::Iterator iter;
                    // イテレーションする範囲をランダムに決めます。
                    const uint32_t indexBegin
                        = (bits < 3) ? 0 : std::uniform_int_distribution<uint32_t>(0, bits - 2)(mt);
                    const uint32_t indexSize
                        = std::uniform_int_distribution<uint32_t>(0, bits - indexBegin - 1)(mt);
                    if( indexSize > 0 )
                    {
                        // 計算上の値と実際の値を比較します。
                        bitmap.IterateBegin(&iter, indexBegin, indexSize);

                        // 計算上はイテレーション開始位置から fill し終えた位置まで
                        // 1 のビットが続きます。
                        uint32_t onesCalculated = (i + 1 < indexBegin) ? 0 : i + 1 - indexBegin;
                        if( onesCalculated > indexSize )
                        {
                            onesCalculated = indexSize;
                        }
                        uint32_t zerosCalculated = indexSize - onesCalculated;

                        if( (indexBegin + indexSize > i + 1) && (indexBegin < i + 1) )
                        {
                            // イテレーション範囲と fill し終えた範囲が重なれば、1 が連続します。
                            size_t zeros = 1;
                            size_t ones = 0;
                            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeros, &ones, &iter));
                            ASSERT_EQ(0, zeros);
                            ASSERT_EQ(onesCalculated, ones);
                            // 1 の連続が終わると残りは fill されていないので 0 だけが連続します。
                            zeros = 0;
                            ones = 1;
                            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeros, &ones, &iter));
                            ASSERT_EQ(zerosCalculated, zeros);
                            ASSERT_EQ(0, ones);
                        }
                        else
                        {
                            // イテレーション範囲と fill し終えた範囲が重ならなければ、
                            // 1 は出現せず 0 だけが連続します。
                            size_t zeros = 0;
                            size_t ones = 1;
                            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeros, &ones, &iter));
                            ASSERT_EQ(zerosCalculated, zeros);
                            ASSERT_EQ(onesCalculated, ones);
                        }
                    }
                }
            }
        }

        // 0、1 の個数を数えます。
        {
            // メタデータビットマップは fill したので 1 のみが連続します。
            nn::fssystem::dbm::DuplexBitmap::Iterator iter;
            size_t zeros = 1;
            size_t ones = 0;
            bitmap.IterateBegin(&iter, 0, bits);
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeros, &ones, &iter));
            ASSERT_EQ(0, zeros);
            ASSERT_EQ(bits, ones);

            // オリジナルは 0 のみが連続します。
            bitmap.IterateOriginalBegin(&iter, 0, bits);
            // 出力用引数が書き換わることをテストするために一度クリアします。
            zeros = 0;
            ones = 1;
            NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateOriginalNext(&zeros, &ones, &iter));
            ASSERT_EQ(bits, zeros);
            ASSERT_EQ(0, ones);
        }

        // テスト高速化のため歯抜けにまわします。
        if( (bits >= 16) && (bits <= 4096 - 16) )
        {
            bits += 3;
            if( (bits >= 48) && (bits <= 4096 - 512) )
            {
                bits += std::uniform_int_distribution<uint32_t>(256, 511)(mt);
            }
        }
    }
} // NOLINT(impl/function_size)

//! 二重化ビットマップの拡張をテストします。
TEST(DuplexBitmapTest, TestExpand)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( uint32_t oldBitmapCount = 1; oldBitmapCount <= 1024; oldBitmapCount <<= 2 )
    {
        for( uint32_t newBitmapCount = oldBitmapCount + 1;
                newBitmapCount <= 2048;
                newBitmapCount <<= 1
            )
        {
            nn::fssystem::dbm::DuplexBitmap bitmap;

            // 拡張前のストレージを確保します。
            const int64_t sizeOld = nn::fssystem::dbm::DuplexBitmap::QuerySize(oldBitmapCount);
            nnt::fs::util::SafeMemoryStorage originalStorageOld(sizeOld);
            nnt::fs::util::SafeMemoryStorage storageOld(sizeOld);
            nn::fs::SubStorage originalSubStorageOld(&originalStorageOld, 0, sizeOld);
            nn::fs::SubStorage subStorageOld(&storageOld, 0, sizeOld);

            // ビットマップを初期化します。
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::dbm::DuplexBitmap::Format(
                    oldBitmapCount,
                    subStorageOld,
                    originalSubStorageOld
                )
            );
            bitmap.Initialize(oldBitmapCount, subStorageOld, originalSubStorageOld);

            // ランダムにビットを反転させます。
            nnt::fs::util::Vector<bool> modifiedBits(newBitmapCount, false);
            for( uint32_t index = 0; index < oldBitmapCount; ++index )
            {
                if( std::uniform_int_distribution<>(0, 1)(mt) == 0 )
                {
                    // ビットを反転させます。
                    NNT_ASSERT_RESULT_SUCCESS(bitmap.MarkModified(index, 1));

                    // ビットを反転させたことを記録します。
                    modifiedBits[index] = true;
                }
            }

            // ビットが反転したことを確認します。
            {
                nn::fssystem::dbm::DuplexBitmap::Iterator iter;
                bitmap.IterateBegin(&iter, 0, oldBitmapCount);

                size_t oneCount = 0;
                size_t zeroCount = 0;

                for( size_t index = 0; index < oldBitmapCount; )
                {
                    NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
                    ASSERT_LE(index + zeroCount + oneCount, oldBitmapCount);

                    // 0 の続く範囲内にビットを反転させた記録が無いことを確認します。
                    for( size_t zeroIndex = index; zeroIndex < index + zeroCount; ++zeroIndex )
                    {
                        ASSERT_FALSE(modifiedBits[zeroIndex]);
                    }

                    index += zeroCount;

                    // 1 の続く範囲内にビットを反転させた記録があることを確認します。
                    for( size_t oneIndex = index; oneIndex < index + oneCount; ++oneIndex )
                    {
                        ASSERT_TRUE(modifiedBits[oneIndex]);
                    }

                    index += oneCount;
                }

                // イテレーションが終了します。
                NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
                ASSERT_EQ(zeroCount, 0);
                ASSERT_EQ(oneCount, 0);
            }

            // 拡張のために一旦アンマウントします。
            bitmap.Finalize();

            // 拡張後のストレージを確保します。
            const int64_t sizeNew = nn::fssystem::dbm::DuplexBitmap::QuerySize(newBitmapCount);
            nnt::fs::util::SafeMemoryStorage originalStorageNew(sizeNew);
            nnt::fs::util::SafeMemoryStorage storageNew(sizeNew);
            nn::fs::SubStorage originalSubStorageNew(&originalStorageNew, 0, sizeNew);
            nn::fs::SubStorage subStorageNew(&storageNew, 0, sizeNew);

            // 拡張前のストレージの内容を書き込みます。
            NNT_ASSERT_RESULT_SUCCESS(
                originalStorageNew.Write(
                    0,
                    originalStorageOld.GetBuffer(),
                    static_cast<size_t>(originalStorageOld.GetSize())
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(
                storageNew.Write(
                    0,
                    storageOld.GetBuffer(),
                    static_cast<size_t>(storageOld.GetSize())
                )
            );
            NNT_ASSERT_RESULT_SUCCESS(originalStorageNew.Flush());
            NNT_ASSERT_RESULT_SUCCESS(subStorageNew.Flush());

            // ビットマップを拡張します。
            NNT_ASSERT_RESULT_SUCCESS(
                bitmap.Expand(
                    oldBitmapCount,
                    newBitmapCount,
                    subStorageNew,
                    originalSubStorageNew
                )
            );

            // 再度マウントします。
            bitmap.Initialize(newBitmapCount, subStorageNew, originalSubStorageNew);

            // ビットの状態が変化していないことを確認します。
            {
                nn::fssystem::dbm::DuplexBitmap::Iterator iter;
                bitmap.IterateBegin(&iter, 0, newBitmapCount);

                size_t oneCount = 0;
                size_t zeroCount = 0;

                for( size_t index = 0; index < newBitmapCount; )
                {
                    NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
                    ASSERT_LE(index + zeroCount + oneCount, newBitmapCount);

                    // 0 の続く範囲内にビットを反転させた記録が無いことを確認します。
                    for( size_t zeroIndex = index; zeroIndex < index + zeroCount; ++zeroIndex )
                    {
                        if( zeroIndex < oldBitmapCount )
                        {
                            ASSERT_FALSE(modifiedBits[zeroIndex]);
                        }
                    }

                    index += zeroCount;

                    if( oneCount > 0 )
                    {
                        // 1 の続く範囲内に拡張された領域はありません。
                        ASSERT_LE(index + oneCount, oldBitmapCount);
                    }

                    // 1 の続く範囲内にビットを反転させた記録があることを確認します。
                    for( size_t oneIndex = index; oneIndex < index + oneCount; ++oneIndex )
                    {
                        ASSERT_TRUE(modifiedBits[oneIndex]);
                    }

                    index += oneCount;
                }

                // イテレーションが終了します。
                NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
                ASSERT_EQ(zeroCount, 0);
                ASSERT_EQ(oneCount, 0);
            }

            // ランダムにビットを反転させます。
            for( uint32_t index = 0; index < newBitmapCount; ++index )
            {
                if( std::uniform_int_distribution<>(0, 1)(mt) == 0 )
                {
                    // ビットを反転させます。
                    NNT_ASSERT_RESULT_SUCCESS(bitmap.MarkModified(index, 1));

                    // ビットを反転させたことを記録します。
                    modifiedBits[index] = true;
                }
            }

            // ビットが反転したことを確認します。
            {
                nn::fssystem::dbm::DuplexBitmap::Iterator iter;
                bitmap.IterateBegin(&iter, 0, newBitmapCount);

                size_t oneCount = 0;
                size_t zeroCount = 0;

                for( size_t index = 0; index < newBitmapCount; )
                {
                    NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
                    ASSERT_LE(index + zeroCount + oneCount, newBitmapCount);

                    // 0 の続く範囲内にビットを反転させた記録が無いことを確認します。
                    for( size_t zeroIndex = index; zeroIndex < index + zeroCount; ++zeroIndex )
                    {
                        ASSERT_FALSE(modifiedBits[zeroIndex]);
                    }

                    index += zeroCount;

                    // 1 の続く範囲内にビットを反転させた記録があることを確認します。
                    for( size_t oneIndex = index; oneIndex < index + oneCount; ++oneIndex )
                    {
                        ASSERT_TRUE(modifiedBits[oneIndex]);
                    }

                    index += oneCount;
                }

                // イテレーションが終了します。
                NNT_ASSERT_RESULT_SUCCESS(bitmap.IterateNext(&zeroCount, &oneCount, &iter));
                ASSERT_EQ(zeroCount, 0);
                ASSERT_EQ(oneCount, 0);
            }

            bitmap.Finalize();
        }
    }
} // NOLINT(impl/function_size)
