﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <numeric>

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>

#include <nn/fssystem/buffers/fs_FileSystemBufferManager.h>
#include <nn/fssystem/dbm/fs_BufferedAllocationTableStorage.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

template<typename TStorage>
class BufferedAllocationTableStorageTestTemplate : public ::testing::Test
{
protected:
    static const auto BlockSize = 16 * 1024;

protected:
    nn::Result Initialize(uint32_t countBlocks, int64_t sizeBody) NN_NOEXCEPT
    {
        const auto sizeTable = nn::fssystem::dbm::AllocationTable::QuerySize(countBlocks);

        m_TableStorage.Initialize(sizeTable);
        m_BodyStorage.Initialize(sizeBody);

        static const auto CacheCount = 4;
        static const auto BufferSize = 256 * 1024;
        m_Buffer.reset(new char[BufferSize]);
        NN_RESULT_THROW_UNLESS(m_Buffer != nullptr, nn::fs::ResultAllocationMemoryFailed());
        NN_RESULT_DO(
            m_BufferManager.Initialize(
                CacheCount,
                reinterpret_cast<uintptr_t>(m_Buffer.get()),
                BufferSize,
                BlockSize
            )
        );

        nn::fs::SubStorage tableStorage(&m_TableStorage, 0, sizeTable);
        uint32_t index = 0;
        NN_RESULT_DO(m_AllocationTable.Format(tableStorage, countBlocks));
        m_AllocationTable.Initialize(tableStorage, countBlocks);
        NN_RESULT_DO(m_AllocationTable.Allocate(&index, countBlocks));
        NN_RESULT_DO(
            m_Storage.InitializeBuffered(
                &m_AllocationTable,
                index,
                BlockSize,
                nn::fs::SubStorage(&m_BodyStorage, 0, sizeBody),
                &m_BufferManager
            )
        );

        NN_RESULT_SUCCESS;
    }

    void Finalize() NN_NOEXCEPT
    {
        m_Storage.Finalize();
        m_BufferManager.Finalize();
    }

    nn::fssystem::dbm::BufferedAllocationTableStorage& GetStorage() NN_NOEXCEPT
    {
        return m_Storage;
    }

    const TStorage& GetBodyStorage() const NN_NOEXCEPT
    {
        return m_BodyStorage;
    }

private:
    std::unique_ptr<char[]> m_Buffer;
    nn::fssystem::FileSystemBufferManager m_BufferManager;
    TStorage m_TableStorage;
    TStorage m_BodyStorage;
    nn::fssystem::dbm::AllocationTable m_AllocationTable;
    nn::fssystem::dbm::BufferedAllocationTableStorage m_Storage;
};

typedef BufferedAllocationTableStorageTestTemplate<nnt::fs::util::AccessCountedMemoryStorage> BufferedAllocationTableStorageTest;
typedef BufferedAllocationTableStorageTestTemplate<nnt::fs::util::VirtualMemoryStorage> BufferedAllocationTableStorageLargeTest;

//! シーケンシャルリードをテストします。
TEST_F(BufferedAllocationTableStorageTest, SequentialRead)
{
    // 初期化します
    static const uint32_t CountBlocks = 64;
    static const auto SizeBody = 16 * 1024 * CountBlocks;
    static const auto CountCache = 4;
    NNT_ASSERT_RESULT_SUCCESS(Initialize(CountBlocks, SizeBody));
    GetStorage().EnableCacheBuffer(CountCache);

    // バッファを用意します
    nnt::fs::util::Vector<char> writeBuffer(SizeBody);
    nnt::fs::util::Vector<char> readBuffer(SizeBody, -1);
    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x0');

    // 読み書きできることを確認します
    static const auto SizeRead = 64 * 1024;
    GetStorage().EnableCacheBuffer(CountCache);
    NNT_ASSERT_RESULT_SUCCESS(
        GetStorage().Write(
            0,
            writeBuffer.data(),
            SizeBody
        )
    );
    for( auto offset = 0; offset < SizeBody; offset += SizeRead )
    {
        NNT_ASSERT_RESULT_SUCCESS(
            GetStorage().Read(
                offset,
                readBuffer.data() + offset,
                SizeRead
            )
        );
    }
    GetStorage().DisableCacheBuffer();
    EXPECT_TRUE(std::equal(readBuffer.begin(), readBuffer.end(), writeBuffer.begin()));

    // 終了処理を行います
    GetStorage().DisableCacheBuffer();
    Finalize();
}

//! ランダムリードをテストします。
TEST_F(BufferedAllocationTableStorageTest, RandomRead)
{
    // 初期化します
    static const uint32_t CountBlocks = 64;
    static const auto SizeBody = 16 * 1024 * CountBlocks;
    static const auto CountCache = 4;
    NNT_ASSERT_RESULT_SUCCESS(Initialize(CountBlocks, SizeBody));
    GetStorage().EnableCacheBuffer(CountCache);

    // バッファを用意します
    nnt::fs::util::Vector<char> writeBuffer(SizeBody);
    nnt::fs::util::Vector<char> readBuffer(SizeBody, -1);
    std::iota(writeBuffer.begin(), writeBuffer.end(), '\x0');

    // 読み書きできることを確認します
    static const auto LoopCount = 256;
    std::mt19937 rng(nnt::fs::util::GetRandomSeed());
    NNT_ASSERT_RESULT_SUCCESS(
        GetStorage().Write(
            0,
            writeBuffer.data(),
            SizeBody
        )
    );
    for( auto count = 0; count < LoopCount; ++count )
    {
        const auto offset = std::uniform_int_distribution<int64_t>(0, SizeBody - 1)(rng);
        const auto readableSize = static_cast<size_t>(SizeBody - offset);
        const auto size = std::uniform_int_distribution<size_t>(1, readableSize)(rng);
        NNT_ASSERT_RESULT_SUCCESS(
            GetStorage().Read(
                offset,
                readBuffer.data() + offset,
                size
            )
        );
        EXPECT_TRUE(
            std::equal(
                std::next(readBuffer.begin(), static_cast<ptrdiff_t>(offset)),
                std::next(readBuffer.begin(), static_cast<ptrdiff_t>(offset + size)),
                std::next(writeBuffer.begin(), static_cast<ptrdiff_t>(offset))
            )
        );
    }

    // 終了処理を行います
    GetStorage().DisableCacheBuffer();
    Finalize();
}

//! リードがキャッシュされることをテストします。
TEST_F(BufferedAllocationTableStorageTest, Cache)
{
    // 初期化します
    static const uint32_t CountBlocks = 64;
    static const auto SizeBody = 16 * 1024 * CountBlocks;
    static const auto CountCache = 1;
    NNT_ASSERT_RESULT_SUCCESS(Initialize(CountBlocks, SizeBody));
    GetStorage().EnableCacheBuffer(CountCache);

    // バッファを用意します
    static const auto SizeBlock = 16 * 1024;
    nnt::fs::util::Vector<char> buffer(SizeBlock);

    // 同じブロックの読み込みはキャッシュされるため初回以外は実ストレージへのアクセスが発生しません
    static const auto LoopCount = 256;
    std::mt19937 rng(nnt::fs::util::GetRandomSeed());
    for( auto count = 0; count < LoopCount; ++count )
    {
        const auto offset = std::uniform_int_distribution<int64_t>(0, SizeBlock - 1)(rng);
        const auto readableSize = static_cast<size_t>(SizeBlock - offset);
        const auto size = std::uniform_int_distribution<size_t>(1, readableSize)(rng);
        NNT_ASSERT_RESULT_SUCCESS(
            GetStorage().Read(
                offset,
                buffer.data() + offset,
                size
            )
        );
        EXPECT_EQ(1, GetBodyStorage().GetReadTimes());
    }

    // 書き込むとキャッシュが無効化され実ストレージへのアクセスが発生します
    {
        const auto offset = std::uniform_int_distribution<int64_t>(0, SizeBlock - 1)(rng);
        const auto readableSize = static_cast<size_t>(SizeBlock - offset);
        const auto size = std::uniform_int_distribution<size_t>(1, readableSize)(rng);
        NNT_ASSERT_RESULT_SUCCESS(
            GetStorage().Write(
                offset,
                buffer.data() + offset,
                size
            )
        );
        NNT_ASSERT_RESULT_SUCCESS(
            GetStorage().Read(
                offset,
                buffer.data() + offset,
                size
            )
        );
        EXPECT_EQ(2, GetBodyStorage().GetReadTimes());
    }

    // 別ブロックの読み込みは実ストレージへのアクセスが発生します
    {
        const auto offset = std::uniform_int_distribution<int64_t>(0, SizeBlock - 1)(rng);
        const auto readableSize = static_cast<size_t>(SizeBlock - offset);
        const auto size = std::uniform_int_distribution<size_t>(1, readableSize)(rng);
        NNT_ASSERT_RESULT_SUCCESS(
            GetStorage().Read(
                offset + SizeBlock,
                buffer.data() + offset,
                size
            )
        );
        EXPECT_EQ(3, GetBodyStorage().GetReadTimes());
    }

    // 終了処理を行います
    GetStorage().DisableCacheBuffer();
    Finalize();
}

TEST_F(BufferedAllocationTableStorageLargeTest, ReadWriteLargeOffset)
{
    static const auto SizeBody = nnt::fs::util::LargeOffsetMax + BlockSize;
    static const uint32_t CountBlocks = static_cast<uint32_t>(SizeBody / BlockSize);
    static const auto CountCache = 4;

    NNT_ASSERT_RESULT_SUCCESS(Initialize(CountBlocks, SizeBody));
    GetStorage().EnableCacheBuffer(CountCache);
    NN_UTIL_SCOPE_EXIT
    {
        GetStorage().DisableCacheBuffer();
    };

    nnt::fs::util::TestStorageAccessWithLargeOffset(&GetStorage(), BlockSize);
}

