﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <random>
#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/dbm/fs_Bitmap.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

TEST(BitmapQueryTest, QuerySize)
{
    ASSERT_EQ(nn::fssystem::dbm::Bitmap::QuerySize(0), 0);
    ASSERT_EQ(nn::fssystem::dbm::Bitmap::QuerySize(1), 4);
    ASSERT_EQ(nn::fssystem::dbm::Bitmap::QuerySize(8), 4);
    ASSERT_EQ(nn::fssystem::dbm::Bitmap::QuerySize(9), 4);
    ASSERT_EQ(nn::fssystem::dbm::Bitmap::QuerySize(31), 4);
    ASSERT_EQ(nn::fssystem::dbm::Bitmap::QuerySize(32), 4);
    ASSERT_EQ(nn::fssystem::dbm::Bitmap::QuerySize(33), 8);
    ASSERT_EQ(nn::fssystem::dbm::Bitmap::QuerySize(64), 8);
    ASSERT_EQ(nn::fssystem::dbm::Bitmap::QuerySize(65), 12);
}

namespace
{
    // フォーマットテスト用のパラメータです。
    struct FormatTestParam
    {
        int64_t offset;    // SubStorage のオフセット
        uint32_t bitCount; // Bitmap のビット数
    };

    // フォーマットテスト用のパラメータを生成します。
    nnt::fs::util::Vector<FormatTestParam> MakeFormatTestParam() NN_NOEXCEPT
    {
        nnt::fs::util::Vector<FormatTestParam> params;

        // ビット数のテーブル
        static const int BitCountTable[] = {
            1, 15, 1023, 1024, 1025, 100000
        };

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        // オフセットをランダムに決め、テーブルにある各ビット数でパラメータを追加します。
        for( int i = 0; i < 10; ++i )
        {
            const int64_t offset = std::uniform_int_distribution<int64_t>(0, 1024)(mt);
            for( auto bitCount : BitCountTable )
            {
                FormatTestParam data;
                data.offset = offset;
                data.bitCount = bitCount;
                params.push_back(data);
            }
        }

        return params;
    }
}

class BitmapFormatTest : public ::testing::TestWithParam< FormatTestParam > { };

// 複数のパラメータでフォーマットするだけのテスト
TEST_P(BitmapFormatTest, Format)
{
    const FormatTestParam& param = GetParam();

    // パラメータのビット数、オフセットに従ってストレージを作ります。
    const int64_t size = nn::fssystem::dbm::Bitmap::QuerySize(param.bitCount);
    nnt::fs::util::SafeMemoryStorage storageBase(size + param.offset);
    nn::fs::SubStorage storage(&storageBase, param.offset, size);

    // Bitmap のストレージとしてフォーマットします。
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::dbm::Bitmap::Format(param.bitCount, storage)
    );

    // ストレージの範囲外アクセスが発生していないかテストします。
    ASSERT_TRUE(storageBase.CheckValid());
}

// フォーマットとマウントを繰り返すテスト
TEST_P(BitmapFormatTest, FormatAndInitializeAging)
{
    const FormatTestParam& param = GetParam();

    // パラメータのビット数、オフセットに従ってストレージを作ります。
    const int64_t size = nn::fssystem::dbm::Bitmap::QuerySize(param.bitCount);
    nnt::fs::util::SafeMemoryStorage storageBase(size + param.offset);
    nn::fs::SubStorage storage(&storageBase, param.offset, size);

    for( int i = 0; i < 10; ++i )
    {
        // Bitmap のストレージとしてフォーマットします。
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::dbm::Bitmap::Format(param.bitCount, storage)
        );

        // Bitmap をマウントします。
        nn::fssystem::dbm::Bitmap bitmap;
        bitmap.Initialize(param.bitCount, storage);

        // ストレージの範囲外アクセスが発生していないかテストします。
        ASSERT_TRUE(storageBase.CheckValid());
    }
}

INSTANTIATE_TEST_CASE_P(
    BitmapTest,
    BitmapFormatTest,
    ::testing::ValuesIn(MakeFormatTestParam())
);

namespace
{
    // ビット操作テスト用のパラメータです。
    struct BitmapTestParam
    {
        int64_t offset; // SubStorage のオフセット
    };

    // ビット操作テスト用のパラメータを生成します。
    nnt::fs::util::Vector<BitmapTestParam> MakeBitmapTestParam()
    {
        nnt::fs::util::Vector<BitmapTestParam> params;

        // オフセットなし
        {
            BitmapTestParam data;
            data.offset = 0;
            params.push_back(data);
        }

        // 適当なオフセット
        {
            BitmapTestParam data;
            data.offset = 190;
            params.push_back(data);
        }

        return params;
    }
}

// ビット操作テストで Bitmap とそれが使うストレージを初期化、保持するテストフィクスチャです。
class BitmapAPITest : public ::testing::TestWithParam< BitmapTestParam >
{
protected:
    /**
    * @brief テスト終了時にストレージの範囲外アクセスが発生していないかテストします。
    */
    virtual void TearDown() NN_OVERRIDE
    {
        ASSERT_TRUE(m_Storage.CheckValid());
    }

protected:
    /**
    * @brief      指定されたビット数で Bitmap とストレージを初期化します。
    *
    * @param[in]  bitCount    Bitmap のビット数
    */
    void SetupBitmap(uint32_t bitCount) NN_NOEXCEPT
    {
        const BitmapTestParam& param = GetParam();

        // パラメータに従ってストレージを初期化します。
        m_Offset = param.offset;
        const int64_t size = nn::fssystem::dbm::Bitmap::QuerySize(bitCount);
        m_Storage.Initialize(size + m_Offset);
        nn::fs::SubStorage storage(&m_Storage, m_Offset, size);

        // Bitmap のストレージとしてフォーマットします。
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::dbm::Bitmap::Format(bitCount, storage)
        );

        // Bitmap をマウントします。
        m_Bitmap.reset(new nn::fssystem::dbm::Bitmap());
        m_Bitmap->Initialize(bitCount, storage);
    }

    /**
    * @brief        指定したインデックスから 32 ビット分のビットマップを取得します。
    *
    * @param[out]   outValue    ビットマップ
    * @param[in]    index       インデックス
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outValue != nullptr
    */
    nn::Result GetBitmap32(uint32_t* outValue, uint32_t index) NN_NOEXCEPT
    {
        return m_Bitmap->GetBitmap32(outValue, index);
    }

    /**
    * @brief        指定したインデックスのビットが 0 か 1 かを取得します。
    *
    * @param[out]   outValue    1 ならば true、0 ならば false
    * @param[in]    index       インデックス
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outValue != nullptr
    */
    nn::Result GetBit(bool* outValue, uint32_t index) NN_NOEXCEPT
    {
        return m_Bitmap->GetBit(outValue, index);
    }

protected:
    std::unique_ptr<nn::fssystem::dbm::Bitmap> m_Bitmap;

private:
    nnt::fs::util::SafeMemoryStorage m_Storage;
    int64_t m_Offset;
};

INSTANTIATE_TEST_CASE_P(
    BitmapTest,
    BitmapAPITest,
    ::testing::ValuesIn(MakeBitmapTestParam())
);

// 1 ビットのビットマップを反転した時の挙動を確認します。
TEST_P(BitmapAPITest, ReverseOneBit)
{
    // 1 ビット分のビットマップとして初期化します。
    SetupBitmap(1);

    uint32_t value;
    bool bit;

    // 初期値は 0 であることを確認します。
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 0));
    ASSERT_EQ(0x00000000, value);
    NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, 0));
    ASSERT_FALSE(bit);

    // 範囲外にはアクセスできないことを確認します。
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultOutOfRange, GetBit(&bit, 1));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultOutOfRange, GetBit(&bit, 10));

    // 1 ビットを書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(0, 1));

    // ビットの反転に成功していることを確認します。
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 0));
    ASSERT_EQ(0x80000000, value);
    NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, 0));
    ASSERT_TRUE(bit);

    // 範囲外のビットは書き換えられません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultOutOfRange,
        m_Bitmap->Reverse(2, 1)
    );

    // 範囲外のビットを書き換えようとしても値が書き換わってないことを確認します。
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 0));
    ASSERT_EQ(0x80000000, value);
    NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, 0));
    ASSERT_TRUE(bit);
}

// 1 バイトのビットマップを反転した時の挙動を確認します。
TEST_P(BitmapAPITest, ReverseOneByte)
{
    // 1 バイト分のビットマップとして初期化します。
    static const uint32_t BitCount = 8;
    SetupBitmap(BitCount);

    uint32_t value;
    bool bit;

    // 初期値は 0 であることを確認します。
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 0));
    ASSERT_EQ(0x00000000, value);
    for( int index = 0; index < BitCount; ++index )
    {
        NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, index));
        ASSERT_FALSE(bit);
    }

    // 範囲外にはアクセスできないことを確認します。
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultOutOfRange, GetBit(&bit, 9));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultOutOfRange, GetBit(&bit, 10));

    // ビットを書き換えます。
    // 変更前: 00000000
    // 変更後: 00011000
    static const uint32_t ReverseIndex = 3;
    static const uint32_t ReverseCount = 2;
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(ReverseIndex, ReverseCount));

    // 書き換えたビットのみ反転していることを確認します。
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 0));
    ASSERT_EQ(0x18000000, value);
    for( int index = 0; index < 8; ++index )
    {
        NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, index));
        if( (ReverseIndex <= index) && (index < ReverseIndex + ReverseCount) )
        {
            ASSERT_TRUE(bit);
        }
        else
        {
            ASSERT_FALSE(bit);
        }
    }

    // ビットを書き換えます。
    // 変更前: 00011000
    // 変更後: 10011000
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(0, 1));

    // 書き換えた状態を確認します。
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 0));
    ASSERT_EQ(0x98000000, value);
    NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, 0));
    ASSERT_TRUE(bit);
    NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, 1));
    ASSERT_FALSE(bit);

    // ビットを書き換えます。
    // 変更前: 10011000
    // 変更後: 10010110
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(4, 3));

    // 書き換えた状態を確認します。
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 0));
    ASSERT_EQ(0x96000000, value);
    NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, 3));
    ASSERT_TRUE(bit);
    NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, 4));
    ASSERT_FALSE(bit);

    // 範囲外のビットは書き換えられません。
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultOutOfRange, m_Bitmap->Reverse(9, 3));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultOutOfRange, m_Bitmap->Reverse(34, 1));

    // 範囲外のビットを書き換えようとしても値が書き換わってないことを確認します。
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 0));
    ASSERT_EQ(0x96000000, value);
}

// 16 バイトのビットマップを反転した時の挙動を確認します。
TEST_P(BitmapAPITest, Reverse16Byte)
{
    // 16 バイト分のビットマップとして初期化します。
    static const uint32_t BitCount = 8 * 16;
    SetupBitmap(BitCount);

    uint32_t value;
    bool bit;

    // 初期値は 0 であることを確認します。
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 0));
    ASSERT_EQ(0, value);
    for( int index = 0; index < BitCount; ++index )
    {
        NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, index));
        ASSERT_FALSE(bit);
    }

    // 範囲外にはアクセスできないことを確認します。
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultOutOfRange, GetBit(&bit, BitCount));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultOutOfRange, GetBit(&bit, 1024));

    // 先頭 4 バイト範囲内のビットを書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(3, 2));

    // 書き換えた状態を確認します。
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 0));
    ASSERT_EQ(0x18000000, value);
    NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, 0));
    ASSERT_FALSE(bit);
    NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, 3));
    ASSERT_TRUE(bit);
    NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, 4));
    ASSERT_TRUE(bit);

    // 2 番目以降の 4 バイトは初期値であることを確認します。
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 32));
    ASSERT_EQ(0x00000000, value);
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 64));
    ASSERT_EQ(0x00000000, value);
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 96));
    ASSERT_EQ(0x00000000, value);
    for( int index = 32; index < 96; ++index )
    {
        NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, index));
        ASSERT_FALSE(bit);
    }

    // 2 番目の 4 バイト範囲内のビットを書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(32, 1));

    // 書き換えた状態を確認します。
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 0));
    ASSERT_EQ(0x18000000, value);
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 32));
    ASSERT_EQ(0x80000000, value);
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 64));
    ASSERT_EQ(0x00000000, value);
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 96));
    ASSERT_EQ(0x00000000, value);
    NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, 32));
    ASSERT_TRUE(bit);
    NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, 33));
    ASSERT_FALSE(bit);

    // 2 番目と 3番目の 4 バイトを跨ぐ範囲を書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(60, 10));

    // 書き換えた状態を確認します。
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 0));
    ASSERT_EQ(0x18000000, value);
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 32));
    ASSERT_EQ(0x8000000F, value);
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 64));
    ASSERT_EQ(0xFC000000, value);
    NNT_ASSERT_RESULT_SUCCESS(GetBitmap32(&value, 96));
    ASSERT_EQ(0x00000000, value);

    // 範囲外のビットは書き換えられません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultOutOfRange,
        m_Bitmap->Reverse(1000, 4)
    );
}

// 1 ビットのビットマップのイテレーションテストを行います。
TEST_P(BitmapAPITest, IterateOneBit)
{
    // 1 ビット分のビットマップとして初期化します。
    SetupBitmap(1);

    uint32_t oneCount = 0;
    uint32_t zeroCount = 0;

    // ビットマップに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 先頭に 0 が 1 個だけあることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(1, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // 1 ビットを書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(0, 1));

    // ビットマップに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 先頭に 1 が 1 個だけあることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(1, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
}

// 1 バイトのビットマップのイテレーションテストを行います。
TEST_P(BitmapAPITest, IterateOneByte)
{
    // 1 バイト分のビットマップとして初期化します。
    SetupBitmap(8);

    uint32_t zeroCount = 0;
    uint32_t oneCount = 0;

    // ビットマップに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 0 が 1 バイト分連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(8, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // ビットを書き換えます。
    // 変更前: 00000000
    // 変更後: 00011000
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(3, 2));

    // 先頭からイテレーションを行います。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 0 が 3 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(3, zeroCount);
        ASSERT_EQ(0, oneCount);

        // 1 が 2 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(2, oneCount);

        // 0 が 3 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(3, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // 途中からイテレーションを行います。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 2));

        // 0 が 1 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(1, zeroCount);
        ASSERT_EQ(0, oneCount);

        // 1 が 2 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(2, oneCount);

        // 0 が 3 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(3, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
}

// 4 バイト以上かつ中途半端な長さのビットマップに対するイテレーションのテスト
TEST_P(BitmapAPITest, Iterate19Bit)
{
    // 19 ビット分のビットマップとして初期化します。
    SetupBitmap(19);

    uint32_t zeroCount = 0;
    uint32_t oneCount = 0;

    // ビットマップに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 0 が 19 ビット分連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(19, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // ビットを書き換えます。
    // 変更前: 0000000000000000000
    // 変更後: 0000011111111100000
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(5, 9));

    // 先頭からイテレーションを行います。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 0 が 5 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(zeroCount, 5);
        ASSERT_EQ(oneCount, 0);

        // 1 が 9 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(zeroCount, 0);
        ASSERT_EQ(oneCount, 9);

        // 0 が 5 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(zeroCount, 5);
        ASSERT_EQ(oneCount, 0);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(zeroCount, 0);
        ASSERT_EQ(oneCount, 0);
    }

    // 途中からイテレーションを行います。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 6));

        // 1 が 8 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(zeroCount, 0);
        ASSERT_EQ(oneCount, 8);

        // 0 が 5 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(zeroCount, 5);
        ASSERT_EQ(oneCount, 0);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(zeroCount, 0);
        ASSERT_EQ(oneCount, 0);
    }
}

namespace {

    /**
    * @brief        読み込むサイズを制限したイテレーションで連続した 0 または 1 を読み込みます。
    *
    * @param[in]    pBitmap     ビットマップ
    * @param[in]    pIter       イテレータ
    * @param[in]    isReadZero  連続した 0 を読み込むなら true, 1 を読み込むなら false
    * @param[in]    count       0 または 1 が連続する回数
    * @param[in]    countMax    0 または 1 を読み込む最大回数
    *
    * @pre
    *               - pBitmap != nullptr
    *               - pIter != nullptr
    */
    void LimitedIterateTest(
        nn::fssystem::dbm::Bitmap* pBitmap,
        nn::fssystem::dbm::Bitmap::Iterator* pIter,
        bool isReadZero,
        uint32_t count,
        uint32_t countMax
    ) NN_NOEXCEPT
    {
        ASSERT_NE(nullptr, pBitmap);
        ASSERT_NE(nullptr, pIter);

        for( uint32_t read = 0; read < count; )
        {
            uint32_t zeroCount = 0;
            uint32_t oneCount = 0;
            if( isReadZero )
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pBitmap->LimitedIterateNext(&zeroCount, &oneCount, pIter, countMax, 0)
                );
                ASSERT_EQ(std::min(countMax, count - read), zeroCount);
                ASSERT_EQ(0, oneCount);
                read += zeroCount;
            }
            else
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    pBitmap->LimitedIterateNext(&zeroCount, &oneCount, pIter, 0, countMax)
                );
                ASSERT_EQ(0, zeroCount);
                ASSERT_EQ(std::min(countMax, count - read), oneCount);
                read += oneCount;
            }
        }
    }
}

// 1 ビットのビットマップの読み込むサイズを制限したイテレーションテストを行います。
TEST_P(BitmapAPITest, LimitedIterateOneBit)
{
    // 1 ビット分のビットマップとして初期化します。
    SetupBitmap(1);

    uint32_t oneCount = 0;
    uint32_t zeroCount = 0;

    // ビットマップに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 先頭に 0 が 1 個だけあることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 1, 1);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // 1 ビットを書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(0, 1));

    // ビットマップに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 先頭に 1 が 1 個だけあることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, false, 1, 1);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
}

// 1 バイトのビットマップの読み込むサイズを制限したイテレーションテストを行います。
TEST_P(BitmapAPITest, LimitedIterateOneByte)
{
    // 1 バイト分のビットマップとして初期化します。
    SetupBitmap(8);

    uint32_t zeroCount = 0;
    uint32_t oneCount = 0;

    for( uint32_t countMax = 1; countMax <= 8; ++countMax )
    {
        // ビットマップに対するイテレーションをテストします。
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 0 が 1 バイト分連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 8, countMax);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // ビットを書き換えます。
    // 変更前: 00000000
    // 変更後: 00011000
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(3, 2));

    // 先頭からイテレーションを行います。
    for( uint32_t countMax = 1; countMax <= 8; ++countMax )
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 0 が 3 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 3, countMax);

        // 1 が 2 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, false, 2, countMax);

        // 0 が 3 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 3, countMax);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(
            m_Bitmap->LimitedIterateNext(&zeroCount, &oneCount, &iter, countMax, countMax));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // 途中からイテレーションを行います。
    for( uint32_t countMax = 1; countMax <= 8; ++countMax )
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 2));

        // 0 が 1 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 1, countMax);

        // 1 が 2 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, false, 2, countMax);

        // 0 が 3 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 3, countMax);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
}

// 4 バイト以上かつ中途半端な長さのビットマップに対する読み込むサイズを制限したイテレーションのテスト
TEST_P(BitmapAPITest, LimitedIterate19Bit)
{
    // 19 ビット分のビットマップとして初期化します。
    SetupBitmap(19);

    uint32_t zeroCount = 0;
    uint32_t oneCount = 0;

    // ビットマップに対するイテレーションをテストします。
    for( uint32_t countMax = 1; countMax <= 19; ++countMax )
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 0 が 19 ビット分連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 19, countMax);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // ビットを書き換えます。
    // 変更前: 0000000000000000000
    // 変更後: 0000011111111100000
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(5, 9));

    // 先頭からイテレーションを行います。
    for( uint32_t countMax = 1; countMax <= 19; ++countMax )
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 0 が 5 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 5, countMax);

        // 1 が 9 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, false, 9, countMax);

        // 0 が 5 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 5, countMax);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(zeroCount, 0);
        ASSERT_EQ(oneCount, 0);
    }

    // 途中からイテレーションを行います。
    for( uint32_t countMax = 1; countMax <= 19; ++countMax )
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 6));

        // 1 が 8 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, false, 8, countMax);

        // 0 が 5 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 5, countMax);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(zeroCount, 0);
        ASSERT_EQ(oneCount, 0);
    }
}

// 1 ビットのビットマップに対するクリア
TEST_P(BitmapAPITest, ClearOneBit)
{
    SetupBitmap(1);

    bool bit;

    // ビットが 0 であることを確認します。
    NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, 0));
    ASSERT_FALSE(bit);

    // ビットを書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(0, 1));

    // ビットが 1 になったことを確認します。
    NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, 0));
    ASSERT_TRUE(bit);

    // ビットをクリアします。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Clear());

    // ビットが 0 であることを確認します。
    NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, 0));
    ASSERT_FALSE(bit);
}

// 4 バイト以上かつ中途半端な長さのビットマップに対するクリア
TEST_P(BitmapAPITest, Clear19bit)
{
    static const uint32_t BitCount = 19;
    SetupBitmap(BitCount);

    bool bit;

    // 全てのビットが 0 であることを確認します。
    for( int index = 0; index < BitCount; ++index )
    {
        NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, index));
        ASSERT_FALSE(bit);
    }

    // ビットを書き換えます。
    // 変更前: 0000000000000000000
    // 変更後: 0000011111111100000
    static const uint32_t ReverseIndex = 5;
    static const uint32_t ReverseCount = 9;
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(ReverseIndex, ReverseCount));

    // 判定した領域内のビットが 1 であることを確認します。
    for( int index = 0; index < BitCount; ++index )
    {
        NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, index));
        if( (ReverseIndex <= index) && (index < ReverseIndex + ReverseCount) )
        {
            ASSERT_TRUE(bit);
        }
        else
        {
            ASSERT_FALSE(bit);
        }
    }

    // ビットをクリアします。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Clear());

    // 全てのビットが 0 であることを確認します。
    for( int index = 0; index < BitCount; ++index )
    {
        NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, index));
        ASSERT_FALSE(bit);
    }
}

namespace
{
    // 拡張のテスト用のパラメータです
    struct ExpandTestParam
    {
        int64_t offset;       // SubStorage のオフセット
        uint32_t maxBitCount; // 拡張できる最大ビット数
    };

    // 拡張のテスト用パラメータを生成します。
    nnt::fs::util::Vector<ExpandTestParam> MakeExpandTestParam()
    {
        nnt::fs::util::Vector<ExpandTestParam> params;

        // オフセット無し
        {
            ExpandTestParam data;
            data.offset = 0;
            data.maxBitCount = 1000;
            params.push_back(data);
        }

        // 適当なオフセット
        {
            ExpandTestParam data;
            data.offset = 190;
            data.maxBitCount = 1000;
            params.push_back(data);
        }

        return params;
    }
}

// 拡張のテストで Bitmap とそのストレージを初期化、保持するテストフィクスチャです。
class BitmapExpandTest : public ::testing::TestWithParam< ExpandTestParam >
{
protected:
    /**
    *   @brief テスト終了時にストレージの範囲外アクセスが発生していないかテストします。
    */
    virtual void TearDown() NN_OVERRIDE
    {
        ASSERT_TRUE(m_Storage.CheckValid());
    }

protected:
    /**
    * @brief      指定されたビット数で Bitmap とストレージを初期化します。
    *
    * @param[in]  bitCount    Bitmap のビット数
    */
    void SetupBitmap(uint32_t bitCount) NN_NOEXCEPT
    {
        const ExpandTestParam& param = GetParam();
        m_Offset = param.offset;
        m_MaxBitCount = param.maxBitCount;
        m_BitCount = bitCount;

        // パラメータに従ってストレージを初期化します。
        const int64_t sizeMax = nn::fssystem::dbm::Bitmap::QuerySize(m_MaxBitCount);
        const int64_t size = nn::fssystem::dbm::Bitmap::QuerySize(bitCount);
        m_Storage.Initialize(sizeMax + m_Offset);
        nn::fs::SubStorage storage(&m_Storage, m_Offset, size);

        // Bitmap のストレージとしてフォーマットします。
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::dbm::Bitmap::Format(bitCount, storage)
        );

        // Bitmap をマウントします。
        m_Bitmap.reset(new nn::fssystem::dbm::Bitmap());
        m_Bitmap->Initialize(bitCount, storage);
    }

    /**
    *   @brief      Bitmap を拡張します。
    *
    *   @param[in]  bitCount    拡張後のビット数
    */
    void ExpandBitmap(uint32_t bitCount) NN_NOEXCEPT
    {
        // 初期化時にパラメータで設定された最大ビット数を超えない範囲で拡張できます。
        NN_ASSERT(bitCount <= m_MaxBitCount);

        // 今までの Bitmap オブジェクトを破棄し新しく生成します。
        m_Bitmap.reset(new nn::fssystem::dbm::Bitmap());

        // 拡張するビット数に合わせて SubStorage を作ります。
        const int64_t size = nn::fssystem::dbm::Bitmap::QuerySize(bitCount);
        nn::fs::SubStorage storage(&m_Storage, m_Offset, size);

        // ストレージを拡張します。
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::dbm::Bitmap::Expand(m_BitCount, bitCount, storage)
        );
        m_BitCount = bitCount;

        // 拡張したストレージをマウントします。
        m_Bitmap->Initialize(bitCount, storage);
    }

    /**
    * @brief        指定したインデックスから 32 ビット分のビットマップを取得します。
    *
    * @param[out]   outValue    ビットマップ
    * @param[in]    index       インデックス
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outValue != nullptr
    */
    nn::Result GetBitmap32(uint32_t* outValue, uint32_t index) NN_NOEXCEPT
    {
        return m_Bitmap->GetBitmap32(outValue, index);
    }

    /**
    * @brief        指定したインデックスのビットが 0 か 1 かを取得します。
    *
    * @param[out]   outValue    1 ならば true、0 ならば false
    * @param[in]    index       インデックス
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outValue != nullptr
    */
    nn::Result GetBit(bool* outValue, uint32_t index) NN_NOEXCEPT
    {
        return m_Bitmap->GetBit(outValue, index);
    }

protected:
    std::unique_ptr<nn::fssystem::dbm::Bitmap> m_Bitmap;

private:
    nnt::fs::util::SafeMemoryStorage m_Storage;
    int64_t m_Offset;
    uint32_t m_MaxBitCount;
    uint32_t m_BitCount;
};

// 1 ビットのビットマップを拡張しない場合のイテレーションテストを行います。
TEST_P(BitmapExpandTest, NoExpandOneBit)
{
    // 1 ビット分のビットマップとして初期化します。
    SetupBitmap(1);

    uint32_t oneCount = 0;
    uint32_t zeroCount = 0;

    // ビットマップに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 先頭に 0 が 1 個だけあることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(1, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // 1 ビットを書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(0, 1));

    // ビットマップに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 先頭に 1 が 1 個だけあることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(1, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
};

// 1 ビットのビットマップを 2 ビットに拡張するテストを行います。
TEST_P(BitmapExpandTest, IterateOneBit)
{
    // 1 ビット分のビットマップとして初期化します。
    SetupBitmap(1);

    uint32_t oneCount = 0;
    uint32_t zeroCount = 0;

    // 1 ビットを書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(0, 1));

    // 2 ビットのビットマップに拡張します。
    ExpandBitmap(2);

    // ビットマップに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 先頭に 1 が 1 個あることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(1, oneCount);

        // 拡張された領域として、0 が 1 個あることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(1, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // 1 ビットを書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(1, 1));

    // ビットマップに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 先頭に 1 が 2 個だけあることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(2, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
}

// 1 バイトのビットマップを拡張しない場合のイテレーションテストを行います。
TEST_P(BitmapExpandTest, NoExpandOneByte)
{
    // 1 バイト分のビットマップとして初期化します。
    SetupBitmap(8);

    uint32_t zeroCount = 0;
    uint32_t oneCount = 0;

    // ビットマップに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 0 が 1 バイト分連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(8, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // ビットを書き換えます。
    // 変更前: 00000000
    // 変更後: 00011000
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(3, 2));

    // 先頭からイテレーションを行います。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 0 が 3 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(3, zeroCount);
        ASSERT_EQ(0, oneCount);

        // 1 が 2 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(2, oneCount);

        // 0 が 3 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(3, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // 途中からイテレーションを行います。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 2));

        // 0 が 1 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(1, zeroCount);
        ASSERT_EQ(0, oneCount);

        // 1 が 2 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(2, oneCount);

        // 0 が 3 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(3, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
}

// 1 バイトのビットマップを 10 バイトに拡張するテストを行います。
TEST_P(BitmapExpandTest, IterateOneByte)
{
    uint32_t zeroCount = 0;
    uint32_t oneCount = 0;

    // 1 バイト分のビットマップとして初期化します。
    SetupBitmap(8);

    // ビットを書き換えます。
    // 変更前: 00000000
    // 変更後: 00011000
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(3, 2));

    // 10 バイト分のビットマップに拡張します。
    ExpandBitmap(8 * 10);

    // 先頭からイテレーションを行います。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 0 が 3 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(3, zeroCount);
        ASSERT_EQ(0, oneCount);

        // 1 が 2 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(2, oneCount);

        // 0 が 75 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(75, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // 途中からイテレーションを行います。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 2));

        // 0 が 1 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(1, zeroCount);
        ASSERT_EQ(0, oneCount);

        // 1 が 2 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(2, oneCount);

        // 0 が 75 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(75, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // 拡張した領域からイテレーションを開始します。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 22));

        // 0 が 80 - 22 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(80 - 22, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // ビットを書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(40, 10));

    // 先頭からイテレーションを行います。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 0 が 3 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(3, zeroCount);
        ASSERT_EQ(0, oneCount);

        // 1 が 2 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(2, oneCount);

        // 0 が 35 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(35, zeroCount);
        ASSERT_EQ(0, oneCount);

        // 1 が 10 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(10, oneCount);

        // 0 が 30 個連続していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(30, zeroCount);
        ASSERT_EQ(0, oneCount);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
}

// 1 ビットのビットマップを拡張しない場合の読み込むサイズを制限したイテレーションテストを行います。
TEST_P(BitmapExpandTest, LimitedNoExpandOneBit)
{
    // 1 ビット分のビットマップとして初期化します。
    SetupBitmap(1);

    uint32_t oneCount = 0;
    uint32_t zeroCount = 0;

    // ビットマップに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 先頭に 0 が 1 個だけあることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 1, 1);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // 1 ビットを書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(0, 1));

    // ビットマップに対するイテレーションをテストします。
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 先頭に 1 が 1 個だけあることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, false, 1, 1);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
};

// 1 ビットのビットマップを 2 ビットに拡張して読み込むサイズを制限したイテレーションテストを行います。
TEST_P(BitmapExpandTest, LimitedIterateOneBit)
{
    // 1 ビット分のビットマップとして初期化します。
    SetupBitmap(1);

    uint32_t oneCount = 0;
    uint32_t zeroCount = 0;

    // 1 ビットを書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(0, 1));

    // 2 ビットのビットマップに拡張します。
    ExpandBitmap(2);

    // ビットマップに対するイテレーションをテストします。
    for( uint32_t countMax = 1; countMax <= 2; ++countMax )
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 先頭に 1 が 1 個あることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, false, 1, countMax);

        // 拡張された領域として、0 が 1 個あることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 1, countMax);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // 1 ビットを書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(1, 1));

    // ビットマップに対するイテレーションをテストします。
    for( uint32_t countMax = 1; countMax <= 2; ++countMax )
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 先頭に 1 が 2 個だけあることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, false, 2, countMax);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
}

// 1 バイトのビットマップを拡張しない場合の読み込むサイズを制限したイテレーションテストを行います。
TEST_P(BitmapExpandTest, LimitedNoExpandOneByte)
{
    // 1 バイト分のビットマップとして初期化します。
    SetupBitmap(8);

    uint32_t zeroCount = 0;
    uint32_t oneCount = 0;

    // ビットマップに対するイテレーションをテストします。
    for( uint32_t countMax = 1; countMax <= 8; ++countMax )
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 0 が 1 バイト分連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 8, countMax);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // ビットを書き換えます。
    // 変更前: 00000000
    // 変更後: 00011000
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(3, 2));

    // 先頭からイテレーションを行います。
    for( uint32_t countMax = 1; countMax <= 8; ++countMax )
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 0 が 3 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 3, countMax);

        // 1 が 2 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, false, 2, countMax);

        // 0 が 3 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 3, countMax);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // 途中からイテレーションを行います。
    for( uint32_t countMax = 1; countMax <= 8; ++countMax )
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 2));

        // 0 が 1 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 1, countMax);

        // 1 が 2 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, false, 2, countMax);

        // 0 が 3 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 3, countMax);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
}

// 1 バイトのビットマップを 10 バイトに拡張して読み込むサイズを制限したイテレーションテストを行います。
TEST_P(BitmapExpandTest, LimitedIterateOneByte)
{
    uint32_t zeroCount = 0;
    uint32_t oneCount = 0;

    // 1 バイト分のビットマップとして初期化します。
    SetupBitmap(8);

    // ビットを書き換えます。
    // 変更前: 00000000
    // 変更後: 00011000
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(3, 2));

    // 10 バイト分のビットマップに拡張します。
    ExpandBitmap(8 * 10);

    // 先頭からイテレーションを行います。
    for( uint32_t countMax = 1; countMax <= 80; ++countMax )
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 0 が 3 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 3, countMax);

        // 1 が 2 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, false, 2, countMax);

        // 0 が 75 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 75, countMax);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // 途中からイテレーションを行います。
    for( uint32_t countMax = 1; countMax <= 80; ++countMax )
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 2));

        // 0 が 1 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 1, countMax);

        // 1 が 2 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, false, 2, countMax);

        // 0 が 75 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 75, countMax);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // 拡張した領域からイテレーションを開始します。
    for( uint32_t countMax = 1; countMax <= 80; ++countMax )
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 22));

        // 0 が 80 - 22 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 80 - 22, countMax);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }

    // ビットを書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->Reverse(40, 10));

    // 先頭からイテレーションを行います。
    for( uint32_t countMax = 1; countMax <= 80; ++countMax )
    {
        nn::fssystem::dbm::Bitmap::Iterator iter;

        // イテレータを初期化します。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateBegin(&iter, 0));

        // 0 が 3 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 3, countMax);

        // 1 が 2 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, false, 2, countMax);

        // 0 が 35 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 35, countMax);

        // 1 が 10 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, false, 10, countMax);

        // 0 が 30 個連続していることをチェックします。
        LimitedIterateTest(m_Bitmap.get(), &iter, true, 30, countMax);

        // イテレーションが完了していることをチェックします。
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap->IterateNext(&zeroCount, &oneCount, &iter));
        ASSERT_EQ(0, zeroCount);
        ASSERT_EQ(0, oneCount);
    }
}

INSTANTIATE_TEST_CASE_P(
    BitmapTest,
    BitmapExpandTest,
    ::testing::ValuesIn(MakeExpandTestParam())
);

// ランダムテストで Bitmap とそれが使うストレージを初期化、保持するテストフィクスチャです。
class BitmapAgingTest : public ::testing::TestWithParam< uint32_t >
{
protected:
    /**
    * @brief  指定したビット数で Bitmap とストレージを初期化します。
    */
    void SetupBitmap(uint32_t bitCount) NN_NOEXCEPT
    {
        // オフセット 0 でストレージを初期化します。
        m_Offset = 0;
        const int64_t size = nn::fssystem::dbm::Bitmap::QuerySize(bitCount);
        m_Storage.Initialize(size + m_Offset);
        nn::fs::SubStorage storage(&m_Storage, m_Offset, size);

        // Bitmap のストレージとしてフォーマットします。
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::dbm::Bitmap::Format(bitCount, storage)
        );

        // Bitmap をマウントします。
        m_Bitmap.Initialize(bitCount, storage);
    }

    /**
    * @brief        指定したインデックスから 32 ビット分のビットマップを取得します。
    *
    * @param[out]   outValue    ビットマップ
    * @param[in]    index       インデックス
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outValue != nullptr
    */
    nn::Result GetBitmap32(uint32_t* outValue, uint32_t index) NN_NOEXCEPT
    {
        return m_Bitmap.GetBitmap32(outValue, index);
    }

    /**
    * @brief        指定したインデックスのビットが 0 か 1 かを取得します。
    *
    * @param[out]   outValue    1 ならば true、0 ならば false
    * @param[in]    index       インデックス
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outValue != nullptr
    */
    nn::Result GetBit(bool* outValue, uint32_t index) NN_NOEXCEPT
    {
        return m_Bitmap.GetBit(outValue, index);
    }

protected:
    nn::fssystem::dbm::Bitmap m_Bitmap;

private:
    nnt::fs::util::SafeMemoryStorage m_Storage;
    int64_t m_Offset;
};

// ランダムな箇所を反転させ、反転した範囲のビットのみが
// 書き換わっていることをチェックします。
TEST_P(BitmapAgingTest, RandomReverse)
{
    // パラメータのビット数で Bitmap を初期化します。
    const uint32_t bitCount = GetParam();
    SetupBitmap(bitCount);

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int loop = 0; loop <= 100; ++loop )
    {
        // すべてのビットを 0 クリアしておきます。
        m_Bitmap.Clear();

        // 0 クリアされていることを確認します。
        for( uint32_t index = 0; index < bitCount; ++index )
        {
            bool bit;
            NNT_ASSERT_RESULT_SUCCESS(GetBit(&bit, index));
            ASSERT_FALSE(bit);
        }

        // 反転させた範囲を保存する構造体と配列です。
        struct ReverseInfo
        {
            uint32_t offset;
            uint32_t size;
        };
        nnt::fs::util::Vector<ReverseInfo> reverseInfoArray;

        // ランダムに複数箇所反転します。
        static const int ReverseCountMax = 10;
        for( int reverseCount = 0; reverseCount < ReverseCountMax; ++reverseCount )
        {
            // ビットを反転する範囲を乱数で決定します。
            const uint32_t offset = std::uniform_int_distribution<uint32_t>(0, bitCount - 2)(mt);
            const uint32_t size
                = std::uniform_int_distribution<uint32_t>(1, bitCount - offset - 1)(mt);

            // 既に反転させた範囲と重なっていたらビットの反転を終了します。
            bool conflict = false;
            for( const auto reverseInfo : reverseInfoArray )
            {
                if( (reverseInfo.offset <= offset)
                    && (offset < reverseInfo.offset + reverseInfo.size))
                {
                    conflict = true;
                    break;
                }
                if( (offset <= reverseInfo.offset)
                    && (reverseInfo.offset < offset + size))
                {
                    conflict = true;
                    break;
                }
            }
            if( conflict )
            {
                break;
            }

            // ビットを反転します。
            ReverseInfo reverseInfo;
            reverseInfo.offset = offset;
            reverseInfo.size = size;
            reverseInfoArray.push_back(reverseInfo);
            NNT_ASSERT_RESULT_SUCCESS(m_Bitmap.Reverse(offset, size));

            // ビットの反転に成功したことを確認します。
            for( uint32_t index = 0; index < bitCount; ++index )
            {
                if( (offset <= index) && (index < offset + size) )
                {
                    bool bitCheck;
                    GetBit(&bitCheck, index);
                    ASSERT_TRUE(bitCheck);
                }
            }
        }

        // 反転させた範囲のみが書き換わっていることを確認します。
        for( uint32_t bitIndex = 0; bitIndex < bitCount; ++bitIndex )
        {
            // bitIndex が反転させた範囲内か調べます。
            bool isInsideReverse = false;
            for( const auto reverseInfo : reverseInfoArray )
            {
                if( (reverseInfo.offset <= bitIndex)
                    && (bitIndex < reverseInfo.offset + reverseInfo.size) )
                {
                    // 範囲内の値です。
                    isInsideReverse = true;
                    break;
                }
            }

            // 反転範囲ならビットが立っています。
            bool bitCheck;
            GetBit(&bitCheck, bitIndex);
            ASSERT_EQ(isInsideReverse, bitCheck);
        }
    }
}

// ランダムな箇所を反転させ、反転後にイテレーションが正常に出来ることをチェックします。
TEST_P(BitmapAgingTest, RandomReverseAndIterate)
{
    const uint32_t bitCount = GetParam();
    SetupBitmap(bitCount);

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int loop = 0; loop <= 10; ++loop )
    {
        // ランダムに複数箇所反転します。
        for( int reverseCount = 0; reverseCount < 3; ++reverseCount )
        {
            // ビットを反転する範囲を乱数で決定します。
            uint32_t offset = std::uniform_int_distribution<uint32_t>(0, bitCount - 2)(mt);
            uint32_t size = std::uniform_int_distribution<uint32_t>(1, bitCount - offset - 1)(mt);
            NNT_ASSERT_RESULT_SUCCESS(m_Bitmap.Reverse(offset, size));
        }

        // ビットマップに対するイテレーションを開始します。
        // 0 と 1 の出現回数の合計が全体のビット数と一致するか確かめます。
        uint32_t totalCount = 0;
        nn::fssystem::dbm::Bitmap::Iterator iter;
        NNT_ASSERT_RESULT_SUCCESS(m_Bitmap.IterateBegin(&iter, 0));
        for( ; ; )
        {
            uint32_t oneCount = 0;
            uint32_t zeroCount = 0;
            NNT_ASSERT_RESULT_SUCCESS(m_Bitmap.IterateNext(&zeroCount, &oneCount, &iter));
            if( (oneCount == 0) && (zeroCount == 0) )
            {
                break;
            }
            totalCount += zeroCount;
            totalCount += oneCount;
        }
        ASSERT_EQ(bitCount, totalCount);
    }
}

// ランダムテストで使う、Bitmap のビット数のパラメータです。
static const uint32_t SizeTableAging[] = {
    32, 1023, 1024, 1025, 4 * 1024
};

INSTANTIATE_TEST_CASE_P(
    BitmapTest,
    BitmapAgingTest,
    ::testing::ValuesIn(SizeTableAging)
);

