﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "testFs_Stress_TestContext.h"

class StressTestMountInfoBase
{
public:
    StressTestMountInfoBase() NN_NOEXCEPT
    : m_Mutex(false)
    {
    }

    virtual ~StressTestMountInfoBase() NN_NOEXCEPT
    {
    }

    //! マウント名を取得
    const char* GetMountName() const NN_NOEXCEPT
    {
        return m_MountName.c_str();
    }

    //! マウント名を設定
    void SetMountName(const char* name) NN_NOEXCEPT
    {
        m_MountName = name;
    }

    //! リードオンリーかどうかを取得
    virtual bool IsReadOnly() const NN_NOEXCEPT
    {
        return false;
    }

    //! セーブデータかどうかを取得 (コミット操作判定で使用)
    virtual bool IsSaveData() const NN_NOEXCEPT
    {
        return false;
    }

    //! アプリケーションIDを取得
    virtual uint64_t GetApplicationId() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(false);
        static const uint64_t ApplicationId = 0;
        return ApplicationId;
    }

    //! ユーザーIDを取得
    virtual nn::fs::UserId GetUserId() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(false);
        static const nn::fs::UserId UserId = { { 0, 0 } };
        return UserId;
    }

    //! FAT への直接アクセスが発生するかどうかを取得
    virtual bool IsAccessesFatDirectly() const NN_NOEXCEPT
    {
        return false;
    }

    void SetUpTestDirectory() NN_NOEXCEPT
    {
        if( !IsReadOnly() )
        {
            nn::time::PosixTime posixTime;
            nn::time::CalendarTime calenderTime;
            char path[nn::fs::EntryNameLengthMax];

            for( ; ; )
            {
                NNT_ASSERT_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime));
                NNT_ASSERT_RESULT_SUCCESS(nn::time::ToCalendarTime(&calenderTime, nullptr, posixTime));
                nn::util::SNPrintf(
                    m_TestDirectoryPath,
                    sizeof(m_TestDirectoryPath),
                    "/%02d%02d%02d%02d%02d%02d",
                    calenderTime.year % 100,
                    calenderTime.month,
                    calenderTime.day,
                    calenderTime.hour,
                    calenderTime.minute,
                    calenderTime.second
                );

                nn::util::SNPrintf(path, sizeof(path), "%s:%s", GetMountName(), m_TestDirectoryPath);
                const auto result = nn::fs::CreateDirectory(path);
                if( nn::fs::ResultPathAlreadyExists::Includes(result) )
                {
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                    continue;
                }
                NNT_ASSERT_RESULT_SUCCESS(result);

                break;
            }
        }
    }

    void TearDownTestDirectory() NN_NOEXCEPT
    {
        char path[nn::fs::EntryNameLengthMax];
        nn::util::SNPrintf(path, sizeof(path), "%s:%s", GetMountName(), m_TestDirectoryPath);
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(path));
    }

    const char* GetTestDirectoryName() const NN_NOEXCEPT
    {
        return m_TestDirectoryPath;
    }

    nn::os::Mutex& GetMutex() NN_NOEXCEPT
    {
        return m_Mutex;
    }

private:
    nn::os::Mutex m_Mutex;
    nnt::fs::util::String m_MountName;
    char m_TestDirectoryPath[nn::fs::EntryNameLengthMax];
};

