﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testFs_Stress_StressTest.h"

namespace {

    char* CreateThreadUniqueDirectoryName(const StressTestMountInfoBase* pTest, int threadIndex)
    {
        NN_ASSERT(pTest != nullptr);
        char* path = new char[nn::fs::EntryNameLengthMax];
        NN_ASSERT(path != nullptr);
        nn::util::SNPrintf(
            path,
            nn::fs::EntryNameLengthMax,
            "%s:/dir%03d",
            pTest->GetMountName(),
            threadIndex
        );
        return path;
    }

    char* CreateThreadUniqueDirectoryName(const StressTestMountInfoBase* pTest, int threadIndex, int entryIndex)
    {
        NN_ASSERT(pTest != nullptr);
        char* path = new char[nn::fs::EntryNameLengthMax];
        NN_ASSERT(path != nullptr);
        nn::util::SNPrintf(
            path,
            nn::fs::EntryNameLengthMax,
            "%s:/dir%03d_%d",
            pTest->GetMountName(),
            threadIndex,
            entryIndex
        );
        return path;
    }

}

// 複数スレッドからそれぞれ異なるディレクトリをオープン、クローズします。
class TestOpenUniqueDirectory : public TestRunner
{
public:
    virtual void RunTest(TestContext* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        std::unique_ptr<char[]> path(
            CreateThreadUniqueDirectoryName(
                pTest->GetTarget(),
                threadIndex
            )
        );

        nn::fs::SetPriorityRawOnCurrentThread(pTest->GetPriority());

        pTest->Fail();

        nn::fs::DirectoryHandle directory;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::fs::OpenDirectory(
                &directory,
                path.get(),
                nn::fs::OpenDirectoryMode_All
            )
        );
        nn::fs::CloseDirectory(directory);

        pTest->Succeed();
    }
};

namespace {

    auto runOpenUniqueDirectory = [](const nnt::fs::util::Vector<StressTestMountInfoBase*>& list) NN_NOEXCEPT
    {
        TestOpenUniqueDirectory test;

        const auto ThreadCount = StressTestParam::ThreadCountMax;
        int indexTable[ThreadCount];

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());
        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            indexTable[threadIndex] = std::uniform_int_distribution<int>(0, static_cast<int>(list.size()) - 1)(mt);
        }

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            if( !list[indexTable[threadIndex]]->IsReadOnly() )
            {
                std::unique_ptr<char[]> path(
                    CreateThreadUniqueDirectoryName(
                        list[indexTable[threadIndex]],
                        threadIndex
                    )
                );
                (void)nn::fs::DeleteDirectoryRecursively(path.get());
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory(path.get()));
            }
        }
        NN_UTIL_SCOPE_EXIT
        {
            for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
            {
                if( !list[indexTable[threadIndex]]->IsReadOnly() )
                {
                    std::unique_ptr<char[]> path(
                        CreateThreadUniqueDirectoryName(
                            list[indexTable[threadIndex]],
                            threadIndex
                        )
                    );
                    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(path.get()));
                }
            }
        };

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            auto pContext = new TestContext(list[indexTable[threadIndex]]);
            pContext->SetPriorityRandom();
            test.AddContext(pContext);
        }
        test.RunThreadTest();
    };

}

TEST(BisStressTest, OpenUniqueDirectory)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountBis fs;
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenUniqueDirectory(list);
}

TEST(HostFsStressTest, OpenUniqueDirectory)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("host");
    StressTestMountHostFs fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenUniqueDirectory(list);
}

TEST(RomFsStressTest, OpenUniqueDirectory)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("rom");
    StressTestMountRomFs fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenUniqueDirectory(list);
}

TEST(SaveDataFsStressTest, OpenUniqueDirectory)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("save");
    StressTestMountSaveData fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenUniqueDirectory(list);
}

TEST(SdCardStressTest, OpenUniqueDirectory)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("sd");
    StressTestMountSdCard fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenUniqueDirectory(list);
}

TEST(MultiFsStressTest, OpenUniqueDirectory)
{
    nnt::fs::util::Vector<StressTestMountInfoBase *> list;

    StressTestMountBis fsBis;
    {
        fsBis.Mount();
        list.push_back(&fsBis);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsBis.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountCacheStorage fsCache;
    {
        const nnt::fs::util::String mountName("cache");
        fsCache.SetMountName(mountName.c_str());
        fsCache.Mount();
        list.push_back(&fsCache);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsCache.Unmount();
    };
#endif

    StressTestMountRomFs fsRom;
    {
        const nnt::fs::util::String mountName("rom");
        fsRom.SetMountName(mountName.c_str());
        fsRom.Mount();
        list.push_back(&fsRom);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsRom.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountOtherSaveData fsOther;
    {
        const nnt::fs::util::String mountName("other");
        fsOther.SetMountName(mountName.c_str());
        fsOther.Mount();
        list.push_back(&fsOther);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsOther.Unmount();
    };
#endif

    StressTestMountSaveData fsSave;
    {
        const nnt::fs::util::String mountName("save");
        fsSave.SetMountName(mountName.c_str());
        fsSave.Mount();
        list.push_back(&fsSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSave.Unmount();
    };

    StressTestMountSdCard fsSdCard;
    {
        const nnt::fs::util::String mountName("sd");
        fsSdCard.SetMountName(mountName.c_str());
        fsSdCard.Mount();
        list.push_back(&fsSdCard);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSdCard.Unmount();
    };

    runOpenUniqueDirectory(list);
}

// 複数スレッドからそれぞれディレクトリを作成、削除します。
class TestCreateAndDeleteDirectory : public TestRunner
{
public:
    virtual void RunTest(TestContext* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fs::SetPriorityRawOnCurrentThread(pTest->GetPriority());

        pTest->Fail();

        for( auto entryIndex = 0; entryIndex < EntryCountMax; ++entryIndex )
        {
            std::unique_ptr<char[]> path(
                CreateThreadUniqueDirectoryName(
                    pTest->GetTarget(),
                    threadIndex,
                    entryIndex
                )
            );
            (void)nn::fs::DeleteDirectoryRecursively(path.get());
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory(path.get()));
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectory(path.get()));
        }

        pTest->Succeed();
    }

    static const int EntryCountMax = 20;
};

namespace {

    auto runCreateAndDeleteDirectory = [](const nnt::fs::util::Vector<StressTestMountInfoBase*>& list) NN_NOEXCEPT
    {
        TestCreateAndDeleteDirectory test;

        const auto ThreadCount = StressTestParam::ThreadCountMax;
        int indexTable[ThreadCount];

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());
        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            indexTable[threadIndex] = std::uniform_int_distribution<int>(0, static_cast<int>(list.size()) - 1)(mt);
        }

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            auto pContext = new TestContext(list[indexTable[threadIndex]]);
            pContext->SetPriorityRandom();
            test.AddContext(pContext);
        }
        test.RunThreadTest();
    };

}

TEST(BisStressTest, CreateAndDeleteDirectory)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountBis fs;
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateAndDeleteDirectory(list);
}

TEST(HostFsStressTest, CreateAndDeleteDirectory)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("host");
    StressTestMountHostFs fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateAndDeleteDirectory(list);
}

TEST(SaveDataFsStressTest, CreateAndDeleteDirectory)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("save");
    StressTestMountSaveData fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateAndDeleteDirectory(list);
}

TEST(SdCardStressTest, CreateAndDeleteDirectory)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("sd");
    StressTestMountSdCard fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateAndDeleteDirectory(list);
}

TEST(MultiFsStressTest, CreateAndDeleteDirectory)
{
    nnt::fs::util::Vector<StressTestMountInfoBase *> list;

    StressTestMountBis fsBis;
    {
        fsBis.Mount();
        list.push_back(&fsBis);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsBis.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountCacheStorage fsCache;
    {
        const nnt::fs::util::String mountName("cache");
        fsCache.SetMountName(mountName.c_str());
        fsCache.Mount();
        list.push_back(&fsCache);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsCache.Unmount();
    };
#endif

    StressTestMountHostFs fsHost;
    {
        const nnt::fs::util::String mountName("host");
        fsHost.SetMountName(mountName.c_str());
        fsHost.Mount();
        list.push_back(&fsHost);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsHost.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountOtherSaveData fsOther;
    {
        const nnt::fs::util::String mountName("other");
        fsOther.SetMountName(mountName.c_str());
        fsOther.Mount();
        list.push_back(&fsOther);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsOther.Unmount();
    };
#endif

    StressTestMountSaveData fsSave;
    {
        const nnt::fs::util::String mountName("save");
        fsSave.SetMountName(mountName.c_str());
        fsSave.Mount();
        list.push_back(&fsSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSave.Unmount();
    };

    StressTestMountSdCard fsSdCard;
    {
        const nnt::fs::util::String mountName("sd");
        fsSdCard.SetMountName(mountName.c_str());
        fsSdCard.Mount();
        list.push_back(&fsSdCard);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSdCard.Unmount();
    };

    runCreateAndDeleteDirectory(list);
}

// 複数スレッドからそれぞれ深い階層のディレクトリ操作を行います。
class TestAccessDeepDirectory : public TestRunner
{
public:
    virtual void RunTest(TestContext* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fs::SetPriorityRawOnCurrentThread(pTest->GetPriority());

        const auto MaxDepth = 20;

        std::unique_ptr<char[]> path(
            CreateThreadUniqueDirectoryName(
                pTest->GetTarget(),
                threadIndex
            )
        );
        nnt::fs::util::String dirPath(path.get());
        if( pTest->IsReadOnly() )
        {
            for( auto depth = 0; depth < MaxDepth; ++depth )
            {
                nn::fs::DirectoryEntryType type;
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetEntryType(&type, dirPath.c_str()));
                ASSERT_EQ(nn::fs::DirectoryEntryType_Directory, type);
                std::unique_ptr<char[]> pathCurrent(new char[nn::fs::EntryNameLengthMax]);
                nn::util::SNPrintf(pathCurrent.get(), nn::fs::EntryNameLengthMax, "/%d", depth + 1);
                dirPath += pathCurrent.get();
            }
        }
        else
        {
            (void)nn::fs::DeleteDirectoryRecursively(path.get());

            // エントリを作成する
            for( auto depth = 0; depth < MaxDepth; ++depth )
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory(dirPath.c_str()));
                std::unique_ptr<char[]> pathCurrent(new char[nn::fs::EntryNameLengthMax]);
                nn::util::SNPrintf(pathCurrent.get(), nn::fs::EntryNameLengthMax, "/%d", depth + 1);
                dirPath += pathCurrent.get();
            }
            // エントリを削除する
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(path.get()));
        }
    }
};

namespace {

    auto runAccessDeepDirectory = [](const nnt::fs::util::Vector<StressTestMountInfoBase*>& list) NN_NOEXCEPT
    {
        TestAccessDeepDirectory test;

        const auto ThreadCount = StressTestParam::ThreadCountMax;
        int indexTable[ThreadCount];

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());
        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            indexTable[threadIndex] = std::uniform_int_distribution<int>(0, static_cast<int>(list.size()) - 1)(mt);
        }

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            auto pContext = new TestContext(list[indexTable[threadIndex]]);
            pContext->SetPriorityRandom();
            test.AddContext(pContext);
        }
        test.RunThreadTest();
    };

}

TEST(BisStressTest, AccessDeepDirectory)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountBis fs;
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runAccessDeepDirectory(list);
}

TEST(HostFsStressTest, AccessDeepDirectory)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("host");
    StressTestMountHostFs fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runAccessDeepDirectory(list);
}

TEST(SaveDataFsStressTest, AccessDeepDirectory)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("save");
    StressTestMountSaveData fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runAccessDeepDirectory(list);
}

TEST(SdCardStressTest, AccessDeepDirectory)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("sd");
    StressTestMountSdCard fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runAccessDeepDirectory(list);
}

TEST(MultiFsStressTest, AccessDeepDirectory)
{
    nnt::fs::util::Vector<StressTestMountInfoBase *> list;

    StressTestMountBis fsBis;
    {
        fsBis.Mount();
        list.push_back(&fsBis);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsBis.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountCacheStorage fsCache;
    {
        const nnt::fs::util::String mountName("cache");
        fsCache.SetMountName(mountName.c_str());
        fsCache.Mount();
        list.push_back(&fsCache);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsCache.Unmount();
    };
#endif

    StressTestMountHostFs fsHost;
    {
        const nnt::fs::util::String mountName("host");
        fsHost.SetMountName(mountName.c_str());
        fsHost.Mount();
        list.push_back(&fsHost);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsHost.Unmount();
    };

    StressTestMountRomFs fsRom;
    {
        const nnt::fs::util::String mountName("rom");
        fsRom.SetMountName(mountName.c_str());
        fsRom.Mount();
        list.push_back(&fsRom);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsRom.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountOtherSaveData fsOther;
    {
        const nnt::fs::util::String mountName("other");
        fsOther.SetMountName(mountName.c_str());
        fsOther.Mount();
        list.push_back(&fsOther);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsOther.Unmount();
    };
#endif

    StressTestMountSaveData fsSave;
    {
        const nnt::fs::util::String mountName("save");
        fsSave.SetMountName(mountName.c_str());
        fsSave.Mount();
        list.push_back(&fsSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSave.Unmount();
    };

    StressTestMountSdCard fsSdCard;
    {
        const nnt::fs::util::String mountName("sd");
        fsSdCard.SetMountName(mountName.c_str());
        fsSdCard.Mount();
        list.push_back(&fsSdCard);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSdCard.Unmount();
    };

    runAccessDeepDirectory(list);
}

