﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NN_ENABLE_HTC

#include <nn/init.h>
#include <nn/ae.h>
#include <nn/time.h>
#include <nn/htc.h>
#include <nn/fs.h>
#include <nn/fs/fs_SaveDataForDebug.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_TemporaryStorage.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/detail/fs_AccessLog.h>

#include "testFs_Stress_StressTest.h"

// スレッドの情報 (数やアクセス優先度) が欲しいときは false を true に書き換える
// (ログ出力による副作用が気になるので通常は false にする)
#define NNT_FS_TRACE_THREAD_INFOMATION(...) \
    if(NN_STATIC_CONDITION(false)) \
    { \
        NN_LOG(__VA_ARGS__); \
    }

namespace nnt { namespace fs {

namespace {

#if defined(NNT_FS_STRESS_TEST_A_ENSURESAVEDATA_ONLY)
    const uint64_t OwnerIds[2] = { 0x0005000c10000002, 0x0005000c10000003 };
#elif defined(NNT_FS_STRESS_TEST_A_EXCLUDING_ENSURESAVEDATA)
    const uint64_t OwnerIds[2] = { 0x0005000c10000004, 0x0005000c10000005 };
#elif defined(NNT_FS_STRESS_TEST_A_EXCLUDING_ENSURESAVEDATA2)
    const uint64_t OwnerIds[2] = { 0x0005000c10000006, 0x0005000c10000007 };
#else
    const uint64_t OwnerIds[2] = { 0x0005000c10000000, 0x0005000c10000001 };
#endif

    const nn::ncm::ApplicationId ApplicationIds[2] = { { OwnerIds[0] }, { OwnerIds[1] } };
    const nn::fs::UserId UserIds[2] = { { { 0, 1 } }, { { 0, 2 } } };

    const auto SaveDataSize = 16 * 1024 * 1024;
    const auto JournalSize = 16 * 1024 * 1024;

    void SetUpSaveData() NN_NOEXCEPT
    {
        nn::Result result;

        result = nn::fs::CreateSaveData(
            ApplicationIds[0],
            UserIds[0],
            OwnerIds[0],
            SaveDataSize,
            JournalSize,
            0);
        if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        result = nn::fs::CreateSaveData(
            ApplicationIds[0],
            UserIds[1],
            OwnerIds[0],
            SaveDataSize,
            JournalSize,
            0);
        if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }

        result = nn::fs::CreateSaveData(
            ApplicationIds[1],
            UserIds[0],
            OwnerIds[1],
            SaveDataSize,
            JournalSize,
            0);
        if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    }

    int g_Priority = 0;
}

NN_DEFINE_STATIC_CONSTANT(const int FsStressTest::ThreadCountMax);
NN_DEFINE_STATIC_CONSTANT(const int FsStressTest::ThreadStackSize);
NN_DEFINE_STATIC_CONSTANT(const int FsStressTest::EntryCountMax);

FsStressTest::TestCase::TestCase() NN_NOEXCEPT
    : m_MountName(nullptr)
{
}

FsStressTest::TestCase::~TestCase() NN_NOEXCEPT
{
}

int FsStressTest::TestCase::GetLoopCount() const NN_NOEXCEPT
{
    return 100;
}


int FsStressTest::TestCase::GetThreadCount() const NN_NOEXCEPT
{
    return 1;
}

int FsStressTest::TestCase::GetPriority(int) const NN_NOEXCEPT
{
    return nn::os::DefaultThreadPriority;
}

void FsStressTest::TestCase::SetUp(FsStressTest*) NN_NOEXCEPT
{
}

void FsStressTest::TestCase::TearDown(FsStressTest*) NN_NOEXCEPT
{
}

FsStressTest::~FsStressTest() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(GetMountedFileSystemCount(), 0);
}

void FsStressTest::SetUp() NN_NOEXCEPT
{
    if( m_IsProcessNameNeeded )
    {
#if defined(NNT_FS_STRESS_TEST_A_ENSURESAVEDATA_ONLY)
        NN_LOG("[EnsureSaveData Only]\n");
#elif defined(NNT_FS_STRESS_TEST_A_EXCLUDING_ENSURESAVEDATA)
        NN_LOG("[Excluding EnsureSaveData1]\n");
#elif defined(NNT_FS_STRESS_TEST_A_EXCLUDING_ENSURESAVEDATA2)
        NN_LOG("[Excluding EnsureSaveData2]\n");
#else
        NN_LOG("[All]\n");
#endif
    }
}

void FsStressTest::TearDown() NN_NOEXCEPT
{
}

void FsStressTest::ResetTestStatus() NN_NOEXCEPT
{
    m_IsProcessNameNeeded = false;
    TearDown();
    SetUp();
    m_IsProcessNameNeeded = true;
}

bool FsStressTest::IsReadOnly(int) const NN_NOEXCEPT
{
    return false;
}

bool FsStressTest::IsSaveData(int) const NN_NOEXCEPT
{
    return false;
}

uint64_t FsStressTest::GetApplicationId(int) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(false);
    static const uint64_t ApplicationId = 0;
    return ApplicationId;
}

nn::fs::UserId FsStressTest::GetUserId(int) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(false);
    static const nn::fs::UserId UserId = { { 0, 0 } };
    return UserId;
}

bool FsStressTest::AccessesFatDirectly(int) const NN_NOEXCEPT
{
    return false;
}

void FsStressTest::SetUpTestDirectory(TestCase** ppTestCases, int testCaseCount) NN_NOEXCEPT
{
    nn::time::StandardUserSystemClock clock;
    nn::time::PosixTime posixTime;
    nn::time::CalendarTime calenderTime;
    char path[32];
    NN_UNUSED(clock);

    for( ; ; )
    {
        NNT_ASSERT_RESULT_SUCCESS(clock.GetCurrentTime(&posixTime));
        NNT_ASSERT_RESULT_SUCCESS(nn::time::ToCalendarTime(&calenderTime, nullptr, posixTime));
        nn::util::SNPrintf(
            m_TestDirectoryPath,
            sizeof(m_TestDirectoryPath),
            "/%02d%02d%02d%02d%02d%02d",
            calenderTime.year % 100,
            calenderTime.month,
            calenderTime.day,
            calenderTime.hour,
            calenderTime.minute,
            calenderTime.second);

        auto testCaseIndex = 0;
        for( ; testCaseIndex < testCaseCount; ++testCaseIndex )
        {
            if( !IsReadOnly(testCaseIndex) && testCaseIndex < GetMountedFileSystemCount() )
            {
                nn::util::SNPrintf(
                    path,
                    sizeof(path),
                    "%s:%s",
                    ppTestCases[testCaseIndex]->GetMountName(),
                    m_TestDirectoryPath);
                const auto result = nn::fs::CreateDirectory(path);
                if( nn::fs::ResultPathAlreadyExists::Includes(result) )
                {
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                    break;
                }
                else
                {
                    NNT_ASSERT_RESULT_SUCCESS(result);
                }
            }
        }

        if( testCaseIndex < testCaseCount )
        {
            for( --testCaseIndex ; 0 <= testCaseIndex; --testCaseIndex )
            {
                if( !IsReadOnly(testCaseIndex) && testCaseIndex < GetMountedFileSystemCount() )
                {
                    nn::util::SNPrintf(
                        path,
                        sizeof(path),
                        "%s:%s",
                        ppTestCases[testCaseIndex]->GetMountName(),
                        m_TestDirectoryPath);
                    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteDirectory(path));
                }
            }
        }
        else
        {
            break;
        }
    }
}

void FsStressTest::TearDownTestDirectory(TestCase** ppTestCases, int testCaseCount) NN_NOEXCEPT
{
    char path[64];
    for( auto testCaseIndex = 0; testCaseIndex < testCaseCount; ++testCaseIndex )
    {
        if( !IsReadOnly(testCaseIndex) && testCaseIndex < GetMountedFileSystemCount() )
        {
            nn::util::SNPrintf(
                path,
                sizeof(path),
                "%s:%s",
                ppTestCases[testCaseIndex]->GetMountName(),
                m_TestDirectoryPath);
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(path));
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::Commit(ppTestCases[testCaseIndex]->GetMountName()));
        }
    }
}

void FsStressTest::TestCore(TestCase** ppTestCases, int testCaseCount) NN_NOEXCEPT
{
    ASSERT_NE(nullptr, ppTestCases);
    auto threadCount = 0;
    for( auto testIndex = 0; testIndex < testCaseCount; ++testIndex )
    {
        ASSERT_NE(nullptr, ppTestCases[testIndex]);
        ASSERT_GT(ppTestCases[testIndex]->GetThreadCount(), 0);
        threadCount += ppTestCases[testIndex]->GetThreadCount();
    }
    ASSERT_LE(threadCount, ThreadCountMax);
    NNT_FS_TRACE_THREAD_INFOMATION("thread count %d\n", threadCount);

    SetUpTestDirectory(ppTestCases, testCaseCount);
    NN_UTIL_SCOPE_EXIT
    {
        TearDownTestDirectory(ppTestCases, testCaseCount);
    };

    if( threadCount == 1 )
    {
        // こちらは ForSingleThreadTest.UpdatePriority で更新される
        auto oldPriority = nn::fs::GetPriorityRawOnCurrentThread();
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::SetPriorityRawOnCurrentThread(oldPriority);
        };
        nn::fs::SetPriorityRawOnCurrentThread(static_cast<nn::fs::PriorityRaw>(g_Priority));
        NNT_FS_TRACE_THREAD_INFOMATION("access priority %s\n", nn::fs::detail::IdString().ToString(nn::fs::GetPriorityRawOnCurrentThread()));

        const auto loopCount = ppTestCases[0]->GetLoopCount();
        ASSERT_GT(loopCount, 0);
        ASSERT_EQ(1, testCaseCount);
        ppTestCases[0]->SetUp(this);
        for( auto count = 0; count < loopCount && !ppTestCases[0]->Failed(0); ++count )
        {
            ppTestCases[0]->Test(this, 0);
        }
        ppTestCases[0]->TearDown(this);
    }
    else
    {
        struct Argument
        {
            FsStressTest* pTest;
            TestCase* pTestCase;
            int threadIndex;
        };

        static NN_OS_ALIGNAS_THREAD_STACK char s_ThreadStack[ThreadStackSize * ThreadCountMax];
        static nn::os::ThreadType s_Threads[ThreadCountMax];
        static Argument s_Arguments[ThreadCountMax];

        for( auto testIndex = 0; testIndex < testCaseCount; ++testIndex )
        {
            ppTestCases[testIndex]->SetUp(this);
        }

        auto ppTestCase = ppTestCases - 1;
        auto remainThreadCount = 1;
        for( auto threadIndex = 0; threadIndex < threadCount; ++threadIndex )
        {
            --remainThreadCount;
            if( remainThreadCount == 0 )
            {
                ++ppTestCase;
                remainThreadCount = (*ppTestCase)->GetThreadCount();
            }

            s_Arguments[threadIndex].pTest = this;
            s_Arguments[threadIndex].pTestCase = *ppTestCase;
            s_Arguments[threadIndex].threadIndex
                = (*ppTestCase)->GetThreadCount() - remainThreadCount;

            NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                s_Threads + threadIndex,
                [](void* pArgument) NN_NOEXCEPT
                {
                    const auto& argument = *reinterpret_cast<Argument*>(pArgument);

                    {
                        auto priority = static_cast<nn::fs::PriorityRaw>(argument.threadIndex % 4);
                        NNT_FS_TRACE_THREAD_INFOMATION("access priority %s\n", nn::fs::detail::IdString().ToString(priority));
                        nn::fs::SetPriorityRawOnCurrentThread(priority);
                    }

                    const auto loopCount = argument.pTestCase->GetLoopCount();
                    ASSERT_GT(loopCount, 0);
                    for( auto count = 0;
                        count < loopCount && !argument.pTestCase->Failed(argument.threadIndex);
                        ++count )
                    {
                        argument.pTestCase->Test(argument.pTest, argument.threadIndex);
                    }
                },
                s_Arguments + threadIndex,
                s_ThreadStack + ThreadStackSize * threadIndex,
                ThreadStackSize,
                (*ppTestCase)->GetPriority(threadIndex)));
        }

        for( auto threadIndex = 0; threadIndex < threadCount; ++threadIndex )
        {
            nn::os::StartThread(s_Threads + threadIndex);
        }

        for( auto threadIndex = 0; threadIndex < threadCount; ++threadIndex )
        {
            nn::os::WaitThread(s_Threads + threadIndex);
            nn::os::DestroyThread(s_Threads + threadIndex);
        }

        for( auto testIndex = 0; testIndex < testCaseCount; ++testIndex )
        {
            ppTestCases[testIndex]->TearDown(this);
        }
    }
} // NOLINT(impl/function_size)

SaveDataFsStressTest::~SaveDataFsStressTest() NN_NOEXCEPT
{
}

void SaveDataFsStressTest::SetUp() NN_NOEXCEPT
{
    FsStressTest::SetUp();

#ifdef NN_BUILD_CONFIG_OS_WIN
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(
        GetMountName(),
        ApplicationIds[0],
        UserIds[0]));
#else
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(GetMountName(), UserIds[0]));
#endif
    NotifyMountFileSystem();
}

void SaveDataFsStressTest::TearDown() NN_NOEXCEPT
{
    FsStressTest::TearDown();

    if( 0 < GetMountedFileSystemCount() )
    {
        nn::fs::Unmount(GetMountName());
        NotifyUnmountFileSystem();
    }
}

bool SaveDataFsStressTest::IsSaveData(int) const NN_NOEXCEPT
{
    return true;
}

uint64_t SaveDataFsStressTest::GetApplicationId(int) const NN_NOEXCEPT
{
    return OwnerIds[0];
}

nn::fs::UserId SaveDataFsStressTest::GetUserId(int) const NN_NOEXCEPT
{
    return UserIds[0];
}

MultipleSaveDataFsStressTest::~MultipleSaveDataFsStressTest() NN_NOEXCEPT
{
}

void MultipleSaveDataFsStressTest::SetUp() NN_NOEXCEPT
{
    FsStressTest::SetUp();

#ifdef NN_BUILD_CONFIG_OS_WIN
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(
        GetMountName(0),
        ApplicationIds[0],
        UserIds[0]));
    NotifyMountFileSystem();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(
        GetMountName(1),
        ApplicationIds[0],
        UserIds[1]));
    NotifyMountFileSystem();
#else
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(GetMountName(0), UserIds[0]));
    NotifyMountFileSystem();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(GetMountName(1), UserIds[1]));
    NotifyMountFileSystem();
#endif
}

void MultipleSaveDataFsStressTest::TearDown() NN_NOEXCEPT
{
    FsStressTest::TearDown();

    if( 1 < GetMountedFileSystemCount() )
    {
        nn::fs::Unmount(GetMountName(1));
        NotifyUnmountFileSystem();
    }
    if( 0 < GetMountedFileSystemCount() )
    {
        nn::fs::Unmount(GetMountName(0));
        NotifyUnmountFileSystem();
    }
}

bool MultipleSaveDataFsStressTest::IsSaveData(int) const NN_NOEXCEPT
{
    return true;
}

uint64_t MultipleSaveDataFsStressTest::GetApplicationId(int) const NN_NOEXCEPT
{
    return OwnerIds[0];
}

nn::fs::UserId MultipleSaveDataFsStressTest::GetUserId(int fsIndex) const NN_NOEXCEPT
{
    switch( fsIndex )
    {
    case 0:
        return UserIds[0];
    case 1:
        return UserIds[1];
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

OtherApplicationSaveDataFsStressTest::~OtherApplicationSaveDataFsStressTest() NN_NOEXCEPT
{
}

void OtherApplicationSaveDataFsStressTest::SetUp() NN_NOEXCEPT
{
    FsStressTest::SetUp();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(
        GetMountName(),
        ApplicationIds[1],
        UserIds[0]));
    NotifyMountFileSystem();
}

void OtherApplicationSaveDataFsStressTest::TearDown() NN_NOEXCEPT
{
    FsStressTest::TearDown();

    if( 0 < GetMountedFileSystemCount() )
    {
        nn::fs::Unmount(GetMountName());
        NotifyUnmountFileSystem();
    }
}

bool OtherApplicationSaveDataFsStressTest::IsSaveData(int) const NN_NOEXCEPT
{
    return true;
}

uint64_t OtherApplicationSaveDataFsStressTest::GetApplicationId(int) const NN_NOEXCEPT
{
    return OwnerIds[1];
}

nn::fs::UserId OtherApplicationSaveDataFsStressTest::GetUserId(int) const NN_NOEXCEPT
{
    return UserIds[0];
}

SaveDataFsRomFsStressTest::~SaveDataFsRomFsStressTest() NN_NOEXCEPT
{
}

void SaveDataFsRomFsStressTest::SetUp() NN_NOEXCEPT
{
    FsStressTest::SetUp();

#ifdef NN_BUILD_CONFIG_OS_WIN
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(
        GetMountName(0),
        ApplicationIds[0],
        UserIds[0]));
#else
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(GetMountName(0), UserIds[0]));
#endif
    NotifyMountFileSystem();

    size_t cacheSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    m_CacheBuffer.reset(new char[cacheSize]);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom(GetMountName(1), m_CacheBuffer.get(), cacheSize));
    NotifyMountFileSystem();
}

void SaveDataFsRomFsStressTest::TearDown() NN_NOEXCEPT
{
    FsStressTest::TearDown();

    if( 1 < GetMountedFileSystemCount() )
    {
        nn::fs::Unmount(GetMountName(1));
        NotifyUnmountFileSystem();
    }
    if( 0 < GetMountedFileSystemCount() )
    {
        nn::fs::Unmount(GetMountName(0));
        NotifyUnmountFileSystem();
    }
}

bool SaveDataFsRomFsStressTest::IsReadOnly(int fsIndex) const NN_NOEXCEPT
{
    return fsIndex == 1;
}

bool SaveDataFsRomFsStressTest::IsSaveData(int fsIndex) const NN_NOEXCEPT
{
    return fsIndex == 0;
}

uint64_t SaveDataFsRomFsStressTest::GetApplicationId(int fsIndex) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(fsIndex, 0);
    NN_UNUSED(fsIndex);
    return OwnerIds[0];
}

nn::fs::UserId SaveDataFsRomFsStressTest::GetUserId(int fsIndex) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(fsIndex, 0);
    NN_UNUSED(fsIndex);
    return UserIds[0];
}

RomFsStressTest::~RomFsStressTest() NN_NOEXCEPT
{
}

void RomFsStressTest::SetUp() NN_NOEXCEPT
{
    FsStressTest::SetUp();

    size_t cacheSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    m_CacheBuffer.reset(new char[cacheSize]);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom(GetMountName(), m_CacheBuffer.get(), cacheSize));
    NotifyMountFileSystem();
}

void RomFsStressTest::TearDown() NN_NOEXCEPT
{
    FsStressTest::TearDown();

    if( 0 < GetMountedFileSystemCount() )
    {
        nn::fs::Unmount(GetMountName());
        NotifyUnmountFileSystem();
    }
}

bool RomFsStressTest::IsReadOnly(int) const NN_NOEXCEPT
{
    return true;
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)
HostFsStressTest::~HostFsStressTest() NN_NOEXCEPT
{
}

void HostFsStressTest::SetUp() NN_NOEXCEPT
{
    FsStressTest::SetUp();

    m_HostDirectory.Create();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost(GetMountName(), m_HostDirectory.GetPath().c_str()));
    NotifyMountFileSystem();
}

void HostFsStressTest::TearDown() NN_NOEXCEPT
{
    FsStressTest::TearDown();

    if( 0 < GetMountedFileSystemCount() )
    {
        nn::fs::Unmount(GetMountName());
        NotifyUnmountFileSystem();
    }
    m_HostDirectory.Delete();
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)

SdCardFsStressTest::~SdCardFsStressTest() NN_NOEXCEPT
{
}

void SdCardFsStressTest::SetUp() NN_NOEXCEPT
{
    FsStressTest::SetUp();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug(GetMountName()));
    NotifyMountFileSystem();
}

void SdCardFsStressTest::TearDown() NN_NOEXCEPT
{
    FsStressTest::TearDown();

    if( 0 < GetMountedFileSystemCount() )
    {
        nn::fs::Unmount(GetMountName());
        NotifyUnmountFileSystem();
    }
}

bool SdCardFsStressTest::AccessesFatDirectly(int) const NN_NOEXCEPT
{
    return true;
}

SdCardFsRomFsStressTest::~SdCardFsRomFsStressTest() NN_NOEXCEPT
{
}

void SdCardFsRomFsStressTest::SetUp() NN_NOEXCEPT
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCard(GetMountName(0)));
    NotifyMountFileSystem();
    size_t cacheSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    m_CacheBuffer.reset(new char[cacheSize]);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom(GetMountName(1), m_CacheBuffer.get(), cacheSize));
    NotifyMountFileSystem();
}

void SdCardFsRomFsStressTest::TearDown() NN_NOEXCEPT
{
    if( 0 < GetMountedFileSystemCount() )
    {
        nn::fs::Unmount(GetMountName(0));
        NotifyUnmountFileSystem();
        nn::fs::Unmount(GetMountName(1));
        NotifyUnmountFileSystem();
    }
}

bool SdCardFsRomFsStressTest::IsReadOnly(int fsIndex) const NN_NOEXCEPT
{
    switch( fsIndex )
    {
    case 0:
        return false;
    case 1:
        return true;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool SdCardFsRomFsStressTest::AccessesFatDirectly(int fsIndex) const NN_NOEXCEPT
{
    switch( fsIndex )
    {
    case 0:
        return true;
    case 1:
        return false;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)
TemporaryStorageFsStressTest::~TemporaryStorageFsStressTest() NN_NOEXCEPT
{
}

void TemporaryStorageFsStressTest::SetUp() NN_NOEXCEPT
{
    FsStressTest::SetUp();

    auto result = nn::fs::CreateTemporaryStorage(
        ApplicationIds[0],
        OwnerIds[0],
        SaveDataSize,
        0
    );
    if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
    {
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountTemporaryStorage(GetMountName()));

    NotifyMountFileSystem();
}

void TemporaryStorageFsStressTest::TearDown() NN_NOEXCEPT
{
    FsStressTest::TearDown();

    if( 0 < GetMountedFileSystemCount() )
    {
        nn::fs::Unmount(GetMountName());
        NotifyUnmountFileSystem();
    }
}

bool TemporaryStorageFsStressTest::IsSaveData(int) const NN_NOEXCEPT
{
    return true;
}

uint64_t TemporaryStorageFsStressTest::GetApplicationId(int) const NN_NOEXCEPT
{
    return OwnerIds[0];
}

nn::fs::UserId TemporaryStorageFsStressTest::GetUserId(int) const NN_NOEXCEPT
{
    return nn::fs::InvalidUserId;
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)

TEST(ForSingleThreadTest, UpdatePriority)
{
    g_Priority = ((g_Priority + 1) % 4);
}

}}

extern "C" void nninitStartup()
{
    static char s_Buffer[64 * 1024 * 1024];
    nn::init::InitializeAllocator(s_Buffer, sizeof(s_Buffer));
}

extern "C" void nnMain()
{
#ifdef NNT_FS_STRESS_TEST_A_ENSURESAVEDATA_ONLY
    // nn::fs::EnsureSaveData() を呼び出すためには am に接続している必要がある
    // appletOE に同時に接続することはできないため、_Other 側は appletAE に接続する
    nn::ae::InitializeAsSystemApplication();
#endif

#if !defined(NNT_FS_STRESS_TEST_A_ENSURESAVEDATA_ONLY)
    // 起動直後に死ぬとログをとりこぼすので Sleep
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
#endif // !defined(NNT_FS_STRESS_TEST_A_ENSURESAVEDATA_ONLY)

    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

#if defined(NN_BUILD_CONFIG_OS_WIN)
    // NX では bat の中で実行しているセーブデータ一括削除を代わりにここで実行する
    {
        const nn::fs::SaveDataSpaceId SpaceIds[] = {
            nn::fs::SaveDataSpaceId::User,
            nn::fs::SaveDataSpaceId::System,
            nn::fs::SaveDataSpaceId::SdSystem,
            nn::fs::SaveDataSpaceId::Temporary
        };
        for( auto spaceId : SpaceIds )
        {
            while( NN_STATIC_CONDITION(true) )
            {
                nn::util::optional<nn::fs::SaveDataInfo> info;
                nnt::fs::util::FindSaveData(&info, spaceId, [](const nn::fs::SaveDataInfo&) NN_NOEXCEPT { return true; });
                if( info == nn::util::nullopt )
                {
                    break;
                }

                auto saveId = info->saveDataId;
                NN_LOG("delete save data (%x, %llx).\n", spaceId, saveId);
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::DeleteSaveData(spaceId, saveId));
            }
        }
    }
#endif // defined(NN_BUILD_CONFIG_OS_WIN)

    nnt::fs::SetUpSaveData();

    nn::time::Initialize();

    auto testResult = RUN_ALL_TESTS();

    nn::time::Finalize();

#if !defined(NNT_FS_STRESS_TEST_A_ENSURESAVEDATA_ONLY)
    // テスト完了を検知するため、ログを出す。変更には注意。
    // ログ解析中のスキマにあたらないように繰り返しログを出す
    for( int i = 0; i < 5; ++i )
    {
        NN_LOG("[ StressTest End ]\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    }
#endif // !defined(NNT_FS_STRESS_TEST_A_ENSURESAVEDATA_ONLY)

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
