﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <numeric>
#include <nn/time.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SystemSaveData.h>
#include "testFs_Stress_OpenEntriesTestCase.h"

namespace nnt { namespace fs {

class CommitSaveData : public OpenEntriesTestCase
{
public:
    CommitSaveData()
        : OpenEntriesTestCase(1, 50, nn::fs::DirectoryEntryType_File)
    {
    }

    virtual ~CommitSaveData() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
#ifdef NN_BUILD_CONFIG_OS_WIN
        return 5;
#else
        return 100;
#endif
    }

    virtual void Test(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);

        ASSERT_TRUE(pTest->IsSaveData(GetTestCaseIndex()));

        for( auto entryIndex = 0; entryIndex < GetEntryCount(); ++entryIndex )
        {
            char path[PathLength];
            MakePath(path, entryIndex);
            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(
                &file,
                path,
                nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };
            const char buffer = -1;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteFile(
                file,
                0,
                &buffer,
                1,
                nn::fs::WriteOption()));
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::FlushFile(file));
        }

        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(GetMountName()));
        }

        Succeed(threadIndex);
    }
};

class RollbackSaveData : public OpenEntriesTestCase
{
public:
    RollbackSaveData()
        : OpenEntriesTestCase(1, 50, nn::fs::DirectoryEntryType_File)
    {
    }

    virtual ~RollbackSaveData() NN_NOEXCEPT NN_OVERRIDE {}

    virtual void SetUp(FsStressTest* pTest) NN_NOEXCEPT NN_OVERRIDE
    {
        FailAll();
        for( auto entryIndex = 0; entryIndex < GetEntryCount(); ++entryIndex )
        {
            char path[PathLength];
            MakePath(path, entryIndex);
            const auto result = nn::fs::DeleteFile(path);
            if( !nn::fs::ResultPathNotFound::Includes(result) )
            {
                NNT_ASSERT_RESULT_SUCCESS(result);
            }
        }
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(GetMountName()));
        SucceedAll();

        OpenEntriesTestCase::SetUp(pTest);
    }

    virtual void Test(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);

        ASSERT_TRUE(pTest->IsSaveData(GetTestCaseIndex()));

        // ファイルを作成する
        for( auto entryIndex = 0; entryIndex < GetEntryCount(); ++entryIndex )
        {
            char path[PathLength];
            MakePath(path, entryIndex);

            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            const auto result = nn::fs::CreateFile(path, 1024);
            if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
            {
                NNT_ASSERT_RESULT_SUCCESS(result);
            }
        }

        // コミットせずにアンマウントすることで巻き戻す
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            pTest->ResetTestStatus();
        }

        // ファイル作成前に巻き戻っていることを確認する
        for( auto entryIndex = 0; entryIndex < GetEntryCount(); ++entryIndex )
        {
            char path[PathLength];
            MakePath(path, entryIndex);
            nn::fs::DirectoryEntryType type;

            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathNotFound, nn::fs::GetEntryType(
                &type,
                path));
        }

        Succeed(threadIndex);
    }
};

class AccessSaveDataThumbnail : public FsStressTest::TestCase
{
public:
    virtual ~AccessSaveDataThumbnail() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 1000;
    }

    virtual void Test(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);
        ASSERT_TRUE(pTest->IsSaveData(GetTestCaseIndex()));
        static const auto HeaderSize = 256;
        static const auto BodySize = 16 * 1024 - HeaderSize;
        std::unique_ptr<char[]> buffer(new char[HeaderSize + BodySize]);
        std::iota(buffer.get(), buffer.get() + HeaderSize + BodySize, static_cast<char>(0));
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteSaveDataThumbnailFile(
                pTest->GetApplicationId(GetTestCaseIndex()),
                pTest->GetUserId(GetTestCaseIndex()),
                buffer.get(),
                HeaderSize,
                buffer.get() + HeaderSize,
                BodySize));
        }
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadSaveDataThumbnailFile(
                pTest->GetApplicationId(GetTestCaseIndex()),
                pTest->GetUserId(GetTestCaseIndex()),
                buffer.get(),
                HeaderSize,
                buffer.get() + HeaderSize,
                BodySize));
        }
        Succeed(threadIndex);
    }
};

class GetSaveDataManagementData : public FsStressTest::TestCase
{
public:
    virtual ~GetSaveDataManagementData() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 200;
    }

    virtual void Test(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);
        ASSERT_TRUE(pTest->IsSaveData(GetTestCaseIndex()));
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        {
            std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(
                &iter,
                nn::fs::SaveDataSpaceId::User));
        }
        const nn::ncm::ApplicationId applicationId = { pTest->GetApplicationId(GetTestCaseIndex()) };
        const auto userId = pTest->GetUserId(GetTestCaseIndex());
        for( ; ; )
        {
            int64_t readCount = 0;
            nn::fs::SaveDataInfo info;
            {
                std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
                NNT_ASSERT_RESULT_SUCCESS(iter->ReadSaveDataInfo(&readCount, &info, 1));
            }
            if( readCount == 0 )
            {
                FAIL();
                break;
            }
            else if( info.applicationId == applicationId && info.saveDataUserId == userId )
            {
                nn::Bit64 ownerId = 0;
                uint32_t flags = 0;
                nn::time::PosixTime time;
                int64_t size = 0;
                static const auto LoopCount = 100;
                for( auto count = 0; count < LoopCount; ++count )
                {
                    std::lock_guard<nn::os::Mutex> lock(pTest->GetMutex());
                    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataOwnerId(
                        &ownerId,
                        info.saveDataId));
                    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataFlags(
                        &flags,
                        info.saveDataId));
                    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataTimeStamp(
                        &time,
                        info.saveDataId));
                    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataAvailableSize(
                        &size,
                        info.saveDataId));
                    NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataJournalSize(
                        &size,
                        info.saveDataId));
                }
                Succeed(threadIndex);
                return;
            }
        }
    }
};

class CreateSystemSaveData : public FsStressTest::TestCase
{
public:
    virtual ~CreateSystemSaveData() NN_NOEXCEPT NN_OVERRIDE {}

    virtual void Test(FsStressTest*, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);

        static const auto DataSize = 64 * 1024;
        static const auto JournalSize = 32 * 1024;
        auto systemSaveDataId = 0x80000000f0000000;

        for( ; ; )
        {
            auto result = nn::fs::CreateSystemSaveData(systemSaveDataId, DataSize, JournalSize, 0);
            if( nn::fs::ResultPathAlreadyExists::Includes(result) )
            {
                ++systemSaveDataId;
            }
            else
            {
                NNT_ASSERT_RESULT_SUCCESS(result);
                break;
            }
        }

        for( ; ; )
        {
            std::unique_ptr<nn::fs::SaveDataIterator> iter;
            {
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(
                    &iter,
                    nn::fs::SaveDataSpaceId::System));
            }
            for( ; ; )
            {
                int64_t readCount = 0;
                nn::fs::SaveDataInfo info;
                NNT_ASSERT_RESULT_SUCCESS(iter->ReadSaveDataInfo(&readCount, &info, 1));
                if( readCount == 0 )
                {
                    FAIL();
                    return;
                }
                else if( info.systemSaveDataId == systemSaveDataId )
                {
                    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(info.saveDataId));
                    Succeed(threadIndex);
                    return;
                }
            }
        }
    }
};

TEST_F(SaveDataFsStressTest, CommitSaveData)
{
    Test<CommitSaveData>(GetMountName());
}

TEST_F(SaveDataFsStressTest, RollbackSaveData)
{
    Test<RollbackSaveData>(GetMountName());
}

TEST_F(SaveDataFsStressTest, AccessSaveDataThumbnail)
{
    Test<AccessSaveDataThumbnail>(GetMountName());
}

TEST_F(SaveDataFsStressTest, GetSaveDataManagementData)
{
    Test<GetSaveDataManagementData>(GetMountName());
}

TEST_F(MultipleSaveDataFsStressTest, CommitSaveData)
{
    Test<CommitSaveData, CommitSaveData>(GetMountName(0), GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, RollbackSaveData)
{
    Test<RollbackSaveData, RollbackSaveData>(GetMountName(0), GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, AccessSaveDataThumbnail)
{
    Test<AccessSaveDataThumbnail, AccessSaveDataThumbnail>(GetMountName(0), GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, GetSaveDataManagementData)
{
    Test<GetSaveDataManagementData, GetSaveDataManagementData>(GetMountName(0), GetMountName(1));
}

TEST_F(OtherApplicationSaveDataFsStressTest, CommitSaveData)
{
    Test<CommitSaveData>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, RollbackSaveData)
{
    Test<RollbackSaveData>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, AccessSaveDataThumbnail)
{
    Test<AccessSaveDataThumbnail>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, GetSaveDataManagementData)
{
    Test<GetSaveDataManagementData>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)
TEST_F(TemporaryStorageFsStressTest, CommitSaveData)
{
    Test<CommitSaveData>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)

// TemporaryStorageFsStressTest は RollbackSaveData と GetSaveDataManagementData 不要

TEST_F(FsStressTest, CreateSystemSaveData)
{
    Test<CreateSystemSaveData>(nullptr);
}

}}
