﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <numeric>
#include "testFs_Stress_AccessFilesTestCase.h"

namespace nnt { namespace fs {

class GetFilePropertiesTestCase : public AccessFilesTestCase
{
public:
    static const auto FileSize = 1 * 1024;

public:
    GetFilePropertiesTestCase(int threadCount, int fileCount, bool isConcurrent) NN_NOEXCEPT
        : AccessFilesTestCase(
            threadCount,
            fileCount,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)),
          m_IsConcurrent(isConcurrent)
    {
    }

    virtual ~GetFilePropertiesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

    virtual int GetLoopCount() const NN_NOEXCEPT NN_OVERRIDE
    {
        return 1000;
    }

    virtual void Test(FsStressTest* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        Fail(threadIndex);
        const auto entryCount
            = m_IsConcurrent ? GetEntryCount() * GetThreadCount() : GetEntryCount();
        for( auto entryIndex = 0; entryIndex < entryCount; ++entryIndex )
        {
            auto succeeded = false;
            GetFileProperty(
                &succeeded,
                m_IsConcurrent ? entryIndex : GetEntryIndex(threadIndex, entryIndex),
                pTest->IsReadOnly(GetTestCaseIndex()));
            ASSERT_TRUE(succeeded);
        }
        Succeed(threadIndex);
    }

protected:
    virtual void SetUpFile(bool* outSucceeded, int entryIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetFileSize(file, FileSize));

        std::unique_ptr<char[]> buffer(new char[FileSize]);
        std::iota(buffer.get(), buffer.get() + FileSize, static_cast<char>(entryIndex));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(
            file,
            0,
            buffer.get(),
            FileSize,
            nn::fs::WriteOption()));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(file));

        *outSucceeded = true;
    }

    virtual void GetFileProperty(
        bool* outSucceeded,
        int entryIndex,
        bool isReadOnly) NN_NOEXCEPT = 0;

private:
    bool m_IsConcurrent;
};

NN_DEFINE_STATIC_CONSTANT(const int GetFilePropertiesTestCase::FileSize);

class GetFileSizesTestCase : public GetFilePropertiesTestCase
{
public:
    GetFileSizesTestCase(int threadCount, int fileCount, bool isConcurrent) NN_NOEXCEPT
        : GetFilePropertiesTestCase(threadCount, fileCount, isConcurrent)
    {
    }

    virtual ~GetFileSizesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

protected:
    virtual void GetFileProperty(bool* outSucceeded, int entryIndex, bool) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        int64_t size = 0;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetFileSize(&size, file));
        EXPECT_EQ(FileSize, size);

        *outSucceeded = true;
    }
};

class GetFileOpenModesTestCase : public GetFilePropertiesTestCase
{
public:
    GetFileOpenModesTestCase(int threadCount, int fileCount, bool isConcurrent) NN_NOEXCEPT
        : GetFilePropertiesTestCase(threadCount, fileCount, isConcurrent)
    {
    }

    virtual ~GetFileOpenModesTestCase() NN_NOEXCEPT NN_OVERRIDE {}

protected:
    virtual void GetFileProperty(
        bool* outSucceeded,
        int entryIndex,
        bool isReadOnly) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outSucceeded);
        *outSucceeded = false;

        const auto file = GetFile(entryIndex);
        const auto mode = nn::fs::GetFileOpenMode(file);
        EXPECT_EQ(nn::fs::OpenMode_Read | (isReadOnly ? 0 : nn::fs::OpenMode_Write), mode);

        *outSucceeded = true;
    }
};

class GetFileSizes : public GetFileSizesTestCase
{
public:
    GetFileSizes() NN_NOEXCEPT
        : GetFileSizesTestCase(FsStressTest::ThreadCountMax, 5, false)
    {
    }
};

class GetFewerFileSizes : public GetFileSizesTestCase
{
public:
    GetFewerFileSizes() NN_NOEXCEPT
        : GetFileSizesTestCase(FsStressTest::ThreadCountMax / 2, 5, false)
    {
    }
};

class ConcurrentGetFileSizes : public GetFileSizesTestCase
{
public:
    ConcurrentGetFileSizes() NN_NOEXCEPT
        : GetFileSizesTestCase(FsStressTest::ThreadCountMax, 5, true)
    {
    }
};

class ConcurrentGetFewerFileSizes : public GetFileSizesTestCase
{
public:
    ConcurrentGetFewerFileSizes() NN_NOEXCEPT
        : GetFileSizesTestCase(FsStressTest::ThreadCountMax / 2, 5, true)
    {
    }
};

class GetFileOpenModes : public GetFileOpenModesTestCase
{
public:
    GetFileOpenModes() NN_NOEXCEPT
        : GetFileOpenModesTestCase(FsStressTest::ThreadCountMax, 5, false)
    {
    }
};

class GetFewerFileOpenModes : public GetFileOpenModesTestCase
{
public:
    GetFewerFileOpenModes() NN_NOEXCEPT
        : GetFileOpenModesTestCase(FsStressTest::ThreadCountMax / 2, 5, false)
    {
    }
};

class ConcurrentGetFileOpenModes : public GetFileOpenModesTestCase
{
public:
    ConcurrentGetFileOpenModes() NN_NOEXCEPT
        : GetFileOpenModesTestCase(FsStressTest::ThreadCountMax, 5, true)
    {
    }
};

class ConcurrentGetFewerFileOpenModes : public GetFileOpenModesTestCase
{
public:
    ConcurrentGetFewerFileOpenModes() NN_NOEXCEPT
        : GetFileOpenModesTestCase(FsStressTest::ThreadCountMax / 2, 5, true)
    {
    }
};

TEST_F(SaveDataFsStressTest, GetFileSizes)
{
    Test<GetFileSizes>(GetMountName());
}

TEST_F(SaveDataFsStressTest, ConcurrentGetFileSizes)
{
    Test<ConcurrentGetFileSizes>(GetMountName());
}

TEST_F(SaveDataFsStressTest, GetFileOpenModes)
{
    Test<GetFileOpenModes>(GetMountName());
}

TEST_F(SaveDataFsStressTest, ConcurrentGetFileOpenModes)
{
    Test<ConcurrentGetFileOpenModes>(GetMountName());
}

TEST_F(MultipleSaveDataFsStressTest, GetFileSizes)
{
    Test<GetFewerFileSizes, GetFewerFileSizes>(GetMountName(0), GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, ConcurrentGetFileSizes)
{
    Test<ConcurrentGetFewerFileSizes, ConcurrentGetFewerFileSizes>(
        GetMountName(0),
        GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, GetFileOpenModes)
{
    Test<GetFewerFileOpenModes, GetFewerFileOpenModes>(GetMountName(0), GetMountName(1));
}

TEST_F(MultipleSaveDataFsStressTest, ConcurrentGetFileOpenModes)
{
    Test<ConcurrentGetFewerFileOpenModes, ConcurrentGetFewerFileOpenModes>(
        GetMountName(0),
        GetMountName(1));
}

TEST_F(OtherApplicationSaveDataFsStressTest, GetFileSizes)
{
    Test<GetFileSizes>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, ConcurrentGetFileSizes)
{
    Test<ConcurrentGetFileSizes>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, GetFileOpenModes)
{
    Test<GetFileOpenModes>(GetMountName());
}

TEST_F(OtherApplicationSaveDataFsStressTest, ConcurrentGetFileOpenModes)
{
    Test<ConcurrentGetFileOpenModes>(GetMountName());
}

TEST_F(SaveDataFsRomFsStressTest, GetFileSizes)
{
    Test<GetFewerFileSizes, GetFewerFileSizes>(GetMountName(0), GetMountName(1));
}

TEST_F(SaveDataFsRomFsStressTest, ConcurrentGetFileSizes)
{
    Test<ConcurrentGetFewerFileSizes, ConcurrentGetFewerFileSizes>(
        GetMountName(0),
        GetMountName(1));
}

TEST_F(SaveDataFsRomFsStressTest, GetFileOpenModes)
{
    Test<GetFewerFileOpenModes, GetFewerFileOpenModes>(GetMountName(0), GetMountName(1));
}

TEST_F(SaveDataFsRomFsStressTest, ConcurrentGetFileOpenModes)
{
    Test<ConcurrentGetFewerFileOpenModes, ConcurrentGetFewerFileOpenModes>(
        GetMountName(0),
        GetMountName(1));
}


TEST_F(RomFsStressTest, GetFileSizes)
{
    Test<GetFileSizes>(GetMountName());
}

TEST_F(RomFsStressTest, ConcurrentGetFileSizes)
{
    Test<ConcurrentGetFileSizes>(GetMountName());
}

TEST_F(RomFsStressTest, GetFileOpenModes)
{
    Test<GetFileOpenModes>(GetMountName());
}

TEST_F(RomFsStressTest, ConcurrentGetFileOpenModes)
{
    Test<ConcurrentGetFileOpenModes>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)
TEST_F(HostFsStressTest, GetFileSizes)
{
    Test<GetFileSizes>(GetMountName());
}

TEST_F(HostFsStressTest, ConcurrentGetFileSizes)
{
    Test<ConcurrentGetFileSizes>(GetMountName());
}

TEST_F(HostFsStressTest, GetFileOpenModes)
{
    Test<GetFileOpenModes>(GetMountName());
}

TEST_F(HostFsStressTest, ConcurrentGetFileOpenModes)
{
    Test<ConcurrentGetFileOpenModes>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_HOST_FS)

TEST_F(SdCardFsStressTest, GetFileSizes)
{
    Test<GetFileSizes>(GetMountName());
}

TEST_F(SdCardFsStressTest, ConcurrentGetFileSizes)
{
    Test<ConcurrentGetFileSizes>(GetMountName());
}

TEST_F(SdCardFsStressTest, GetFileOpenModes)
{
    Test<GetFileOpenModes>(GetMountName());
}

TEST_F(SdCardFsStressTest, ConcurrentGetFileOpenModes)
{
    Test<ConcurrentGetFileOpenModes>(GetMountName());
}

#if defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)
TEST_F(TemporaryStorageFsStressTest, GetFileSizes)
{
    Test<GetFileSizes>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, ConcurrentGetFileSizes)
{
    Test<ConcurrentGetFileSizes>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, GetFileOpenModes)
{
    Test<GetFileOpenModes>(GetMountName());
}

TEST_F(TemporaryStorageFsStressTest, ConcurrentGetFileOpenModes)
{
    Test<ConcurrentGetFileOpenModes>(GetMountName());
}
#endif // defined(NNT_FS_STRESS_TEST_SUPPORTS_TEMPORARY_STORAGE)

}}
