﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testFs_Stress_AccessDeepDirectoriesTestCase.h"

namespace nnt { namespace fs {

NN_DEFINE_STATIC_CONSTANT(const int AccessDeepDirectoriesTestCase::PathLength);

AccessDeepDirectoriesTestCase::AccessDeepDirectoriesTestCase(
    int threadCount,
    int depthMax) NN_NOEXCEPT
    : m_ThreadCount(threadCount),
      m_DepthMax(depthMax)
{
}

AccessDeepDirectoriesTestCase::~AccessDeepDirectoriesTestCase() NN_NOEXCEPT
{
}

int AccessDeepDirectoriesTestCase::GetThreadCount() const NN_NOEXCEPT
{
    return m_ThreadCount;
}

void AccessDeepDirectoriesTestCase::SetUp(FsStressTest* pTest) NN_NOEXCEPT
{
    FailAll();
    std::unique_ptr<char[]> path(new char[PathLength]);
    for( auto threadIndex = 0; threadIndex < GetThreadCount(); ++threadIndex )
    {
        if( pTest->IsReadOnly(GetTestCaseIndex()) )
        {
            // エントリの存在を確認する
            for( auto depth = 0; depth < GetDepthMax(); ++depth )
            {
                MakePath(path.get(), PathLength, threadIndex, depth);
                nn::fs::DirectoryEntryType type;
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetEntryType(&type, path.get()));
                ASSERT_EQ(nn::fs::DirectoryEntryType_Directory, type);
            }
        }
        else
        {
            // エントリを作成する
            CreateDirectories(path.get(), PathLength, threadIndex);
        }
    }
    SucceedAll();
}

void AccessDeepDirectoriesTestCase::TearDown(FsStressTest* pTest) NN_NOEXCEPT
{
    std::unique_ptr<char[]> path(new char[PathLength]);
    if( !pTest->IsReadOnly(GetTestCaseIndex()) )
    {
        // エントリを削除する
        for( auto threadIndex = 0; threadIndex < GetThreadCount(); ++threadIndex )
        {
            MakePath(path.get(), PathLength, threadIndex, 0);
            const auto result = nn::fs::DeleteDirectoryRecursively(path.get());
            if( !nn::fs::ResultPathNotFound::Includes(result) )
            {
                NNT_ASSERT_RESULT_SUCCESS(result);
            }
        }
    }
}

void AccessDeepDirectoriesTestCase::CreateDirectories(
    char* buffer,
    size_t bufferLength,
    int entryIndex) NN_NOEXCEPT
{
    for( auto depth = 0; depth < GetDepthMax(); ++depth )
    {
        MakePath(buffer, bufferLength, entryIndex, depth);
        {
            const auto result = nn::fs::CreateDirectory(buffer);
            if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
            {
                NNT_ASSERT_RESULT_SUCCESS(result);
            }
        }

        const auto length = strnlen(buffer, bufferLength - 1);
        nn::util::SNPrintf(buffer + length, bufferLength - length, "/sub");
        {
            const auto result = nn::fs::CreateFile(buffer, 0);
            if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
            {
                NNT_ASSERT_RESULT_SUCCESS(result);
            }
        }
    }
}

}}
