﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_SystemSaveData.h>

using namespace nn;

namespace {
    const size_t SaveDataSize = 16 * 1024 * 1024;
    const size_t SaveDataJournalSize = 16 * 1024 * 1024;
    const nn::fs::UserId TestUserId1 = { { 0, 1 } };
    const nn::fs::UserId TestUserId2 = { { 0, 2 } };
    const nn::fs::SystemSaveDataId TestSystemSaveDataId1 = 0x8000000000004000ULL;
    const nn::fs::SystemSaveDataId TestSystemSaveDataId2 = 0x8000000000004001ULL;

    const size_t StackSize = 64 * 1024;
    bool g_MultiThreadTestExit;
}


namespace {

bool IsTestSaveDataInfo(nn::fs::SaveDataInfo& info) NN_NOEXCEPT
{
    return ((info.saveDataUserId == TestUserId1) || (info.saveDataUserId == TestUserId2)) &&
           ((info.applicationId == nnt::fs::util::TestApplicationId0) || (info.applicationId == nnt::fs::util::TestApplicationId1));
}

Result ReadTestSaveDataInfo(std::unique_ptr<nn::fs::SaveDataIterator>& iter, nn::fs::SaveDataInfo* pInfo) NN_NOEXCEPT
{
    int64_t count = 0;
    NN_RESULT_DO(iter->ReadSaveDataInfo(&count, pInfo, 1));
    EXPECT_EQ(1, count);
    NN_LOG("ApplicationId=%016llx, UserId=%016llx:%016llx SaveDataId=%016llx\n", pInfo->applicationId.value, pInfo->saveDataUserId._data[0], pInfo->saveDataUserId._data[1], pInfo->saveDataId);
    EXPECT_TRUE(IsTestSaveDataInfo(*pInfo));
    NN_RESULT_SUCCESS;
}

Result ReadTestSaveDataInfoEnd(std::unique_ptr<nn::fs::SaveDataIterator>& iter) NN_NOEXCEPT
{
    nn::fs::SaveDataInfo info;
    int64_t count = 0;
    NN_RESULT_DO(iter->ReadSaveDataInfo(&count, &info, 1));
    EXPECT_EQ(0, count);
    NN_RESULT_SUCCESS;
}

void ReadSaveDataInfoThreadFunction(void* arg) NN_NOEXCEPT
{
    auto id = reinterpret_cast<int64_t>(arg);
    int loop = 0;
    while (!g_MultiThreadTestExit)
    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));

        while (NN_STATIC_CONDITION(true))
        {
            nn::fs::SaveDataInfo info;
            int64_t count = 0;
            NNT_ASSERT_RESULT_SUCCESS(iter->ReadSaveDataInfo(&count, &info, 1));
            if (count == 0)
            {
                break;
            }
        }

        loop++;
        if ((loop % 100) == 0)
        {
            NN_LOG("ReadSaveDataInfoThreadFunction(%lld) : %d\n", id, loop);
        }
    }
}

void CreateUserSaveDataThreadFunction(void* arg) NN_NOEXCEPT
{
    auto id = reinterpret_cast<int64_t>(arg);
    int loop = 0;
    while (!g_MultiThreadTestExit)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId0, TestUserId2, 0, SaveDataSize, SaveDataJournalSize, 0));

        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));
        nn::fs::SaveDataInfo info;
        int64_t count;
        while (NN_STATIC_CONDITION(true))
        {
            count = 0;
            NNT_ASSERT_RESULT_SUCCESS(iter->ReadSaveDataInfo(&count, &info, 1));
            if (count == 0)
            {
                break;
            }
            if ((info.saveDataUserId == TestUserId2) && (info.applicationId == nnt::fs::util::TestApplicationId0))
            {
                break;
            }
        }
        EXPECT_EQ(1, count);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(info.saveDataId));

        loop++;
        if ((loop % 100) == 0)
        {
            NN_LOG("CreateUserSaveDataThreadFunction(%lld) : %d\n", id, loop);
        }
    }
}

void CreateSystemSaveDataThreadFunction(void* arg) NN_NOEXCEPT
{
    auto id = reinterpret_cast<int64_t>(arg);
    int loop = 0;
    while (!g_MultiThreadTestExit)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(TestSystemSaveDataId2, SaveDataSize, SaveDataJournalSize, 0));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(TestSystemSaveDataId2));

        loop++;
        if ((loop % 100) == 0)
        {
            NN_LOG("CreateSystemSaveDataThreadFunction(%lld) : %d\n", id, loop);
        }
    }
}

}

TEST(SaveDataIterator, ReadSaveDataInfo)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId0, TestUserId1, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId0, TestUserId2, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId1, TestUserId1, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(TestSystemSaveDataId1, SaveDataSize, SaveDataJournalSize, 0));

    // ユーザアカウントセーブを列挙
    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));

        nn::fs::SaveDataInfo info;
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfoEnd(iter));
    }
    nnt::fs::util::DeleteAllTestSaveData();
}

TEST(SaveDataIterator, ReadAndDeleteUserSaveData)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId0, TestUserId1, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId0, TestUserId2, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId1, TestUserId1, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(TestSystemSaveDataId1, SaveDataSize, SaveDataJournalSize, 0));

    // ユーザアカウントセーブを列挙しながら削除
    nn::fs::SaveDataId lastSaveDataId;
    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));

        nn::fs::SaveDataInfo info;
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(info.saveDataId));

        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(info.saveDataId));

        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        lastSaveDataId = info.saveDataId;

        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfoEnd(iter));
    }
    // 列挙前に削除
    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(lastSaveDataId));

        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfoEnd(iter));
    }
}

TEST(SaveDataIterator, ReadAndCreateUserSaveData)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId1, TestUserId1, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId1, TestUserId2, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(TestSystemSaveDataId1, SaveDataSize, SaveDataJournalSize, 0));

    // ユーザアカウントセーブの列挙中に作成
    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId0, TestUserId1, 0, SaveDataSize, SaveDataJournalSize, 0));

        nn::fs::SaveDataInfo info;
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId0, TestUserId2, 0, SaveDataSize, SaveDataJournalSize, 0));

        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfoEnd(iter));
    }
    nnt::fs::util::DeleteAllTestSaveData();
}

TEST(SaveDataIterator, ReadAndDeleteSystemSaveData)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId0, TestUserId1, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId0, TestUserId2, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId1, TestUserId1, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(TestSystemSaveDataId1, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(TestSystemSaveDataId2, SaveDataSize, SaveDataJournalSize, 0));

    // ユーザアカウントセーブの列挙中にシステムセーブを削除
    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(TestSystemSaveDataId1));

        nn::fs::SaveDataInfo info;
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(TestSystemSaveDataId2));

        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfoEnd(iter));
    }
    nnt::fs::util::DeleteAllTestSaveData();
}

TEST(SaveDataIterator, ReadAndCreateSystemSaveData)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId0, TestUserId1, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId0, TestUserId2, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId1, TestUserId1, 0, SaveDataSize, SaveDataJournalSize, 0));

    // ユーザアカウントセーブの列挙中にシステムセーブを作成
    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(TestSystemSaveDataId1, SaveDataSize, SaveDataJournalSize, 0));

        nn::fs::SaveDataInfo info;
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(TestSystemSaveDataId2, SaveDataSize, SaveDataJournalSize, 0));

        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfoEnd(iter));
    }
    nnt::fs::util::DeleteAllTestSaveData();
}

/* 複数のイテレータでアカウントセーブを列挙中に操作 */
TEST(SaveDataIterator, MultipleOpen)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId0, TestUserId1, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId0, TestUserId2, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId1, TestUserId1, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(TestSystemSaveDataId1, SaveDataSize, SaveDataJournalSize, 0));

    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter1;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter1, nn::fs::SaveDataSpaceId::User));

        nn::fs::SaveDataInfo info;
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter1, &info));

        // 列挙途中に別のイテレータで列挙
        {
            std::unique_ptr<nn::fs::SaveDataIterator> iter2;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter2, nn::fs::SaveDataSpaceId::User));

            NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter2, &info));
            NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter2, &info));

            // 削除
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(info.saveDataId));

            NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter2, &info));
            NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfoEnd(iter2));
        }

        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter1, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfoEnd(iter1));
    }
    nnt::fs::util::DeleteAllTestSaveData();
}

/* 一度オープンしたイテレータを途中で再オープンし、セーブを列挙 */
TEST(SaveDataIterator, ReuseIterator)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId0, TestUserId1, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId0, TestUserId2, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId1, TestUserId1, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(TestSystemSaveDataId1, SaveDataSize, SaveDataJournalSize, 0));

    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));

        nn::fs::SaveDataInfo info;
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));

        // 列挙途中に再オープン
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));

        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));

        // 削除
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(info.saveDataId));

        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfoEnd(iter));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));

        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfoEnd(iter));
    }
    nnt::fs::util::DeleteAllTestSaveData();
}

/* セーブの作成、削除、列挙を複数スレッドで輻輳させても問題ないことの確認 */
TEST(SaveDataIterator, MultiThreads)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId0, TestUserId1, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId1, TestUserId1, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::TestApplicationId1, TestUserId2, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(TestSystemSaveDataId1, SaveDataSize, SaveDataJournalSize, 0));

    nn::os::ThreadFunction function[] = {ReadSaveDataInfoThreadFunction, ReadSaveDataInfoThreadFunction, CreateUserSaveDataThreadFunction, CreateSystemSaveDataThreadFunction};
    const int ThreadCount = sizeof(function) / sizeof(function[0]);

    static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
    nn::os::ThreadType threads[ThreadCount];

    g_MultiThreadTestExit = false;

    // スレッドの生成、起動
    for (int i = 0; i < ThreadCount; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(&threads[i], function[i], reinterpret_cast<void*>(i), stack[i], StackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&threads[i]);
    }

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(60 * 3));
    g_MultiThreadTestExit = true;

    // スレッドの終了待機
    for (int i = 0; i < ThreadCount; i++)
    {
        nn::os::WaitThread(&threads[i]);
        nn::os::DestroyThread(&threads[i]);
    }

    /* Verify */
    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));
        nn::fs::SaveDataInfo info;
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfo(iter, &info));
        NNT_ASSERT_RESULT_SUCCESS(ReadTestSaveDataInfoEnd(iter));
    }
    nnt::fs::util::DeleteAllTestSaveData();
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetEnabledAutoAbort(false);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
