﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SdCardForDebug.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

namespace {

char* g_SignedSystemPartitionFilePath = nullptr;

// usage:
//  - $SIGLO_ROOT/Tools/CommandLineTools/RunOnTarget.exe $SIGLO_ROOT/Tests/Outputs/NX-NXFP2-a64/Tests/testFs_Integration_SafeMode_SetUpVerify/Develop/testFs_Integration_SafeMode_SetUpVerify.nsp -- --gtest_filter="*SetUp*" $SIGLO_ROOT/Tests/Outputs/NX-NXFP2-a64/SystemImages/SystemPartition/TestSignedSafeModePartition/Develop/TestSignedSafeModePartition.fatimg
//  - シャットダウン
//  - safe mode 起動
//  - 再起動
//  - $SIGLO_ROOT/Tools/CommandLineTools/RunOnTarget.exe $SIGLO_ROOT/Tests/Outputs/NX-NXFP2-a64/Tests/testFs_Integration_SafeMode_SetUpVerify/Develop/testFs_Integration_SafeMode_SetUpVerify.nsp -- --gtest_filter="*Verify*"

void SetUpSignedSystemPartitionOnSd()
{
    // テスト結果削除
    (void)(DeleteFile("sd:/safeModeTestResult.bin"));

    EXPECT_NE(nullptr, g_SignedSystemPartitionFilePath);

    // SignedSystemPartition on sd
    NNT_ASSERT_RESULT_SUCCESS(MountHostRoot());
    const int BufferSize = 1024 * 1024;
    auto buffer = AllocateBuffer(BufferSize);
    (void)(DeleteFile("sd:/recovery"));
    NN_LOG("Copy %s -> %s\n", g_SignedSystemPartitionFilePath, "sd:/recovery");

    NNT_ASSERT_RESULT_SUCCESS(CopyFile(g_SignedSystemPartitionFilePath, "sd:/recovery", buffer.get(), BufferSize));
}

// 指定ファイルの指定オフセットの LSB 1bit を反転する
void CorruptFile(const char* path, int64_t offset)
{
    FileHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(OpenFile(&handle, path, OpenMode_Write | OpenMode_Read));

    char buffer[1];
    size_t readSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadFile(&readSize, handle, offset, buffer, sizeof(buffer), ReadOption()));
    ASSERT_EQ(1, readSize);
    buffer[0] ^= 0x1;
    NN_ABORT_UNLESS_RESULT_SUCCESS(WriteFile(handle, offset, buffer, sizeof(buffer), nn::fs::WriteOption::MakeValue(WriteOptionFlag_Flush)));
    CloseFile(handle);
}


TEST(SafeMode, SetUp)
{
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd"));
    SetUpSignedSystemPartitionOnSd();
    Unmount("sd");

    NNT_ASSERT_RESULT_SUCCESS(CreateAndMountSystemSaveData("ssave", 0x8000000000004000, TestUserId1));
    Unmount("ssave");
}

// 署名改竄
TEST(SafeMode, SetUpCorruptedSignature)
{
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd"));
    SetUpSignedSystemPartitionOnSd();
    CorruptFile("sd:/recovery", 0);
    Unmount("sd");
}

// ヘッダ改竄（先頭
TEST(SafeMode, SetUpCorruptedHeader1)
{
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd"));
    SetUpSignedSystemPartitionOnSd();
    CorruptFile("sd:/recovery", 256);
    Unmount("sd");
}

// ヘッダ改竄（末尾
TEST(SafeMode, SetUpCorruptedHeader2)
{
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd"));
    SetUpSignedSystemPartitionOnSd();
    CorruptFile("sd:/recovery", 511);
    Unmount("sd");
}

// encrypted content (fatimg) 改竄
TEST(SafeMode, SetUpCorruptedEncryptedContent)
{
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd"));
    SetUpSignedSystemPartitionOnSd();
    CorruptFile("sd:/recovery", 512);
    Unmount("sd");
}


// TestSafeMode の結果の確認 ＆ TearDown
TEST(SafeMode, Verify)
{
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd"));

    size_t readSize = 0;
    auto buffer = AllocateBuffer(32);
    NNT_EXPECT_RESULT_SUCCESS(ReadFileAtOnce(&readSize, &buffer, "sd:/safeModeTestResult.bin"));
    EXPECT_EQ(readSize, 20);
    DumpBuffer(buffer.get(), readSize);
    EXPECT_TRUE(IsFilledWithValue(buffer.get(), readSize, 0));

    (void)(DeleteFile("sd:/safeModeTestResult.bin"));
    Unmount("sd");
}


// 改竄検出により TestSafeMode が走らなかったことの確認 ＆ TearDown
TEST(SafeMode, VerifyFailure)
{
    NNT_ASSERT_RESULT_SUCCESS(MountSdCardForDebug("sd"));
    DirectoryEntryType type;
    NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetEntryType(&type, "sd:/safeModeTestResult.bin"));
    Unmount("sd");
}


}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    nn::fs::SetEnabledAutoAbort(false);

    ::testing::InitGoogleTest(&argc, argv);

    if (argc > 1)
    {
        g_SignedSystemPartitionFilePath = argv[1];
    }

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
