﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_SaveDataForDebug.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_ResultHandler.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

namespace {

    const uint64_t OwnerId = 0x0005000c10000000;
    const nn::ncm::ApplicationId ApplicationId = { OwnerId };
    const nn::fs::UserId UserId = { { 0, 1 } };

    const auto SaveDataSize = 16 * 1024 * 1024;
    const auto JournalSize = 16 * 1024 * 1024;
    const auto ExtendDataSize = 16 * 1024 * 1024;
    const auto ExtendJournalSize = 16 * 1024 * 1024;
    const auto SaveDataCount = 30;

    const auto FileSize = 256 * 1024;
    const auto FileCount = 60;

    int s_Buffer[FileSize / sizeof(int)] = {};

    std::pair<nn::fs::SaveDataId, bool> s_SaveDataIds[SaveDataCount] = {};

    void MakePath(char* buffer, size_t length, int index) NN_NOEXCEPT
    {
        nn::util::SNPrintf(buffer, length, "save:/%d", index);
    }

}

void CreateSaveDataAndFiles(nn::fs::UserId userId)
{
    nn::Result result;

    result = nn::fs::CreateSaveData(
        ApplicationId,
        userId,
        OwnerId,
        SaveDataSize,
        JournalSize,
        0);
    if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
    {
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    result = nn::fs::MountSaveData("save", ApplicationId, userId);
    NNT_ASSERT_RESULT_SUCCESS(result);
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("save");
    };

    {
        std::mt19937 rng(nnt::fs::util::GetRandomSeed());
        std::generate(std::next(std::begin(s_Buffer)), std::end(s_Buffer), [&rng]() NN_NOEXCEPT
        {
            return std::uniform_int_distribution<>(0, 0xff)(rng);
        });
        s_Buffer[0] = std::accumulate(std::next(std::begin(s_Buffer)), std::end(s_Buffer), 0);
    }

    for( auto fileCount = 0; fileCount < FileCount; ++fileCount )
    {
        char path[16];
        MakePath(path, sizeof(path), fileCount);

        result = nn::fs::CreateFile(path, FileSize);
        if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
        {
            NNT_ASSERT_RESULT_SUCCESS(result);
        }

        nn::fs::FileHandle file;
        result = nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write);
        NNT_ASSERT_RESULT_SUCCESS(result);
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(file);
        };

        int64_t fileSize = 0;
        result = nn::fs::GetFileSize(&fileSize, file);
        NNT_ASSERT_RESULT_SUCCESS(result);
        if( fileSize < FileSize )
        {
            result = nn::fs::SetFileSize(file, FileSize);
            NNT_ASSERT_RESULT_SUCCESS(result);
        }

        result = nn::fs::WriteFile(
            file,
            0,
            s_Buffer,
            FileSize,
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    result = nn::fs::CommitSaveData("save");
    NNT_ASSERT_RESULT_SUCCESS(result);
}

TEST(PowerOffTest, PrepareReadWrite)
{
    nnt::fs::util::DeleteAllTestSaveData();
    CreateSaveDataAndFiles(UserId);
}

TEST(PowerOffTest, RepeatReadWrite)
{
    nn::Result result;

    result = nn::fs::MountSaveData("save", ApplicationId, UserId);
    NNT_ASSERT_RESULT_SUCCESS(result);
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("save");
    };

    for( ; ; )
    {
        for( auto fileIndex = 0; fileIndex < FileCount; ++fileIndex )
        {
            {
                char path[16];
                MakePath(path, sizeof(path), fileIndex);

                nn::fs::FileHandle file;
                result = nn::fs::OpenFile(
                    &file,
                    path,
                    nn::fs::OpenMode_Read | nn::fs::OpenMode_Write);
                NNT_ASSERT_RESULT_SUCCESS(result);
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseFile(file);
                };

                result = nn::fs::ReadFile(file, 0, s_Buffer, FileSize);
                NNT_ASSERT_RESULT_SUCCESS(result);

                EXPECT_EQ(
                    s_Buffer[0],
                    std::accumulate(std::next(std::begin(s_Buffer)), std::end(s_Buffer), 0));

                auto sum = 0;
                s_Buffer[0] += fileIndex;
                for( auto index = 1; index < sizeof(s_Buffer) / sizeof(s_Buffer[0]); ++index )
                {
                    s_Buffer[index] = (s_Buffer[index] + s_Buffer[index - 1]) % 0x10000;
                    sum += s_Buffer[index];
                }
                s_Buffer[0] = sum;

                result = nn::fs::WriteFile(
                    file,
                    0,
                    s_Buffer,
                    FileSize,
                    nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
                NNT_ASSERT_RESULT_SUCCESS(result);
            }

            result = nn::fs::CommitSaveData("save");
            NNT_ASSERT_RESULT_SUCCESS(result);
        }
    }
}

TEST(PowerOffTest, PrepareCreateDelete)
{
    nn::Result result;

    for( auto userId = UserId; userId._data[1] <= SaveDataCount; ++userId._data[1] )
    {
        result = nn::fs::CreateSaveData(
            ApplicationId,
            userId,
            OwnerId,
            SaveDataSize,
            JournalSize,
            0);
        if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
        {
            NNT_ASSERT_RESULT_SUCCESS(result);
        }
    }
}

TEST(PowerOffTest, RepeatCreateDelete)
{
    nn::Result result;

    for( ; ; )
    {
        for( auto& saveDataId : s_SaveDataIds )
        {
            saveDataId.second = false;
        }

        {
            std::unique_ptr<nn::fs::SaveDataIterator> iterator;

            result = nn::fs::OpenSaveDataIterator(&iterator, nn::fs::SaveDataSpaceId::User);
            NNT_ASSERT_RESULT_SUCCESS(result);

            nn::fs::SaveDataInfo information = {};
            int64_t readCount = -1;

            while( readCount != 0 )
            {
                result = iterator->ReadSaveDataInfo(&readCount, &information, 1);
                NNT_ASSERT_RESULT_SUCCESS(result);

                if( information.applicationId == ApplicationId
                    && 0 < information.saveDataUserId._data[1]
                    && information.saveDataUserId._data[1] <= SaveDataCount )
                {
                    s_SaveDataIds[information.saveDataUserId._data[1] - 1]
                        = std::make_pair(information.saveDataId, true);
                }
            }
        }

        EXPECT_LE(
            std::count_if(
                std::begin(s_SaveDataIds),
                std::end(s_SaveDataIds),
                [](const std::pair<nn::fs::SaveDataId, bool>& saveDataId) NN_NOEXCEPT
                {
                    return !saveDataId.second;
                }),
            1);

        for( const auto& saveDataId : s_SaveDataIds )
        {
            if( saveDataId.second )
            {
                result = nn::fs::DeleteSaveData(saveDataId.first);
                NNT_ASSERT_RESULT_SUCCESS(result);
            }

            const nn::fs::UserId userId
                = { { 0, static_cast<nn::Bit64>(&saveDataId - s_SaveDataIds + 1) } };
            result = nn::fs::CreateSaveData(
                ApplicationId,
                userId,
                OwnerId,
                SaveDataSize,
                JournalSize,
                0);
            NNT_ASSERT_RESULT_SUCCESS(result);
        }
    }
}

TEST(PowerOffTest, PrepareExtend)
{
    for( int i = 0; i < SaveDataCount; ++i )
    {
        const nn::fs::UserId userId = { { 0, static_cast<nn::Bit64>(i + 1) } };
        CreateSaveDataAndFiles(userId);
    }
}

TEST(PowerOffTest, RepeatExtend)
{
    nn::Result result;

    std::mt19937 rng(nnt::fs::util::GetRandomSeed());
    {
        std::generate(std::next(std::begin(s_Buffer)), std::end(s_Buffer), [&rng]() NN_NOEXCEPT
        {
            return std::uniform_int_distribution<>(0, 0xff)(rng);
        });
        s_Buffer[0] = std::accumulate(std::next(std::begin(s_Buffer)), std::end(s_Buffer), 0);
    }

    nn::fs::SaveDataInfo info[SaveDataCount];
    int extendCountList[SaveDataCount];

    NN_LOG("Start Verifying\n");
    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User)
        );
        int64_t saveDataCount = 0;
        NNT_ASSERT_RESULT_SUCCESS(
            iter->ReadSaveDataInfo(&saveDataCount, info, SaveDataCount)
        );
        ASSERT_EQ(SaveDataCount, saveDataCount);

        for( int saveIndex = 0; saveIndex < SaveDataCount; ++saveIndex )
        {
            extendCountList[saveIndex]
                = (info[saveIndex].saveDataSize - SaveDataSize + JournalSize)
                      / (ExtendDataSize + ExtendJournalSize);
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::MountSaveData("save", info[saveIndex].saveDataUserId)
            );
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CommitSaveData("save");
                nn::fs::Unmount("save");
            };

            int MaxFileCount = FileCount + extendCountList[saveIndex];
            for( int fileIndex = 0; fileIndex < MaxFileCount; ++fileIndex )
            {
                // 各セーブの各ファイルについて、想定外の上書きが行われていないかチェック
                char path[32] = {};
                MakePath(path, sizeof(path) / sizeof(path[0]), fileIndex);
                nn::fs::FileHandle file;
                result = nn::fs::OpenFile(
                    &file,
                    path,
                    nn::fs::OpenMode_Read | nn::fs::OpenMode_Write);

                if( nn::fs::ResultPathNotFound::Includes(result) )
                {
                    // 拡張後ファイル作成前に電源断された場合はファイルが存在しない
                    continue;
                }
                NNT_ASSERT_RESULT_SUCCESS(result);
                NN_UTIL_SCOPE_EXIT
                {
                    nn::fs::CloseFile(file);
                };

                std::unique_ptr<int[]> buffer(new int[FileSize / sizeof(int)]);
                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fs::ReadFile(
                        file,
                        0,
                        buffer.get(),
                        FileSize
                    )
                );
                int accumulateValue
                    = std::accumulate(&buffer[1], &buffer[FileSize / sizeof(int)], 0);
                if( buffer[0] != accumulateValue)
                {
                    if( fileIndex != FileCount + extendCountList[saveIndex])
                    {
                        // ファイル作成後ファイル書込前に電源断された場合でないならエラー
                        ASSERT_EQ(buffer[0], accumulateValue);
                    }
                    // 書き換えておく
                    NNT_ASSERT_RESULT_SUCCESS(
                        nn::fs::WriteFile(
                            file,
                            0,
                            &accumulateValue,
                            sizeof(int),
                            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)
                        )
                    );
                }
            }
        }
    }

    NN_LOG("Start ExtendSaveData\n");
    for( ;; )
    {
        int saveIndex = std::uniform_int_distribution<int>(0, SaveDataCount - 1)(rng);

        extendCountList[saveIndex] += 1;
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::ExtendSaveData(
               nn::fs::SaveDataSpaceId::User,
               info[saveIndex].saveDataId,
               SaveDataSize + ExtendDataSize * extendCountList[saveIndex],
               JournalSize + ExtendJournalSize * extendCountList[saveIndex]
           )
        );

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::MountSaveData("save", info[saveIndex].saveDataUserId)
        );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CommitSaveData("save");
            nn::fs::Unmount("save");
        };

        char path[32] = {};
        MakePath(path, sizeof(path) / sizeof(path[0]), FileCount + extendCountList[saveIndex]);

        // 拡張回数を名前にしたファイルを作成する
        // ファイルが既に存在する=拡張処理が無効化された
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::CreateFile(path, FileSize)
        );
        {
            nn::fs::FileHandle file;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::OpenFile(
                    &file,
                    path,
                    nn::fs::OpenMode_Write
                )
            );
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::WriteFile(
                    file,
                    0,
                    s_Buffer,
                    FileSize,
                    nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)
                )
            );
        }
    }
} // NOLINT(impl/function_size)

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
