﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fssystem/fs_AesXtsStorage.h>
#include <nn/fssystem/fs_AesCtrStorage.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/crypto/crypto_Aes128XtsEncryptor.h>
#include <nn/crypto/crypto_Aes128XtsDecryptor.h>
#include <nn/crypto/crypto_Aes128CtrEncryptor.h>
#include <nn/crypto/crypto_Aes128CtrDecryptor.h>



using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;
using namespace nn::fssystem;

const int TotalAccessSize = 2 * 8 * 1024 * 1024;
const int BufferSize = 4 * 1024 * 1024;

void ReadWrite(IStorage* pStorage, size_t startSize)
{
    auto buffer = AllocateBuffer(BufferSize);
    NN_ASSERT(buffer != nullptr);

    NN_LOG("buffer address: %p\n", buffer.get());

    const int base = 0; // 開始地点のオフセット
    NN_LOG("base offset: %d\n", base);
    NN_LOG("total size : %d\n", TotalAccessSize);
    NN_LOG("buf size     R[us]    W[us]   R[MB/s] W[MB/s]\n");


    for (auto accessSize = startSize; accessSize <= BufferSize; accessSize *= 2)
    {
        NN_ASSERT(accessSize <= BufferSize);

        nn::TimeSpan writeTime = 0;
        nn::TimeSpan readTime = 0;

        {
            for (int i = 0; i < static_cast<int>(TotalAccessSize / accessSize); i++)
            {
                nnt::fs::util::FillBufferWith32BitCount(buffer.get(), accessSize, i * accessSize);

                auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
                {
                    NNT_EXPECT_RESULT_SUCCESS(pStorage->Write(base + i * accessSize, buffer.get(), accessSize));
                }
                auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());

                writeTime += end - start;
            }

        }


        memset(buffer.get(), 0xCD, accessSize);

        {
            for (int i = 0; i < static_cast<int>(TotalAccessSize / accessSize); i++)
            {
                auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
                {
                    NNT_EXPECT_RESULT_SUCCESS(pStorage->Read(base + i * accessSize, buffer.get(), accessSize));
                }
                auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());

                readTime += end - start;
            }

        }

        // Summary
        NN_LOG("%8d: %8lld %8lld   %7.3f %7.3f\n", accessSize, readTime.GetMicroSeconds(), writeTime.GetMicroSeconds(),
            static_cast<double>(TotalAccessSize) / (1024 * 1024) / readTime.GetMicroSeconds() * 1000 * 1000,
            static_cast<double>(TotalAccessSize) / (1024 * 1024) / writeTime.GetMicroSeconds() * 1000 * 1000
        );
    }

}



class VacantStorage : public nn::fs::IStorage, public nn::fs::detail::Newable
{
public:

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(offset);
        NN_UNUSED(buffer);
        NN_UNUSED(size);
        NN_RESULT_SUCCESS;
    }

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(offset);
        NN_UNUSED(buffer);
        NN_UNUSED(size);
        NN_RESULT_SUCCESS;
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }

    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(size);
        NN_RESULT_SUCCESS;
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(outValue);
        NN_RESULT_SUCCESS;
    }
};


TEST(AesXtsStoragePerformance, readWrite)
{
    VacantStorage baseStorage;
    char key[2][AesXtsStorage::KeySize];
    char iv[AesXtsStorage::IvSize];
    memset(key, 1, sizeof(key));
    memset(iv, 0, sizeof(iv));

    for (int encryptionBlockSize = 512; encryptionBlockSize <= 128 * 1024; encryptionBlockSize *= 2)
    {
        NN_LOG("block size: %d\n", encryptionBlockSize);
        AesXtsStorage aesXtsStorage(&baseStorage, key[0], key[1], AesXtsStorage::KeySize, iv, AesXtsStorage::IvSize, encryptionBlockSize);
        ReadWrite(&aesXtsStorage, std::max(4 * 1024, encryptionBlockSize));
        NN_LOG("\n\n\n", encryptionBlockSize);
    }
}



TEST(AesCtrStoragePerformance, readWrite)
{
    VacantStorage baseStorage;
    char key[2][AesCtrStorage::KeySize];
    char iv[AesCtrStorage::IvSize];
    memset(key, 1, sizeof(key));
    memset(iv, 0, sizeof(iv));

    AesCtrStorage aesCtrStorage(&baseStorage, key[0], AesCtrStorage::KeySize, iv, AesCtrStorage::IvSize);
    ReadWrite(&aesCtrStorage, 4 * 1024);
}


TEST(MemoryStoragePerformance, readWrite)
{
    auto storageBuffer = AllocateBuffer(TotalAccessSize);
    MemoryStorage memoryStorage(storageBuffer.get(), TotalAccessSize);

    char key[2][AesCtrStorage::KeySize];
    char iv[AesCtrStorage::IvSize];
    memset(key, 1, sizeof(key));
    memset(iv, 0, sizeof(iv));

    ReadWrite(&memoryStorage, 512);
}


TEST(EncryptAes128XtsPerformance, readWrite)
{
    char key[2][AesXtsStorage::KeySize];
    char iv[AesXtsStorage::IvSize];
    memset(key, 1, sizeof(key));
    memset(iv, 0, sizeof(iv));
    auto storageBuffer = AllocateBuffer(TotalAccessSize);

    auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());

    nn::crypto::EncryptAes128Xts(storageBuffer.get(), TotalAccessSize, key[0], key[1], 16, iv, 16, storageBuffer.get(), TotalAccessSize);

    auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
    NN_LOG("elapsed: %8lld us.\n", (end - start).GetMicroSeconds());
    NN_LOG("velocity: %lf MB/s.\n", static_cast<double>(TotalAccessSize) / 1024 / 1024 / (end - start).GetMicroSeconds() * 1000 * 1000);
}


TEST(EncryptAes128CtrPerformance, readWrite)
{
    char key[2][AesXtsStorage::KeySize];
    char iv[AesXtsStorage::IvSize];
    memset(key, 1, sizeof(key));
    memset(iv, 0, sizeof(iv));
    auto storageBuffer = AllocateBuffer(TotalAccessSize);

    auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());

    nn::crypto::EncryptAes128Ctr(storageBuffer.get(), TotalAccessSize, key[0], 16, iv, 16, storageBuffer.get(), TotalAccessSize);

    auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
    NN_LOG("elapsed: %8lld us.\n", (end - start).GetMicroSeconds());
    NN_LOG("velocity: %lf MB/s.\n", static_cast<double>(TotalAccessSize) / 1024 / 1024 / (end - start).GetMicroSeconds() * 1000 * 1000);
}


TEST(AesXtsAddCounterPerformance, 32K)
{
    char key[2][AesXtsStorage::KeySize];
    char iv[AesXtsStorage::IvSize];
    memset(key, 1, sizeof(key));
    memset(iv, 0, sizeof(iv));

    char counter[AesXtsStorage::IvSize] = { 0 };

    auto start = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());

    for (int i = 0; i < 32768; i++)
    {
        AddCounter(counter, AesXtsStorage::IvSize, 1);
    }

    auto end = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
    NN_LOG("elapsed: %8lld us.\n", (end - start).GetMicroSeconds());
}


extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    const size_t BufferPoolSize = 8 * 1024 * 1024;
    static NN_ALIGNAS(4096) char g_BufferPool[BufferPoolSize];
    nn::fssystem::InitializeBufferPool(g_BufferPool, sizeof(g_BufferPool));

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
