﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_MmcPrivate.h>

#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>

void* operator new(size_t size) NN_NOEXCEPT
{
    return malloc(size);
}

void operator delete(void* p) NN_NOEXCEPT
{
    free(p);
}

namespace{


const int DivisionCountMax = 1   * 1024; // 小さいバッファサイズの場合、この数以上の read/write はしない
const int BufferSizeMin    = 1   * 1024;
const int BufferSizeMax    = 4   * 1024 * 1024;

} // namespace

// ROM をマウント後 ROM 内の特定ファイルを各バッファサイズで読み込む時間を計測する
TEST(PerformanceTestReadGapPatch, ReadFile)
{
    const char* TestFileName = "0_0";

    nnt::fs::util::PerformanceReadTestForPatchedRom(
        TestFileName, BufferSizeMin, BufferSizeMax, DivisionCountMax);
}

TEST(PerformanceTestReadGapPatch, ReadPatchedFiles)
{
    const int GapArray[] = {
        0, 16, 32, 48, 64, 80, 96, 112, 128, 256, 512
    };
    const int GapArrayLength = sizeof(GapArray) / sizeof(int);

    for( int i = 0; i < GapArrayLength; ++i )
    {
        NN_LOG("Gap Size: %d bytes\n", GapArray[i]);
        char fileName[256];
        nn::util::SNPrintf(fileName, sizeof(fileName), "%d_%d", i, GapArray[i]);
        nnt::fs::util::PerformanceReadTestForPatchedRom(
            fileName, BufferSizeMin, BufferSizeMax, DivisionCountMax);
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nnt::fs::util::SetFsTestPerformanceConfiguration();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    // MmcPatrol 休止
    nn::fs::SuspendMmcPatrol();

    auto testResult = RUN_ALL_TESTS();

    // MmcPatrol 再開
    nn::fs::ResumeMmcPatrol();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
