﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/os.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fssystem/fs_AesXtsFile.h>
#include <nn/util/util_FormatString.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsApi/testFs_Integration_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {
    const int HeapSize = 128 * 1024;
    char g_HeapStack[HeapSize];
    char g_RootDirPath[256];
}

namespace nnt { namespace fs { namespace api {
    class FsApiSdContentStorage : public FsApiIntegrationTestFileSystem
    {
    public:
        FsApiSdContentStorage() NN_NOEXCEPT
            : FsApiIntegrationTestFileSystem(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::SdCard))
        {
            NN_ABORT_UNLESS(nn::fs::MountContentStorage(nn::fs::ContentStorageId::SdCard).IsSuccess());
        }
        virtual ~FsApiSdContentStorage() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::fs::Unmount(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::SdCard));
        }
    };

    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, int index)
    {
        NN_UNUSED(index);
        std::unique_ptr<FsApiSdContentStorage> sdFs(new FsApiSdContentStorage());

        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Integration));
        info->fileSystem = std::move(sdFs);
        info->type = FileSystemType_FatFileSystem;
        info->rootDirPath = g_RootDirPath;

        auto& attribute = info->attribute;

        static const int FatNameLengthMax       = 255;     // prfile2 の LONG_NAME_CHARS
        static const int FatPathLengthMax       = 260 - 1; // prfile2 の LONG_NAME_PATH_CHARS
        static const int ContentRootPathLength  = 20;      // ContentStorage 内部で使用されるルートディレクトリ名の文字数 ("C:\Nintendo/Contents/")
        static const int ContentMountNameLength = 15;      // ContentStorage の内部マウント名の文字数 ("@SdCardContent:")
        attribute.directoryNameLengthMax = FatNameLengthMax;
        attribute.fileNameLengthMax      = FatNameLengthMax;
        attribute.directoryPathLengthMax
            = FatPathLengthMax + ContentMountNameLength - ContentRootPathLength;
        attribute.filePathLengthMax
            = FatPathLengthMax + ContentMountNameLength - ContentRootPathLength;

        attribute.fileSizeMax = 0xFFFFFFFF - fssystem::AesXtsFileHeader::Size;
        attribute.storageSize = 12 * 1024 * 1024 * 1024LL;

        attribute.fileOpenMax      = 30;
        attribute.directoryOpenMax = 20;

        attribute.speedEmulationTypeFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::MmcStorageContextEnabled)
                                         | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::SdCardStorageContextEnabled)
                                         | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::GameCardStorageContextEnabled)
                                         | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::UsbStorageContextEnabled);

        attribute.isSupportedMultiBytePath = false;
        attribute.isSupportedGetFreeSpaceSize = true;
        attribute.isConcatenationFileSystem = false;
        attribute.isSupportedQueryRange = true;
        attribute.isReadOverloadsSupported = false; // ストレージが遅いので無効化

        attribute.fileSizeAlignment = 1;

        *outValue = std::move(info);
    }
}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::api::LoadAllTests();

    nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountContentStorage(nn::fs::ContentStorageId::SdCard));
    nn::util::SNPrintf(g_RootDirPath, sizeof(g_RootDirPath), "%s:/API_TEST", nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::SdCard));
    nn::fs::DeleteDirectoryRecursively(g_RootDirPath);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(g_RootDirPath));
    nn::fs::Unmount(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::SdCard));

    auto ret = RUN_ALL_TESTS();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountContentStorage(nn::fs::ContentStorageId::SdCard));
    nn::fs::DeleteDirectory(g_RootDirPath);
    nn::fs::Unmount(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::SdCard));

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
