﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/os.h>
#include <nn/util/util_FormatString.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsApi/testFs_Integration_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {
    const int HeapSize = 128 * 1024;
    char g_HeapStack[HeapSize];
    char g_RootDirPath[256];
    nn::fs::BisPartitionId g_PartitionId = nn::fs::BisPartitionId::User;
    int64_t g_StorageSize;
}

namespace nnt { namespace fs { namespace api {
    class FsApiBuiltInStorageFileSystem : public FsApiIntegrationTestFileSystem
    {
    public:
        FsApiBuiltInStorageFileSystem() NN_NOEXCEPT
            : FsApiIntegrationTestFileSystem(GetBisMountName(g_PartitionId))
        {
            NN_ABORT_UNLESS(nn::fs::MountBis(g_PartitionId, nullptr).IsSuccess());
        }
        ~FsApiBuiltInStorageFileSystem() NN_NOEXCEPT
        {
            nn::fs::Unmount(GetBisMountName(g_PartitionId));
        }
    };

    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, int index)
    {
        NN_UNUSED(index);
        std::unique_ptr<FsApiBuiltInStorageFileSystem> bisFs(new FsApiBuiltInStorageFileSystem());

        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Integration));
        info->fileSystem = std::move(bisFs);
        info->type = FileSystemType_FatFileSystem;
        info->rootDirPath = g_RootDirPath;

        auto& attribute = info->attribute;

        static const int FatNameLengthMax   = 255;     // prfile2 の LONG_NAME_CHARS
        static const int FatPathLengthMax   = 260 - 1; // prfile2 の LONG_NAME_PATH_CHARS
        static const int DriveLetterLength  = 2;       // FatFileSystem で付与されるドライブレターの文字数 ("B:")
        const int BisMountNameLength = std::strlen(nn::fs::GetBisMountName(g_PartitionId)) + 1; // BIS の内部マウント名の文字数 (例: @User:)
        attribute.directoryNameLengthMax = FatNameLengthMax;
        attribute.fileNameLengthMax      = FatNameLengthMax;
        attribute.directoryPathLengthMax
            = FatPathLengthMax + BisMountNameLength - DriveLetterLength;
        attribute.filePathLengthMax
            = FatPathLengthMax + BisMountNameLength - DriveLetterLength;

        attribute.fileSizeMax = 0xFFFFFFFF;
        attribute.storageSize = g_StorageSize;

        attribute.fileOpenMax      = 30;
        attribute.directoryOpenMax = 20;

        attribute.speedEmulationTypeFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::MmcStorageContextEnabled);

        attribute.isSupportedMultiBytePath = false;
        attribute.isSupportedGetFreeSpaceSize = true;
        attribute.isConcatenationFileSystem = false;
        attribute.isSupportedQueryRange = true;

        attribute.fileSizeAlignment = 1;

        *outValue = std::move(info);
    }
}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    g_PartitionId = nn::fs::BisPartitionId::User;
    if (argc >= 2)
    {
        auto partitionName = argv[1];
        if (std::strcmp(partitionName, "User") == 0)
        {
            g_PartitionId = nn::fs::BisPartitionId::User;
        }
        else if (std::strcmp(partitionName, "CalibrationFile") == 0)
        {
            g_PartitionId = nn::fs::BisPartitionId::CalibrationFile;
        }
        else if (std::strcmp(partitionName, "SafeMode") == 0)
        {
            g_PartitionId = nn::fs::BisPartitionId::SafeMode;
        }
        else if (std::strcmp(partitionName, "System") == 0)
        {
            g_PartitionId = nn::fs::BisPartitionId::System;
        }
        else
        {
            NN_LOG("Unknown Partition Name : %s\n", partitionName);
            nnt::Exit(1);
        }
        NN_LOG("Partition is set: %s\n", partitionName);
    }

    nnt::fs::api::LoadAllTests();

    nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    static const auto StorageSizeMax = 12 * 1024 * 1024 * 1024LL;
    g_StorageSize = std::min<int64_t>(StorageSizeMax, nnt::fs::api::GetBisFreeSize(g_PartitionId));

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(g_PartitionId, nullptr));
    nn::util::SNPrintf(g_RootDirPath, sizeof(g_RootDirPath), "%s:/API_TEST", nn::fs::GetBisMountName(g_PartitionId));
    nn::fs::DeleteDirectoryRecursively(g_RootDirPath);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(g_RootDirPath));
    nn::fs::Unmount(GetBisMountName(g_PartitionId));

    auto ret = RUN_ALL_TESTS();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(g_PartitionId, nullptr));
    nn::fs::DeleteDirectory(g_RootDirPath);
    nn::fs::Unmount(GetBisMountName(g_PartitionId));

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
