﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <map>
#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/aoc.h>
#include <nn/fs.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IDirectory.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fs/fs_AddOnContent.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemData.h>
#include <nn/fs/fs_TemporaryStorage.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_ApplicationSaveDataManagement.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Debug.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_Optional.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/fsUtil/testFs_util_adapter.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include "testFs_Integration_AgingRw_TestCase.h"
#include "testFs_Integration_AgingRw_TestStat.h"
#include "testFs_Integration_AgingRw_IFileSystemProvider.h"

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

namespace {
    const ncm::ApplicationId TestApplicationId = { 0x010000000000B1C9 };

    os::Mutex g_MountIdMutex(false);
    int g_MountId = 0;
    int GetUniqueMountId()
    {
        std::lock_guard<os::Mutex> scopedLock(g_MountIdMutex);
        return g_MountId++;
    }

    os::Mutex g_RunnerIdMutex(false);
    int g_RunnerId = 0;
    int GetUniqueRunnerId()
    {
        std::lock_guard<os::Mutex> scopedLock(g_RunnerIdMutex);
        return g_RunnerId++;
    }

    int GetBalancedCoreNumber()
    {
        static os::Mutex s_Mutex(false);
        static int coreIndex = 0;
        std::lock_guard<os::Mutex> scopedLock(s_Mutex);

        coreIndex = (coreIndex + 1) % 3;
        return coreIndex;
    }

    nn::fs::PriorityRaw GetBalancedPriorityRaw()
    {
        static os::Mutex s_Mutex(false);
        static int priorityIndex = 0;
        static const nn::fs::PriorityRaw priorityList[] = {
            nn::fs::PriorityRaw_Realtime,
            nn::fs::PriorityRaw_Normal,
            nn::fs::PriorityRaw_Low,
            nn::fs::PriorityRaw_Background
        };
        std::lock_guard<os::Mutex> scopedLock(s_Mutex);

        priorityIndex = (priorityIndex + 1) % (sizeof(priorityList) / sizeof(nn::fs::PriorityRaw));
        return priorityList[priorityIndex];
    }

    class OptionParser
    {
    public:
        OptionParser(int argc, char** argv)
        {
            for(int i = 0; i < argc; ++i)
            {
                auto pArg = argv[i];
                auto pNextArg = argv[i + 1];
                if(pArg[0] == '-')
                {
                    if(i == argc - 1 || pNextArg[0] == '-')
                    {
                        m_Map[pArg] = "";
                    }
                    else
                    {
                        m_Map[pArg] = pNextArg;
                    }
                }
            }
        }
        bool HasKey(const char* key)
        {
            return m_Map.find(key) != m_Map.cend();
        }
        const char* GetValue(const char* key)
        {
            auto it = m_Map.find(key);
            return it != m_Map.cend() ? (*it).second.c_str() : nullptr;
        }
    private:
        std::map<String, String> m_Map;
    };
}

// 与えられた FsProvider から得られる fs 上で、与えられたテストケースを走らせる
// 1 fs 上でスレッド数いくつで回すか、マウントアンマウントをいつ行うか等を定める
class ITestRunner
{
public:
    virtual void Run(IFileSystemProvider* pFsProvider, TestCaseFunction function, TestStatManager* pStatManager) = 0;

protected:
    struct TestParam
    {
        IFileSystemProvider* pFsProvider;
        TestStatHolder* pTestStatHolder;
        TestCaseFunction function;
        int initialRandomSeed;
    };

    static void RunTest(void* arg) NN_NOEXCEPT
    {
        auto pFsProvider = static_cast<TestParam*>(arg)->pFsProvider;
        auto testCaseFunction = static_cast<TestParam*>(arg)->function;
        auto pTestStatHolder = static_cast<TestParam*>(arg)->pTestStatHolder;
        auto initialRandomSeed = static_cast<TestParam*>(arg)->initialRandomSeed;

        // スレッド名を設定
        os::SetThreadName(os::GetCurrentThread(), pTestStatHolder->GetName().c_str());

        // テストスレッドの用の RandomSeed を設定
        SetInitialThreadLocalRandomSeed(initialRandomSeed);

        // FS の優先度を設定
        nn::fs::PriorityRaw priority = GetBalancedPriorityRaw();
        nn::fs::SetPriorityRawOnCurrentThread(priority);
        NN_ABORT_UNLESS_EQUAL(priority, nn::fs::GetPriorityRawOnCurrentThread());
        pTestStatHolder->SetPriority(priority);

        // テストを実行
        testCaseFunction(pFsProvider, pTestStatHolder);
    }
};





// 実装

class FsProviderCommon : public IFileSystemProvider
{
public:
    const int MountRandomNameLength = 7;
public:
    explicit FsProviderCommon(const char* prefix)
        : m_MountMutex(false)
        , m_MountCount(0)
        , m_InitializeMutex(false)
        , m_InitializeCount(-1)
    {
        m_MountName = prefix;
        m_MountName += "_" + ToString(GetUniqueMountId());
    }

    virtual void Clean()
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_InitializeMutex);
        if (m_InitializeCount < 0)
        {
            CleanImpl();
            m_InitializeCount = 0;
            return;
        }

        if (m_InitializeCount > 0)
        {
            if (m_InitializeCount == 1)
            {
                CleanImpl();
            }
            m_InitializeCount--;
        }
    }

    virtual void Initialize()
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_InitializeMutex);
        if (m_InitializeCount < 0)
        {
            InitializeImpl();
            m_InitializeCount = 1;
            return;
        }

        if (m_InitializeCount == 0)
        {
            InitializeImpl();
        }
        m_InitializeCount++;
    }

    virtual void Mount()
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_MountMutex);
        ASSERT_TRUE(m_MountCount >= 0);
        if (m_MountCount == 0)
        {
            MountImpl();
        }
        m_MountCount++;
    }

    virtual void Unmount()
    {
        ASSERT_TRUE(m_MountCount > 0);
        std::lock_guard<nn::os::Mutex> scopedLock(m_MountMutex);
        if (m_MountCount == 1)
        {
            fs::Unmount(m_MountName.c_str());
        }
        m_MountCount--;
    }

    virtual String GetMountName()
    {
        return m_MountName;
    }

    virtual fsa::IFileSystem* GetFileSystem()
    {
        NN_ABORT("GetFileSystem() is not implemented");
    }

private:
    virtual void CleanImpl()
    {
    }
    virtual void InitializeImpl()
    {
    }
    virtual void MountImpl()
    {
    }

private:
    String m_MountName;
    nn::os::Mutex m_MountMutex;
    int m_MountCount;
    nn::os::Mutex m_InitializeMutex;
    int m_InitializeCount;

};

// Host
class HostFsProvider : public FsProviderCommon
{
public:
    explicit HostFsProvider()
        : FsProviderCommon("host")
    {
        m_HostDirectory.Create();
    }

private:
    virtual void MountImpl()
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(MountHost(GetMountName().c_str(), m_HostDirectory.GetPath().c_str()), "<%s>\n", GetMountName().c_str());
    }

private:
    nnt::fs::util::TemporaryHostDirectory m_HostDirectory;
};

class HostDirectoryFsProvider : public FsProviderCommon
{
public:
    explicit HostDirectoryFsProvider()
        : FsProviderCommon("hostdir")
        , m_RootDir("C:\\Windows\\Temp")
    {
    }

    void SetRootDir(const char* rootDir)
    {
        m_RootDir = rootDir;
    }

private:
    virtual void MountImpl()
    {
        // create root directory
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHostRoot());
        nn::Result result = nn::fs::CreateDirectory(m_RootDir.c_str());
        if( nn::fs::ResultPathAlreadyExists::Includes(result)
            || nn::fs::ResultTargetLocked::Includes(result) )
        {}
        else
        {
            NNT_ASSERT_RESULT_SUCCESS(result);
        }
        nn::fs::UnmountHostRoot();

        MY_ABORT_UNLESS_RESULT_SUCCESS(MountHost(GetMountName().c_str(), m_RootDir.c_str()), "<%s>\n", GetMountName().c_str());
    }

private:
    String m_RootDir;
};

// Sd
class SdFsProvider : public FsProviderCommon
{
public:
    SdFsProvider()
        : FsProviderCommon("sd")
    {
    }

private:
    virtual void MountImpl()
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(MountSdCardForDebug(GetMountName().c_str()), "<%s>\n", GetMountName().c_str());
    }
};

// Sd content storage
class SdContentStorageFsProvider : public FsProviderCommon
{
public:
    SdContentStorageFsProvider()
        : FsProviderCommon("sdcn")
    {
    }

private:
    virtual void MountImpl()
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(MountContentStorage(GetMountName().c_str(), ContentStorageId::SdCard), "<%s>\n", GetMountName().c_str());
    }
};

// Bis
class BisFsProvider : public FsProviderCommon
{
public:
    explicit BisFsProvider(BisPartitionId id)
        : FsProviderCommon("bis")
        , m_Id(id)
    {
    }

private:
    virtual void InitializeImpl()
    {
        // 内容ランダム生成 // TODO: runner に移動
        Mount();

        // TORIAEZU: 決め打ちで作業ディレクトリ削除
        DeleteDirectoryRecursively((GetMountName() + ":/0").c_str());
        DeleteDirectoryRecursively((GetMountName() + ":/1").c_str());
        DeleteDirectoryRecursively((GetMountName() + ":/2").c_str());

        Unmount();
    }

    virtual void MountImpl()
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(MountBis(GetMountName().c_str(), m_Id), "<%s>\n", GetMountName().c_str());
    }
private:
    const BisPartitionId m_Id;
};


// Save
class SaveFsProvider : public FsProviderCommon
{
public:
    // TORIAEZU: 固定サイズ
    const int64_t SaveDataSize = 128 * 1024 * 1024;
    const ncm::ApplicationId AppId = TestApplicationId0;

public:
    explicit SaveFsProvider(UserId userId)
        : FsProviderCommon("save")
        , m_UserId(userId)
    {
    }

private:
    virtual void CleanImpl()
    {
        nn::util::optional<SaveDataInfo> info;
        FindSaveData(&info, SaveDataSpaceId::User, [&](const SaveDataInfo& i) NN_NOEXCEPT
        {
            return (i.applicationId == AppId && i.saveDataUserId == m_UserId);
        });
        if (info != util::nullopt)
        {
            DeleteSaveData(info->saveDataId);
        }
    }

    virtual void InitializeImpl()
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(CreateSaveData(AppId, m_UserId, 0, SaveDataSize, SaveDataSize, 0), "<%s>\n", GetMountName().c_str());
    }

    virtual void MountImpl()
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(MountSaveData(GetMountName().c_str(), AppId, m_UserId), "<%s>\n", GetMountName().c_str());
    }

    // TODO: teardown

private:
    const UserId m_UserId;
};

// System save
class SystemSaveFsProvider : public FsProviderCommon
{
public:
    // TORIAEZU: 固定サイズ
    const int64_t SaveDataSize = 128 * 1024 * 1024;

public:
    explicit SystemSaveFsProvider(SystemSaveDataId id)
        : FsProviderCommon("ssave")
        , m_SpaceId(SaveDataSpaceId::System)
        , m_Id(id)
    {
    }

    SystemSaveFsProvider(SaveDataSpaceId spaceId, SystemSaveDataId id)
        : FsProviderCommon((String("ssave") + ToString(static_cast<int>(spaceId))).c_str())
        , m_SpaceId(spaceId)
        , m_Id(id)
    {
    }

private:
    virtual void CleanImpl()
    {
        DeleteSaveData(m_SpaceId, m_Id);
    }

    virtual void InitializeImpl()
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(CreateSystemSaveData(m_SpaceId, m_Id, 0, SaveDataSize, SaveDataSize, 0), "<%s>\n", GetMountName().c_str());
    }

    // TODO: teardown

    virtual void MountImpl()
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(MountSystemSaveData(GetMountName().c_str(), m_SpaceId, m_Id), "<%s>\n", GetMountName().c_str());
    }

private:
    const SaveDataSpaceId  m_SpaceId;
    const SystemSaveDataId m_Id;
};

// Temporary storage
class TemporaryStorageFsProvider : public FsProviderCommon
{
public:
    // TORIAEZU: 固定サイズ
    const int64_t TemporaryStorageSize = 256 * 1024 * 1024;

public:
    TemporaryStorageFsProvider()
        : FsProviderCommon("temp")
    {
    }

private:
    virtual void CleanImpl()
    {
        CleanUpTemporaryStorage();
    }

    virtual void InitializeImpl()
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(CreateTemporaryStorage(TestApplicationId, 0, TemporaryStorageSize, 0), "<%s>\n", GetMountName().c_str());
    }

    // TODO: teardown

    virtual void MountImpl()
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(MountTemporaryStorage(GetMountName().c_str()), "<%s>\n", GetMountName().c_str());
    }
};



// System data
class SystemDataFsProvider : public FsProviderCommon
{
public:
    explicit SystemDataFsProvider(ncm::SystemDataId id)
        : FsProviderCommon("data")
        , m_Id(id)
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(QueryMountSystemDataCacheSize(&m_BufferSize, m_Id), "<%s>\n", GetMountName().c_str());
        auto buffer = AllocateBuffer(m_BufferSize);
        m_Buffer.emplace(std::move(buffer));
    }

private:
    virtual void MountImpl()
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(MountSystemData(GetMountName().c_str(), m_Id, m_Buffer.value().get(), m_BufferSize), "<%s>\n", GetMountName().c_str());
    }

private:
    const ncm::SystemDataId m_Id;
    size_t m_BufferSize;
    util::optional<decltype(AllocateBuffer(0))> m_Buffer;
};


// RomFs
class RomFsProvider : public FsProviderCommon
{
public:
    RomFsProvider()
        : FsProviderCommon("rom")
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(QueryMountRomCacheSize(&m_BufferSize), "<%s>\n", GetMountName().c_str());
        auto buffer = AllocateBuffer(m_BufferSize);
        m_Buffer.emplace(std::move(buffer));
    }

private:
    virtual void MountImpl()
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(MountRom(GetMountName().c_str(), m_Buffer.value().get(), m_BufferSize), "<%s>\n", GetMountName().c_str());
    }

private:
    size_t m_BufferSize;
    util::optional<decltype(AllocateBuffer(0))> m_Buffer;
};

// SubDirectory fs
class SubDirFsProvider : public FsProviderCommon
{
public:
    explicit SubDirFsProvider(const char* rootPath)
        : FsProviderCommon("subfs")
        , m_RootPath(rootPath)
    {

    }

private:
    virtual void MountImpl()
    {
        std::unique_ptr<::SubdirectoryFileSystem> fs(new ::SubdirectoryFileSystem(m_RootPath.c_str()));
        NN_ABORT_UNLESS_NOT_NULL(fs);
        MY_ABORT_UNLESS_RESULT_SUCCESS(fsa::Register(GetMountName().c_str(), std::move(fs)), "<%s>\n", GetMountName().c_str());
    }

private:
    String m_RootPath;
};

// AOC
class AocFsProvider : public FsProviderCommon
{
public:
    AocFsProvider(nn::aoc::AddOnContentIndex aocIndex, int index)
        : FsProviderCommon((String("aoc") + ToString(index)).c_str())
        , m_IsValid(true)
        , m_AocIndex(aocIndex)
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountAddOnContentCacheSize(&m_BufferSize, m_AocIndex), "<%s>\n", GetMountName().c_str());

        auto buffer = AllocateBuffer(m_BufferSize);
        NN_ASSERT_NOT_NULL(buffer);
        m_Buffer.emplace(std::move(buffer));
    }

    AocFsProvider()
        : FsProviderCommon("invalid aoc")
        , m_IsValid(false)
    {
    }

private:
    virtual void MountImpl()
    {
        NN_ABORT_UNLESS(m_IsValid, "Error: aoc is not installed.\n");
        MY_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountAddOnContent(GetMountName().c_str(), m_AocIndex, m_Buffer.value().get(), m_BufferSize), "<%s>\n", GetMountName().c_str());
    }

private:
    bool m_IsValid;
    size_t m_BufferSize;
    nn::aoc::AddOnContentIndex m_AocIndex;
    util::optional<decltype(AllocateBuffer(0))> m_Buffer;
};

// インデックス付きキャッシュストレージ
class CacheStorageFsProvider : public FsProviderCommon
{
public:
    // TORIAEZU: 固定サイズ
    const int64_t CacheStorageSize = 64 * 1024 * 1024;
    const int64_t CacheStorageJournalSize = 64 * 1024 * 1024;

public:
    explicit CacheStorageFsProvider(int index)
        : FsProviderCommon("cache")
        , m_Index(index)
    {
    }

private:
    virtual void CleanImpl()
    {
        DeleteCacheStorage(m_Index);
    }

    virtual void InitializeImpl()
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(CreateCacheStorage(m_Index, CacheStorageSize, CacheStorageJournalSize), "<%s>\n", GetMountName().c_str());
    }

    // TODO: teardown

    virtual void MountImpl()
    {
        MY_ABORT_UNLESS_RESULT_SUCCESS(MountCacheStorage(GetMountName().c_str(), m_Index), "<%s>\n", GetMountName().c_str());
    }

    int m_Index;
};


// 渡された fs 上で渡されたテストケースを複数スレッドで回すランナー
template <int ThreadCount>
class MultiThreadRunner : public ITestRunner
{
public:
    const size_t StackSize = 48 * 1024;

    MultiThreadRunner()
        : m_Mutex(false)
        , m_RunnerId(GetUniqueRunnerId())
    {
    }

    virtual void Run(IFileSystemProvider* pFsProvider, TestCaseFunction function, TestStatManager* pStatManager)
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        TestStatHolder* pHolder[ThreadCount];
        TestParam testParam[ThreadCount];

        std::mt19937 random(nnt::fs::util::GetRandomSeed());

        for (int i = 0; i < ThreadCount; i++)
        {
            TestStat tmpStat;
            util::SNPrintf(tmpStat.message, tmpStat.MessageLegnthMax, "Initializing");
            pHolder[i] = pStatManager->Register();
            pHolder[i]->SetTestStat(tmpStat);
        }

        // fs 初期化・マウント
        pFsProvider->Initialize();
        pFsProvider->Mount();

        // ランナー固有の初期化


        // スレッドは上から渡してもいいかもだがとりあえず
        for (int i = 0; i < ThreadCount; i++)
        {
            pHolder[i]->SetName(pFsProvider->GetMountName().c_str());

            testParam[i].pFsProvider = pFsProvider;
            testParam[i].pTestStatHolder = pHolder[i];
            testParam[i].function = function;
            testParam[i].initialRandomSeed = random();

            auto buffer = AllocateAlignedBuffer(StackSize, os::ThreadStackAlignment);
            m_StackArray[i].emplace(std::move(buffer));

            MY_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_Threads[i], RunTest, &testParam[i], m_StackArray[i].value().get(), StackSize, nn::os::DefaultThreadPriority, GetBalancedCoreNumber()), pFsProvider->GetMountName().c_str());
        }

        // 他のランナーの初期化を少し待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(180));

        // 開始
        for (int i = 0; i < ThreadCount; i++)
        {
            os::StartThread(&m_Threads[i]);
        }

        // todo: 定期的にスレッド合流してアンマウント・再マウントとか
        // pFsProvider->Unmount();
        // pFsProvider->Mount();

        // 終了待ち
        for (int i = 0; i < ThreadCount; i++)
        {
            os::WaitThread(&m_Threads[i]);
            os::DestroyThread(&m_Threads[i]);
        }

        pFsProvider->Unmount();
    }
private:
    nn::os::Mutex m_Mutex;
    nn::os::ThreadType m_Threads[ThreadCount];
    util::optional<decltype(AllocateBuffer(0))> m_StackArray[ThreadCount];
    int m_RunnerId;
};


// 複数スレッド向けにディレクトリを掘ってツリーを生成してその中でテストケースを走らせるランナー
template <int ThreadCount>
class MultiThreadRwOnEachDirectoryTreeRunner : public ITestRunner
{
public:
    const size_t StackSize = 48 * 1024;

    MultiThreadRwOnEachDirectoryTreeRunner()
        : m_Mutex(false)
        , m_RunnerId(GetUniqueRunnerId())
    {
    }

    virtual void Run(IFileSystemProvider* pFsProvider, TestCaseFunction function, TestStatManager* pStatManager)
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        TestStatHolder* pHolder[ThreadCount];
        TestParam testParam[ThreadCount];

        std::mt19937 random(nnt::fs::util::GetRandomSeed());

        for (int i = 0; i < ThreadCount; i++)
        {
            TestStat tmpStat;
            util::SNPrintf(tmpStat.message, tmpStat.MessageLegnthMax, "Initializing");
            pHolder[i] = pStatManager->Register();
            pHolder[i]->SetTestStat(tmpStat);
        }

        // fs 初期化・マウント
        pFsProvider->Initialize();
        pFsProvider->Mount();

        // スレッドごとにディレクトリ作成、ツリー作成、SubDirFs かぶせ
        for (int i = 0; i < ThreadCount; i++)
        {
            auto testDirectoryPath = pFsProvider->GetMountName() + ":/" + ToString(m_RunnerId) + "_" + ToString(i) + "/";

            (void)(DeleteDirectoryRecursively(testDirectoryPath.c_str()), pFsProvider->GetMountName().c_str());

            pHolder[i]->SetName(testDirectoryPath.c_str());
            MY_ABORT_UNLESS_RESULT_SUCCESS(CreateDirectory(testDirectoryPath.c_str()), "<%s>\n", pFsProvider->GetMountName().c_str());

            int64_t totalSize = 1024 * 1024 * 40;
            while (NN_STATIC_CONDITION(true))
            {
                auto result = CreateTestDirectoryTreeRandomly(testDirectoryPath.c_str(), totalSize, 30);
                if (result.IsSuccess())
                {
                    break;
                }
                else if (ResultUsableSpaceNotEnough::Includes(result))
                {
                    NN_LOG("Retry CreateTestDirectoryTreeRandomly() at <%s>. \n", pFsProvider->GetMountName().c_str());
                    (void)(CleanDirectoryRecursively(testDirectoryPath.c_str()));
                    totalSize /= 2;
                    continue;
                }
                else
                {
                    MY_ABORT_UNLESS_RESULT_SUCCESS(result, pFsProvider->GetMountName().c_str());
                }
            }

            m_SubDirFsProviderArray[i].emplace(testDirectoryPath.c_str());

            m_SubDirFsProviderArray[i]->Initialize();
            m_SubDirFsProviderArray[i]->Mount();
        }


        // スレッドは上から渡してもいいかもだがとりあえず
        for (int i = 0; i < ThreadCount; i++)
        {
            testParam[i].pFsProvider = &(m_SubDirFsProviderArray[i].value());
            testParam[i].pTestStatHolder = pHolder[i];
            testParam[i].function = function;
            testParam[i].initialRandomSeed = random();

            auto buffer = AllocateAlignedBuffer(StackSize, os::ThreadStackAlignment);
            m_StackArray[i].emplace(std::move(buffer));

            MY_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&m_Threads[i], RunTest, &testParam[i], m_StackArray[i].value().get(), StackSize, nn::os::DefaultThreadPriority, GetBalancedCoreNumber()), pFsProvider->GetMountName().c_str());
        }

        // 他のランナーの初期化を少し待つ
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(120));

        // 開始
        for (int i = 0; i < ThreadCount; i++)
        {
            os::StartThread(&m_Threads[i]);
        }

        // todo: 定期的にスレッド合流してアンマウント・再マウントとか
        // pFsProvider->Unmount();
        // pFsProvider->Mount();

        // 終了待ち
        for (int i = 0; i < ThreadCount; i++)
        {
            os::WaitThread(&m_Threads[i]);
            os::DestroyThread(&m_Threads[i]);
            m_SubDirFsProviderArray[i]->Unmount();
        }

        pFsProvider->Unmount();
    }
private:
    nn::os::Mutex m_Mutex;
    nn::os::ThreadType m_Threads[ThreadCount];
    util::optional<decltype(AllocateBuffer(0))> m_StackArray[ThreadCount];
    util::optional<SubDirFsProvider> m_SubDirFsProviderArray[ThreadCount];
    int m_RunnerId;
};


struct Test {
    IFileSystemProvider* pFsProvider;
    ITestRunner*         pRunner;
    TestCaseFunction     pTestCase;
};



struct TestSet {

private:
    struct TestParam {
        TestStatManager* pStatManager;
        Test*            pTest;
        int              initialRandomSeed;
    };
    struct MonitorStatusParam {
        TestStatManager*    pStatManager;
        os::LightEventType* pExitEvent;
        int                 checkInterval;
    };

private:
    TestStatManager* m_pStatManager;
    Vector<Test> m_TestArray;
    String m_Name;

private:
    static void RunTest(void* arg)
    {
        auto pTestParam = static_cast<TestParam*>(arg);
        auto pTest = pTestParam->pTest;
        auto initialRandomSeed = pTestParam->initialRandomSeed;

        SetInitialThreadLocalRandomSeed(initialRandomSeed);
        pTest->pRunner->Run(pTest->pFsProvider, pTest->pTestCase, pTestParam->pStatManager);
    }

    static void MonitorStatus(void* arg)
    {
        auto pMonitorStatusParam = static_cast<MonitorStatusParam*>(arg);

        // LoopCount チェック用のタイマーを作成
        const int checkInterval = pMonitorStatusParam->checkInterval;
        nn::os::TimerEvent loopCountTimer(nn::os::EventClearMode_AutoClear);
        if(checkInterval > 0 )
        {
            loopCountTimer.StartPeriodic(nn::TimeSpan::FromMinutes(checkInterval), nn::TimeSpan::FromMinutes(checkInterval));
        }

        while(NN_STATIC_CONDITION(true))
        {
            // 各テストのステータスを表示
            pMonitorStatusParam->pStatManager->ShowAllStat();

            // 終了条件を満たしていたら抜ける
            if(os::TryWaitLightEvent(pMonitorStatusParam->pExitEvent))
            {
                break;
            }

            // LoopCount チェック用のタイマーが発火しているか確認
            if(loopCountTimer.TryWait())
            {
                // 各テストの LoopCount が更新されていることをチェック
                pMonitorStatusParam->pStatManager->CheckAllLoopCount();
            }

            nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
        }
    }

public:
    TestSet(const char* name, TestStatManager* pStatManager, Vector<Test>&& testArray)
        : m_pStatManager(pStatManager)
        , m_TestArray(std::move(testArray))
        , m_Name(name)
    {
    }

    void RunAllTest(int testDuration, int checkInterval, int randomSeed)
    {
        const size_t StackSize = 48 * 1024;

        // ステータス表示スレッドを生成・実行
        os::LightEventType monitorStatusThreadExitEvent;
        InitializeLightEvent(&monitorStatusThreadExitEvent, false, os::EventClearMode_ManualClear);

        nn::os::ThreadType monitorStatusThread;
        MonitorStatusParam monitorStatusThreadParam = { m_pStatManager, &monitorStatusThreadExitEvent, checkInterval };
        auto monitorStatusThreadStack = AllocateAlignedBuffer(StackSize, os::ThreadStackAlignment);

        os::CreateThread(&monitorStatusThread, MonitorStatus, &monitorStatusThreadParam, monitorStatusThreadStack.get(), StackSize, nn::os::DefaultThreadPriority, GetBalancedCoreNumber());
        os::StartThread(&monitorStatusThread);

        // クリーン
        for (int i = 0; i < static_cast<int>(m_TestArray.size()); i++)
        {
            m_TestArray[i].pFsProvider->Clean();
        }

        // 各テストランナー用のスレッドを生成・実行
        Vector<decltype(AllocateBuffer(0))> stackArray;
        Vector<std::unique_ptr<os::ThreadType>> threadArray;
        Vector<TestParam> testParamArray;

        std::mt19937 random(randomSeed);

        for (int i = 0; i < static_cast<int>(m_TestArray.size()); i++)
        {
            auto& test = m_TestArray[i];

            auto buffer = AllocateAlignedBuffer(StackSize, os::ThreadStackAlignment);
            std::unique_ptr<os::ThreadType> thread(new os::ThreadType());

            NN_ASSERT_NOT_NULL(buffer);
            NN_ASSERT_NOT_NULL(thread);
            threadArray.push_back(std::move(thread));
            stackArray.push_back(std::move(buffer));

            TestParam param = { m_pStatManager, &test, static_cast<int>(random()) };
            testParamArray.push_back(param);
        }

        for (int i = 0; i < static_cast<int>(m_TestArray.size()); i++)
        {
            MY_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(threadArray[i].get(), RunTest, &testParamArray[i], stackArray[i].get(), StackSize, nn::os::DefaultThreadPriority, GetBalancedCoreNumber()), "");
            os::StartThread(threadArray[i].get());

        }

        // 終了用のタイマー設定
        os::TimerEvent exitTimer(os::EventClearMode::EventClearMode_ManualClear);
        if(testDuration > 0)
        {
            exitTimer.StartOneShot(nn::TimeSpan::FromSeconds(testDuration));
        }

        // 終了条件待ち
        while(!exitTimer.TryWait())
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
        }

        // テストの終了要求
        m_pStatManager->RequireExitAll();

        // 全てのテストが終了するのを待つ
        for (int i = 0; i < static_cast<int>(threadArray.size()); i++)
        {
            os::WaitThread( threadArray[i].get() );
            os::DestroyThread( threadArray[i].get() );
        }

        // クリーン
        for (int i = 0; i < static_cast<int>(m_TestArray.size()); i++)
        {
            m_TestArray[i].pFsProvider->Clean();
        }

        // ステータス表示スレッドを終了
        os::SignalLightEvent(&monitorStatusThreadExitEvent);
        os::WaitThread( &monitorStatusThread );
        os::DestroyThread( &monitorStatusThread );

        // 各テストのステータスを表示して終了
        m_pStatManager->ShowAllStat();
    }

    const char* GetName()
    {
        return m_Name.c_str();
    }
};

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    auto optionParser = OptionParser(argc, argv);

    // ThreadLocalRandomSeed を初期化
    InitializeThreadLocalRandomSeed();
    SetInitialThreadLocalRandomSeed();

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();
    nnt::fs::util::DeleteAllTestSaveData();

    AllocateTestParameterTlsSlot();

    const UserId TestUserId1 = { { 0x00ull, 0x01ull } };
    const UserId TestUserId2 = { { 0x00ull, 0x01ull } };

    // fs provider のインスタンス作成
    // TODO: この辺が結局面倒なのをマシにする
    RomFsProvider romFsProvider;
    HostFsProvider hostFsProvider1;
    HostFsProvider hostFsProvider2;
    HostFsProvider hostFsProvider3;
    HostDirectoryFsProvider dataSrcFsProvider;
    BisFsProvider bisFsProvider1(BisPartitionId::User);
    BisFsProvider bisFsProvider2(BisPartitionId::System);
    SdFsProvider sdFsProvider1;
    SdContentStorageFsProvider sdContentStorageFsProvider1;
    SaveFsProvider saveFsProvider1(TestUserId1);
    SaveFsProvider saveFsProvider2(TestUserId2);
    SystemSaveFsProvider sysSaveFsProvider1(0x8000000000004001);
    SystemSaveFsProvider sysSaveFsProvider2(0x8000000000004002);
    SystemSaveFsProvider sdSysSaveFsProvider3(SaveDataSpaceId::SdSystem, 0x8000000000004003);
    SystemSaveFsProvider sysSaveFsProvider4(0x8000000000004004);
    SystemSaveFsProvider sdSysSaveFsProvider5(SaveDataSpaceId::SdSystem, 0x8000000000004005);
    SystemSaveFsProvider sysSaveFsProvider6(0x8000000000004006);
    SystemSaveFsProvider sysSaveFsProvider7(0x8000000000004007);
    TemporaryStorageFsProvider tempStorageFsProvider;
    CacheStorageFsProvider cacheStorageFsProvider1(0);
    CacheStorageFsProvider cacheStorageFsProvider2(1);

#if defined(NN_BUILD_CONFIG_OS_WIN)
#else
    ncm::SystemDataId systemDataIdNgWord = { 0x0100000000000806 };
    SystemDataFsProvider systemDataFsProvider1(systemDataIdNgWord);

    // 追加コンテンツ数の取得
    int aocCount = nn::aoc::CountAddOnContent();
    NN_LOG("CountAddOnCountent -> %d\n", aocCount);

    // 追加コンテンツのリストアップ
    const int MaxListupCount = 256;
    nn::aoc::AddOnContentIndex aocList[MaxListupCount] = {0};
    int listupCount = nn::aoc::ListAddOnContent(aocList, 0, MaxListupCount);

    nn::util::optional<AocFsProvider> aocFsProvider1;
    nn::util::optional<AocFsProvider> aocFsProvider2;
    nn::util::optional<AocFsProvider> aocFsProvider3;
    if(listupCount >= 3)
    {
        aocFsProvider1.emplace(aocList[0], 0);
        aocFsProvider2.emplace(aocList[1], 1);
        aocFsProvider3.emplace(aocList[2], 2);
    }
    else
    {
        aocFsProvider1.emplace();
        aocFsProvider2.emplace();
        aocFsProvider3.emplace();
    }
#endif

    // runner のインスタンス作成
    MultiThreadRunner<3> runner1;
    MultiThreadRunner<3> runner2;
    MultiThreadRunner<3> runner3;
    MultiThreadRunner<3> runner4;
    MultiThreadRwOnEachDirectoryTreeRunner<3> runner5;
    MultiThreadRwOnEachDirectoryTreeRunner<3> runner6;
    MultiThreadRwOnEachDirectoryTreeRunner<3> runner7;
    MultiThreadRwOnEachDirectoryTreeRunner<3> runner8;
    MultiThreadRwOnEachDirectoryTreeRunner<3> runner9;
    MultiThreadRwOnEachDirectoryTreeRunner<3> runner10;
    MultiThreadRwOnEachDirectoryTreeRunner<3> runner11;
    MultiThreadRunner<3> runner12;
    MultiThreadRwOnEachDirectoryTreeRunner<3> runner13;
    MultiThreadRunner<1> runner14;
    MultiThreadRwOnEachDirectoryTreeRunner<1> runner15;
    MultiThreadRwOnEachDirectoryTreeRunner<1> runner16;
    MultiThreadRunner<1> runner17;
    MultiThreadRunner<1> runner18;
    MultiThreadRunner<1> runner19;
    MultiThreadRwOnEachDirectoryTreeRunner<1> runner20;
    MultiThreadRwOnEachDirectoryTreeRunner<1> runner21;

    MultiThreadRwOnEachDirectoryTreeRunner<2> runner22;
    MultiThreadRwOnEachDirectoryTreeRunner<2> runner23;
    MultiThreadRwOnEachDirectoryTreeRunner<2> runner24;
    MultiThreadRwOnEachDirectoryTreeRunner<2> runner25;
    MultiThreadRwOnEachDirectoryTreeRunner<2> runner26;
    MultiThreadRwOnEachDirectoryTreeRunner<2> runner27;
    MultiThreadRwOnEachDirectoryTreeRunner<2> runner28;
    MultiThreadRwOnEachDirectoryTreeRunner<2> runner29;

    MultiThreadRwOnEachDirectoryTreeRunner<3> runner30;
    MultiThreadRwOnEachDirectoryTreeRunner<2> runner31;

    TestStatManager statManager;

    // テストセット（プリセット）を引数で選ぶなど
    TestSet defaultTestSet(
            "default",
            &statManager,
            {
#if defined(NN_BUILD_CONFIG_OS_WIN)

                // Test{ &hostFsProvider1, &runner14, TestDisturbDirectoryTreeForPatchTest },
                // Test{ &hostFsProvider1, &runner14, TestCreateRandomDirectoryTree },
                //Test{ &hostFsProvider1, &runner1, TestCaseReadOnlyDumpDirectoryRecursive },
                //Test{ &hostFsProvider2, &runner2, TestCaseReadOnlyDumpDirectoryRecursive },
                //Test{ &hostFsProvider3, &runner3, TestCaseReadOnlyDumpDirectoryRecursive },
                Test{ &romFsProvider,                  &runner4,  TestCaseReadOnlyRandomWith32BitCountCheck },

                //Test{ &bisFsProvider1,   &runner5,  TestCaseReadOnlyDumpDirectoryRecursive }, // TODO: DumpDirectoryRecursiveWithWhiteList
                //Test{ &bisFsProvider2,   &runner6,  TestCaseReadOnlyDumpDirectoryRecursive },
                //Test{ &sdContentStorageFsProvider1,    &runner7,  TestCaseReadWriteRandom },
                Test{ &sdFsProvider1,                  &runner7,  TestCaseReadWriteWith32BitCount },

                Test{ &saveFsProvider1,                &runner8,  TestCaseReadWriteRandom },
                Test{ &sysSaveFsProvider1,             &runner9,  TestCaseReadWriteRandom },
                Test{ &sysSaveFsProvider2,             &runner10, TestCaseReadWriteWith32BitCount },

                //Test{ &tempStorageFsProvider,          &runner11, TestCaseReadWriteRandom },


#else
                Test{ &romFsProvider,                  &runner4,  TestCaseReadOnlyRandomWith32BitCountCheckAtCornerByFileNameOffset },
                Test{ &aocFsProvider1.value(),         &runner17, TestCaseReadOnlyRandomWith32BitCountCheckAtCornerByFileNameOffset },
                Test{ &aocFsProvider2.value(),         &runner18, TestCaseReadOnlyRandomWith32BitCountCheckAtCornerByFileNameOffset },
                Test{ &aocFsProvider3.value(),         &runner19, TestCaseReadOnlyRandomWith32BitCountCheckAtCornerByFileNameOffset },

                Test{ &bisFsProvider1,                 &runner5,  TestCaseReadWriteWith32BitCountAtCorner },
                Test{ &bisFsProvider2,                 &runner6,  TestCaseReadWriteRandomAtCorner },
                Test{ &sdContentStorageFsProvider1,    &runner7,  TestCaseReadWriteRandomAtCorner },

                Test{ &saveFsProvider1,                &runner8,  TestCaseReadWriteRandomAtCorner },
                Test{ &sysSaveFsProvider1,             &runner9,  TestCaseReadWriteRandomAtCorner },
                Test{ &sysSaveFsProvider2,             &runner10, TestCaseReadWriteWith32BitCountAtCorner },

                Test{ &tempStorageFsProvider,          &runner11, TestCaseReadWriteRandomAtCorner },

                Test{ &systemDataFsProvider1,          &runner12, TestCaseReadOnlyRandomAtCorner },

                Test{ &sdSysSaveFsProvider3,           &runner13, TestCaseReadWriteWith32BitCountAtCorner },
                Test{ &sysSaveFsProvider4,             &runner15,  TestCaseReadWriteCommitRandomAtCorner },
                Test{ &sdSysSaveFsProvider5,           &runner16,  TestCaseReadWriteCommitRandomAtCorner },
                Test{ &sysSaveFsProvider6,             &runner20,  TestCaseReadWriteCommitRandomAtCorner },
                Test{ &sysSaveFsProvider7,             &runner21,  TestCaseReadWriteCommitRandomAtCorner },

                Test{ &bisFsProvider1,                 &runner22, TestCaseCreateDeleteRandom },
                Test{ &bisFsProvider2,                 &runner23, TestCaseCreateDeleteRandom },
                Test{ &sdContentStorageFsProvider1,    &runner24, TestCaseCreateDeleteRandom },

                Test{ &saveFsProvider1,                &runner25, TestCaseCreateDeleteRandom },
                Test{ &sysSaveFsProvider1,             &runner26, TestCaseCreateDeleteRandom },
                Test{ &sysSaveFsProvider2,             &runner27, TestCaseCreateDeleteRandom },

                Test{ &tempStorageFsProvider,          &runner28, TestCaseCreateDeleteRandom },
                Test{ &sdSysSaveFsProvider3,           &runner29, TestCaseCreateDeleteRandom },

                Test{ &cacheStorageFsProvider1,        &runner30, TestCaseReadWriteRandomAtCorner },
                Test{ &cacheStorageFsProvider2,        &runner31, TestCaseCreateDeleteRandom },
#endif
        }
    );

    // または引数に応じて構築するなど

    TestSet presetTestSetArray[] = {

        defaultTestSet,

        {
            // rom 作成用プリセット：
            // C:/Windows/Temp/dataSrc が生成されるので、そのまま dataSrcV0, V1 にコピーして再ビルド
            "createrom",
            &statManager,
            {
                Test{ &dataSrcFsProvider, &runner14, TestCreateRandomDirectoryTree },
            }
        },

        {
            // パッチ時の rom v0 作成用プリセット:
            // c:/Windows/Temp/dataSrc 生成済みの状態で実行し、改変された dataSrc を dataSrvV0 にコピーしてパッチを作成
            "disturbrom",
            &statManager,
            {
                Test{ &dataSrcFsProvider, &runner14, TestDisturbDirectoryTreeForPatchTest },
            }
        },

        {
            // fs プロセス側のメモリ枯渇狙いのパラメータ
            "depletememory",
            &statManager,
            {
#if defined(NN_BUILD_CONFIG_OS_WIN)
#else
                Test{ &romFsProvider,                  &runner4,  TestCaseReadOnlyRandomWith32BitCountCheckAtCornerByFileNameOffset },
                Test{ &aocFsProvider1.value(),         &runner17, TestCaseReadOnlyRandomWith32BitCountCheckAtCornerByFileNameOffset },
                Test{ &aocFsProvider2.value(),         &runner18, TestCaseReadOnlyRandomWith32BitCountCheckAtCornerByFileNameOffset },
                Test{ &aocFsProvider3.value(),         &runner19, TestCaseReadOnlyRandomWith32BitCountCheckAtCornerByFileNameOffset },

                Test{ &sdContentStorageFsProvider1,    &runner7,  TestCaseReadOnlyRandom },
                Test{ &sysSaveFsProvider1,             &runner9,  TestCaseReadOnlyRandom },
                Test{ &tempStorageFsProvider,          &runner11, TestCaseReadOnlyRandom },
#endif
            }
        },


    };

    NN_LOG("----------------------------------------------------------- \n");
    NN_LOG("usage: \n");
    NN_LOG("RunOnTarget testFs_Integration_AgingRw.nsp -- [--test-set PresetTestSetName] [--duration TestDuration] [--check-interval CheckInterval] [--rand-seed RandomSeed] [--data-src DataSrcRoot] [--contents-on-sd ContentsNumOnSd] [--global-file-cache-size GlobalFileCacheSize]\n");
    NN_LOG("RunOnTarget 0x010000000000B1C9 -- [--test-set PresetTestSetName] [--duration TestDuration] [--check-interval CheckInterval] [--rand-seed RandomSeed] [--data-src DataSrcRoot] [--contents-on-sd ContentsNumOnSd] [--global-file-cache-size GlobalFileCacheSize]\n");
    NN_LOG("\n");
    NN_LOG("preset test set list:\n");
    for (auto& presetTestSet : presetTestSetArray)
    {
        NN_LOG("%s\n", presetTestSet.GetName());
    }
    NN_LOG("\n");
    NN_LOG("----------------------------------------------------------- \n");

    /*
        Options
        --test-set:                 テストセット名
        --duration:                 テスト実行時間 (sec)
        --rand-seed:                ランダムシード
        --global-file-cache-size:   ファイルキャッシュサイズ (MB)
        --data-src:                 データの読み込み先
        --contents-on-sd:           SD 上に置くコンテンツ数
    */
    const char* testSetOpt                  = "--test-set";
    const char* durationOpt                 = "--duration";
    const char* checkIntervalOpt            = "--check-interval";
    const char* randSeedOpt                 = "--rand-seed";
    const char* globalFileDataCacheSizeOpt  = "--global-file-cache-size";
    const char* dataSrcOpt                  = "--data-src";
    const char* contentsOnSdOpt             = "--contents-on-sd";

    // テストセット
    TestSet* pTestSet = nullptr;
    if(optionParser.HasKey(testSetOpt))
    {
        auto testSet = optionParser.GetValue(testSetOpt);
        for (auto& presetTestSet : presetTestSetArray)
        {
            if (strncmp(testSet, presetTestSet.GetName(), strlen(presetTestSet.GetName()) + 1) == 0)
            {
                pTestSet = &presetTestSet;
                break;
            }
        }
        if (pTestSet == nullptr)
        {
            NN_ABORT("preset %s is not found.\n", testSet);
        }
    }
    else
    {
        pTestSet = &defaultTestSet;
    }

    // テスト実行時間
    int testDuration = 0;
    if(optionParser.HasKey(durationOpt))
    {
        testDuration = std::atoi(optionParser.GetValue(durationOpt));
    }

    // チェック間隔
    int checkInterval = 0;
    if(optionParser.HasKey(checkIntervalOpt))
    {
        checkInterval = std::atoi(optionParser.GetValue(checkIntervalOpt));
    }

    // ランダムシード
    int randomSeed = 0;
    if(optionParser.HasKey(randSeedOpt))
    {
        randomSeed = std::atoi(optionParser.GetValue(randSeedOpt));
    }
    if (randomSeed == 0)
    {
        randomSeed = nnt::fs::util::GetRandomSeed();
    }
    NN_LOG("RandomSeed: %d\n", randomSeed);

    // グローバルファイルキャッシュ、個別ファイルデータキャッシュ
    util::optional<decltype(AllocateBuffer(0))> globalFileDataCacheBuffer;
    int globalFileDataCacheBufferSize = 0;
    if(optionParser.HasKey(globalFileDataCacheSizeOpt))
    {
        globalFileDataCacheBufferSize = std::atoi(optionParser.GetValue(globalFileDataCacheSizeOpt));
    }
    if(globalFileDataCacheBufferSize > 0)
    {
        NN_LOG("EnableGlobalFileDataCache (%dMB)\n", globalFileDataCacheBufferSize);

        // バッファを確保
        size_t buffer_size = static_cast<size_t>(globalFileDataCacheBufferSize) * 1024 * 1024;
        auto buffer = AllocateBuffer(buffer_size);
        NN_ASSERT_NOT_NULL(buffer);
        globalFileDataCacheBuffer.emplace(std::move(buffer));

        // ファイルキャッシュを有効にする
        EnableGlobalFileDataCache(globalFileDataCacheBuffer.value().get(), buffer_size);

        const size_t SizeThresholdForFileDataCache = 2 * 1024 * 1024;
        SetBufferSizeForIndividualFileDataCache(SizeThresholdForFileDataCache);
    }

    // データの読み込み先
    if(optionParser.HasKey(dataSrcOpt))
    {
        dataSrcFsProvider.SetRootDir(optionParser.GetValue(dataSrcOpt));
    }

    // SD カード上に置くコンテンツの数
    if(optionParser.HasKey(contentsOnSdOpt))
    {
        double contentsNumOnSd = std::atof(optionParser.GetValue(contentsOnSdOpt));
        if (contentsNumOnSd > 0)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("sd"));
            int64_t sdFreeSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(GetFreeSpaceSize(&sdFreeSize, "sd:/"));
            nn::fs::Unmount("sd");

            // SD カードの空き容量が 8 GB 以下の場合は失敗とする
            EXPECT_TRUE(sdFreeSize > static_cast<int64_t>(1024) * 1024 * 1024 * 8);

            // SD カードの空き容量の 8 割をテスト用のコンテンツが利用可能とする
            SetMaxDirectoryTreeSize(static_cast<int64_t>(sdFreeSize * 0.8 / contentsNumOnSd));
        }
    }


    // テストセット実行
    pTestSet->RunAllTest(testDuration, checkInterval, randomSeed);


    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(0);
} // NOLINT(impl/function_size)


