﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Debug.h>
#include <nn/fs/fs_SaveDataTransfer.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_DebugPrivate.h>
#include "detail/fssrv_SaveDataTransferManager.h"
#include <nn/fssystem/save/fs_JournalIntegritySaveDataFileSystem.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/crypto/crypto_Csrng.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_Aes128GcmEncryptor.h>
#include <nn/crypto/crypto_RsaPssSha256Signer.h>
#include <nn/spl/spl_Api.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>

namespace nn { namespace fssystem { namespace save {

class FsJournalIntegritySaveDataFileSystemTest
{
public:
    static const auto MasterHeaderSize = sizeof(JournalIntegritySaveDataFileSystem::MasterHeader);
};

}}}

void GenerateAesKeyForSaveDataTransfer(void* pBuffer, size_t bufferSize, nn::fssrv::SaveDataTransferCryptoConfiguration::KeyIndex keyIndex, const void* pKeySource, size_t keySourceSize) NN_NOEXCEPT;

// Generic 用ダミー
#if defined(NN_BUILD_CONFIG_OS_WIN)

namespace nn{

namespace spl {

void InitializeForCrypto() NN_NOEXCEPT
{
}

void Finalize() NN_NOEXCEPT
{
}

}

namespace fs { namespace detail {

void GenerateAesKeyForSaveDataTransfer(void* pBuffer, size_t bufferSize, nn::fssrv::SaveDataTransferCryptoConfiguration::KeyIndex keyIndex, const void* pKeySource, size_t keySourceSize) NN_NOEXCEPT;

}}

}

void GenerateAesKeyForSaveDataTransfer(void* pBuffer, size_t bufferSize, nn::fssrv::SaveDataTransferCryptoConfiguration::KeyIndex keyIndex, const void* pKeySource, size_t keySourceSize) NN_NOEXCEPT
{
    return nn::fs::detail::GenerateAesKeyForSaveDataTransfer(pBuffer, bufferSize, keyIndex, pKeySource, keySourceSize);
}

#endif

namespace
{
    const size_t DefaultSaveDataSize        = 1024 * 1024;
    const size_t DefaultSaveDataJournalSize = 1024 * 1024;

    nn::Bit64 GetSaveDataApplicationId() NN_NOEXCEPT
    {
        return 0x0005000C10000000;
    }

    nn::fs::UserId GetUserIdByIndex(int index) NN_NOEXCEPT
    {
        nn::fs::UserId userId = {{ 0xFull, static_cast<uint32_t>(index) }};
        return userId;
    }

    nn::Bit64 GetOwnerIdByIndex(int index) NN_NOEXCEPT
    {
        return GetSaveDataApplicationId() + index;
    }

    uint32_t GetFlagsByIndex(int index) NN_NOEXCEPT
    {
        return 1024 - index;
    }

    // テスト用署名検証鍵 (PublicModulus)
    const unsigned char TestPublicModulus[] =
    {
        0xd8, 0xf1, 0x18, 0xef, 0x32, 0x72, 0x4c, 0xa7, 0x47, 0x4c, 0xb9, 0xea, 0xb3, 0x04, 0xa8, 0xa4,
        0xac, 0x99, 0x08, 0x08, 0x04, 0xbf, 0x68, 0x57, 0xb8, 0x43, 0x94, 0x2b, 0xc7, 0xb9, 0x66, 0x49,
        0x85, 0xe5, 0x8a, 0x9b, 0xc1, 0x00, 0x9a, 0x6a, 0x8d, 0xd0, 0xef, 0xce, 0xff, 0x86, 0xc8, 0x5c,
        0x5d, 0xe9, 0x53, 0x7b, 0x19, 0x2a, 0xa8, 0xc0, 0x22, 0xd1, 0xf3, 0x22, 0x0a, 0x50, 0xf2, 0x2b,
        0x65, 0x05, 0x1b, 0x9e, 0xec, 0x61, 0xb5, 0x63, 0xa3, 0x6f, 0x3b, 0xba, 0x63, 0x3a, 0x53, 0xf4,
        0x49, 0x2f, 0xcf, 0x03, 0xcc, 0xd7, 0x50, 0x82, 0x1b, 0x29, 0x4f, 0x08, 0xde, 0x1b, 0x6d, 0x47,
        0x4f, 0xa8, 0xb6, 0x6a, 0x26, 0xa0, 0x83, 0x3f, 0x1a, 0xaf, 0x83, 0x8f, 0x0e, 0x17, 0x3f, 0xfe,
        0x44, 0x1c, 0x56, 0x94, 0x2e, 0x49, 0x83, 0x83, 0x03, 0xe9, 0xb6, 0xad, 0xd5, 0xde, 0xe3, 0x2d,
        0xa1, 0xd9, 0x66, 0x20, 0x5d, 0x1f, 0x5e, 0x96, 0x5d, 0x5b, 0x55, 0x0d, 0xd4, 0xb4, 0x77, 0x6e,
        0xae, 0x1b, 0x69, 0xf3, 0xa6, 0x61, 0x0e, 0x51, 0x62, 0x39, 0x28, 0x63, 0x75, 0x76, 0xbf, 0xb0,
        0xd2, 0x22, 0xef, 0x98, 0x25, 0x02, 0x05, 0xc0, 0xd7, 0x6a, 0x06, 0x2c, 0xa5, 0xd8, 0x5a, 0x9d,
        0x7a, 0xa4, 0x21, 0x55, 0x9f, 0xf9, 0x3e, 0xbf, 0x16, 0xf6, 0x07, 0xc2, 0xb9, 0x6e, 0x87, 0x9e,
        0xb5, 0x1c, 0xbe, 0x97, 0xfa, 0x82, 0x7e, 0xed, 0x30, 0xd4, 0x66, 0x3f, 0xde, 0xd8, 0x1b, 0x4b,
        0x15, 0xd9, 0xfb, 0x2f, 0x50, 0xf0, 0x9d, 0x1d, 0x52, 0x4c, 0x1c, 0x4d, 0x8d, 0xae, 0x85, 0x1e,
        0xea, 0x7f, 0x86, 0xf3, 0x0b, 0x7b, 0x87, 0x81, 0x98, 0x23, 0x80, 0x63, 0x4f, 0x2f, 0xb0, 0x62,
        0xcc, 0x6e, 0xd2, 0x46, 0x13, 0x65, 0x2b, 0xd6, 0x44, 0x33, 0x59, 0xb5, 0x8f, 0xb9, 0x4a, 0xa9
    };

    // テスト用署名鍵 (PrivateExponent)
    const unsigned char TestPrivateExponent[] =
    {
        0x0c, 0x05, 0xb5, 0x6d, 0xe9, 0x0f, 0xe6, 0x41, 0x55, 0x6d, 0x52, 0x36, 0xc8, 0x57, 0xb3, 0x60,
        0x57, 0xdb, 0xcd, 0xb3, 0x03, 0x0f, 0x57, 0xf1, 0x17, 0x8a, 0x30, 0x33, 0x8a, 0x68, 0x92, 0xfb,
        0x73, 0x57, 0x04, 0x8a, 0xcb, 0xe3, 0xf4, 0x8a, 0xbf, 0xe3, 0xf2, 0xac, 0x38, 0x23, 0x30, 0x26,
        0x95, 0x42, 0x3d, 0x50, 0xfa, 0xb4, 0xaf, 0x60, 0x21, 0x75, 0xdc, 0xd9, 0x57, 0xb4, 0xc3, 0x6c,
        0xe5, 0xf6, 0xe5, 0xe0, 0x55, 0x65, 0x77, 0x4b, 0xc7, 0xa6, 0x7e, 0x0a, 0xfe, 0xdd, 0x80, 0x42,
        0x4f, 0x0d, 0x7e, 0x15, 0x8d, 0xf4, 0x27, 0x37, 0x24, 0x99, 0xf2, 0x12, 0x31, 0xdb, 0xd7, 0x7f,
        0x1e, 0x92, 0x21, 0x14, 0xca, 0x21, 0xf6, 0x50, 0x08, 0x92, 0xae, 0x31, 0xde, 0xf4, 0x29, 0x24,
        0xd6, 0x41, 0xb3, 0x47, 0x18, 0x37, 0x14, 0xf9, 0x8d, 0x5d, 0x95, 0xf4, 0xf5, 0x7f, 0x99, 0xfb,
        0x86, 0xda, 0x65, 0xe9, 0x72, 0xa9, 0x77, 0x65, 0xc8, 0xc5, 0x29, 0x5a, 0x19, 0x2b, 0x51, 0x1c,
        0x72, 0xeb, 0x49, 0xd1, 0x0b, 0x73, 0x8b, 0x3e, 0x2e, 0xc8, 0x7e, 0xff, 0xd8, 0xfe, 0xf4, 0xf4,
        0xf6, 0x92, 0x27, 0x7f, 0xa0, 0xdb, 0xc1, 0x25, 0xbc, 0xec, 0x5f, 0x0b, 0x2d, 0x99, 0xeb, 0xdd,
        0x9e, 0x5d, 0x42, 0x75, 0xb5, 0xe3, 0x24, 0xcb, 0xe9, 0xeb, 0xd9, 0x00, 0x4b, 0x12, 0x5d, 0xa3,
        0xa6, 0x25, 0xac, 0x20, 0x82, 0x25, 0x53, 0x1f, 0xc6, 0x2f, 0x27, 0xf1, 0x99, 0x7a, 0x99, 0xdc,
        0xa5, 0xc0, 0x5e, 0x63, 0x0f, 0x78, 0x03, 0x2a, 0x18, 0xd9, 0xe1, 0x06, 0x3b, 0xdf, 0xb2, 0x95,
        0x19, 0x32, 0xb4, 0x65, 0xd2, 0xd0, 0xfe, 0x18, 0xc7, 0x54, 0x5c, 0xa4, 0xf6, 0xd8, 0xfd, 0xdb,
        0x6d, 0xd8, 0xda, 0xf2, 0x9a, 0x55, 0x5c, 0x3e, 0xec, 0x17, 0x72, 0x09, 0xa3, 0x1a, 0x0a, 0xc1
    };

    // データ移行サーバの代わりに challenge から token を生成する。
    // 署名はテスト用の鍵を使用する。
    // keySeedSeed には export/import で共通値を入れる。
    void GenerateSaveDataTransferToken(void* pOutBuffer, size_t outBufferSize, const void* pChallengeBuffer, size_t challengeBufferSize, int64_t keySeedSeed, bool falsify = false)
    {
        NN_ABORT_UNLESS(outBufferSize == sizeof(nn::fssrv::detail::SaveDataTransferToken));
        NN_ABORT_UNLESS(challengeBufferSize == nn::fssrv::detail::Challenge::Size);

        auto pToken = reinterpret_cast<nn::fssrv::detail::SaveDataTransferToken*>(pOutBuffer);

        // version zero
        memset(pToken, 0x00, sizeof(nn::fssrv::detail::SaveDataTransferToken));

        // challenge
        memcpy(pToken->encryptedArea.signedArea.challenge, pChallengeBuffer, challengeBufferSize);

        // token iv
        nn::crypto::GenerateCryptographicallyRandomBytes(pToken->tokenIv, sizeof(pToken->tokenIv));

        // iv, keyseed for transfer
        char hash[32];
        nn::crypto::GenerateSha256Hash(hash, sizeof(hash), &keySeedSeed, sizeof(keySeedSeed));
        memcpy(pToken->encryptedArea.signedArea.iv,      &hash[0],  sizeof(pToken->encryptedArea.signedArea.iv));
        memcpy(pToken->encryptedArea.signedArea.keySeed, &hash[16], sizeof(pToken->encryptedArea.signedArea.keySeed));

        char RsaInterimSalt[nn::crypto::Sha256Generator::HashSize];
        nnt::fs::util::FillBufferWithRandomValue(RsaInterimSalt, sizeof(RsaInterimSalt));

        // sign
        nn::crypto::SignRsa2048PssSha256(
            pToken->encryptedArea.sign,        sizeof(pToken->encryptedArea.sign),
            TestPublicModulus,                 sizeof(TestPublicModulus),
            TestPrivateExponent,               sizeof(TestPrivateExponent),
            &pToken->encryptedArea.signedArea, sizeof(pToken->encryptedArea.signedArea),
            RsaInterimSalt, sizeof(RsaInterimSalt));

        // 署名後の改竄
        if (falsify)
        {
            pToken->encryptedArea.signedArea.reserverd2[0] ^= 0xFF;
        }

        // AES-GCM
        char key[16];
        GenerateAesKeyForSaveDataTransfer(key, 16, nn::fssrv::SaveDataTransferCryptoConfiguration::KeyIndex::SaveDataTransferToken, nullptr, 0);
        auto encryptedSize = nn::crypto::EncryptAes128Gcm(
            &pToken->encryptedArea, sizeof(pToken->encryptedArea),
            &pToken->tokenMac, sizeof(pToken->tokenMac),
            key, sizeof(key),
            pToken->tokenIv, sizeof(pToken->tokenIv),
            &pToken->encryptedArea, sizeof(pToken->encryptedArea),
            nullptr, 0
        );

        EXPECT_EQ(sizeof(pToken->encryptedArea), encryptedSize);
    }

    bool IsMatchSaveDataInfo(const nn::fs::SaveDataInfo& info1, const nn::fs::SaveDataInfo& info2)
    {
        bool isMatched = info1.saveDataId == info2.saveDataId &&
            info1.saveDataSpaceId == info2.saveDataSpaceId &&
            info1.saveDataType == info2.saveDataType &&
            info1.saveDataUserId == info2.saveDataUserId &&
            info1.systemSaveDataId == info2.systemSaveDataId &&
            info1.applicationId.value == info2.applicationId.value &&
            info1.saveDataSize == info2.saveDataSize;
        if (!isMatched)
        {
            nnt::fs::util::DumpBufferDiff(&info1, &info2, sizeof(info1));
        }
        return isMatched;
    }

    void SaveDataTransferBasicCore(nn::fs::SaveDataTransferManager* manager, const nn::fs::SaveDataInfo& info, int i, int num, bool falsify, size_t bufferSize = 16 * 1024, bool isExportFromV4 = false)
    {
        auto exportSize = nn::fs::SaveDataTransferSizeCalculator::QuerySaveDataExportSize(info);

        std::unique_ptr<nn::fs::SaveDataExporter> exporter;
        NNT_ASSERT_RESULT_SUCCESS(manager->OpenSaveDataExporter(&exporter, info.saveDataSpaceId, info.saveDataId));
        {
            auto tmpInfo = exporter->GetSaveDataInfo();
            EXPECT_TRUE(IsMatchSaveDataInfo(info, tmpInfo));
        }

        auto exportRestSizeBeforePullInitial = exporter->GetRestSize();

        char initialData[nn::fs::SaveDataExporter::InitialDataSize];
        NNT_EXPECT_RESULT_SUCCESS(exporter->PullInitialData(initialData, sizeof(initialData)));

        auto exportRestSize = exporter->GetRestSize();
        EXPECT_GT(exportRestSize, 0);
        EXPECT_EQ(exportRestSize, exportRestSizeBeforePullInitial); // export size は initial を含まない

        EXPECT_EQ(exportRestSize, exportSize);


        std::unique_ptr<nn::fs::SaveDataImporter> importer;
        int64_t requiredSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(manager->OpenSaveDataImporter(&importer, &requiredSize, initialData, sizeof(initialData), GetUserIdByIndex(i + num), info.saveDataSpaceId));
        {
            auto tmpInfo = importer->GetSaveDataInfo();
            auto expectedInfo = info;
            {
                expectedInfo.saveDataId = tmpInfo.saveDataId;
                expectedInfo.saveDataUserId = GetUserIdByIndex(i + num);
            }

            if (isExportFromV4)
            {
                // 16GB までのセーブの場合、 v4 -> v5 に移すと 64KB 大きくなる
                expectedInfo.saveDataSize += 64 * 1024;
            }
            EXPECT_TRUE(IsMatchSaveDataInfo(expectedInfo, tmpInfo));

            auto importSize = exportSize; // importer->GetSaveDataInfo() からの QuerySaveDataExportSize() は非対応化
            auto importRestSize = importer->GetRestSize();
            EXPECT_EQ(importRestSize, importSize);
        }

        auto buffer = nnt::fs::util::AllocateBuffer(bufferSize);
        uint64_t totalPulledSize = 0;

        bool falsified = false;

        while (NN_STATIC_CONDITION(true))
        {
            // export
            size_t pulledSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(exporter->Pull(&pulledSize, buffer.get(), bufferSize));
            totalPulledSize += pulledSize;
            exportRestSize -= pulledSize;
            EXPECT_EQ(exporter->GetRestSize(), exportRestSize);

            // 改竄指定時は先頭1バイトを改竄
            if (falsify && !falsified)
            {
                buffer.get()[0] ^= 0xFF;
                falsified = true;
            }

            // import
            NNT_EXPECT_RESULT_SUCCESS(importer->Push(buffer.get(), pulledSize));
            EXPECT_EQ(exportRestSize, importer->GetRestSize());

            if (pulledSize == 0)
            {
                // complete
                EXPECT_EQ(0, exportRestSize);
                if (falsify)
                {
                    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSaveDataTransferImportMacVerificationFailed, importer->Finalize());
                }
                else
                {
                    NNT_EXPECT_RESULT_SUCCESS(importer->Finalize());
                }
                break;
            }
        }
    }

    void SetUpDefaultSaveDataTransferManager(nn::fs::SaveDataTransferManager* pManager, int64_t keySeedSeed = 0)
    {
        char challenge[nn::fs::SaveDataTransferManager::ChallengeSize];
        NNT_EXPECT_RESULT_SUCCESS(pManager->GetChallenge(challenge, sizeof(challenge)));


        nn::fssrv::detail::SaveDataTransferToken token;
        GenerateSaveDataTransferToken(&token, sizeof(token), challenge, sizeof(challenge), keySeedSeed);

        NNT_EXPECT_RESULT_SUCCESS(pManager->SetToken(&token, sizeof(token)));
    }


}

// --- 検証鍵の未差し替えを期待するテストはここより上に配置する ---

#if !defined(NN_BUILD_CONFIG_OS_WIN)

TEST(SaveDataTransferTest, UsableSpaceNotEnoughHeavy)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    nnt::fs::util::DeleteAllTestSaveData();

    const size_t SaveDataSize        = 64 * 1024 * 1024;
    const size_t SaveDataJournalSize = 16 * 1024 * 1024;

    nn::ncm::ApplicationId appId = { GetSaveDataApplicationId() };

    nn::fs::SaveDataInfo info;
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(appId, GetUserIdByIndex(1), GetOwnerIdByIndex(1), SaveDataSize, SaveDataJournalSize, GetFlagsByIndex(1)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info, appId, GetUserIdByIndex(1)));
    }

    nn::fs::SaveDataTransferManager manager;
    std::unique_ptr<nn::fs::SaveDataExporter> exporter;

    char challenge[nn::fs::SaveDataTransferManager::ChallengeSize];
    NNT_EXPECT_RESULT_SUCCESS(manager.GetChallenge(challenge, sizeof(challenge)));

    nn::fssrv::detail::SaveDataTransferToken token;
    GenerateSaveDataTransferToken(&token, sizeof(token), challenge, sizeof(challenge), 0);
    NNT_EXPECT_RESULT_SUCCESS(manager.SetToken(&token, sizeof(token)));

    NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataExporter(&exporter, info.saveDataSpaceId, info.saveDataId));

    char initialData[nn::fs::SaveDataExporter::InitialDataSize];
    NNT_EXPECT_RESULT_SUCCESS(exporter->PullInitialData(initialData, sizeof(initialData)));

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::User, nullptr));
    {
        nnt::fs::util::ScopedPadder padder("@User:/", 64 * 1024 * 1024); // 残り 64MB 以下に埋める
        std::unique_ptr<nn::fs::SaveDataImporter> importer;
        int64_t requiredSize = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultUsableSpaceNotEnoughForSaveData, manager.OpenSaveDataImporter(&importer, &requiredSize, initialData, sizeof(initialData), GetUserIdByIndex(2), info.saveDataSpaceId));
        EXPECT_GT(requiredSize, static_cast<int64_t>(SaveDataJournalSize));
    }

    exporter.reset();

    nn::fs::Unmount("@User");

    nnt::fs::util::DeleteAllTestSaveData();
}

#endif


TEST(SaveDataTransferTest, NoFinalize)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId appId = { GetSaveDataApplicationId() };

    nn::fs::SaveDataInfo info;
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(appId, GetUserIdByIndex(1), GetOwnerIdByIndex(1), DefaultSaveDataSize, DefaultSaveDataJournalSize, GetFlagsByIndex(1)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info, appId, GetUserIdByIndex(1)));
    }

    nn::fs::SaveDataTransferManager manager;
    SetUpDefaultSaveDataTransferManager(&manager);

    std::unique_ptr<nn::fs::SaveDataExporter> exporter;

    NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataExporter(&exporter, info.saveDataSpaceId, info.saveDataId));

    char initialData[nn::fs::SaveDataExporter::InitialDataSize];
    NNT_EXPECT_RESULT_SUCCESS(exporter->PullInitialData(initialData, sizeof(initialData)));

    {
        std::unique_ptr<nn::fs::SaveDataImporter> importer;
        int64_t requiredSize = 0;
        NNT_EXPECT_RESULT_SUCCESS(manager.OpenSaveDataImporter(&importer, &requiredSize, initialData, sizeof(initialData), GetUserIdByIndex(2), info.saveDataSpaceId));

        // インポート中は DB、実体ともに存在
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info, appId, GetUserIdByIndex(2)));
    }

    // インポートを中断すると DB、実体ともに削除される
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nnt::fs::util::FindSaveDataByApplicationId(&info, appId, GetUserIdByIndex(2)));
}

TEST(SaveDataTransferTest, Invalid)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId appId = { GetSaveDataApplicationId() };

    nn::fs::SaveDataInfo info;
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(appId, GetUserIdByIndex(1), GetOwnerIdByIndex(1), DefaultSaveDataSize, DefaultSaveDataJournalSize, GetFlagsByIndex(1)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info, appId, GetUserIdByIndex(1)));
    }

    nn::fs::SaveDataTransferManager manager;
    std::unique_ptr<nn::fs::SaveDataExporter> exporter;
    std::unique_ptr<nn::fs::SaveDataImporter> importer;
    int64_t requiredSize = 0;

    char challenge[nn::fs::SaveDataTransferManager::ChallengeSize];
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidSize, manager.GetChallenge(challenge, sizeof(challenge) - 1));
    NNT_EXPECT_RESULT_SUCCESS(manager.GetChallenge(challenge, sizeof(challenge)));

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPreconditionViolation, manager.OpenSaveDataExporter(&exporter, info.saveDataSpaceId, info.saveDataId));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPreconditionViolation, manager.OpenSaveDataImporter(&importer, &requiredSize, challenge, sizeof(challenge), GetUserIdByIndex(2), info.saveDataSpaceId));

    nn::fssrv::detail::SaveDataTransferToken token;
    GenerateSaveDataTransferToken(&token, sizeof(token), challenge, sizeof(challenge), 0);

    NNT_EXPECT_RESULT_SUCCESS(manager.SetToken(&token, sizeof(token)));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPreconditionViolation, manager.SetToken(&token, sizeof(token)));

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, manager.OpenSaveDataExporter(&exporter, nn::fs::SaveDataSpaceId::System, info.saveDataId));
    NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataExporter(&exporter, info.saveDataSpaceId, info.saveDataId));

    char initialData[nn::fs::SaveDataExporter::InitialDataSize];
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidSize, exporter->PullInitialData(initialData, sizeof(initialData) - 1));
    NNT_EXPECT_RESULT_SUCCESS(exporter->PullInitialData(initialData, sizeof(initialData)));

    auto exportRestSize = exporter->GetRestSize();
    std::unique_ptr<char[]> buffer(new char[static_cast<size_t>(exportRestSize) + 16]);
    size_t pulledSize = 0;
    NNT_EXPECT_RESULT_SUCCESS(exporter->Pull(&pulledSize, buffer.get(), static_cast<size_t>(exportRestSize) + 16)); // Pull は RestSize を超えても OK
    EXPECT_EQ(pulledSize, exportRestSize);

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidSize, manager.OpenSaveDataImporter(&importer, &requiredSize, initialData, sizeof(initialData) - 1, GetUserIdByIndex(2), info.saveDataSpaceId));
    NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataImporter(&importer, &requiredSize, initialData, sizeof(initialData), GetUserIdByIndex(2), info.saveDataSpaceId));

    auto importRestSize = importer->GetRestSize();
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidSize, importer->Push(buffer.get(), importRestSize + 16)); // Push は RestSize を超えると NG
    EXPECT_EQ(importRestSize, importer->GetRestSize());

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPreconditionViolation, importer->Finalize());

    NNT_EXPECT_RESULT_SUCCESS(importer->Push(buffer.get(), importRestSize));
    NNT_EXPECT_RESULT_SUCCESS(importer->Finalize());

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPreconditionViolation, importer->Push(buffer.get(), importRestSize));
}


// token の検証
TEST(SaveDataTransferTest, InvalidToken)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    nn::fs::SaveDataTransferManager manager;

    // invalid challenge
    {
        char challenge[nn::fs::SaveDataTransferManager::ChallengeSize];
        NNT_EXPECT_RESULT_SUCCESS(manager.GetChallenge(challenge, sizeof(challenge)));
        challenge[0] ^= 0xFF;

        nn::fssrv::detail::SaveDataTransferToken token;
        GenerateSaveDataTransferToken(&token, sizeof(token), challenge, sizeof(challenge), 0);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSaveDataTransferTokenChallengeVerificationFailed, manager.SetToken(&token, sizeof(token)));
    }

    // invalid mac
    {
        char challenge[nn::fs::SaveDataTransferManager::ChallengeSize];
        NNT_EXPECT_RESULT_SUCCESS(manager.GetChallenge(challenge, sizeof(challenge)));

        nn::fssrv::detail::SaveDataTransferToken token;
        GenerateSaveDataTransferToken(&token, sizeof(token), challenge, sizeof(challenge), 0);

        // keySeed
        token.encryptedArea.signedArea.keySeed[0] ^= 0xFF;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSaveDataTransferTokenMacVerificationFailed, manager.SetToken(&token, sizeof(token)));
    }

    // invalid sign
    {
        char challenge[nn::fs::SaveDataTransferManager::ChallengeSize];
        NNT_EXPECT_RESULT_SUCCESS(manager.GetChallenge(challenge, sizeof(challenge)));

        nn::fssrv::detail::SaveDataTransferToken token;
        GenerateSaveDataTransferToken(&token, sizeof(token), challenge, sizeof(challenge), 0, true);
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSaveDataTransferTokenSignatureVerificationFailed, manager.SetToken(&token, sizeof(token)));
    }

}


TEST(SaveDataTransferTest, SystemSaveData)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    nnt::fs::util::DeleteAllTestSaveData();

    const int SaveDataNum = 4;

    nn::fs::SystemSaveDataId systemSaveDataId = 0x8000000000000003;

    nn::fs::SaveDataInfo info;

    for (int i = 1; i <= SaveDataNum; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(systemSaveDataId, GetUserIdByIndex(i), GetOwnerIdByIndex(i), DefaultSaveDataSize, DefaultSaveDataJournalSize, GetFlagsByIndex(i)));
    }

    nn::fs::SaveDataTransferManager manager;
    SetUpDefaultSaveDataTransferManager(&manager);

    for (int i = 1; i <= SaveDataNum; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataBySystemSaveDataId(&info, systemSaveDataId, GetUserIdByIndex(i)));

#if !defined(NN_BUILD_CONFIG_OS_WIN)
        auto freeSpaceSize = nn::fs::SaveDataTransferSizeCalculator::GetFreeSpaceSize(info.saveDataSpaceId);
        auto requiredSizeForImport = nn::fs::SaveDataTransferSizeCalculator::QuerySaveDataRequiredSizeForImport(info);
        {
            auto exportSize = nn::fs::SaveDataTransferSizeCalculator::QuerySaveDataExportSize(info);
            EXPECT_GT(requiredSizeForImport, exportSize);
            EXPECT_GT(freeSpaceSize, requiredSizeForImport);
        }
#endif

        SaveDataTransferBasicCore(&manager, info, i, SaveDataNum, false);

#if !defined(NN_BUILD_CONFIG_OS_WIN)
        EXPECT_LE(nn::fs::SaveDataTransferSizeCalculator::GetFreeSpaceSize(info.saveDataSpaceId) + info.saveDataSize, freeSpaceSize);

        // QuerySaveDataRequiredSizeForImport() 以上に消費していないこと
        EXPECT_LE(freeSpaceSize - nn::fs::SaveDataTransferSizeCalculator::GetFreeSpaceSize(info.saveDataSpaceId), requiredSizeForImport);
#endif

        // TODO: 内容の一致確認
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataBySystemSaveDataId(&info, systemSaveDataId, GetUserIdByIndex(i + SaveDataNum)));
    }

    nnt::fs::util::DeleteAllTestSaveData();
}

// データ転送経路の改竄を検知できること
TEST(SaveDataTransferTest, IntegrityVerification)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    nnt::fs::util::DeleteAllTestSaveData();

    const int SaveDataNum = 4;

    nn::fs::SystemSaveDataId systemSaveDataId = 0x8000000000000003;

    nn::fs::SaveDataInfo info;

    for (int i = 1; i <= SaveDataNum; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(systemSaveDataId, GetUserIdByIndex(i), GetOwnerIdByIndex(i), DefaultSaveDataSize, DefaultSaveDataJournalSize, GetFlagsByIndex(i)));
    }

    nn::fs::SaveDataTransferManager manager;
    SetUpDefaultSaveDataTransferManager(&manager);

    for (int i = 1; i <= SaveDataNum; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataBySystemSaveDataId(&info, systemSaveDataId, GetUserIdByIndex(i)));

#if !defined(NN_BUILD_CONFIG_OS_WIN)
        auto freeSpaceSize = nn::fs::SaveDataTransferSizeCalculator::GetFreeSpaceSize(info.saveDataSpaceId);
        {
            auto exportSize = nn::fs::SaveDataTransferSizeCalculator::QuerySaveDataExportSize(info);
            auto importSize = nn::fs::SaveDataTransferSizeCalculator::QuerySaveDataRequiredSizeForImport(info);
            EXPECT_GT(importSize, exportSize);
            EXPECT_GT(freeSpaceSize, importSize);
        }
#endif

        SaveDataTransferBasicCore(&manager, info, i, SaveDataNum, true);

        // ロールバック済み
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nnt::fs::util::FindSaveDataBySystemSaveDataId(&info, systemSaveDataId, GetUserIdByIndex(i + SaveDataNum)));
    }

    nnt::fs::util::DeleteAllTestSaveData();
}


// Initial data の改竄を検知できること
TEST(SaveDataTransferTest, InvalidInitialData)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId appId = { GetSaveDataApplicationId() };

    nn::fs::SaveDataInfo info;
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(appId, GetUserIdByIndex(1), GetOwnerIdByIndex(1), DefaultSaveDataSize, DefaultSaveDataJournalSize, GetFlagsByIndex(1)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info, appId, GetUserIdByIndex(1)));
    }

    nn::fs::SaveDataTransferManager manager;
    SetUpDefaultSaveDataTransferManager(&manager);

    std::unique_ptr<nn::fs::SaveDataExporter> exporter;
    NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataExporter(&exporter, info.saveDataSpaceId, info.saveDataId));

    char initialData[nn::fs::SaveDataExporter::InitialDataSize];
    NNT_EXPECT_RESULT_SUCCESS(exporter->PullInitialData(initialData, sizeof(initialData)));

    // initial data の改竄
    initialData[sizeof(initialData) - 1] ^= 0xFF;

    {
        std::unique_ptr<nn::fs::SaveDataImporter> importer;
        int64_t requiredSize = 0;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSaveDataTransferInitialDataMacVerificationFailed, manager.OpenSaveDataImporter(&importer, &requiredSize, initialData, sizeof(initialData), GetUserIdByIndex(2), info.saveDataSpaceId));
    }
}

// KeySeed 違いを検知できること
TEST(SaveDataTransferTest, InvalidKeySeed)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId appId = { GetSaveDataApplicationId() };

    nn::fs::SaveDataInfo info;
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(appId, GetUserIdByIndex(1), GetOwnerIdByIndex(1), DefaultSaveDataSize, DefaultSaveDataJournalSize, GetFlagsByIndex(1)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info, appId, GetUserIdByIndex(1)));
    }

    nn::fs::SaveDataTransferManager manager[2];
    for (int i = 0; i < 2; i++)
    {
        SetUpDefaultSaveDataTransferManager(&manager[i], i); // keyseed 変える
    }

    std::unique_ptr<nn::fs::SaveDataExporter> exporter;
    NNT_EXPECT_RESULT_SUCCESS(manager[0].OpenSaveDataExporter(&exporter, info.saveDataSpaceId, info.saveDataId));

    char initialData[nn::fs::SaveDataExporter::InitialDataSize];
    NNT_EXPECT_RESULT_SUCCESS(exporter->PullInitialData(initialData, sizeof(initialData)));


    std::unique_ptr<nn::fs::SaveDataImporter> importer;
    int64_t requiredSize = 0;
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSaveDataTransferInitialDataMacVerificationFailed, manager[1].OpenSaveDataImporter(&importer, &requiredSize, initialData, sizeof(initialData), GetUserIdByIndex(2), info.saveDataSpaceId));
}

// Importer/Expoter が同一 index 同士のみで通信できること（IV ずらし）
TEST(SaveDataTransferTest, InvalidPorterPair)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId appId = { GetSaveDataApplicationId() };

    nn::fs::SaveDataTransferManager manager;
    SetUpDefaultSaveDataTransferManager(&manager);

    static const auto ExportSaveDataIndex = 1;
    static const auto ExportSaveDataCount = 4;
    nn::fs::SaveDataInfo info[ExportSaveDataCount];
    std::unique_ptr<nn::fs::SaveDataExporter> exporter[ExportSaveDataCount];
    char initialData[ExportSaveDataCount][nn::fs::SaveDataExporter::InitialDataSize];
    for (int i = 0; i < ExportSaveDataCount; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(appId, GetUserIdByIndex(1 + i), GetOwnerIdByIndex(1 + i), DefaultSaveDataSize, DefaultSaveDataJournalSize, GetFlagsByIndex(1 + i)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info[i], appId, GetUserIdByIndex(1 + i)));

        NNT_EXPECT_RESULT_SUCCESS(manager.OpenSaveDataExporter(&exporter[i], info[i].saveDataSpaceId, info[i].saveDataId));
        NNT_EXPECT_RESULT_SUCCESS(exporter[i]->PullInitialData(initialData[i], sizeof(initialData[i])));
    }

    static const auto ImportSaveDataIndex = ExportSaveDataIndex + ExportSaveDataCount;
    std::unique_ptr<nn::fs::SaveDataImporter> importer;
    int64_t requiredSize = 0;

    // (importer 0: 破棄)
    {
        const auto currentIndex = ImportSaveDataIndex + 0;
        NNT_EXPECT_RESULT_SUCCESS(manager.OpenSaveDataImporter(&importer, &requiredSize, initialData[0], sizeof(initialData[0]), GetUserIdByIndex(currentIndex), info[0].saveDataSpaceId));
    }

    // importer 1 vs exporter 0 : NG (initial data)
    {
        const auto currentIndex = ImportSaveDataIndex + 1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSaveDataTransferInitialDataMacVerificationFailed, manager.OpenSaveDataImporter(&importer, &requiredSize, initialData[0], sizeof(initialData[0]), GetUserIdByIndex(currentIndex), info[0].saveDataSpaceId));
    }

    // importer 2 vs exporter 2 : OK (initial data)
    {
        const auto currentIndex = ImportSaveDataIndex + 2;
        NNT_EXPECT_RESULT_SUCCESS(manager.OpenSaveDataImporter(&importer, &requiredSize, initialData[2], sizeof(initialData[2]), GetUserIdByIndex(currentIndex), info[2].saveDataSpaceId));
    }

    // importer 3 vs exporter 3 : OK (initial data)
    {
        const auto currentIndex = ImportSaveDataIndex + 3;
        NNT_EXPECT_RESULT_SUCCESS(manager.OpenSaveDataImporter(&importer, &requiredSize, initialData[3], sizeof(initialData[3]), GetUserIdByIndex(currentIndex), info[3].saveDataSpaceId));
    }

    // importer 3 vs exporter 2 : NG (data)
    {
        auto exportSize = exporter[2]->GetRestSize();
        auto buffer = nnt::fs::util::AllocateBuffer(static_cast<size_t>(exportSize));
        ASSERT_TRUE(buffer.get() != nullptr);

        // export at once
        size_t pulledSize = 0;
        NNT_EXPECT_RESULT_SUCCESS(exporter[2]->Pull(&pulledSize, buffer.get(), static_cast<size_t>(exportSize)));

        // import at once
        NNT_EXPECT_RESULT_SUCCESS(importer->Push(buffer.get(), pulledSize));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultSaveDataTransferImportMacVerificationFailed, importer->Finalize());
    }

}


struct ParamsStructure {
    size_t  bufferSize;
    int64_t saveDataSize;
    int     saveDataCount;
};


class SaveDataTransferTestFixture : public ::testing::TestWithParam <ParamsStructure>
{
};

// アプリのセーブ＋サムネの移行
TEST_P(SaveDataTransferTestFixture, Basic)
{
    auto param = GetParam();
    auto bufferSize          = param.bufferSize;
    auto saveDataSize        = param.saveDataSize / 2;
    auto saveDataJournalSize = param.saveDataSize / 2;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    nnt::fs::util::DeleteAllTestSaveData();

    const int SaveDataNum = param.saveDataCount;

    nn::ncm::ApplicationId appId = { GetSaveDataApplicationId() };

    nn::fs::SaveDataInfo info;

    for (int i = 1; i <= SaveDataNum; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(appId, GetUserIdByIndex(i), GetOwnerIdByIndex(i), saveDataSize, saveDataJournalSize, GetFlagsByIndex(i)));

        {
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountSaveData("save", appId, GetUserIdByIndex(i)));
            auto filePath = nnt::fs::util::String("save:/file") + nnt::fs::util::ToString(i);
            NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateFileWith32BitCount(filePath.c_str(), i * 1024, i * 1024 * 1024));
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));
            nn::fs::Unmount("save");

            char thumbnailFileHeader[32];
            nnt::fs::util::FillBufferWith8BitCount(thumbnailFileHeader, sizeof(thumbnailFileHeader), i);
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteSaveDataThumbnailFileHeader(appId.value, GetUserIdByIndex(i), thumbnailFileHeader, sizeof(thumbnailFileHeader)));
        }
    }

    nn::fs::SaveDataTransferManager manager;
    SetUpDefaultSaveDataTransferManager(&manager);

    for (int i = 1; i <= SaveDataNum; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info, appId, GetUserIdByIndex(i)));

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
        auto requiredSizeForImport = nn::fs::SaveDataTransferSizeCalculator::QuerySaveDataRequiredSizeForImport(info);
        auto freeSpaceSize = nn::fs::SaveDataTransferSizeCalculator::GetFreeSpaceSize(info.saveDataSpaceId);
        {
            auto exportSize = nn::fs::SaveDataTransferSizeCalculator::QuerySaveDataExportSize(info);
            EXPECT_GT(requiredSizeForImport, exportSize);
            EXPECT_GT(freeSpaceSize, requiredSizeForImport);
        }
#endif

        SaveDataTransferBasicCore(&manager, info, i, SaveDataNum, false, bufferSize);

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
        EXPECT_LE(nn::fs::SaveDataTransferSizeCalculator::GetFreeSpaceSize(info.saveDataSpaceId) + info.saveDataSize + nn::fs::detail::ThumbnailFileSize, freeSpaceSize);

        // QuerySaveDataRequiredSizeForImport() 以上に消費していないこと
        EXPECT_LE(freeSpaceSize - nn::fs::SaveDataTransferSizeCalculator::GetFreeSpaceSize(info.saveDataSpaceId), requiredSizeForImport);
#endif

        nn::fs::SaveDataInfo dstInfo;
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&dstInfo, appId, GetUserIdByIndex(i + SaveDataNum)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info,    appId, GetUserIdByIndex(i)));

        nnt::fs::util::CheckSaveData(info, dstInfo, i);
    }

    nnt::fs::util::DeleteAllTestSaveData();
}

const int64_t ExpectedExportSizeForBorderTest = 344248;
const int64_t MacSize = 16;

// bufferSize, saveDataSize, count
ParamsStructure patternParamSmoke[] = {
    {    4095,   1024 * 1024,          4 },
    { 0x40001,   1024 * 1024,          2 },
};

ParamsStructure patternParamBorderHeavy[] = {
    { ExpectedExportSizeForBorderTest,               128 * 1024,          1 },
    { ExpectedExportSizeForBorderTest + 1,           128 * 1024,          1 },
    { ExpectedExportSizeForBorderTest - MacSize,     128 * 1024,          1 },
    { ExpectedExportSizeForBorderTest - MacSize + 1, 128 * 1024,          1 },
    { ExpectedExportSizeForBorderTest - MacSize - 1, 128 * 1024,          1 },
    { 16,                                            128 * 1024,          1 },
};

ParamsStructure patternParamSizeHeavy[] = {
    { 100,       128 * 1024,           2 },
    { 10,        128 * 1024,           1 }, // < mac size
    { 0x200000,  4000LL * 1024 * 1024, 2 },
    { 0x3FFFFF,  4000LL * 1024 * 1024, 1 },
    { 0x6400000, 4000LL * 1024 * 1024, 1 },
};

INSTANTIATE_TEST_CASE_P(WithVariousPattern,            SaveDataTransferTestFixture, ::testing::ValuesIn(patternParamSmoke));
INSTANTIATE_TEST_CASE_P(WithVariousPatternBorderHeavy, SaveDataTransferTestFixture, ::testing::ValuesIn(patternParamBorderHeavy));
INSTANTIATE_TEST_CASE_P(WithVariousPatternHeavy,       SaveDataTransferTestFixture, ::testing::ValuesIn(patternParamSizeHeavy));



typedef SaveDataTransferTestFixture SaveDataTransferTestFixture2;

// 拡張した経緯のあるセーブの移行
TEST_P(SaveDataTransferTestFixture2, ExtendedSaveData)
{
    auto param = GetParam();
    auto bufferSize = param.bufferSize;
    auto saveDataSize = param.saveDataSize / 2;

    const int InitialSaveDataSize = 32 * 1024;
    const int InitialSaveDataJournalSize = InitialSaveDataSize;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId appId = { GetSaveDataApplicationId() };

    const int ExtendCountMax = 64;
    nn::fs::SaveDataInfo info;
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndExtendSaveData(&info, appId, GetUserIdByIndex(1), InitialSaveDataSize, InitialSaveDataJournalSize, saveDataSize, saveDataSize, ExtendCountMax));

    const int SaveDataNum = 1;

    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountSaveData("save", appId, GetUserIdByIndex(1)));
        auto filePath = nnt::fs::util::String("save:/file") + nnt::fs::util::ToString(1);
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateFileWith32BitCount(filePath.c_str(), 1 * 1024, 1 * 1024 * 1024));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));
        nn::fs::Unmount("save");

        char thumbnailFileHeader[32];
        nnt::fs::util::FillBufferWith8BitCount(thumbnailFileHeader, sizeof(thumbnailFileHeader), 1);
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteSaveDataThumbnailFileHeader(appId.value, GetUserIdByIndex(1), thumbnailFileHeader, sizeof(thumbnailFileHeader)));
    }

    nn::fs::SaveDataTransferManager manager;
    SetUpDefaultSaveDataTransferManager(&manager);

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
    auto requiredSizeForImport = nn::fs::SaveDataTransferSizeCalculator::QuerySaveDataRequiredSizeForImport(info);
    auto freeSpaceSize = nn::fs::SaveDataTransferSizeCalculator::GetFreeSpaceSize(info.saveDataSpaceId);
    {
        auto exportSize = nn::fs::SaveDataTransferSizeCalculator::QuerySaveDataExportSize(info);
        EXPECT_GT(requiredSizeForImport, exportSize);
        EXPECT_GT(freeSpaceSize, requiredSizeForImport);
    }
#endif

    SaveDataTransferBasicCore(&manager, info, 1, 1, false, bufferSize);

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
    EXPECT_LE(nn::fs::SaveDataTransferSizeCalculator::GetFreeSpaceSize(info.saveDataSpaceId) + info.saveDataSize + nn::fs::detail::ThumbnailFileSize, freeSpaceSize);

    // QuerySaveDataRequiredSizeForImport() 以上に消費していないこと
    EXPECT_LE(freeSpaceSize - nn::fs::SaveDataTransferSizeCalculator::GetFreeSpaceSize(info.saveDataSpaceId), requiredSizeForImport);
#endif

    nn::fs::SaveDataInfo dstInfo;
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&dstInfo, appId, GetUserIdByIndex(1 + SaveDataNum)));

    nnt::fs::util::CheckSaveData(info, dstInfo, 1);

    nnt::fs::util::DeleteAllTestSaveData();
}


// bufferSize,  saveDataSize,      count
ParamsStructure patternParamSmoke2[] = {
    { 0x200000,  256 * 1024,           1 },
};

ParamsStructure patternParamSizeHeavy2[] = {
    { 0x200000,  4000LL * 1024 * 1024, 1 },
};

INSTANTIATE_TEST_CASE_P(WithVariousPattern,            SaveDataTransferTestFixture2, ::testing::ValuesIn(patternParamSmoke2));
INSTANTIATE_TEST_CASE_P(WithVariousPatternHeavy,       SaveDataTransferTestFixture2, ::testing::ValuesIn(patternParamSizeHeavy2));




// version4 セーブの移行（手動テスト）
// @pre
//     - d:/save_v4_32KB32KB.bin にテストデータの配置
//     - UpdateSaveDataMacForDebug 有効化 fw （実機の場合）
TEST_P(SaveDataTransferTestFixture2, DISABLED_SaveDataVersion4)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));
    nnt::fs::util::DeleteAllTestSaveData();

    // 最小構成で作成
    nn::fs::UserId userId = { { 0xFull, static_cast<uint32_t>(1) } };
    nn::ncm::ApplicationId appId = { 0x0005000C10000000ULL };
    NNT_EXPECT_RESULT_SUCCESS(CreateSaveData(appId, userId, 0, 32 * 1024, 32 * 1024, 0));

    nn::fs::SaveDataInfo info;
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info, appId, userId));

    // v4 セーブバイナリで差し替え
    // （この時点では SaveDataInfo の占有サイズに食い違いが発生する点に注意）
    {
        const size_t BufferSize = 1024 * 1024;
        auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountHostRoot());
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountBis("bisuser", nn::fs::BisPartitionId::User));
        char dstPath[256];
        nn::util::SNPrintf(dstPath, sizeof(dstPath), "bisuser:/save/%016llx", info.saveDataId);
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile(dstPath));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CopyFile("d:/save_v4_32KB32KB.bin", dstPath, buffer.get(), BufferSize));
        nn::fs::UnmountHostRoot();
        nn::fs::Unmount("bisuser");
    }

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::UpdateSaveDataMacForDebug(static_cast<uint8_t>(info.saveDataSpaceId), info.saveDataId));
#endif

    auto param = GetParam();
    auto bufferSize = param.bufferSize;
    auto saveDataSize = param.saveDataSize / 2;

    // 対象サイズに拡張
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ExtendSaveData(info.saveDataSpaceId, info.saveDataId, saveDataSize, saveDataSize));
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info, appId, userId));

    const int SaveDataNum = 1;

    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountSaveData("save", appId, userId));
        auto filePath = nnt::fs::util::String("save:/file") + nnt::fs::util::ToString(1);
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateFileWith32BitCount(filePath.c_str(), 1 * 1024, 1 * 1024 * 1024));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));
        nn::fs::Unmount("save");

        char thumbnailFileHeader[32];
        nnt::fs::util::FillBufferWith8BitCount(thumbnailFileHeader, sizeof(thumbnailFileHeader), 1);
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteSaveDataThumbnailFileHeader(appId.value, userId, thumbnailFileHeader, sizeof(thumbnailFileHeader)));
    }

    nn::fs::SaveDataTransferManager manager;
    SetUpDefaultSaveDataTransferManager(&manager);

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    auto freeSpaceSize = nn::fs::SaveDataTransferSizeCalculator::GetFreeSpaceSize(info.saveDataSpaceId);
    auto requiredSizeForImport = nn::fs::SaveDataTransferSizeCalculator::QuerySaveDataRequiredSizeForImport(info);
    {
        auto exportSize = nn::fs::SaveDataTransferSizeCalculator::QuerySaveDataExportSize(info);
        EXPECT_GT(requiredSizeForImport, exportSize);
        EXPECT_GT(freeSpaceSize, requiredSizeForImport);
    }
#endif

    SaveDataTransferBasicCore(&manager, info, 1, 1, false, bufferSize, true);

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
    EXPECT_LE(nn::fs::SaveDataTransferSizeCalculator::GetFreeSpaceSize(info.saveDataSpaceId) + info.saveDataSize + nn::fs::detail::ThumbnailFileSize, freeSpaceSize);

    // QuerySaveDataRequiredSizeForImport() 以上に消費していないこと
    EXPECT_LE(freeSpaceSize - nn::fs::SaveDataTransferSizeCalculator::GetFreeSpaceSize(info.saveDataSpaceId), requiredSizeForImport);
#endif

    nn::fs::SaveDataInfo dstInfo;
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&dstInfo, appId, GetUserIdByIndex(1 + SaveDataNum)));

    nnt::fs::util::CheckSaveData(info, dstInfo, 1);

    nnt::fs::util::DeleteAllTestSaveData();
}



const int64_t KeySeedSeed = 0x1234;

// patternParamBorderHeavy の境界値が適切なこと
TEST(SaveDataTransferTest, CheckTestParam)
{
    nnt::fs::util::DeleteAllTestSaveData();

    const auto SaveDataSize        = 64 * 1024;
    const auto SaveDataJournalSize = 64 * 1024;

    int i = 1;
    nn::ncm::ApplicationId appId = { GetSaveDataApplicationId() };
    nn::fs::SaveDataInfo info;
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(appId, GetUserIdByIndex(i), GetOwnerIdByIndex(i), SaveDataSize, SaveDataJournalSize, GetFlagsByIndex(i)));
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info, appId, GetUserIdByIndex(i)));

    auto exportSize = nn::fs::SaveDataTransferSizeCalculator::QuerySaveDataExportSize(info);

    EXPECT_EQ(ExpectedExportSizeForBorderTest, exportSize);

    nnt::fs::util::DeleteAllTestSaveData();
}

TEST(SaveDataTransferTest, ExportSaveDataMasterHeaderBrokenSaveData)
{
    const nn::ncm::ApplicationId ApplicationId = { GetSaveDataApplicationId() };

    nnt::fs::util::DeleteAllTestSaveData();

    // 壊れたセーブデータの準備
    int i = 1;
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(ApplicationId, GetUserIdByIndex(i), GetOwnerIdByIndex(i), DefaultSaveDataSize, DefaultSaveDataJournalSize, GetFlagsByIndex(i)));

    nn::fs::SaveDataInfo info = {};
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info, ApplicationId, GetUserIdByIndex(i)));

    static const auto BufferSize = nn::fssystem::save::FsJournalIntegritySaveDataFileSystemTest::MasterHeaderSize * 2;
    auto pBuffer = nnt::fs::util::AllocateBuffer(BufferSize);

    {
        // 先頭にマスターヘッダーが二重で配置されていることを確認
        size_t size;
        NNT_FS_ASSERT_NO_FATAL_FAILURE(nnt::fs::util::SaveDataFileSystemMounter::ReadFile(&size, info.saveDataSpaceId, info.saveDataId, 0, pBuffer.get(), BufferSize));
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(pBuffer.get(), pBuffer.get() + BufferSize / 2, BufferSize / 2);

        // 0 を壊す
        std::memset(pBuffer.get(), 0xFE, BufferSize / 2);
        NNT_FS_ASSERT_NO_FATAL_FAILURE(nnt::fs::util::SaveDataFileSystemMounter::WriteFile(info.saveDataSpaceId, info.saveDataId, 0, pBuffer.get(), BufferSize));
    }

    // エクスポートは成功する
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

        nn::fs::SaveDataTransferManager manager;
        SetUpDefaultSaveDataTransferManager(&manager, KeySeedSeed);

        std::unique_ptr<nn::fs::SaveDataExporter> exporter;
        NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataExporter(&exporter, info.saveDataSpaceId, info.saveDataId));
    }

    // 修正されていることを確認する
    {
        size_t size;
        NNT_FS_ASSERT_NO_FATAL_FAILURE(nnt::fs::util::SaveDataFileSystemMounter::ReadFile(&size, info.saveDataSpaceId, info.saveDataId, 0, pBuffer.get(), BufferSize));
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(pBuffer.get(), pBuffer.get() + BufferSize / 2, BufferSize / 2);

        // 完全に壊す
        std::memset(pBuffer.get(), 0xFE, BufferSize);
        NNT_FS_ASSERT_NO_FATAL_FAILURE(nnt::fs::util::SaveDataFileSystemMounter::WriteFile(info.saveDataSpaceId, info.saveDataId, 0, pBuffer.get(), BufferSize));
    }

    // エクスポートは失敗する
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

        nn::fs::SaveDataTransferManager manager;
        SetUpDefaultSaveDataTransferManager(&manager, KeySeedSeed);

        std::unique_ptr<nn::fs::SaveDataExporter> exporter;
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultJournalIntegritySaveDataControlAreaVerificationFailed,
            manager.OpenSaveDataExporter(&exporter, info.saveDataSpaceId, info.saveDataId)
        );
    }

    nnt::fs::util::DeleteAllTestSaveData();
}

TEST(SaveDataTransferTest, OpenExporterOnlyOneForOneSaveData)
{
    nnt::fs::util::DeleteAllTestSaveData();

    // セーブデータ準備
    nn::fs::SaveDataInfo info = {};
    {
        const nn::ncm::ApplicationId ApplicationId = { GetSaveDataApplicationId() };

        int i = 1;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(ApplicationId, GetUserIdByIndex(i), GetOwnerIdByIndex(i), DefaultSaveDataSize, DefaultSaveDataJournalSize, GetFlagsByIndex(i)));

        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info, ApplicationId, GetUserIdByIndex(i)));
    }

    // Exporter を2つ開く
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

        nn::fs::SaveDataTransferManager manager;
        SetUpDefaultSaveDataTransferManager(&manager, KeySeedSeed);

        std::unique_ptr<nn::fs::SaveDataExporter> exporter0;
        NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataExporter(&exporter0, info.saveDataSpaceId, info.saveDataId));

        std::unique_ptr<nn::fs::SaveDataExporter> exporter1;
        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetLocked, manager.OpenSaveDataExporter(&exporter1, info.saveDataSpaceId, info.saveDataId));
    }

    nnt::fs::util::DeleteAllTestSaveData();
}

TEST(SaveDataTransferTest, RecoverMissingThumbnail)
{
    nnt::fs::util::DeleteAllTestSaveData();

    const nn::ncm::ApplicationId ApplicationId = { GetSaveDataApplicationId() };
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(ApplicationId, GetUserIdByIndex(1), GetOwnerIdByIndex(1), DefaultSaveDataSize, DefaultSaveDataJournalSize, GetFlagsByIndex(1)));

    // 欠落させる
    nn::util::optional<nn::fs::SaveDataInfo> info;
    nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::User, [&](nn::fs::SaveDataInfo info_) {
        return info_.applicationId == ApplicationId && info_.saveDataType == nn::fs::SaveDataType::Account && info_.saveDataUserId == GetUserIdByIndex(1); });

    NNT_ASSERT_RESULT_SUCCESS(MountBis("bis", nn::fs::BisPartitionId::User));

    char path[128];
    nn::util::SNPrintf(path, sizeof(path), "bis:/saveMeta/%016llx/%08x.meta", info->saveDataId, 1);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteFile(path));


    // 欠落状態である
    {
        nn::fs::DirectoryEntryType type;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, nn::fs::GetEntryType(&type, path));
    }

    // OpenSaveDataExporter() で復旧
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

        nn::fs::SaveDataTransferManager manager;
        SetUpDefaultSaveDataTransferManager(&manager);

        std::unique_ptr<nn::fs::SaveDataExporter> exporter;
        NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataExporter(&exporter, info->saveDataSpaceId, info->saveDataId));
    }

    // 再作成されている
    {
        nn::fs::DirectoryEntryType type;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetEntryType(&type, path));
    }

    nn::fs::Unmount("bis");
    nnt::fs::util::DeleteAllTestSaveData();
}

#if !defined(NN_BUILD_CONFIG_OS_WIN)

// exporter/importer を開いたまま終了しても fs プロセス側に問題が発生しないこと
TEST(SaveDataTransferTest, DISABLED_Shutdown)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId appId = { GetSaveDataApplicationId() };

    nn::fs::SaveDataInfo info;
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(appId, GetUserIdByIndex(1), GetOwnerIdByIndex(1), DefaultSaveDataSize, DefaultSaveDataJournalSize, GetFlagsByIndex(1)));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info, appId, GetUserIdByIndex(1)));
    }

    nn::fs::SaveDataTransferManager manager;
    SetUpDefaultSaveDataTransferManager(&manager);

    std::unique_ptr<nn::fs::SaveDataExporter> exporter;
    NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataExporter(&exporter, info.saveDataSpaceId, info.saveDataId));

    char initialData[nn::fs::SaveDataExporter::InitialDataSize];
    NNT_EXPECT_RESULT_SUCCESS(exporter->PullInitialData(initialData, sizeof(initialData)));

    std::unique_ptr<nn::fs::SaveDataImporter> importer;
    int64_t requiredSize = 0;
    NNT_EXPECT_RESULT_SUCCESS(manager.OpenSaveDataImporter(&importer, &requiredSize, initialData, sizeof(initialData), GetUserIdByIndex(2), info.saveDataSpaceId));

    // exporter/importer を開いたまま終了
    nnt::Exit(0);
}

#endif

// save -> sd にエクスポート
TEST(SaveDataTransferTest, ExportBasic)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId appId = { GetSaveDataApplicationId() };
    nn::fs::SaveDataInfo info;

    int i = 1;

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(appId, GetUserIdByIndex(i), GetOwnerIdByIndex(i), DefaultSaveDataSize, DefaultSaveDataJournalSize, GetFlagsByIndex(i)));
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountSaveData("save", appId, GetUserIdByIndex(i)));
        auto filePath = nnt::fs::util::String("save:/file") + nnt::fs::util::ToString(i);
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateFileWith32BitCount(filePath.c_str(), i * 128 * 1024, i * 1024 * 1024));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));
        nn::fs::Unmount("save");
    }
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::FindSaveDataByApplicationId(&info, appId, GetUserIdByIndex(i)));

    nn::fs::SaveDataTransferManager manager;
    SetUpDefaultSaveDataTransferManager(&manager, KeySeedSeed);

    std::unique_ptr<nn::fs::SaveDataExporter> exporter;
    NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataExporter(&exporter, info.saveDataSpaceId, info.saveDataId));

    char initialData[nn::fs::SaveDataExporter::InitialDataSize];
    NNT_EXPECT_RESULT_SUCCESS(exporter->PullInitialData(initialData, sizeof(initialData)));

    auto exportSize = exporter->GetRestSize();
    auto buffer = nnt::fs::util::AllocateBuffer(static_cast<size_t>(exportSize));
    ASSERT_TRUE(buffer.get() != nullptr);

    // export at once
    size_t pulledSize = 0;
    NNT_EXPECT_RESULT_SUCCESS(exporter->Pull(&pulledSize, buffer.get(), static_cast<size_t>(exportSize)));
    EXPECT_EQ(exporter->GetRestSize(), 0);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("sd"));

    nn::fs::DeleteFile("sd:/saveDataTransferTest_initial.bin");
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndWriteFileAtOnce("sd:/saveDataTransferTest_initial.bin", initialData, sizeof(initialData)));

    nn::fs::DeleteFile("sd:/saveDataTransferTest_data.bin");
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndWriteFileAtOnce("sd:/saveDataTransferTest_data.bin", buffer.get(), static_cast<size_t>(exportSize)));

    exporter.reset();

    nn::fs::Unmount("sd");

    nnt::fs::util::DeleteAllTestSaveData();
}


#if defined(NN_BUILD_CONFIG_OS_WIN)

namespace nn { namespace fs { namespace detail {
            void SetSaveDataMacKeyWin(const void* pKey, size_t keySize) NN_NOEXCEPT;
}}}

// 違う固有鍵の本体にインポート出来ることの確認のための鍵差し替え
// Indexer の mac も変わるため要再起動
TEST(SaveDataTransferTest, ChangeMacKeyForImportBasic)
{
    nn::fs::detail::SetSaveDataMacKeyWin("changed_key_____", 16);
}

#endif

// sd -> save にインポート
// @pre SaveDataTransferTest_ExportBasic が実行済み
TEST(SaveDataTransferTest, ImportBasic)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OverrideSaveDataTransferTokenSignVerificationKey(TestPublicModulus, sizeof(TestPublicModulus)));

    nnt::fs::util::DeleteAllTestSaveData();

    nn::ncm::ApplicationId appId = { GetSaveDataApplicationId() };

    int i = 1;

    nn::fs::SaveDataTransferManager manager;
    SetUpDefaultSaveDataTransferManager(&manager, KeySeedSeed);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("sd"));

    auto initialDataBuffer = nnt::fs::util::AllocateBuffer(32);
    size_t initialDataSize;
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::ReadFileAtOnce(&initialDataSize, &initialDataBuffer, "sd:/saveDataTransferTest_initial.bin"));

    auto dataBuffer = nnt::fs::util::AllocateBuffer(32);
    size_t dataSize;
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::ReadFileAtOnce(&dataSize, &dataBuffer, "sd:/saveDataTransferTest_data.bin"));

    nn::fs::Unmount("sd");

    std::unique_ptr<nn::fs::SaveDataImporter> importer;
    int64_t requiredSize;
    NNT_ASSERT_RESULT_SUCCESS(manager.OpenSaveDataImporter(&importer, &requiredSize, initialDataBuffer.get(), initialDataSize, GetUserIdByIndex(i + 1), nn::fs::SaveDataSpaceId::User));

    // import at once
    NNT_EXPECT_RESULT_SUCCESS(importer->Push(dataBuffer.get(), dataSize));
    NNT_EXPECT_RESULT_SUCCESS(importer->Finalize());
    importer.reset();

    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountSaveData("dst", appId, GetUserIdByIndex(i + 1)));

        auto filePath = nnt::fs::util::String("dst:/file") + nnt::fs::util::ToString(i);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith32BitCount(filePath.c_str(), i * 1024 * 1024));

        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("dst:/"));

        nn::fs::Unmount("dst");
    }

    nnt::fs::util::DeleteAllTestSaveData();
}



extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetEnabledAutoAbort(false);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    nn::spl::InitializeForCrypto();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nn::spl::Finalize();
    nnt::Exit(result);
}
