﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>

#include <nn/nn_Common.h>
#include <nn/fs.h>
#include <nn/fs/fs_DeviceSaveData.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_Transaction.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

namespace {

    const char TestFileName[] = "file";

    const int64_t UserSaveDataSize = 8 * 1024 * 1024;
    const int64_t UserSaveDataJournalSize = 8 * 1024 * 1024;
    const int UserSaveDataCountSplit = 5;
    const int UserSaveDataCountLimit = 10;

    const int64_t SystemSaveDataSize = 1 * 1024 * 1024;
    const int64_t SystemSaveDataJournalSize = 1 * 1024 * 1024;
    const int SystemSaveDataCountSplit = 8;
    const int SystemSaveDataCountLimit = 10;

    nn::Result FindSaveDataId(nn::fs::SaveDataId *outValue, nn::ncm::ApplicationId applicationId, nn::fs::UserId userId) NN_NOEXCEPT
    {
        const nn::fs::SaveDataSpaceId SpaceIds[] = {nn::fs::SaveDataSpaceId::User, nn::fs::SaveDataSpaceId::System};
        for( auto spaceId : SpaceIds )
        {
            std::unique_ptr<nn::fs::SaveDataIterator> iter;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, spaceId));

            while( NN_STATIC_CONDITION(true) )
            {
                int64_t count;
                nn::fs::SaveDataInfo info;
                NNT_EXPECT_RESULT_SUCCESS(iter->ReadSaveDataInfo(&count, &info, 1));
                if( count == 0 )
                {
                    break;
                }

                if( info.applicationId.value == applicationId.value && info.saveDataUserId == userId )
                {
                    *outValue = info.saveDataId;
                    NN_RESULT_SUCCESS;
                }
            }
        }
        return nn::fs::ResultTargetNotFound();
    }

    nn::fs::UserId GetUserSaveDataIdByIndex(int index) NN_NOEXCEPT
    {
        nn::fs::UserId userId = {{ 0xFull, static_cast<uint32_t>(index) }};
        return userId;
    }

    nnt::fs::util::String GetUserMountNameByIndex(int index) NN_NOEXCEPT
    {
        nnt::fs::util::String mountName = "save" + nnt::fs::util::ToString(index);
        return mountName;
    }

    nnt::fs::util::String GetSystemMountNameByIndex(int index) NN_NOEXCEPT
    {
        nnt::fs::util::String mountName = "syssave" + nnt::fs::util::ToString(index);
        return mountName;
    }

    nn::fs::SystemSaveDataId GetSystemSaveDataIdByIndex(int index) NN_NOEXCEPT
    {
        const nn::fs::SystemSaveDataId BaseId = 0x8000000000004100;
        return BaseId + index;
    }

    // セーブデータをダーティー化する
    void MakeDirtinessSaveData(nnt::fs::util::String mountName, int64_t sizeSaveData)
    {
        nnt::fs::util::String TestPath = mountName + ":/" + TestFileName;

        int64_t fileSize = sizeSaveData - 16 * 1024 * 2;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(TestPath.c_str(), fileSize));
        char buf[8] = {};

        nn::fs::FileHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, TestPath.c_str(), nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        int64_t offsetMin = 16 * 1024 - static_cast<int64_t>(sizeof(buf) / 2);
        for( int64_t offset = offsetMin; offset + static_cast<int64_t>(sizeof(buf)) < fileSize; offset += 32 * 1024 )
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(
                handle, offset, buf, sizeof(buf),
                nn::fs::WriteOption()));
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(handle));
    }

    // セーブデータにデータを書き込む
    void WriteSequential(nnt::fs::util::String mountName, int sizeSaveData, int seed) NN_NOEXCEPT
    {
        nnt::fs::util::String TestPath = mountName + ":/" + TestFileName;

        int64_t fileSize = sizeSaveData - 16 * 1024 * 2;
        size_t bufferSize = 4 *  1024;
        nn::fs::DeleteFile(TestPath.c_str());
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(TestPath.c_str(), fileSize));

        {
            std::unique_ptr<char[]> writeBuffer(new char[bufferSize]);
            ASSERT_NE(nullptr, writeBuffer.get());
            for( int i = 0; i < static_cast<int>(bufferSize); ++i )
            {
                writeBuffer[i] = static_cast<char>((i + seed) & 0xFF);
            }

            nn::fs::FileHandle handle;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, TestPath.c_str(), nn::fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handle);
            };

            for( int64_t offset = 0; offset < fileSize; offset += bufferSize )
            {
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::WriteFile(handle, offset, writeBuffer.get(), bufferSize, nn::fs::WriteOption()));
            }
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::FlushFile(handle));
        }
    }

    void ReadVerifySequential(nnt::fs::util::String mountName, int sizeSaveData, int seed) NN_NOEXCEPT
    {
        nnt::fs::util::String TestPath = mountName + ":/" + TestFileName;

        int64_t fileSize = sizeSaveData - 16 * 1024 * 2;
        size_t bufferSize = 4 *  1024;

        {
            int64_t totalSize = fileSize;

            nn::fs::FileHandle handle;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&handle, TestPath.c_str(), nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handle);
            };

            std::unique_ptr<char[]> readBuffer(new char[bufferSize]);
            ASSERT_NE(nullptr, readBuffer.get());

            for(int64_t offset = 0; offset < totalSize; offset += bufferSize)
            {
                NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadFile(handle, offset, readBuffer.get(), bufferSize));
                for( int i = 0; i < static_cast<int>(bufferSize); ++i )
                {
                    EXPECT_EQ(static_cast<char>((i + seed) & 0xFF), readBuffer[i]);
                }
            }
        }
    }

}

// 複数コミットとコミット後に他のセーブデータを操作できるかを確認
TEST(MountMultiSaveData, CommitMulti)
{
    // テストで使用するのと同 ID の既存セーブデータを削除
    for( int i = 1; i <= UserSaveDataCountLimit; ++i )
    {
        nn::fs::SaveDataId saveDataId = 0;
        auto result = FindSaveDataId(
            &saveDataId,
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(i));
        if( result.IsSuccess() )
        {
            nn::fs::DeleteSaveData(saveDataId);
        }
    }

    // アプリケーションセーブデータを作成
    for( int i = 1; i <= UserSaveDataCountLimit; ++i )
    {
        NN_LOG("create user save data #%d\n", i);
        auto result = nn::fs::CreateSaveData(
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(i),
            nnt::fs::util::ApplicationId.value,
            UserSaveDataSize,
            UserSaveDataJournalSize,
            0);
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
    NN_UTIL_SCOPE_EXIT
    {
        for( int i = 1; i <= UserSaveDataCountLimit; ++i )
        {
            nn::fs::Unmount(GetUserMountNameByIndex(i).c_str());
            nn::fs::SaveDataId saveDataId = 0;
            auto result = FindSaveDataId(
                &saveDataId,
                nnt::fs::util::ApplicationId,
                GetUserSaveDataIdByIndex(i)
                );
            if( result.IsSuccess() )
            {
                nn::fs::DeleteSaveData(saveDataId);
            }
        }
    };

    // アプリケーションセーブデータをマウント
    for( int i = 1; i <= UserSaveDataCountLimit; ++i )
    {
        NN_LOG("mount user save data #%d\n", i);
        auto result = nn::fs::MountSaveData(
            GetUserMountNameByIndex(i).c_str(),
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(i));
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    // アプリケーションセーブデータを Dirty 化する
    for( int i = 1; i <= UserSaveDataCountSplit; ++i )
    {
        NN_LOG("write user save data #%d\n", i);
        MakeDirtinessSaveData(GetUserMountNameByIndex(i), UserSaveDataSize);
    }

    // 複数コミットを実行する
    NN_LOG("commit user save data\n");
    nnt::fs::util::Vector<nnt::fs::util::String> fileSystemNameListBuffer;
    for( int i = 0; i < UserSaveDataCountSplit; ++i )
    {
        fileSystemNameListBuffer.push_back(GetUserMountNameByIndex(i + 1));
    }
    const char* fileSystemNameList[UserSaveDataCountSplit];
    for( int i = 0; i < UserSaveDataCountSplit; ++i )
    {
        fileSystemNameList[i] = fileSystemNameListBuffer[i].c_str();
    }
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::Commit(&fileSystemNameList[0], UserSaveDataCountSplit));

    // 残りのセーブデータを Dirty 化する
    for( int i = UserSaveDataCountSplit + 1; i <= UserSaveDataCountLimit; ++i )
    {
        NN_LOG("write user save data #%d\n", i);
        MakeDirtinessSaveData(GetUserMountNameByIndex(i), UserSaveDataSize);
    }
} // NOLINT(impl/function_size)

// 複数コミットとデータがコミットされたことの確認
TEST(MountMultiSaveData, CommitMultiVerify)
{
    // テストで使用するのと同 ID の既存セーブデータを削除
    for( int i = 1; i <= UserSaveDataCountLimit; ++i )
    {
        nn::fs::SaveDataId saveDataId = 0;
        auto result = FindSaveDataId(
            &saveDataId,
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(i));
        if( result.IsSuccess() )
        {
            nn::fs::DeleteSaveData(saveDataId);
        }
    }

    // アプリケーションセーブデータを作成
    for( int i = 1; i <= UserSaveDataCountLimit; ++i )
    {
        NN_LOG("create user save data #%d\n", i);
        auto result = nn::fs::CreateSaveData(
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(i),
            nnt::fs::util::ApplicationId.value,
            UserSaveDataSize,
            UserSaveDataJournalSize,
            0);
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
    NN_UTIL_SCOPE_EXIT
    {
        for( int i = 1; i <= UserSaveDataCountLimit; ++i )
        {
            nn::fs::Unmount(GetUserMountNameByIndex(i).c_str());
            nn::fs::SaveDataId saveDataId = 0;
            auto result = FindSaveDataId(
                &saveDataId,
                nnt::fs::util::ApplicationId,
                GetUserSaveDataIdByIndex(i)
                );
            if( result.IsSuccess() )
            {
                nn::fs::DeleteSaveData(saveDataId);
            }
        }
    };

    // アプリケーションセーブデータをマウント
    for( int i = 1; i <= UserSaveDataCountLimit; ++i )
    {
        NN_LOG("mount user save data #%d\n", i);
        auto result = nn::fs::MountSaveData(
            GetUserMountNameByIndex(i).c_str(),
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(i));
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    // アプリケーションセーブデータに固定値を書き込む
    for( int i = 1; i <= UserSaveDataCountSplit; ++i )
    {
        NN_LOG("write user save data #%d\n", i);
        WriteSequential(GetUserMountNameByIndex(i), UserSaveDataSize, i);
    }

    // 複数コミットを実行する
    NN_LOG("commit user save data\n");
    nnt::fs::util::Vector<nnt::fs::util::String> fileSystemNameListBuffer;
    for( int i = 0; i < UserSaveDataCountSplit; ++i )
    {
        fileSystemNameListBuffer.push_back(GetUserMountNameByIndex(i + 1));
    }
    const char* fileSystemNameList[UserSaveDataCountSplit];
    for( int i = 0; i < UserSaveDataCountSplit; ++i )
    {
        fileSystemNameList[i] = fileSystemNameListBuffer[i].c_str();
    }
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::Commit(&fileSystemNameList[0], UserSaveDataCountSplit));

    // 再マウント
    for( int i = 1; i <= UserSaveDataCountLimit; ++i )
    {
        nn::fs::Unmount(GetUserMountNameByIndex(i).c_str());
        auto result = nn::fs::MountSaveData(
            GetUserMountNameByIndex(i).c_str(),
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(i));
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    // 正しくコミットされていることを確認
    for( int i = 1; i <= UserSaveDataCountSplit; ++i )
    {
        NN_LOG("verify user save data #%d\n", i);
        ReadVerifySequential(GetUserMountNameByIndex(i), UserSaveDataSize, i);
    }
} // NOLINT(impl/function_size)

// 単一セーブデータへの複数コミットとデータがコミットされたことの確認
TEST(MountMultiSaveData, CommitMultiVerifySingle)
{
    // テストで使用するのと同 ID の既存セーブデータを削除
    {
        nn::fs::SaveDataId saveDataId = 0;
        auto result = FindSaveDataId(
            &saveDataId,
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(1));
        if( result.IsSuccess() )
        {
            nn::fs::DeleteSaveData(saveDataId);
        }
    }

    // アプリケーションセーブデータを作成
    {
        auto result = nn::fs::CreateSaveData(
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(1),
            nnt::fs::util::ApplicationId.value,
            UserSaveDataSize,
            UserSaveDataJournalSize,
            0);
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
    NN_UTIL_SCOPE_EXIT
    {
        {
            nn::fs::Unmount(GetUserMountNameByIndex(1).c_str());
            nn::fs::SaveDataId saveDataId = 0;
            auto result = FindSaveDataId(
                &saveDataId,
                nnt::fs::util::ApplicationId,
                GetUserSaveDataIdByIndex(1)
                );
            if( result.IsSuccess() )
            {
                nn::fs::DeleteSaveData(saveDataId);
            }
        }
    };

    // アプリケーションセーブデータをマウント
    {
        auto result = nn::fs::MountSaveData(
            GetUserMountNameByIndex(1).c_str(),
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(1));
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    // アプリケーションセーブデータに固定値を書き込む
    {
        WriteSequential(GetUserMountNameByIndex(1), UserSaveDataSize, 0x77);
    }

    // 複数コミットを実行する
    nnt::fs::util::Vector<nnt::fs::util::String> fileSystemNameListBuffer;
    fileSystemNameListBuffer.push_back(GetUserMountNameByIndex(1));
    const char* fileSystemNameList[1];
    fileSystemNameList[0] = fileSystemNameListBuffer[0].c_str();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::Commit(&fileSystemNameList[0], 1));

    // 再マウント
    {
        nn::fs::Unmount(GetUserMountNameByIndex(1).c_str());
        auto result = nn::fs::MountSaveData(
            GetUserMountNameByIndex(1).c_str(),
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(1));
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    // 正しくコミットされていることを確認
    {
        ReadVerifySequential(GetUserMountNameByIndex(1), UserSaveDataSize, 0x77);
    }
} // NOLINT(impl/function_size)

// 何もデータを書き込まずに複数コミットを実行する
TEST(MountMultiSaveData, CommitMultiWithoutWrite)
{
    // テストで使用するのと同 ID の既存セーブデータを削除
    for( int i = 1; i <= UserSaveDataCountLimit; ++i )
    {
        nn::fs::SaveDataId saveDataId = 0;
        auto result = FindSaveDataId(
            &saveDataId,
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(i));
        if( result.IsSuccess() )
        {
            nn::fs::DeleteSaveData(saveDataId);
        }
    }

    // アプリケーションセーブデータを作成
    for( int i = 1; i <= UserSaveDataCountLimit; ++i )
    {
        NN_LOG("create user save data #%d\n", i);
        auto result = nn::fs::CreateSaveData(
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(i),
            nnt::fs::util::ApplicationId.value,
            UserSaveDataSize,
            UserSaveDataJournalSize,
            0);
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
    NN_UTIL_SCOPE_EXIT
    {
        for( int i = 1; i <= UserSaveDataCountLimit; ++i )
        {
            nn::fs::Unmount(GetUserMountNameByIndex(i).c_str());
            nn::fs::SaveDataId saveDataId = 0;
            auto result = FindSaveDataId(
                &saveDataId,
                nnt::fs::util::ApplicationId,
                GetUserSaveDataIdByIndex(i)
                );
            if( result.IsSuccess() )
            {
                nn::fs::DeleteSaveData(saveDataId);
            }
        }
    };

    // アプリケーションセーブデータをマウント
    for( int i = 1; i <= UserSaveDataCountLimit; ++i )
    {
        NN_LOG("mount user save data #%d\n", i);
        auto result = nn::fs::MountSaveData(
            GetUserMountNameByIndex(i).c_str(),
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(i));
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    // 書き込みは一切行わない

    // 複数コミットを実行する
    NN_LOG("commit user save data\n");
    nnt::fs::util::Vector<nnt::fs::util::String> fileSystemNameListBuffer;
    for( int i = 0; i < UserSaveDataCountSplit; ++i )
    {
        fileSystemNameListBuffer.push_back(GetUserMountNameByIndex(i + 1));
    }
    const char* fileSystemNameList[UserSaveDataCountSplit];
    for( int i = 0; i < UserSaveDataCountSplit; ++i )
    {
        fileSystemNameList[i] = fileSystemNameListBuffer[i].c_str();
    }
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::Commit(&fileSystemNameList[0], UserSaveDataCountSplit));

    // 再マウント
    for( int i = 1; i <= UserSaveDataCountLimit; ++i )
    {
        nn::fs::Unmount(GetUserMountNameByIndex(i).c_str());
        auto result = nn::fs::MountSaveData(
            GetUserMountNameByIndex(i).c_str(),
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(i));
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
} // NOLINT(impl/function_size)

// 0個のファイルシステムを渡した時の挙動確認
TEST(MountMultiSaveData, CommitMultiZero)
{
    const char* fileSystemNameList[1];
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::Commit(&fileSystemNameList[0], 0));
}

// 負の個数を渡した時の挙動確認
TEST(MountMultiSaveData, CommitMultiCountLess)
{
    const char* fileSystemNameList[1];
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultOutOfRange, nn::fs::Commit(&fileSystemNameList[0], -1));
}

// 複数コミット可能な個数の上限を超えた時の挙動確認
TEST(MountMultiSaveData, CommitMultiCountOver)
{
    const char* fileSystemNameList[nn::fs::detail::CommittableCountMax + 2];
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultOutOfRange, nn::fs::Commit(&fileSystemNameList[0], nn::fs::detail::CommittableCountMax + 1));
}

// 同じセーブデータを複数コミットに渡した時の挙動確認
TEST(MountMultiSaveData, CommitMultiDouble)
{
    // テストで使用するのと同 ID の既存セーブデータを削除
    for( int i = 1; i <= UserSaveDataCountLimit; ++i )
    {
        nn::fs::SaveDataId saveDataId = 0;
        auto result = FindSaveDataId(
            &saveDataId,
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(1));
        if( result.IsSuccess() )
        {
            nn::fs::DeleteSaveData(saveDataId);
        }
    }

    // アプリケーションセーブデータを作成
    for( int i = 1; i <= 2; ++i )
    {
        auto result = nn::fs::CreateSaveData(
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(i),
            nnt::fs::util::ApplicationId.value,
            UserSaveDataSize,
            UserSaveDataJournalSize,
            0);
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
    NN_UTIL_SCOPE_EXIT
    {
        for( int i = 1; i <= 2; ++i )
        {
            nn::fs::Unmount(GetUserMountNameByIndex(i).c_str());
            nn::fs::SaveDataId saveDataId = 0;
            auto result = FindSaveDataId(
                &saveDataId,
                nnt::fs::util::ApplicationId,
                GetUserSaveDataIdByIndex(i)
                );
            if( result.IsSuccess() )
            {
                nn::fs::DeleteSaveData(saveDataId);
            }
        }
    };

    // アプリケーションセーブデータをマウント
    for( int i = 1; i <= 2; ++i )
    {
        auto result = nn::fs::MountSaveData(
            GetUserMountNameByIndex(i).c_str(),
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(i));
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    // アプリケーションセーブデータに固定値を書き込む
    for( int i = 1; i <= 2; ++i )
    {
        WriteSequential(GetUserMountNameByIndex(i), UserSaveDataSize, 0x77);
    }

    // 複数コミットを実行する
    nnt::fs::util::Vector<nnt::fs::util::String> fileSystemNameListBuffer;
    fileSystemNameListBuffer.push_back(GetUserMountNameByIndex(1));
    fileSystemNameListBuffer.push_back(GetUserMountNameByIndex(2));
    fileSystemNameListBuffer.push_back(GetUserMountNameByIndex(1));
    fileSystemNameListBuffer.push_back(GetUserMountNameByIndex(1));
    int fileSystemCount = static_cast<int>(fileSystemNameListBuffer.size());
    const char* fileSystemNameList[4];
    for( int i = 0; i < fileSystemCount; ++i )
    {
        fileSystemNameList[i] = fileSystemNameListBuffer[i].c_str();
    }
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultMountNameAlreadyExists, nn::fs::Commit(&fileSystemNameList[0], fileSystemCount));
} // NOLINT(impl/function_size)

// 複数コミットに存在しないマウント名を指定する
TEST(MountMultiSaveData, CommitMultiInvalidName)
{
    // テストで使用するのと同 ID の既存セーブデータを削除
    for( int i = 1; i <= UserSaveDataCountLimit; ++i )
    {
        nn::fs::SaveDataId saveDataId = 0;
        auto result = FindSaveDataId(
            &saveDataId,
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(i));
        if( result.IsSuccess() )
        {
            nn::fs::DeleteSaveData(saveDataId);
        }
    }

    // アプリケーションセーブデータを作成
    for( int i = 1; i <= UserSaveDataCountLimit; ++i )
    {
        NN_LOG("create user save data #%d\n", i);
        auto result = nn::fs::CreateSaveData(
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(i),
            nnt::fs::util::ApplicationId.value,
            UserSaveDataSize,
            UserSaveDataJournalSize,
            0);
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
    NN_UTIL_SCOPE_EXIT
    {
        for( int i = 1; i <= UserSaveDataCountLimit; ++i )
        {
            nn::fs::Unmount(GetUserMountNameByIndex(i).c_str());
            nn::fs::SaveDataId saveDataId = 0;
            auto result = FindSaveDataId(
                &saveDataId,
                nnt::fs::util::ApplicationId,
                GetUserSaveDataIdByIndex(i)
                );
            if( result.IsSuccess() )
            {
                nn::fs::DeleteSaveData(saveDataId);
            }
        }
    };

    // アプリケーションセーブデータをマウント
    for( int i = 1; i <= UserSaveDataCountLimit; ++i )
    {
        NN_LOG("mount user save data #%d\n", i);
        auto result = nn::fs::MountSaveData(
            GetUserMountNameByIndex(i).c_str(),
            nnt::fs::util::ApplicationId,
            GetUserSaveDataIdByIndex(i));
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    // アプリケーションセーブデータを Dirty 化する
    for( int i = 1; i <= UserSaveDataCountSplit; ++i )
    {
        NN_LOG("write user save data #%d\n", i);
        MakeDirtinessSaveData(GetUserMountNameByIndex(i), UserSaveDataSize);
    }

    // 存在しないマウント名を混ぜて複数コミットを実行する
    NN_LOG("commit user save data\n");
    nnt::fs::util::Vector<nnt::fs::util::String> fileSystemNameListBuffer;
    for( int i = 0; i < UserSaveDataCountSplit; ++i )
    {
        fileSystemNameListBuffer.push_back(GetUserMountNameByIndex(i + 1));
    }
    fileSystemNameListBuffer[3] = "invalid:";
    const char* fileSystemNameList[UserSaveDataCountSplit];
    for( int i = 0; i < UserSaveDataCountSplit; ++i )
    {
        fileSystemNameList[i] = fileSystemNameListBuffer[i].c_str();
    }
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultNotMounted, nn::fs::Commit(&fileSystemNameList[0], UserSaveDataCountSplit));
} // NOLINT(impl/function_size)

// デバイスセーブデータに対する複数コミットとデータがコミットされたことの確認
TEST(MountMultiDeviceSaveData, CommitMultiVerify)
{
    // デバイスセーブデータをマウント
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountDeviceSaveData("device"));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("device");
    };

    // アプリケーションセーブデータに固定値を書き込む
    WriteSequential("device", UserSaveDataSize, 0x33);

    // 複数コミットを実行する
    NN_LOG("commit user save data\n");
    nnt::fs::util::Vector<nnt::fs::util::String> fileSystemNameListBuffer;
    fileSystemNameListBuffer.push_back("device");
    const char* fileSystemNameList[1];
    fileSystemNameList[0] = fileSystemNameListBuffer[0].c_str();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::Commit(&fileSystemNameList[0], 1));

    // 再マウント
    nn::fs::Unmount("device");
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountDeviceSaveData("device", nnt::fs::util::ApplicationId));

    // 正しくコミットされていることを確認
    ReadVerifySequential("device", UserSaveDataSize, 0x33);
} // NOLINT(impl/function_size)

// 複数コミットに対応していないファイルシステムに対して実行する
TEST(MountMultiSystemSaveData, CommitMulti)
{
    // テストで使用するのと同 ID の既存セーブデータを削除
    for( int i = 1; i <= SystemSaveDataCountLimit; ++i )
    {
        const auto saveDataId = static_cast<nn::fs::SaveDataId>(GetSystemSaveDataIdByIndex(i));
        nn::fs::DeleteSaveData(saveDataId);
    }

    // システムセーブデータを作成
    for( int i = 1; i <= SystemSaveDataCountLimit; ++i )
    {
        NN_LOG("create system save data #%d\n", i);
        auto result = nn::fs::CreateSystemSaveData(
            GetSystemSaveDataIdByIndex(i), SystemSaveDataSize, SystemSaveDataJournalSize, 0);
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
    NN_UTIL_SCOPE_EXIT
    {
        // 全システムセーブデータをアンマウント、削除
        for( int i = 1; i <= SystemSaveDataCountLimit; ++i )
        {
            nn::fs::Unmount(GetSystemMountNameByIndex(i).c_str());
            const auto saveDataId = static_cast<nn::fs::SaveDataId>(GetSystemSaveDataIdByIndex(i));
            nn::fs::DeleteSaveData(saveDataId);
        }
    };

    // システムセーブデータをマウント
    for( int i = 1; i <= SystemSaveDataCountLimit; ++i )
    {
        NN_LOG("mount system save data #%d\n", i);
        auto result = nn::fs::MountSystemSaveData(
            GetSystemMountNameByIndex(i).c_str(),
            GetSystemSaveDataIdByIndex(i));
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    // システムセーブを Dirty 化する
    for( int i = 1; i <= SystemSaveDataCountSplit; ++i )
    {
        NN_LOG("write system save data #%d\n", i);
        MakeDirtinessSaveData(GetSystemMountNameByIndex(i), SystemSaveDataSize);
    }

    // 複数コミットを実行する
    NN_LOG("commit system save data\n");
    nnt::fs::util::Vector<nnt::fs::util::String> fileSystemNameListBuffer;
    for( int i = 0; i < SystemSaveDataCountSplit; ++i )
    {
        fileSystemNameListBuffer.push_back(GetSystemMountNameByIndex(i + 1));
    }
    const char* fileSystemNameList[SystemSaveDataCountSplit];
    for( int i = 0; i < SystemSaveDataCountSplit; ++i )
    {
        fileSystemNameList[i] = fileSystemNameListBuffer[i].c_str();
    }
    // システムセーブデータは複数コミットに対応してない
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultUnsupportedOperation, nn::fs::Commit(&fileSystemNameList[0], SystemSaveDataCountSplit));
} // NOLINT(impl/function_size)

// 複数コミットに対応していないファイルシステム1つに対して実行する
TEST(MountMultiSystemSaveData, CommitMultiSingle)
{
    // テストで使用するのと同 ID の既存セーブデータを削除
    for( int i = 1; i <= SystemSaveDataCountLimit; ++i )
    {
        const auto saveDataId = static_cast<nn::fs::SaveDataId>(GetSystemSaveDataIdByIndex(i));
        nn::fs::DeleteSaveData(saveDataId);
    }

    // システムセーブデータを作成
    {
        auto result = nn::fs::CreateSystemSaveData(
            GetSystemSaveDataIdByIndex(1), SystemSaveDataSize, SystemSaveDataJournalSize, 0);
        NNT_ASSERT_RESULT_SUCCESS(result);
    }
    NN_UTIL_SCOPE_EXIT
    {
        // 全システムセーブデータをアンマウント、削除
        {
            nn::fs::Unmount(GetSystemMountNameByIndex(1).c_str());
            const auto saveDataId = static_cast<nn::fs::SaveDataId>(GetSystemSaveDataIdByIndex(1));
            nn::fs::DeleteSaveData(saveDataId);
        }
    };

    // システムセーブデータをマウント
    {
        auto result = nn::fs::MountSystemSaveData(
            GetSystemMountNameByIndex(1).c_str(),
            GetSystemSaveDataIdByIndex(1));
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    // システムセーブを Dirty 化する
    {
        MakeDirtinessSaveData(GetSystemMountNameByIndex(1), SystemSaveDataSize);
    }

    // 複数コミットを実行する
    nnt::fs::util::Vector<nnt::fs::util::String> fileSystemNameListBuffer;
    fileSystemNameListBuffer.push_back(GetSystemMountNameByIndex(1));
    const char* fileSystemNameList[1];
    fileSystemNameList[0] = fileSystemNameListBuffer[0].c_str();

    // システムセーブデータは複数コミットに対応してない
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultUnsupportedOperation, nn::fs::Commit(&fileSystemNameList[0], 1));
} // NOLINT(impl/function_size)

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();

    nnt::Exit(testResult);
}
