﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>

using namespace nn;

namespace {
    const int MaxPathLength = 256;
    const size_t SaveDataSize = 16 * 1024 * 1024;
    const size_t SaveDataJournalSize = 16 * 1024 * 1024;
    const nn::fs::UserId IndexerOnlySaveUserId = { { 0, 1 } };
    const nn::fs::UserId NoIndexerSaveUserId   = { { 0, 2 } };
    const nn::fs::SystemSaveDataId IndexerOnlySystemSaveDataId = 0x8000000000004000ULL;
    const nn::fs::SystemSaveDataId NoIndexerSystemSaveDataId   = 0x8000000000004001ULL;
}

void CleanSaveDataCache()
{
    /* SaveDataFs のキャッシュ追い出し */
    nn::fs::MountSystemSaveData("save", 0x8000000000000000);
    nn::fs::Unmount("save");
}

void GetSaveDataInfo(nn::fs::SaveDataInfo* pOutValue, nn::fs::SaveDataSpaceId spaceId, nn::fs::SaveDataType saveDataType, nn::ncm::ApplicationId applicationId, nn::fs::UserId userId, nn::fs::SystemSaveDataId systemSaveDataId)
{
    nnt::fs::util::Vector<nn::fs::SaveDataInfo> infoArray;
    nnt::fs::util::FindSaveData(&infoArray, spaceId,
        [=](const nn::fs::SaveDataInfo& info) { return (info.saveDataType == saveDataType) && (info.applicationId == applicationId) && (info.saveDataUserId == userId) && (info.systemSaveDataId == systemSaveDataId); }
    );
    ASSERT_EQ(1, infoArray.size());
    *pOutValue = infoArray[0];
}

void GetUserSaveDataInfo(nn::fs::SaveDataInfo* pOutValue, nn::fs::SaveDataType saveDataType, nn::fs::UserId userId)
{
    GetSaveDataInfo(pOutValue, nn::fs::SaveDataSpaceId::User, saveDataType, nnt::fs::util::ApplicationId, userId, nn::fs::InvalidSystemSaveDataId);
}

void GetSystemSaveDataInfo(nn::fs::SaveDataInfo* pOutValue, nn::fs::SystemSaveDataId systemSaveDataId)
{
    GetSaveDataInfo(pOutValue, nn::fs::SaveDataSpaceId::System, nn::fs::SaveDataType::System, nn::fs::InvalidApplicationId, nn::fs::InvalidUserId, systemSaveDataId);
}

void CreateIndexerOnlyUserSaveData(nn::fs::SaveDataId* pOutValue)
{
    // テスト用セーブデータを作成
    nn::fs::CreateSaveData(nnt::fs::util::ApplicationId, IndexerOnlySaveUserId, 0, SaveDataSize, SaveDataJournalSize, 0);

    nn::fs::SaveDataInfo saveDataInfo;
    GetUserSaveDataInfo(&saveDataInfo, nn::fs::SaveDataType::Account, IndexerOnlySaveUserId);

    // テスト用セーブデータの実ファイルのみ削除
    CleanSaveDataCache();
    nn::fs::BisPartitionId partitonId = nn::fs::BisPartitionId::User;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(partitonId, nullptr));

    char path[MaxPathLength];
    nn::util::SNPrintf(path, sizeof(path), "%s:/save/%016llx", nn::fs::GetBisMountName(partitonId), saveDataInfo.saveDataId);
//    NN_LOG("DeleteFile : %s\n", path);
    nn::fs::DeleteFile(path);

    nn::fs::Unmount(nn::fs::GetBisMountName(partitonId));

    if (pOutValue != nullptr)
    {
        *pOutValue = saveDataInfo.saveDataId;
    }
}


void CreateNoIndexerUserSaveData(nn::fs::SaveDataId* pOutValue)
{
    // テスト用セーブデータを作成
    nn::fs::CreateSaveData(nnt::fs::util::ApplicationId, IndexerOnlySaveUserId, 0, SaveDataSize, SaveDataJournalSize, 0);

    nn::fs::SaveDataInfo saveDataInfo;
    GetUserSaveDataInfo(&saveDataInfo, nn::fs::SaveDataType::Account, IndexerOnlySaveUserId);

    // テスト用セーブデータの実ファイルをリネームし、次に作成されるセーブデータIDの実ファイルにする
    CleanSaveDataCache();
    nn::fs::BisPartitionId partitonId = nn::fs::BisPartitionId::User;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(partitonId, nullptr));

    char srcPath[MaxPathLength];
    char dstPath[MaxPathLength];
    nn::fs::SaveDataId noIndexerSaveDataId = saveDataInfo.saveDataId + 1;

    nn::util::SNPrintf(srcPath, sizeof(srcPath), "%s:/save/%016llx", nn::fs::GetBisMountName(partitonId), saveDataInfo.saveDataId);
    nn::util::SNPrintf(dstPath, sizeof(dstPath), "%s:/save/%016llx", nn::fs::GetBisMountName(partitonId), noIndexerSaveDataId);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::RenameFile(srcPath, dstPath));

    nn::fs::Unmount(nn::fs::GetBisMountName(partitonId));

    nn::fs::DeleteSaveData(saveDataInfo.saveDataId);
    if (pOutValue != nullptr)
    {
        *pOutValue = noIndexerSaveDataId;
    }
}

void CreateIndexerOnlySystemSaveData()
{
    // テスト用システムセーブデータを作成
    nn::fs::CreateSystemSaveData(IndexerOnlySystemSaveDataId, SaveDataSize, SaveDataJournalSize, 0);

    nn::fs::SaveDataInfo saveDataInfo;
    GetSystemSaveDataInfo(&saveDataInfo, IndexerOnlySystemSaveDataId);

    // テスト用システムセーブデータの実ファイルのみ削除
    CleanSaveDataCache();
    nn::fs::BisPartitionId partitonId = nn::fs::BisPartitionId::System;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(partitonId, nullptr));

    char path[MaxPathLength];
    nn::util::SNPrintf(path, sizeof(path), "%s:/save/%016llx", nn::fs::GetBisMountName(partitonId), saveDataInfo.saveDataId);
    nn::fs::DeleteFile(path);

    nn::fs::Unmount(nn::fs::GetBisMountName(partitonId));
}

void CreateNoIndexerSystemSaveData()
{
    // テスト用システムセーブデータを作成
    nn::fs::CreateSystemSaveData(IndexerOnlySystemSaveDataId, SaveDataSize, SaveDataJournalSize, 0);

    nn::fs::SaveDataInfo saveDataInfo;
    GetSystemSaveDataInfo(&saveDataInfo, IndexerOnlySystemSaveDataId);

    // テスト用システムセーブデータの実ファイルをリネームし、NoIndexerSystemSaveData の実ファイルにする
    CleanSaveDataCache();
    nn::fs::BisPartitionId partitonId = nn::fs::BisPartitionId::System;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(partitonId, nullptr));

    char srcPath[MaxPathLength];
    char dstPath[MaxPathLength];
    nn::util::SNPrintf(srcPath, sizeof(srcPath), "%s:/save/%016llx", nn::fs::GetBisMountName(partitonId), saveDataInfo.saveDataId);
    nn::util::SNPrintf(dstPath, sizeof(dstPath), "%s:/save/%016llx", nn::fs::GetBisMountName(partitonId), NoIndexerSystemSaveDataId);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::RenameFile(srcPath, dstPath));

    nn::fs::Unmount(nn::fs::GetBisMountName(partitonId));

    nn::fs::DeleteSaveData(saveDataInfo.saveDataId);
}

TEST(SaveData, IndexerOnlyCreate)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::fs::SaveDataId indexerOnlySaveDataId;
    CreateIndexerOnlyUserSaveData(&indexerOnlySaveDataId);

    // Indexerのみのテスト用セーブデータを再作成した場合、副作用なくエラーになる
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::CreateSaveData(nnt::fs::util::ApplicationId, IndexerOnlySaveUserId, 0, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::CreateSaveData(nnt::fs::util::ApplicationId, IndexerOnlySaveUserId, 0, SaveDataSize, SaveDataJournalSize, 0));

    nn::fs::DeleteSaveData(indexerOnlySaveDataId);
}

TEST(SaveData, NoIndexerCreate)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::fs::SaveDataId noIndexerSaveDataId;
    CreateNoIndexerUserSaveData(&noIndexerSaveDataId);

    // Indexerがないテスト用セーブデータを作成した場合、内部で実ファイルを削除し再作成する
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::ApplicationId, NoIndexerSaveUserId, 0, SaveDataSize, SaveDataJournalSize, 0));

    nn::fs::SaveDataInfo saveDataInfo;
    GetUserSaveDataInfo(&saveDataInfo, nn::fs::SaveDataType::Account, NoIndexerSaveUserId);

    EXPECT_EQ(noIndexerSaveDataId, saveDataInfo.saveDataId);
    nn::fs::DeleteSaveData(saveDataInfo.saveDataId);
}

TEST(SaveData, IndexerOnlyMount)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::fs::SaveDataId indexerOnlySaveDataId;
    CreateIndexerOnlyUserSaveData(&indexerOnlySaveDataId);

    // Indexerのみのテスト用セーブデータをマウントした場合、ロールバック処理で Indexerの登録を削除しつつ、エラーになる
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::MountSaveData("save", nnt::fs::util::ApplicationId, IndexerOnlySaveUserId));

    // 再度、テスト用セーブデータを作成した場合、Indexerの登録は削除されているので成功する
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::ApplicationId, IndexerOnlySaveUserId, 0, SaveDataSize, SaveDataJournalSize, 0));

    // 再作成後は問題なくマウントもできる
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData("save", nnt::fs::util::ApplicationId, IndexerOnlySaveUserId));
    nn::fs::Unmount("save");

    nn::fs::DeleteSaveData(indexerOnlySaveDataId);
}

TEST(SaveData, NoIndexerMount)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::fs::SaveDataId noIndexerSaveDataId;
    CreateNoIndexerUserSaveData(&noIndexerSaveDataId);

    // Indexerがないテスト用セーブデータはマウントでエラーになる
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::MountSaveData("save", nnt::fs::util::ApplicationId, NoIndexerSaveUserId));

    // クリーンナップ処理
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::ApplicationId, NoIndexerSaveUserId, 0, SaveDataSize, SaveDataJournalSize, 0));
    nn::fs::DeleteSaveData(noIndexerSaveDataId);
}

TEST(SaveData, IndexerOnlyDelete)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::fs::SaveDataId indexerOnlySaveDataId;
    CreateIndexerOnlyUserSaveData(&indexerOnlySaveDataId);

    // Indexerのみのテスト用セーブデータの削除はエラーにならない
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(indexerOnlySaveDataId));
}

TEST(SaveData, NoIndexerDelete)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::fs::SaveDataId noIndexerSaveDataId;
    CreateNoIndexerUserSaveData(&noIndexerSaveDataId);

    // Indexerのないテスト用セーブデータの削除はエラーになる
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::DeleteSaveData(noIndexerSaveDataId));

    // クリーンナップ処理
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::ApplicationId, NoIndexerSaveUserId, 0, SaveDataSize, SaveDataJournalSize, 0));
    nn::fs::DeleteSaveData(noIndexerSaveDataId);
}

TEST(SystemSaveData, IndexerOnlyCreate)
{
    nnt::fs::util::DeleteAllTestSaveData();
    CreateIndexerOnlySystemSaveData();

    // Indexerのみのテスト用システムセーブデータを再作成した場合、副作用なくエラーになる
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::CreateSystemSaveData(IndexerOnlySystemSaveDataId, SaveDataSize, SaveDataJournalSize, 0));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::CreateSystemSaveData(IndexerOnlySystemSaveDataId, SaveDataSize, SaveDataJournalSize, 0));

    nn::fs::DeleteSaveData(IndexerOnlySystemSaveDataId);
}

TEST(SystemSaveData, NoIndexerCreate)
{
    nnt::fs::util::DeleteAllTestSaveData();
    CreateNoIndexerSystemSaveData();

    // Indexerがないテスト用セーブデータを作成した場合、内部で実ファイルを削除し再作成する
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(NoIndexerSystemSaveDataId, SaveDataSize, SaveDataJournalSize, 0));

    nn::fs::SaveDataInfo saveDataInfo;
    GetSystemSaveDataInfo(&saveDataInfo, NoIndexerSystemSaveDataId);

    EXPECT_EQ(NoIndexerSystemSaveDataId, saveDataInfo.saveDataId);
    nn::fs::DeleteSaveData(saveDataInfo.saveDataId);
}

TEST(SystemSaveData, IndexerOnlyMount)
{
    nnt::fs::util::DeleteAllTestSaveData();
    CreateIndexerOnlySystemSaveData();

    // Indexerのみのテスト用システムセーブデータをマウントした場合、ロールバック処理で Indexerの登録を削除しつつ、エラーになる
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::MountSystemSaveData("save", IndexerOnlySystemSaveDataId));

    // 再度、テスト用システムセーブデータを作成した場合、Indexerの登録は削除されているので成功する
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(IndexerOnlySystemSaveDataId, SaveDataSize, SaveDataJournalSize, 0));

    // 再作成後は問題なくマウントもできる
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSystemSaveData("save", IndexerOnlySystemSaveDataId));
    nn::fs::Unmount("save");

    nn::fs::DeleteSaveData(IndexerOnlySystemSaveDataId);
}

TEST(SystemSaveData, NoIndexerMount)
{
    nnt::fs::util::DeleteAllTestSaveData();
    CreateNoIndexerSystemSaveData();

    // システムセーブデータはIndexerがなくてもマウントできる
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSystemSaveData("save", NoIndexerSystemSaveDataId));
    nn::fs::Unmount("save");

    // クリーンナップ処理
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(NoIndexerSystemSaveDataId, SaveDataSize, SaveDataJournalSize, 0));
    nn::fs::DeleteSaveData(NoIndexerSystemSaveDataId);
}

TEST(SystemSaveData, IndexerOnlyDelete)
{
    nnt::fs::util::DeleteAllTestSaveData();
    CreateIndexerOnlySystemSaveData();

    // Indexerのみのテスト用システムセーブデータの削除はエラーにならない
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(IndexerOnlySystemSaveDataId));
}

TEST(SystemSaveData, NoIndexerDelete)
{
    nnt::fs::util::DeleteAllTestSaveData();
    CreateNoIndexerSystemSaveData();

    // Indexerのないテスト用システムセーブデータの削除はエラーになる
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::DeleteSaveData(NoIndexerSystemSaveDataId));

    // クリーンナップ処理
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(NoIndexerSystemSaveDataId, SaveDataSize, SaveDataJournalSize, 0));
    nn::fs::DeleteSaveData(NoIndexerSystemSaveDataId);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    nn::fs::SetEnabledAutoAbort(false);
    auto result = RUN_ALL_TESTS();
    nn::fs::SetEnabledAutoAbort(true);

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
