﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>

#include <nn/time/time_PosixTime.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util_function.h>

namespace
{
    nn::fs::SaveDataSpaceId ValidSpaceId = nn::fs::SaveDataSpaceId::System;
    nn::fs::SaveDataSpaceId InvalidSpaceId = static_cast<nn::fs::SaveDataSpaceId>(0xFE);
    nn::fs::UserId TestUserId = nn::fs::InvalidUserId;
    nn::fs::SaveDataId TestSaveDataId = 0x8000000000000005ULL;
}

TEST(InvalidSaveDataSpaceId, CreateSystemSaveData)
{
    nnt::fs::util::DeleteAllTestSaveData();

    // 成功パターン
    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::CreateSystemSaveData(
            ValidSpaceId,
            TestSaveDataId,
            TestUserId,
            0, 0x8000, 0x8000, 0
        )
    );

    nnt::fs::util::DeleteAllTestSaveData();

    // SpaceId のみを誤ったものに変更
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidArgument,
        nn::fs::CreateSystemSaveData(
            InvalidSpaceId,
            TestSaveDataId,
            TestUserId,
            0, 0x8000, 0x8000, 0
        )
    );
}

TEST(InvalidSaveDataSpaceId, MountSystemSaveData)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fs::CreateSystemSaveData(
            ValidSpaceId,
            TestSaveDataId,
            TestUserId,
            0, 0x8000, 0x8000, 0
        )
    );
    NN_UTIL_SCOPE_EXIT
    {
        nnt::fs::util::DeleteAllTestSaveData();
    };

    // 成功パターン
    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::MountSystemSaveData(
            "save",
            ValidSpaceId,
            TestSaveDataId,
            TestUserId
        )
    );
    nn::fs::Unmount("save");

    // SpaceId のみを誤ったものに変更
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidArgument,
        nn::fs::MountSystemSaveData(
            "save",
            InvalidSpaceId,
            TestSaveDataId,
            TestUserId
        )
    );
}

TEST(InvalidSaveDataSpaceId, DeleteSaveData)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NN_UTIL_SCOPE_EXIT
    {
        nnt::fs::util::DeleteAllTestSaveData();
    };

    // 成功パターン
    {
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::CreateSystemSaveData(
                ValidSpaceId,
                TestSaveDataId,
                TestUserId,
                0, 0x8000, 0x8000, 0
            )
        );
        NNT_EXPECT_RESULT_SUCCESS(
            nn::fs::DeleteSaveData(ValidSpaceId, TestSaveDataId)
        );
    }

    // SpaceId のみを誤ったものに変更
    {
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::CreateSystemSaveData(
                ValidSpaceId,
                TestSaveDataId,
                TestUserId,
                0, 0x8000, 0x8000, 0
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidArgument,
            nn::fs::DeleteSaveData(InvalidSpaceId, TestSaveDataId)
        );
    }
}

TEST(InvalidSaveDataSpaceId, SetSaveDataFlags)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fs::CreateSystemSaveData(
            ValidSpaceId,
            TestSaveDataId,
            TestUserId,
            0, 0x8000, 0x8000, 0
        )
    );
    NN_UTIL_SCOPE_EXIT
    {
        nnt::fs::util::DeleteAllTestSaveData();
    };

    uint32_t flag = 0x12345678;

    // 成功パターン
    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::SetSaveDataFlags(TestSaveDataId, ValidSpaceId, flag)
    );

    // SpaceId のみを誤ったものに変更
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidArgument,
        nn::fs::SetSaveDataFlags(TestSaveDataId, InvalidSpaceId, flag)
    );
}

TEST(InvalidSaveDataSpaceId, OpenSaveDataIterator)
{
    nnt::fs::util::DeleteAllTestSaveData();

    // 成功パターン
    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::fs::OpenSaveDataIterator(&iter, ValidSpaceId)
        );
    }

    // SpaceId のみを誤ったものに変更
    {
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidArgument,
            nn::fs::OpenSaveDataIterator(&iter, InvalidSpaceId)
        );
    }
}

TEST(InvalidSaveDataSpaceId, GetSaveDataOwnerId)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fs::CreateSystemSaveData(
            ValidSpaceId,
            TestSaveDataId,
            TestUserId,
            0, 0x8000, 0x8000, 0
        )
    );
    NN_UTIL_SCOPE_EXIT
    {
        nnt::fs::util::DeleteAllTestSaveData();
    };

    nn::Bit64 value = 0;

    // 成功パターン
    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::GetSaveDataOwnerId(&value, ValidSpaceId, TestSaveDataId)
    );

    // SpaceId のみを誤ったものに変更
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidArgument,
        nn::fs::GetSaveDataOwnerId(&value, InvalidSpaceId, TestSaveDataId)
    );
}

TEST(InvalidSaveDataSpaceId, GetSaveDataFlags)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fs::CreateSystemSaveData(
            ValidSpaceId,
            TestSaveDataId,
            TestUserId,
            0, 0x8000, 0x8000, 0
        )
    );
    NN_UTIL_SCOPE_EXIT
    {
        nnt::fs::util::DeleteAllTestSaveData();
    };
    uint32_t value = 0;

    // 成功パターン
    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::GetSaveDataFlags(&value, ValidSpaceId, TestSaveDataId)
    );

    // SpaceId のみを誤ったものに変更
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidArgument,
        nn::fs::GetSaveDataFlags(&value, InvalidSpaceId, TestSaveDataId)
    );
}

TEST(InvalidSaveDataSpaceId, GetSaveDataTimeStamp)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fs::CreateSystemSaveData(
            ValidSpaceId,
            TestSaveDataId,
            TestUserId,
            0, 0x8000, 0x8000, 0
        )
    );
    NN_UTIL_SCOPE_EXIT
    {
        nnt::fs::util::DeleteAllTestSaveData();
    };
    nn::time::PosixTime time;

    // 成功パターン
    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::GetSaveDataTimeStamp(&time, ValidSpaceId, TestSaveDataId)
    );

    // SpaceId のみを誤ったものに変更
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidArgument,
        nn::fs::GetSaveDataTimeStamp(&time, InvalidSpaceId, TestSaveDataId)
    );
}

TEST(InvalidSaveDataSpaceId, GetSaveDataAvailableSize)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fs::CreateSystemSaveData(
            ValidSpaceId,
            TestSaveDataId,
            TestUserId,
            0, 0x8000, 0x8000, 0
        )
    );
    NN_UTIL_SCOPE_EXIT
    {
        nnt::fs::util::DeleteAllTestSaveData();
    };
    int64_t value = 0;

    // 成功パターン
    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::GetSaveDataAvailableSize(
            &value,
            ValidSpaceId,
            TestSaveDataId
        )
    );

    // SpaceId のみを誤ったものに変更
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidArgument,
        nn::fs::GetSaveDataAvailableSize(
            &value,
            InvalidSpaceId,
            TestSaveDataId
        )
    );
}

TEST(InvalidSaveDataSpaceId, GetSaveDataJournalSize)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fs::CreateSystemSaveData(
            ValidSpaceId,
            TestSaveDataId,
            TestUserId,
            0, 0x8000, 0x8000, 0
        )
    );
    NN_UTIL_SCOPE_EXIT
    {
        nnt::fs::util::DeleteAllTestSaveData();
    };
    int64_t value = 0;

    // 成功パターン
    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::GetSaveDataJournalSize(&value, ValidSpaceId, TestSaveDataId)
    );

    // SpaceId のみを誤ったものに変更
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidArgument,
        nn::fs::GetSaveDataJournalSize(&value, InvalidSpaceId, TestSaveDataId)
    );
}

TEST(InvalidSaveDataSpaceId, ExtendSaveDataFileSystem)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_SUCCESS(
        nn::fs::CreateSystemSaveData(
            ValidSpaceId,
            TestSaveDataId,
            TestUserId,
            0, 0x8000, 0x8000, 0
        )
    );
    NN_UTIL_SCOPE_EXIT
    {
        nnt::fs::util::DeleteAllTestSaveData();
    };

    // 成功パターン
    NNT_EXPECT_RESULT_SUCCESS(
        nn::fs::ExtendSaveData(ValidSpaceId, TestSaveDataId, 0xC000, 0xC000)
    );

    // SpaceId のみを誤ったものに変更
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultInvalidArgument,
        nn::fs::ExtendSaveData(InvalidSpaceId, TestSaveDataId, 0x10000, 0x10000)
    );
}

TEST(CheckSaveDataSpaceId, User)
{
    nnt::fs::util::DeleteAllTestSaveData();

    const char* MountName = "save";
    const nn::fs::SaveDataId Id = 0x8000000000004000;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(static_cast<nn::fs::SystemSaveDataId>(Id), 0, 1024 * 1024, 1024 * 1024, 0));

    auto result = nn::fs::MountSystemSaveData(MountName, nn::fs::SaveDataSpaceId::User, Id);
    if (result.IsSuccess())
    {
        nn::fs::Unmount(MountName);
    }
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, result);

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::CreateSystemSaveData(static_cast<nn::fs::SystemSaveDataId>(Id), 0, 1024 * 1024, 1024 * 1024, 0));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::CreateSystemSaveData(static_cast<nn::fs::SystemSaveDataId>(Id), 0, 1024 * 1024, 1024 * 1024, 0));

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSystemSaveData(MountName, nn::fs::SaveDataSpaceId::System, Id));
    nn::fs::Unmount(MountName);

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::DeleteSaveData(nn::fs::SaveDataSpaceId::SdSystem, Id));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::DeleteSaveData(nn::fs::SaveDataSpaceId::User, Id));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteSaveData(Id));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::DeleteSaveData(Id));
}

TEST(CheckSaveDataSpaceId, System)
{
    nnt::fs::util::DeleteAllTestSaveData();

    const char* MountName = "save";
    nn::fs::UserId userId = { { 0, 1 } };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::util::ApplicationId, userId, 0, 1024 * 1024, 1024 * 1024, 0));

    nn::util::optional<nn::fs::SaveDataInfo> info;
    nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::User, [&userId](const nn::fs::SaveDataInfo& info) NN_NOEXCEPT
    {
        return info.saveDataUserId == userId;
    });

    nn::fs::SaveDataId id = info->saveDataId;
    auto result = nn::fs::MountSystemSaveData(MountName, id, userId);
    if (result.IsSuccess())
    {
        nn::fs::Unmount(MountName);
    }
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidArgument, result);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(MountName, userId));
    nn::fs::Unmount(MountName);

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::DeleteSystemSaveData(nn::fs::SaveDataSpaceId::System, id, userId));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::DeleteSaveData(nn::fs::SaveDataSpaceId::System, id));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteSaveData(nn::fs::SaveDataSpaceId::User, id));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::DeleteSaveData(id));
}
