﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataForDebug.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_CacheStorage.h>
#include <nn/fs/fs_CacheStorageWithIndex.h>
#include <nn/fs/fs_CacheStoragePrivate.h>
#include <nn/fs/fs_TemporaryStorage.h>
#include <nn/fs/fs_ApplicationSaveDataManagement.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataFileSystemDriver.h>
#include <nn/time/time_Types.h>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

namespace {

    struct SaveDataInfoAll
    {
        SaveDataSpaceId spaceId;
        SaveDataId id;
        Bit64 ownerId;
        uint32_t flags;
        int64_t availableSize;
        int64_t journalSize;
        nn::ncm::ApplicationId applicationId;
        UserId userId;
    };

    void CreateTestSystemSaveData(SaveDataSpaceId spaceId, const SaveDataInfoAll& infoAll)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(spaceId, infoAll.id, infoAll.ownerId, infoAll.availableSize, infoAll.journalSize, infoAll.flags));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::CreateSystemSaveData(spaceId, infoAll.id, infoAll.ownerId, infoAll.availableSize, infoAll.journalSize, infoAll.flags));
    }

    void VerifySystemSaveData(SaveDataSpaceId spaceId, const SaveDataInfoAll& infoAll)
    {
        nn::util::optional<SaveDataInfo> info;
        FindSaveData(&info, spaceId, [&](const SaveDataInfo& i) NN_NOEXCEPT
        {
            return i.systemSaveDataId == infoAll.id;
        });
        EXPECT_NE(util::nullopt, info);

        const auto Id = infoAll.id;
        NNT_EXPECT_RESULT_SUCCESS(MountSystemSaveData("save", spaceId, Id));
        NNT_EXPECT_RESULT_SUCCESS(CommitSaveData("save"));
        Unmount("save");

        // indexer に永続化される情報は ProperSystem でなく System
        EXPECT_EQ(infoAll.spaceId, info->saveDataSpaceId);

        IsFilledWithValue(info->padding1, sizeof(info->padding1), 0);
        IsFilledWithValue(info->reserved, sizeof(info->reserved), 0);

        ListAllSaveData();

#if !defined(NN_BUILD_CONFIG_OS_WIN) // win 版 未対応
        // 別枠 Get 系
        Bit64 actualOwnerId;
        NNT_EXPECT_RESULT_SUCCESS(GetSaveDataOwnerId(&actualOwnerId, spaceId, Id));
        EXPECT_EQ(infoAll.ownerId, actualOwnerId);

        uint32_t actualFlags;
        NNT_EXPECT_RESULT_SUCCESS(GetSaveDataFlags(&actualFlags, spaceId, Id));
        EXPECT_EQ(infoAll.flags, actualFlags);

        // ついでに SetFlag のテスト
        {
            uint32_t Flags = 0xAAAAAAAA;
            uint32_t actualFlags2;
            NNT_EXPECT_RESULT_SUCCESS(SetSaveDataFlags(Id, spaceId, Flags));
            NNT_EXPECT_RESULT_SUCCESS(GetSaveDataFlags(&actualFlags2, spaceId, Id));
            EXPECT_EQ(Flags, actualFlags2);

            NNT_EXPECT_RESULT_SUCCESS(SetSaveDataFlags(Id, spaceId, actualFlags)); // 戻す
        }

        nn::time::PosixTime actualTimeStamp;
        NNT_EXPECT_RESULT_SUCCESS(GetSaveDataTimeStamp(&actualTimeStamp, spaceId, Id));
        EXPECT_NE(0, actualTimeStamp.value);

        int64_t actualAvailableSize;
        NNT_EXPECT_RESULT_SUCCESS(GetSaveDataAvailableSize(&actualAvailableSize, spaceId, Id));
        EXPECT_EQ(infoAll.availableSize, actualAvailableSize);

        int64_t actualJournalSize;
        NNT_EXPECT_RESULT_SUCCESS(GetSaveDataJournalSize(&actualJournalSize, spaceId, Id));
        EXPECT_EQ(infoAll.journalSize, actualJournalSize);
#endif

    }

    SaveDataInfoAll InfoAllA =
    {
        SaveDataSpaceId::System,
        0x8000000000004000,
        1,
        2,
        2 * 1024 * 1024,
        1 * 1024 * 1024,
        { 0 },
        { { 0 } },
    };

    SaveDataInfoAll InfoAllB =
    {
        SaveDataSpaceId::System,
        0x8000000000004001,
        3,
        4,
        4 * 1024 * 1024,
        2 * 1024 * 1024,
        { 0 },
        { { 0 } },
    };
TEST(SystemSaveData, ListAll)
{
    ListAllSaveData();
}

TEST(DISABLED_SystemSaveData, CreateAtNormal)
{
    ListAllSaveData();
    nnt::fs::util::DeleteAllTestSaveData();

    CreateTestSystemSaveData(SaveDataSpaceId::System, InfoAllA);
    VerifySystemSaveData(SaveDataSpaceId::System, InfoAllA);

    ListAllSaveData();
}

//! @pre SystemSaveData.CreateAtNormal 実行済み＠通常起動
TEST(DISABLED_SystemSaveData, VerifyAtSafe)
{
    ListAllSaveData();
    VerifySystemSaveData(SaveDataSpaceId::ProperSystem, InfoAllA);
}


TEST(DISABLED_SystemSaveData, CreateAtSafe)
{
    ListAllSaveData();
    nnt::fs::util::DeleteAllTestSaveData();

    CreateTestSystemSaveData(SaveDataSpaceId::ProperSystem, InfoAllB);
    VerifySystemSaveData(SaveDataSpaceId::ProperSystem, InfoAllB);

    ListAllSaveData();
}

//! @pre SystemSaveData.CreateAtSafe 実行済み＠セーフモード起動
TEST(DISABLED_SystemSaveData, VerifyAtNormal)
{
    ListAllSaveData();
    VerifySystemSaveData(SaveDataSpaceId::System, InfoAllB);
}

SaveDataInfoAll InfoAllSd =
{
    SaveDataSpaceId::SdSystem,
    0x8000000000004003,
    8,
    9,
    3 * 1024 * 1024,
    6 * 1024 * 1024,
    { 0 },
    { { 0 } },
};

TEST(SystemSaveData, CreateAndVerifySdSystem)
{
    nnt::fs::util::DeleteAllTestSaveData();

    CreateTestSystemSaveData(SaveDataSpaceId::SdSystem, InfoAllSd);
    VerifySystemSaveData(SaveDataSpaceId::SdSystem, InfoAllSd);
}


SaveDataInfoAll InfoAllC =
{
    SaveDataSpaceId::User,
    0,
    5,
    6,
    5 * 1024 * 1024,
    4 * 1024 * 1024,
    TestApplicationId0,
    { { 1 } },
};

SaveDataInfoAll InfoAllD =
{
    SaveDataSpaceId::User,
    0,
    7,
    8,
    7 * 1024 * 1024,
    6 * 1024 * 1024,
    TestApplicationId1,
    { { 2 } },
};


void CreateAppliSaveData(const SaveDataInfoAll& infoAll)
{
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(infoAll.applicationId, infoAll.userId, infoAll.ownerId, infoAll.availableSize, infoAll.journalSize, infoAll.flags));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathAlreadyExists, nn::fs::CreateSaveData(infoAll.applicationId, infoAll.userId, infoAll.ownerId, infoAll.availableSize, infoAll.journalSize, infoAll.flags));
}

void VerifyAppliSaveData(SaveDataSpaceId spaceId, const SaveDataInfoAll& infoAll)
{
    nn::util::optional<SaveDataInfo> info;
    FindSaveData(&info, spaceId, [&](const SaveDataInfo& i) NN_NOEXCEPT
    {
        return i.applicationId.value == infoAll.applicationId.value;
    });
    EXPECT_NE(util::nullopt, info);

    const auto Id = info->saveDataId;
    NNT_EXPECT_RESULT_SUCCESS(MountSaveData("save", infoAll.applicationId, infoAll.userId));
    NNT_EXPECT_RESULT_SUCCESS(CommitSaveData("save"));
    Unmount("save");


    NN_LOG("%s %d: %d\n", __FUNCTION__, __LINE__, info->saveDataSpaceId);
    DumpBuffer(&(info.value()), sizeof(SaveDataInfo));

    EXPECT_EQ(infoAll.spaceId, info->saveDataSpaceId);

    IsFilledWithValue(info->padding1, sizeof(info->padding1), 0);
    IsFilledWithValue(info->reserved, sizeof(info->reserved), 0);

    ListAllSaveData();

    // 別枠 Get 系
    Bit64 actualOwnerId;
    NNT_EXPECT_RESULT_SUCCESS(GetSaveDataOwnerId(&actualOwnerId, Id));
    EXPECT_EQ(infoAll.ownerId, actualOwnerId);

    uint32_t actualFlags;
    NNT_EXPECT_RESULT_SUCCESS(GetSaveDataFlags(&actualFlags, Id));
    EXPECT_EQ(infoAll.flags, actualFlags);

    nn::time::PosixTime actualTimeStamp;
    NNT_EXPECT_RESULT_SUCCESS(GetSaveDataTimeStamp(&actualTimeStamp, Id));
    EXPECT_NE(0, actualTimeStamp.value);

    int64_t actualAvailableSize;
    NNT_EXPECT_RESULT_SUCCESS(GetSaveDataAvailableSize(&actualAvailableSize, Id));
    EXPECT_EQ(infoAll.availableSize, actualAvailableSize);

    int64_t actualJournalSize;
    NNT_EXPECT_RESULT_SUCCESS(GetSaveDataJournalSize(&actualJournalSize, Id));
    EXPECT_EQ(infoAll.journalSize, actualJournalSize);
}

const size_t MetaDataSize = 256;

void WriteMetaData(const SaveDataInfoAll& infoAll, char fillData)
{
    char buffer[MetaDataSize];
    memset(buffer, fillData, MetaDataSize);
    NNT_EXPECT_RESULT_SUCCESS(WriteSaveDataThumbnailFile(infoAll.applicationId.value, infoAll.userId, buffer, MetaDataSize, nullptr, 0));
}

void VerifyMetaData(const SaveDataInfoAll& infoAll, char fillData)
{
    char expectedBuffer[MetaDataSize];
    char actualBuffer[MetaDataSize];
    memset(expectedBuffer, fillData, MetaDataSize);
    NNT_EXPECT_RESULT_SUCCESS(ReadSaveDataThumbnailFile(infoAll.applicationId.value, infoAll.userId, actualBuffer, MetaDataSize, nullptr, 0));

    NNT_FS_UTIL_EXPECT_MEMCMPEQ(expectedBuffer, actualBuffer, MetaDataSize);
}


TEST(DISABLED_SaveData, CreateAtNormal)
{
    ListAllSaveData();
    nnt::fs::util::DeleteAllTestSaveData();

    CreateAppliSaveData(InfoAllC);
    VerifyAppliSaveData(SaveDataSpaceId::User, InfoAllC);

    WriteMetaData(InfoAllC, 0xC);
    VerifyMetaData(InfoAllC, 0xC);

    ListAllSaveData();
}

//! @pre SaveData.CreateAtNormal 実行済み＠通常起動
TEST(DISABLED_SaveData, VerifyAtSafe)
{
    ListAllSaveData();
    VerifyAppliSaveData(SaveDataSpaceId::User, InfoAllC);
    VerifyMetaData(InfoAllC, 0xC);
}


TEST(DISABLED_SaveData, CreateAtSafe)
{
    ListAllSaveData();
    nnt::fs::util::DeleteAllTestSaveData();

    CreateAppliSaveData(InfoAllD);
    VerifyAppliSaveData(SaveDataSpaceId::User, InfoAllD);

    WriteMetaData(InfoAllD, 0xD);
    VerifyMetaData(InfoAllD, 0xD);

    ListAllSaveData();
}

//! @pre SaveData.CreateAtSafe 実行済み＠セーフモード起動
TEST(DISABLED_SaveData, VerifyAtNormal)
{
    ListAllSaveData();
    VerifyAppliSaveData(SaveDataSpaceId::User, InfoAllD);
    VerifyMetaData(InfoAllD, 0xD);
}



TEST(SystemSaveData, SdSaveTest)
{
    nnt::fs::util::DeleteAllTestSaveData();

    const SaveDataId Id = 0x8000000000000003;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(nn::fs::SaveDataSpaceId::SdSystem, Id, 0, 4 * 1024 * 1024, 4 * 1024 * 1024, 0));
    nn::util::optional<SaveDataInfo> info;
    FindSaveData(&info, SaveDataSpaceId::SdSystem, [&](const SaveDataInfo& i) NN_NOEXCEPT
    {
        return i.systemSaveDataId == Id;
    });
    EXPECT_NE(util::nullopt, info);
    NNT_ASSERT_RESULT_SUCCESS(MountSystemSaveData("save", nn::fs::SaveDataSpaceId::SdSystem, Id));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile("save:/file", 1024));

    nnt::fs::util::DumpDirectoryRecursive("save:/");
    Unmount("save");

    nnt::fs::util::DeleteAllTestSaveData();
}

TEST(SystemSaveData, SafeSaveTest)
{
    const SaveDataId Id = 0x8000000000000003;
    (void)DeleteSaveData(nn::fs::SaveDataSpaceId::SafeMode, Id);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(nn::fs::SaveDataSpaceId::SafeMode, Id, 0, 4 * 1024 * 1024, 4 * 1024 * 1024, 0));
    nn::util::optional<SaveDataInfo> info;
    FindSaveData(&info, SaveDataSpaceId::SafeMode, [&](const SaveDataInfo& i) NN_NOEXCEPT
    {
        return i.systemSaveDataId == Id;
    });
    EXPECT_NE(util::nullopt, info);
    NNT_ASSERT_RESULT_SUCCESS(MountSystemSaveData("save", nn::fs::SaveDataSpaceId::SafeMode, Id));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile("save:/file", 1024));

    nnt::fs::util::DumpDirectoryRecursive("save:/");
    Unmount("save");

    // 手動確認用
    {
        NNT_ASSERT_RESULT_SUCCESS(MountBis("safe", BisPartitionId::SafeMode));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::ListDirectoryRecursive("safe:/"));
        Unmount("safe");
    }
    ListAllSaveData();
    ListAllSaveData(nn::fs::SaveDataSpaceId::SafeMode);

    NNT_EXPECT_RESULT_SUCCESS(DeleteSaveData(nn::fs::SaveDataSpaceId::SafeMode, Id));
}

// TODO: FsApi テストへ移動
TEST(SystemSaveData, OpenCountLimit)
{
    nnt::fs::util::DeleteAllTestSaveData();

    const SaveDataId Id = 0x8000000000000003;
    NNT_EXPECT_RESULT_SUCCESS(CreateSystemSaveData(Id, 4 * 1024 * 1024, 4 * 1024 * 1024, 0));

    const int OpenCountMax = 256;
    FileHandle handles[OpenCountMax];
    NNT_ASSERT_RESULT_SUCCESS(MountSystemSaveData("save", Id));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile("save:/file", 1024));

    // 256 個まで開けること
    for (int i = 0; i < OpenCountMax; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(OpenFile(&handles[i], "save:/file", OpenMode_Read));
    }
    // それ以上開けないこと
    {
        FileHandle handle;
        NNT_EXPECT_RESULT_FAILURE(ResultOpenCountLimit, OpenFile(&handle, "save:/file", OpenMode_Read));
    }
    // ディレクトリも同じ枠であること
    {
        DirectoryHandle handle;
        NNT_EXPECT_RESULT_FAILURE(ResultOpenCountLimit, OpenDirectory(&handle, "save:/", OpenDirectoryMode_All));
    }
    // クローズによって枠が解放されること
    for (int i = 0; i < OpenCountMax; i++)
    {
        CloseFile(handles[i]);
    }
    for (int i = 0; i < OpenCountMax; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(OpenFile(&handles[i], "save:/file", OpenMode_Read));
    }

    // tear down
    for (int i = 0; i < OpenCountMax; i++)
    {
        CloseFile(handles[i]);
    }
    Unmount("save");
    nnt::fs::util::DeleteAllTestSaveData();
}

TEST(SaveData, OpenCountLimit)
{
    // ユーザアカウントセーブ、本体セーブ、デバッグ用セーブはオープン数制限あり
    nnt::fs::util::DeleteAllTestSaveData();

    UserId userId[2] = { {{0, 1}}, {{0, 2}} };
    NNT_EXPECT_RESULT_SUCCESS(CreateAndMountSaveData("userSave0", userId[0]));
    NNT_EXPECT_RESULT_SUCCESS(CreateAndMountSaveData("userSave1", userId[1]));
    NNT_EXPECT_RESULT_SUCCESS(CreateAndMountDeviceSaveData("deviceSave"));
    NNT_EXPECT_RESULT_SUCCESS(MountSaveDataForDebug("debugSave"));

    NNT_ASSERT_RESULT_SUCCESS(CreateFile("userSave0:/file", 1024));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile("userSave1:/file", 1024));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile("deviceSave:/file", 1024));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile("debugSave:/file", 1024));

    const int OpenCountMax = 64;
    FileHandle handles[4][OpenCountMax];

    // あわせて 256 個まで開けること
    for (int i = 0; i < OpenCountMax; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(OpenFile(&handles[0][i], "userSave0:/file", OpenMode_Read));
        NNT_EXPECT_RESULT_SUCCESS(OpenFile(&handles[1][i], "userSave1:/file", OpenMode_Read));
        NNT_EXPECT_RESULT_SUCCESS(OpenFile(&handles[2][i], "deviceSave:/file", OpenMode_Read));
        NNT_EXPECT_RESULT_SUCCESS(OpenFile(&handles[3][i], "debugSave:/file", OpenMode_Read));
    }

    // それ以上開けないこと
    {
        FileHandle handle;
        NNT_EXPECT_RESULT_FAILURE(ResultOpenCountLimit, OpenFile(&handle, "userSave0:/file", OpenMode_Read));
    }

    // ディレクトリも同じ枠であること
    {
        DirectoryHandle handle;
        NNT_EXPECT_RESULT_FAILURE(ResultOpenCountLimit, OpenDirectory(&handle, "userSave0:/", OpenDirectoryMode_All));
    }

    // クローズによって枠が解放されること
    for (int i = 0; i < OpenCountMax; i++)
    {
        CloseFile(handles[0][i]);
    }
    for (int i = 0; i < OpenCountMax; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(OpenFile(&handles[0][i], "userSave0:/file", OpenMode_Read));
    }


    // tear down
    for (int i = 0; i < OpenCountMax; i++)
    {
        CloseFile(handles[0][i]);
        CloseFile(handles[1][i]);
        CloseFile(handles[2][i]);
        CloseFile(handles[3][i]);
    }
    Unmount("userSave0");
    Unmount("userSave1");
    Unmount("deviceSave");
    Unmount("debugSave");
    nnt::fs::util::DeleteAllTestSaveData();
}

TEST(SaveData, IsSaveDataExisting)
{
    nnt::fs::util::DeleteAllTestSaveData();
    UserId userId[2] = { { { 0, 1 } },{ { 0, 2 } } };

    EXPECT_FALSE(IsSaveDataExisting(userId[0]));

    NNT_EXPECT_RESULT_SUCCESS(CreateAndMountSaveData("save", userId[0]));
    NNT_EXPECT_RESULT_SUCCESS(CreateSaveData(nnt::fs::util::ApplicationId, userId[1], 0, 1024 * 1024, 1024 * 1024, 0));

    EXPECT_TRUE(IsSaveDataExisting(userId[0])); // 存在、マウント中
    EXPECT_TRUE(IsSaveDataExisting(userId[1])); // 存在、非マウント中

    Unmount("save");
    nnt::fs::util::DeleteAllTestSaveData();
}

#ifndef NN_BUILD_CONFIG_OS_WIN

TEST(SaveData, Extend)
{
    const auto TrialCount = 10;

    const auto BlockSize = 16 * 1024;
    const auto BlockCountMin = 4;
    const auto BlockCountMax = 64;

    nnt::fs::util::DeleteAllTestSaveData();

    std::mt19937 rng(nnt::fs::util::GetRandomSeed());
    std::uniform_int_distribution<> distribution(BlockCountMin, BlockCountMax);

    for( Bit64 count = 0; count < TrialCount; ++count )
    {
        const UserId userId = { { 0, count } };

        EXPECT_FALSE(IsSaveDataExisting(userId));

        const auto availableSize = distribution(rng) * BlockSize;
        const auto journalSize = distribution(rng) * BlockSize;
        const auto extendedAvailableSize = availableSize + distribution(rng) * BlockSize;
        const auto extendedJournalSize = journalSize + distribution(rng) * BlockSize;
        const auto fileSize = extendedAvailableSize - 2 * BlockSize;

        NNT_ASSERT_RESULT_SUCCESS(CreateSaveData(
            nnt::fs::util::ApplicationId,
            userId,
            0,
            availableSize,
            journalSize,
            0));

        NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", userId));
        NNT_EXPECT_RESULT_FAILURE(
            ResultUsableSpaceNotEnough,
            CreateFile("save:/file", fileSize));
        Unmount("save");

        {
            nn::util::optional<SaveDataInfo> info;
            FindSaveData(&info, SaveDataSpaceId::User, [&userId](const SaveDataInfo& info) NN_NOEXCEPT
            {
                return info.saveDataUserId == userId;
            });
            ASSERT_NE(nn::util::nullopt, info);

            NNT_ASSERT_RESULT_SUCCESS(ExtendSaveData(
                info->saveDataSpaceId,
                info->saveDataId,
                extendedAvailableSize,
                extendedJournalSize));
        }

        {
            nn::util::optional<SaveDataInfo> info;
            FindSaveData(&info, SaveDataSpaceId::User, [&userId](const SaveDataInfo& info) NN_NOEXCEPT
            {
                return info.saveDataUserId == userId;
            });
            ASSERT_NE(nn::util::nullopt, info);

            int64_t expectedSize = 0;
            const auto BlockSize = 16 * 1024;
            const auto parameters = fssystem::save::JournalIntegritySaveDataFileSystemDriver::SetUpSaveDataParameters(
                BlockSize,
                extendedAvailableSize,
                extendedJournalSize);
            fssystem::save::JournalIntegritySaveDataFileSystemDriver::QueryTotalSize(
                &expectedSize,
                parameters.blockSize,
                parameters.countExpandMax,
                parameters.paramDuplex,
                parameters.paramIntegrity,
                parameters.countDataBlock,
                parameters.countJournalBlock);
            EXPECT_EQ(expectedSize, info->saveDataSize);

            int64_t actualAvailableSize = 0;
            int64_t actualJournalSize = 0;
            GetSaveDataAvailableSize(&actualAvailableSize, info->saveDataId);
            GetSaveDataJournalSize(&actualJournalSize, info->saveDataId);
            EXPECT_EQ(extendedAvailableSize, actualAvailableSize);
            EXPECT_EQ(extendedJournalSize, actualJournalSize);
        }

        NNT_ASSERT_RESULT_SUCCESS(MountSaveData("save", userId));
        NNT_EXPECT_RESULT_SUCCESS(CreateFile("save:/file", fileSize));
        Unmount("save");
    }

    nnt::fs::util::DeleteAllTestSaveData();
}

TEST(SaveData, ExtendFailure)
{
    nnt::fs::util::DeleteAllTestSaveData();

    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, ExtendSaveData(
        SaveDataSpaceId::User,
        std::numeric_limits<int64_t>::max(),
        1024 * 1024,
        1024 * 1024));
}

#endif

TEST(SaveData, Remount)
{
    nnt::fs::util::DeleteAllTestSaveData();

    UserId userId = { { 0, 1 } };

    const size_t FileSize = 1024;
    const char* MountName = "save";
    const char* FileName1 = "save:/file";
    const char* FileName2 = "save:/directory/file";

    NNT_ASSERT_RESULT_SUCCESS(CreateAndMountSaveData(MountName, userId));
    NN_UTIL_SCOPE_EXIT
    {
        Unmount(MountName);
        nnt::fs::util::DeleteAllTestSaveData();
    };

    NNT_ASSERT_RESULT_SUCCESS(CreateFile(FileName1, FileSize));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory("save:/directory"));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile(FileName2, FileSize));

    std::unique_ptr<char[]> readBuffer1(new char[FileSize]);
    std::unique_ptr<char[]> readBuffer2(new char[FileSize]);
    std::unique_ptr<char[]> writeBuffer1(new char[FileSize]);
    std::unique_ptr<char[]> writeBuffer2(new char[FileSize]);
    nnt::fs::util::FillBufferWithRandomValue(writeBuffer1.get(), FileSize);
    nnt::fs::util::FillBufferWithRandomValue(writeBuffer2.get(), FileSize);

    // Write、Flush、Commit
    {
        nn::fs::FileHandle file1;
        nn::fs::FileHandle file2;
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&file1, FileName1, nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            CloseFile(file1);
        };
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&file2, FileName2, nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            CloseFile(file2);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            WriteFile(file1, 0, writeBuffer1.get(), FileSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush))
        );
        NNT_ASSERT_RESULT_SUCCESS(
            WriteFile(file2, 0, writeBuffer2.get(), FileSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush))
        );
    }
    NNT_ASSERT_RESULT_SUCCESS(CommitSaveData(MountName));

    // ファイルの中身が書き換わっていないか確かめるラムダ式
    auto checkFileContent = [&]() NN_NOEXCEPT
    {
        nn::fs::FileHandle file1;
        nn::fs::FileHandle file2;
        memset(readBuffer1.get(), 0, FileSize);
        memset(readBuffer2.get(), 0, FileSize);
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&file1, FileName1, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            CloseFile(file1);
        };
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&file2, FileName2, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            CloseFile(file2);
        };

        NNT_ASSERT_RESULT_SUCCESS(
            ReadFile(file1, 0, readBuffer1.get(), FileSize)
        );

        NNT_ASSERT_RESULT_SUCCESS(
            ReadFile(file2, 0, readBuffer2.get(), FileSize)
        );
        ASSERT_EQ(0, memcmp(writeBuffer1.get(), readBuffer1.get(), FileSize));
        ASSERT_EQ(0, memcmp(writeBuffer2.get(), readBuffer2.get(), FileSize));
    };

    // Write と Flush を行い Commit せずに再マウントすると、ファイルの中身は元に戻る
    {
        nn::fs::FileHandle file;
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&file, FileName1, nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            CloseFile(file);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            WriteFile(file, 0, writeBuffer2.get(), FileSize, nn::fs::WriteOption::MakeValue(0))
        );

        NNT_ASSERT_RESULT_SUCCESS(
            FlushFile(file)
        );
    }
    Unmount(MountName);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(MountName, userId));
    checkFileContent();

    // Flush 込みで Write を行い Commit せずに再マウントすると、ファイルの中身は元に戻る
    {
        nn::fs::FileHandle file;
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&file, FileName2, nn::fs::OpenMode_Write));
        NN_UTIL_SCOPE_EXIT
        {
            CloseFile(file);
        };
        NNT_ASSERT_RESULT_SUCCESS(
            WriteFile(file, 1, writeBuffer1.get(), FileSize - 1, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush))
        );
    }
    Unmount(MountName);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(MountName, userId));
    checkFileContent();
}

TEST(SaveDataDeathTest, Commit)
{
    nnt::fs::util::DeleteAllTestSaveData();

    UserId userId = { { 0, 1 } };

    NNT_EXPECT_RESULT_SUCCESS(CreateAndMountSaveData("save", userId));
    NN_UTIL_SCOPE_EXIT
    {
        Unmount("save");
        nnt::fs::util::DeleteAllTestSaveData();
    };

    NNT_ASSERT_RESULT_SUCCESS(CreateFile("save:/file", 1024));
    NNT_ASSERT_RESULT_SUCCESS(CreateDirectory("save:/directory"));
    NNT_ASSERT_RESULT_SUCCESS(CreateFile("save:/directory/subfile", 1024));

    // ディレクトリを開いていてもコミットできる
    {
        nn::fs::DirectoryHandle directory;
        NNT_ASSERT_RESULT_SUCCESS(OpenDirectory(
            &directory,
            "save:/directory",
            nn::fs::OpenDirectoryMode_All));
        NNT_EXPECT_RESULT_SUCCESS(CommitSaveData("save"));
        CloseDirectory(directory);
    }

    // OpenMode_Read でファイルを開いていてもコミットできる
    {
        nn::fs::FileHandle file;
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&file, "save:/file", nn::fs::OpenMode_Read));
        NNT_EXPECT_RESULT_SUCCESS(CommitSaveData("save"));
        CloseFile(file);
    }

    // OpenMode_Write でファイルを開いているとコミットできない
    {
        nn::fs::FileHandle file;
        const auto data = 42;
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&file, "save:/file", nn::fs::OpenMode_Write));
        NNT_EXPECT_RESULT_SUCCESS(WriteFile(
            file,
            0,
            &data,
            sizeof(data),
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        EXPECT_DEATH_IF_SUPPORTED(CommitSaveData("save"), "");
        CloseFile(file);
    }
}

#if !defined(NN_BUILD_CONFIG_OS_WIN)

TEST(SaveData, CorruptAndVerify)
{
    nnt::fs::util::DeleteAllTestSaveData();

    const SaveDataId Id = 0x8000000000000003;
    DeleteSaveData(Id);

    const int BufferSize = 1024 * 1024;
    auto buffer = AllocateBuffer(BufferSize);

    NNT_EXPECT_RESULT_SUCCESS(CreateSystemSaveData(Id, 4 * 1024 * 1024, 4 * 1024 * 1024, 0));

    {
        NNT_EXPECT_RESULT_SUCCESS(MountSystemSaveData("save", Id));

        NNT_EXPECT_RESULT_SUCCESS(CreateDirectory("save:/dir"));
        NNT_EXPECT_RESULT_SUCCESS(CreateDirectory("save:/dir/subdir"));
        NNT_EXPECT_RESULT_SUCCESS(CreateFileWith32BitCount("save:/dir/subdir/file", 2 * 1024 * 1024 + 1, 0));

        NNT_EXPECT_RESULT_SUCCESS(CommitSaveData("save"));
        Unmount("save");
    }

    bool isValid = true;
    NNT_EXPECT_RESULT_SUCCESS(VerifySaveData(&isValid, Id, buffer.get(), BufferSize));
    EXPECT_TRUE(isValid);
    EXPECT_TRUE(IsFilledWithValue(buffer.get(), BufferSize, 0));

    NNT_EXPECT_RESULT_SUCCESS(VerifySaveData(&isValid, Id, buffer.get(), 16 * 1024));
    EXPECT_TRUE(isValid);
    EXPECT_TRUE(IsFilledWithValue(buffer.get(), BufferSize, 0));

    // 破壊
    NNT_EXPECT_RESULT_SUCCESS(CorruptSaveDataForDebug(Id));
    NNT_EXPECT_RESULT_SUCCESS(VerifySaveData(&isValid, Id, buffer.get(), BufferSize));
    EXPECT_FALSE(isValid);
    NNT_EXPECT_RESULT_FAILURE(ResultDataCorrupted, MountSystemSaveData("save", Id));

    // （再作成で修復できること）
    NNT_EXPECT_RESULT_SUCCESS(DeleteSaveData(Id));
    NNT_EXPECT_RESULT_SUCCESS(CreateSystemSaveData(Id, 64 * 1024, 64 * 1024, 0));
    NNT_EXPECT_RESULT_SUCCESS(VerifySaveData(&isValid, Id, buffer.get(), BufferSize));
    EXPECT_TRUE(isValid);
    EXPECT_TRUE(IsFilledWithValue(buffer.get(), BufferSize, 0));

    NNT_EXPECT_RESULT_SUCCESS(DeleteSaveData(Id));
}


namespace {
    const SaveDataId IdBase = 0x8000000000004000;
    const int SaveDataCount = 512;
    const int DeleteSaveDataCount = SaveDataCount / 2;
    const UserId DummyUserId = { {0, 1} };
}

// RegisterSaveDataAtomicDeletion.DISABLED_VerifyDeletion とセットでのみ実行すべきため DISABLED 化
// 512 + 2 個のセーブを作成し偶数番 (+ 2) を一括遅延削除登録する
TEST(RegisterSaveDataAtomicDeletion, DISABLED_CreateAndRegisterDeletion)
{
    nnt::fs::util::DeleteAllTestSaveData();

    {
        for (int i = 0; i < SaveDataCount; i++)
        {
            NNT_FS_SCOPED_TRACE("i: %d\n", i);
            NNT_EXPECT_RESULT_SUCCESS(CreateSystemSaveData(IdBase + i, 1024 * 1024, 1024 * 1024, 0));
        }

        // ユーザアカウントごとシステムセーブデータ
        NNT_EXPECT_RESULT_SUCCESS(CreateSystemSaveData(IdBase + SaveDataCount, DummyUserId, 1024 * 1024, 1024 * 1024, 0));

        // ユーザアカウントセーブデータ
        NNT_EXPECT_RESULT_SUCCESS(CreateSaveData(TestApplicationId0, DummyUserId, 0, 1024 * 1024, 1024 * 1024, 0));
    }


    {
        SaveDataId deletionSaveDataArray[DeleteSaveDataCount + 2];
        for (int i = 0; i < SaveDataCount; i+=2)
        {
            NNT_FS_SCOPED_TRACE("i: %d\n", i);
            deletionSaveDataArray[i / 2] = IdBase + i;
        }

        nn::util::optional<SaveDataInfo> info;

        FindSaveData(&info, SaveDataSpaceId::System, [&](SaveDataInfo info_) {
            return (info_.systemSaveDataId == IdBase + SaveDataCount);
        });
        ASSERT_NE(util::nullopt, info);
        deletionSaveDataArray[DeleteSaveDataCount] = info->saveDataId;


        FindSaveData(&info, SaveDataSpaceId::User, [&](SaveDataInfo info_) {
            return (info_.applicationId == TestApplicationId0);
        });
        ASSERT_NE(util::nullopt, info);
        deletionSaveDataArray[DeleteSaveDataCount + 1] = info->saveDataId;


        NNT_EXPECT_RESULT_SUCCESS(RegisterSaveDataAtomicDeletion(deletionSaveDataArray, DeleteSaveDataCount + 2));
    }
}


// @pre RegisterSaveDataAtomicDeletion.DISABLED_CreateAndRegisterDeletion の実行後 fs が再起動済み
// 削除できていることを確認する
TEST(RegisterSaveDataAtomicDeletion, DISABLED_VerifyDeletion)
{

    nn::util::optional<SaveDataInfo> info;
    for (int i = 0; i < SaveDataCount; i++)
    {
        NNT_FS_SCOPED_TRACE("i: %d\n", i);
        FindSaveData(&info, SaveDataSpaceId::System, [&](SaveDataInfo info_) {
            return (info_.systemSaveDataId == IdBase + i);
        });

        if (i % 2 == 0)
        {
            EXPECT_EQ(util::nullopt, info); // 遅延削除済み
        }
        else
        {
            EXPECT_NE(util::nullopt, info);
            if (info != util::nullopt)
            {
                NNT_EXPECT_RESULT_SUCCESS(DeleteSaveData(info->saveDataId));
            }
        }
    }

    FindSaveData(&info, SaveDataSpaceId::System, [&](SaveDataInfo info_) {
        return (info_.systemSaveDataId == IdBase + SaveDataCount);
    });
    ASSERT_EQ(util::nullopt, info);

    FindSaveData(&info, SaveDataSpaceId::System, [&](SaveDataInfo info_) {
        return (info_.applicationId == TestApplicationId0);
    });
    ASSERT_EQ(util::nullopt, info);

}



#endif

TEST(TemporaryStorage, Mount)
{
    nnt::fs::util::DeleteAllTestSaveData();
    const int64_t Size = 1024 * 1024;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateTemporaryStorage(nnt::fs::util::ApplicationId, 0, Size, 0));

    // 列挙確認
    nn::util::optional<SaveDataInfo> info;
    FindSaveData(&info, SaveDataSpaceId::Temporary, [&](const SaveDataInfo& i) NN_NOEXCEPT
    {
        return i.saveDataType == SaveDataType::Temporary;
    });
    EXPECT_NE(util::nullopt, info);

    // @BisUser:/temp/ に配置されること
    {
        NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::User, nullptr));

        char path[256];
        util::SNPrintf(path, sizeof(path), "@User:/temp/%016llx", info->saveDataId);
        DirectoryEntryType type;
        NNT_EXPECT_RESULT_SUCCESS(GetEntryType(&type, path));

        nnt::fs::util::ListDirectoryRecursive("@User:/");
        Unmount("@User");
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountTemporaryStorage("temp"));

    nnt::fs::util::DumpDirectoryRecursive("temp:/");
    NNT_EXPECT_RESULT_SUCCESS(CreateFile("temp:/test.file", 32));
    nnt::fs::util::DumpDirectoryRecursive("temp:/");
    NNT_EXPECT_RESULT_SUCCESS(CommitSaveData("temp")); // コミット不問

    NNT_EXPECT_RESULT_SUCCESS(DeleteFile("temp:/test.file"));
    NNT_EXPECT_RESULT_SUCCESS(CommitSaveData("temp"));

    Unmount("temp");
    nnt::fs::util::DeleteAllTestSaveData();

    // 削除後は存在しない
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, MountTemporaryStorage("temp"));

    {
        NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::User, nullptr));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::ListDirectoryRecursive("@User:/"));
        Unmount("@User");
    }

}


// アプリ一時ストレージはジャーナルの容量制限が無いこと
TEST(TemporaryStorage, NoJournal)
{
    nnt::fs::util::DeleteAllTestSaveData();
    const int64_t Size = 32 * 1024 * 1024;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateTemporaryStorage(nnt::fs::util::ApplicationId, 0, Size, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountTemporaryStorage("temp"));

    const int64_t FileSize = Size - 32 * 1024;

    NNT_EXPECT_RESULT_SUCCESS(CreateFile("temp:/test.file", FileSize));
    nnt::fs::util::DumpDirectoryRecursive("temp:/");

    NNT_EXPECT_RESULT_SUCCESS(WriteFileWith32BitCount("temp:/test.file", FileSize, 0));
    NNT_EXPECT_RESULT_SUCCESS(DeleteFile("temp:/test.file"));

    NNT_EXPECT_RESULT_SUCCESS(CreateFileWith32BitCount("temp:/test.file", FileSize, 0));

    Unmount("temp");
    nnt::fs::util::DeleteAllTestSaveData();

    // 削除後は存在しない
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, MountTemporaryStorage("temp"));
}


TEST(TemporaryStorage, CleanUp)
{
    nnt::fs::util::DeleteAllTestSaveData();
    const int64_t Size = 1024 * 1024;

    // 作成、確認
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateTemporaryStorage(nnt::fs::util::ApplicationId, 0, Size, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountTemporaryStorage("temp"));
    Unmount("temp");

    // clean up
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CleanUpTemporaryStorage());

    // 存在しない
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, MountTemporaryStorage("temp"));

    nn::util::optional<SaveDataInfo> info;
    FindSaveData(&info, SaveDataSpaceId::Temporary, [&](const SaveDataInfo& i) NN_NOEXCEPT
    {
        return i.saveDataType == SaveDataType::Temporary;
    });
    EXPECT_EQ(util::nullopt, info);

}
#if !defined(NN_BUILD_CONFIG_OS_WIN)
void VerifyCacheStorage(SaveDataSpaceId spaceId)
{
    nnt::fs::util::DeleteAllTestSaveData();
    const int64_t Size = 1024 * 1024;
    const uint16_t IndexNum = 3;
    for(uint16_t i = 0; i < IndexNum; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateCacheStorage(nnt::fs::util::ApplicationId, spaceId, 0, i, Size, Size, 0));
    }

    // 列挙確認
    nn::fs::CacheStorageListHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenCacheStorageList(&handle));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseCacheStorageList(handle);
    };

    nn::fs::CacheStorageInfo info;
    int64_t count = 0;
    while (NN_STATIC_CONDITION(true))
    {
        int num = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadCacheStorageList(&num, &info, handle, 1));
        if (num == 0)
        {
            break;
        }
        else
        {
            EXPECT_EQ(count, info.index);
        }
        count++;
    }
    EXPECT_EQ(count, IndexNum);

    Vector<String> nameList =
    {
        "cache0", "cache1", "cache2"
    };


    for(uint16_t i = 0; i < IndexNum; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountCacheStorage(nameList[i].c_str(), i));
    }

    for(uint16_t i = 0; i < IndexNum; i++)
    {
        nnt::fs::util::DumpDirectoryRecursive((nameList[i] + ":/").c_str());
        NNT_EXPECT_RESULT_SUCCESS(CreateFile((nameList[i] + ":/test.file").c_str(), 32));
        nnt::fs::util::DumpDirectoryRecursive((nameList[i] + ":/").c_str());
        NNT_EXPECT_RESULT_SUCCESS(CommitSaveData(nameList[i].c_str()));

        NNT_EXPECT_RESULT_SUCCESS(DeleteFile((nameList[i] + ":/test.file").c_str()));
        NNT_EXPECT_RESULT_SUCCESS(CommitSaveData(nameList[i].c_str()));
    }

    for(uint16_t i = 0; i < IndexNum; i++)
    {
        Unmount(nameList[i].c_str());
    }

    nnt::fs::util::DeleteAllTestSaveData();

    // 削除後は存在しない
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, MountCacheStorage("cache"));
}

// アプリキャッシュストレージを作成・マウントできること
TEST(CacheStorage, Mount)
{
    VerifyCacheStorage(SaveDataSpaceId::User);
    VerifyCacheStorage(SaveDataSpaceId::SdUser);
}

TEST(CacheStorage, CreateCacheStorageNegativeSize)
{
    nnt::fs::util::DeleteAllTestSaveData();
    const int64_t Size = 3 * 1024 * 1024;
    const int64_t NegativeSize = -1 * Size;

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidSize, nn::fs::CreateCacheStorage(0, NegativeSize, Size));
    nn::fs::DeleteCacheStorage(0);
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidSize, nn::fs::CreateCacheStorage(nnt::fs::util::ApplicationId, nn::fs::SaveDataSpaceId::SdUser, 0, 0, NegativeSize, Size, 0));
}

void CorruptAndVerifyCacheStorage(SaveDataSpaceId spaceId)
{
    // TODO : CorruptAndVerify との共通化
    nnt::fs::util::DeleteAllTestSaveData();
    const int64_t Size = 3 * 1024 * 1024;
    const int BufferSize = 1024 * 1024;
    auto buffer = AllocateBuffer(BufferSize);

    const uint16_t IndexNum = 3;
    for(uint16_t i = 0; i < IndexNum; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateCacheStorage(nnt::fs::util::ApplicationId, spaceId, 0, i, Size, Size, 0));
    }

    NN_UTIL_SCOPE_EXIT
    {
        nnt::fs::util::DeleteAllTestSaveData();
    };
    Vector<String> nameList =
    {
        "cache0", "cache1", "cache2"
    };

    for(uint16_t i = 0; i < IndexNum; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountCacheStorage(nameList[i].c_str(), i));
    }

    for(uint16_t i = 0; i < IndexNum; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(CreateDirectory((nameList[i] + ":/dir").c_str()));
        NNT_EXPECT_RESULT_SUCCESS(CreateDirectory((nameList[i] + ":/dir/subdir").c_str()));
        NNT_EXPECT_RESULT_SUCCESS(CreateFileWith32BitCount((nameList[i] + ":/dir/subdir/file").c_str(), 2 * 1024 * 1024 + 1, 0));
        NNT_EXPECT_RESULT_SUCCESS(CommitSaveData(nameList[i].c_str()));
    }

    for(uint16_t i = 0; i < IndexNum; i++)
    {
        Unmount(nameList[i].c_str());
    }

    // 列挙確認
    nn::fs::CacheStorageListHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenCacheStorageList(&handle));

    nn::fs::CacheStorageInfo cacheStorageInfo;
    int64_t count = 0;
    while (NN_STATIC_CONDITION(true))
    {
        int num = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadCacheStorageList(&num, &cacheStorageInfo, handle, 1));
        if (num == 0)
        {
            break;
        }
        else
        {
            EXPECT_TRUE(count == cacheStorageInfo.index);
            bool isValid = true;
            nn::util::optional<SaveDataInfo> info;
            FindSaveData(&info, spaceId, [&](const SaveDataInfo& i) NN_NOEXCEPT
            {
                return i.saveDataType == SaveDataType::Cache && i.applicationId == nnt::fs::util::ApplicationId && i.index == cacheStorageInfo.index;
            });
            ASSERT_NE(nn::util::nullopt, info);

            NNT_EXPECT_RESULT_SUCCESS(VerifySaveData(&isValid, spaceId, info->saveDataId, buffer.get(), BufferSize));
            EXPECT_TRUE(isValid);
            EXPECT_TRUE(IsFilledWithValue(buffer.get(), BufferSize, 0));

            // 破壊
            NNT_EXPECT_RESULT_SUCCESS(CorruptSaveDataForDebug(spaceId, info->saveDataId));
            NNT_EXPECT_RESULT_SUCCESS(VerifySaveData(&isValid, spaceId, info->saveDataId, buffer.get(), BufferSize));
            EXPECT_FALSE(isValid);
            NNT_EXPECT_RESULT_FAILURE(ResultDataCorrupted, MountCacheStorage("cache", cacheStorageInfo.index));
        }
        count++;
    }
    EXPECT_TRUE(count == IndexNum);
    nn::fs::CloseCacheStorageList(handle);

}

TEST(CacheStorage, CorruptAndVerify)
{
    CorruptAndVerifyCacheStorage(SaveDataSpaceId::User);
    CorruptAndVerifyCacheStorage(SaveDataSpaceId::SdUser);
}
TEST(CacheStorage, NegativeInfoBufferCount)
{
    nnt::fs::util::DeleteAllTestSaveData();
    nn::fs::CacheStorageListHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenCacheStorageList(&handle));
    int num = 0;
    nn::fs::CacheStorageInfo cacheStorageInfo;
    NNT_EXPECT_RESULT_FAILURE(ResultInvalidArgument, nn::fs::ReadCacheStorageList(&num, &cacheStorageInfo, handle, -30000));
    EXPECT_TRUE(num == 0);
    nn::fs::CloseCacheStorageList(handle);
}


TEST(CacheStorage, NoCacheStorage)
{
    nnt::fs::util::DeleteAllTestSaveData();
    nn::fs::CacheStorageListHandle handle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenCacheStorageList(&handle));
    int num = 0;
    nn::fs::CacheStorageInfo cacheStorageInfo;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadCacheStorageList(&num, &cacheStorageInfo, handle, 1));
    EXPECT_TRUE(num == 0);
    nn::fs::CloseCacheStorageList(handle);
}

TEST(CacheStorage, DeleteNotExistCacheStorage)
{
    nnt::fs::util::DeleteAllTestSaveData();
    NNT_EXPECT_RESULT_FAILURE(ResultTargetNotFound, nn::fs::DeleteCacheStorage(0));
}

TEST(CacheStorage, DeleteWhileOpen)
{
    nnt::fs::util::DeleteAllTestSaveData();
    nn::fs::CacheStorageListHandle handle;
    const int64_t Size = 3 * 1024 * 1024;
    const uint16_t IndexNum = 3;
    nn::fs::SaveDataSpaceId spaceId = nn::fs::SaveDataSpaceId::SdUser;
    for(uint16_t i = 0; i < IndexNum; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateCacheStorage(nnt::fs::util::ApplicationId, spaceId, 0, i, Size, Size, 0));
    }

    // 列挙中の削除は許可される
    nn::fs::CacheStorageInfo cacheStorageInfo;
    int num = 0;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenCacheStorageList(&handle));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteCacheStorage(0));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadCacheStorageList(&num, &cacheStorageInfo, handle, 1));
    nn::fs::CloseCacheStorageList(handle);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenCacheStorageList(&handle));

    int64_t count = 0;
    while (NN_STATIC_CONDITION(true))
    {
        num = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadCacheStorageList(&num, &cacheStorageInfo, handle, 1));
        if (num == 0)
        {
            break;
        }
        else
        {
            EXPECT_TRUE(count == cacheStorageInfo.index - 1);
            nn::util::optional<SaveDataInfo> info;
            FindSaveData(&info, spaceId, [&](const SaveDataInfo& i) NN_NOEXCEPT
            {
                return i.saveDataType == SaveDataType::Cache && i.applicationId == nnt::fs::util::ApplicationId && i.index == cacheStorageInfo.index;
            });
            ASSERT_NE(nn::util::nullopt, info);
        }
        count++;
    }
    EXPECT_TRUE(count == IndexNum - 1);
    nn::fs::CloseCacheStorageList(handle);
}

TEST(CacheStorage, MountWhileOpen)
{
    nnt::fs::util::DeleteAllTestSaveData();
    nn::fs::CacheStorageListHandle handle;
    const int64_t Size = 3 * 1024 * 1024;
    const uint16_t IndexNum = 3;
    nn::fs::SaveDataSpaceId spaceId = nn::fs::SaveDataSpaceId::SdUser;
    for(uint16_t i = 0; i < IndexNum; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateCacheStorage(nnt::fs::util::ApplicationId, spaceId, 0, i, Size, Size, 0));
    }

    Vector<String> nameList =
    {
        "cache0", "cache1", "cache2"
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenCacheStorageList(&handle));

    int64_t count = 0;
    while (NN_STATIC_CONDITION(true))
    {
        int num = 0;
        nn::fs::CacheStorageInfo cacheStorageInfo;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadCacheStorageList(&num, &cacheStorageInfo, handle, 1));
        if (num == 0)
        {
            break;
        }
        else
        {
            EXPECT_TRUE(count == cacheStorageInfo.index);
            NNT_ASSERT_RESULT_SUCCESS(MountCacheStorage(nameList[count].c_str(), cacheStorageInfo.index));
        }
        count++;
    }
    EXPECT_TRUE(count == IndexNum);
    nn::fs::CloseCacheStorageList(handle);
    for(uint16_t i = 0; i < IndexNum; i++)
    {
        Unmount(nameList[i].c_str());
    }
}

TEST(CacheStorage, ExtendSizeOver)
{
    const auto BlockSize = 16 * 1024;

    nnt::fs::util::DeleteAllTestSaveData();

    const UserId userId = { { 0, 0 } };

    const auto AvailableSize = 4 * BlockSize;
    const auto JournalSize = 4 * BlockSize;
    const auto SpaceId = nn::fs::SaveDataSpaceId::SdUser;

    // キャッシュストレージを作成
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateCacheStorage(
        nnt::fs::util::ApplicationId,
        SpaceId,
        0,
        AvailableSize,
        JournalSize,
        0));
    NN_UTIL_SCOPE_EXIT
    {
        nnt::fs::util::DeleteAllTestSaveData();
    };

    // マウントできる
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountCacheStorage("cache", nnt::fs::util::ApplicationId));
    nn::fs::Unmount("cache");

    // 空き容量を取得
    int64_t freeSpaceSize = 0;
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("sd"));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("sd");
        };

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetFreeSpaceSize(&freeSpaceSize, "sd:/"));
    }

    // 空き容量を超えるサイズに拡張（失敗する）
    {
        nn::util::optional<SaveDataInfo> info;
        FindSaveData(
            &info,
            SpaceId,
            [&userId](const SaveDataInfo& info) NN_NOEXCEPT
            {
                return info.saveDataUserId == userId;
            });
        ASSERT_NE(nn::util::nullopt, info);

        const auto result = ExtendSaveData(
                info->saveDataSpaceId,
                info->saveDataId,
                AvailableSize + freeSpaceSize,
            JournalSize + freeSpaceSize);

        // FAT32 で 4 GB 以上に拡張しようとする場合は ResultOutOfRange
        // それ以外は ResultUsableSpaceNotEnough
        if( !nn::fs::ResultOutOfRange::Includes(result) )
        {
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultUsableSpaceNotEnough, result);
    }
    }

    // 拡張に失敗した後もマウントできる
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountCacheStorage("cache", nnt::fs::util::ApplicationId));
    nn::fs::Unmount("cache");
}
#endif

}

class SaveDataSameSaveDataId : public ::testing::TestWithParam<int>
{
public:
    void TestSameSaveDataId(int saveDataCount) NN_NOEXCEPT
    {
    nnt::fs::util::DeleteAllTestSaveData();

    int64_t size = 64 * 1024;

    auto spaceId = SaveDataSpaceId::User;

    // ユーザアカウントセーブ作成
        for (int i = 0; i < saveDataCount; i++)
    {
        UserId userId = { { 0, 0 } };
        userId._data[0] = i;

        NNT_EXPECT_RESULT_SUCCESS(CreateSaveData(TestApplicationId0, userId, 0, size, size, 0));

        nn::util::optional<SaveDataInfo> info;
        FindSaveData(&info, spaceId, [&](const SaveDataInfo& i)
        {
            return i.saveDataUserId == userId;
        });

        NN_LOG("create %d %016llx\n", i, info->saveDataId);
    }

    // 一時ストレージの作成削除
        for (int i = 0; i < saveDataCount * 2; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateTemporaryStorage(TestApplicationId0, 0, size, 0));

        nn::util::optional<SaveDataInfo> info;
        FindSaveData(&info, SaveDataSpaceId::Temporary, [&](const SaveDataInfo& i)
        {
            return i.saveDataType == SaveDataType::Temporary;
        });

        NN_LOG("create and delete TemporaryStorage: %016llx\n", info->saveDataId);

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteSaveData(SaveDataSpaceId::Temporary, info->saveDataId));
    }


    // ユーザアカウントセーブに影響がないこと
    NNT_ASSERT_RESULT_SUCCESS(MountBis("bis", nn::fs::BisPartitionId::User));

        for (int i = 0; i < saveDataCount; i++)
    {
        UserId userId = { { 0, 0 } };
        userId._data[0] = i;

        // マウント出来ること
        NNT_EXPECT_RESULT_SUCCESS(MountSaveData("save", TestApplicationId0, userId));
        Unmount("save");

        // サムネが消えていないこと
        nn::util::optional<SaveDataInfo> info;
        FindSaveData(&info, spaceId, [&](const SaveDataInfo& i)
        {
            return i.saveDataUserId == userId;
        });

        char path[128];
        DirectoryEntryType type;
        nn::util::SNPrintf(path, sizeof(path), "bis:/saveMeta/%016llx/%08x.meta", info->saveDataId, 1);
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetEntryType(&type, path));
    }

    nn::fs::Unmount("bis");
    nnt::fs::util::DeleteAllTestSaveData();
    }


};

// 一時ストレージの削除が他のセーブデータに影響しないこと（SIGLO-71646）
TEST_P(SaveDataSameSaveDataId, SameSaveDataId)
{
    TestSameSaveDataId(GetParam());
}

INSTANTIATE_TEST_CASE_P(
    SaveDataSameSaveDataId,
    SaveDataSameSaveDataId,
    ::testing::Values(16)
);

INSTANTIATE_TEST_CASE_P(
    SaveDataSameSaveDataIdHeavy,
    SaveDataSameSaveDataId,
    ::testing::Values(128)
);


extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    SetEnabledAutoAbort(false);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
