﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_BcatSaveData.h>
#include <nn/fs/fs_SaveDataForDebug.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemBcatSaveData.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>

#include "testFs_FsLib_Mount.h"

namespace {
    const size_t SaveDataSize = 22 * 16 * 1024;
    const size_t SaveDataJournalSize = 3 * 16 * 1024;

    class MountSaveData : public ::testing::Test
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            nnt::fs::util::DeleteAllTestSaveData();
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            nnt::fs::util::DeleteAllTestSaveData();
        }
    };

    typedef MountSaveData SystemSaveData;
}

//! DeleteSaveData で削除
TEST_F(MountSaveData, DeleteSaveData)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("save"));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("save:/test.file", 32));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory("save:/dir"));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("save:/dir/subfile", 32));
    nnt::fs::util::DumpDirectoryRecursive("save:/");

    nn::fs::Unmount("save");
}

TEST_F(MountSaveData, EnumerateSaveData)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("save"));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSaveData("usrSave0", nnt::fs::util::TestUserId0));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSaveData("usrSave1", nnt::fs::util::TestUserId1));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSystemSaveData("sysSave0", 0x8000000000000001));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSystemSaveData("sysSave3", 0x8000000000000003));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountBcatSaveData("bcatSave0", nnt::fs::util::TestApplicationId0));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountBcatSaveData("bcatSave1", nnt::fs::util::TestApplicationId1));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountDeviceSaveData("deviceSave"));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSystemBcatSaveData("sysBcatSave0", 0x8000000000000005));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSystemBcatSaveData("sysBcatSave1", 0x8000000000000006));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("save:/test.file", 32));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("usrSave0:/test.file", 32));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("sysSave0:/save.file", 32));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("bcatSave0:/save.file", 32));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("deviceSave:/save.file", 32));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("sysBcatSave0:/save.file", 32));
    nnt::fs::util::DumpDirectoryRecursive("save:/");
    nnt::fs::util::DumpDirectoryRecursive("usrSave0:/");
    nnt::fs::util::DumpDirectoryRecursive("sysSave0:/");
    nnt::fs::util::DumpDirectoryRecursive("bcatSave0:/");
    nnt::fs::util::DumpDirectoryRecursive("deviceSave:/");
    nnt::fs::util::DumpDirectoryRecursive("sysBcatSave0:/");

    nn::fs::Unmount("save");
    nn::fs::Unmount("usrSave0");
    nn::fs::Unmount("usrSave1");
    nn::fs::Unmount("sysSave0");
    nn::fs::Unmount("sysSave3");
    nn::fs::Unmount("bcatSave0");
    nn::fs::Unmount("bcatSave1");
    nn::fs::Unmount("deviceSave");
    nn::fs::Unmount("sysBcatSave0");
    nn::fs::Unmount("sysBcatSave1");

    {

        { // system
            nnt::fs::util::Vector<nn::fs::SaveDataInfo> info;

            nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::System, nnt::fs::util::IsTestSaveData);
            EXPECT_EQ(4, info.size());
            for( unsigned int i = 0; i < info.size(); i++ )
            {
                EXPECT_TRUE((info[i].saveDataType == nn::fs::SaveDataType::System) || (info[i].saveDataType == nn::fs::SaveDataType::SystemBcat));
            }
        }
        { // user
            nnt::fs::util::Vector<nn::fs::SaveDataInfo> info;
            nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::User, nnt::fs::util::IsTestSaveData);

            EXPECT_EQ(6, info.size());
            for( unsigned int i = 0; i < info.size(); i++ )
            {
                switch( info[i].saveDataType )
                {
                case nn::fs::SaveDataType::Bcat:
                {
                    static bool bcatTestExpectValue[] = { true, true };
                    if( bcatTestExpectValue[0] && info[i].applicationId.value == nnt::fs::util::TestApplicationId0.value ) { bcatTestExpectValue[0] = false; }
                    else if( bcatTestExpectValue[1] && info[i].applicationId.value == nnt::fs::util::TestApplicationId1.value ) { bcatTestExpectValue[1] = false; }
                    else
                    {
                        // かぶりや抜けがあった場合は意図的にエラーを起こす
                        EXPECT_TRUE(false);
                    }
                }
                break;
                case nn::fs::SaveDataType::Device:
                    EXPECT_EQ(nnt::fs::util::UserSaveDataApplicationId, info[i].applicationId.value);
                    break;
                case nn::fs::SaveDataType::Account:
                    EXPECT_EQ(nnt::fs::util::UserSaveDataApplicationId, info[i].applicationId.value);
                    break;
                case nn::fs::SaveDataType::System:
                    EXPECT_NE(nn::fs::SaveDataType::System, info[i].saveDataType);
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
        }
    }
}

TEST_F(MountSaveData, EnumerateAndDeleteSaveData)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("save"));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSaveData("usrSave0", nnt::fs::util::TestUserId0));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSaveData("usrSave1", nnt::fs::util::TestUserId1));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSystemSaveData("sysSave0", 0x8000000000000001));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSystemSaveData("sysSave3", 0x8000000000000003));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountBcatSaveData("bcatSave0", nnt::fs::util::TestApplicationId0));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountBcatSaveData("bcatSave1", nnt::fs::util::TestApplicationId1));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountDeviceSaveData("deviceSave"));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSystemBcatSaveData("sysBcatSave0", 0x8000000000000005));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSystemBcatSaveData("sysBcatSave1", 0x8000000000000006));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("save:/test.file", 32));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("usrSave0:/test.file", 32));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("sysSave0:/save.file", 32));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("bcatSave0:/save.file", 32));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("deviceSave:/save.file", 32));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("sysBcatSave0:/save.file", 32));
    nnt::fs::util::DumpDirectoryRecursive("save:/");
    nnt::fs::util::DumpDirectoryRecursive("usrSave0:/");
    nnt::fs::util::DumpDirectoryRecursive("sysSave0:/");
    nnt::fs::util::DumpDirectoryRecursive("bcatSave0:/");
    nnt::fs::util::DumpDirectoryRecursive("deviceSave:/");
    nnt::fs::util::DumpDirectoryRecursive("sysBcatSave0:/");

    nn::fs::Unmount("save");
    nn::fs::Unmount("usrSave0");
    nn::fs::Unmount("usrSave1");
    nn::fs::Unmount("sysSave0");
    nn::fs::Unmount("sysSave3");
    nn::fs::Unmount("bcatSave0");
    nn::fs::Unmount("bcatSave1");
    nn::fs::Unmount("deviceSave");
    nn::fs::Unmount("sysBcatSave0");
    nn::fs::Unmount("sysBcatSave1");

    {
        { // user
            int saveDeletionCount = 0;
            while( NN_STATIC_CONDITION(true) )
            {
                nn::util::optional<nn::fs::SaveDataInfo> info;
                nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::User, nnt::fs::util::IsTestSaveData);

                if( info == nn::util::nullopt )
                {
                    break;
                }

                NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteSaveData(info->saveDataId));

                NN_LOG("save data id: %llx, user id: %llx_%llx\n", info.value().saveDataId, info.value().saveDataUserId._data[0], info.value().saveDataUserId._data[1]);
                if( info.value().saveDataType == nn::fs::SaveDataType::Bcat )
                {
                    static bool bcatTestExpectValue[] = { true, true };
                    if( bcatTestExpectValue[0] && info.value().applicationId.value == nnt::fs::util::TestApplicationId0.value ) { bcatTestExpectValue[0] = false; }
                    else if( bcatTestExpectValue[1] && info.value().applicationId.value == nnt::fs::util::TestApplicationId1.value ) { bcatTestExpectValue[1] = false; }
                    else
                    {
                        // かぶりや抜けがあった場合は意図的にエラーを起こす
                        EXPECT_TRUE(false);
                    }
                }
                else
                {
                    EXPECT_EQ(nnt::fs::util::UserSaveDataApplicationId, info.value().applicationId.value);
                    EXPECT_TRUE(info.value().saveDataType == nn::fs::SaveDataType::Device || info.value().saveDataType == nn::fs::SaveDataType::Account);
                }
                ++saveDeletionCount;
            }

            EXPECT_EQ(6, saveDeletionCount);
        }

        { // system
            int saveDeletionCount = 0;
            while( NN_STATIC_CONDITION(true) )
            {
                nn::util::optional<nn::fs::SaveDataInfo> info;
                nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::System, nnt::fs::util::IsTestSaveData);

                if( info == nn::util::nullopt )
                {
                    break;
                }

                NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteSaveData(info->saveDataId));

                NN_LOG("save data id: %llx\n", info.value().saveDataId);
                EXPECT_TRUE(info.value().saveDataId == 0x8000000000000001ULL || info.value().saveDataId == 0x8000000000000003ULL ||
                    info.value().saveDataId == 0x8000000000000005ULL || info.value().saveDataId == 0x8000000000000006ULL);
                // TODO: OwnerId を取得して検証
                ++saveDeletionCount;
            }
            EXPECT_EQ(4, saveDeletionCount);
        }

    }
}

TEST_F(MountSaveData, MountSaveData)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("save"));

    nnt::fs::util::DumpDirectoryRecursive("save:/");
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("save:/test.file", 32));
    nnt::fs::util::DumpDirectoryRecursive("save:/");
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile("save:/test.file"));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));

    nn::fs::Unmount("save");
}

TEST_F(MountSaveData, MountUserSaveData)
{
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSaveData("save", nnt::fs::util::TestUserId0));

    nnt::fs::util::DumpDirectoryRecursive("save:/");
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("save:/test.file", 32));
    nnt::fs::util::DumpDirectoryRecursive("save:/");
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile("save:/test.file"));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));

    nn::fs::Unmount("save");
}

TEST_F(MountSaveData, TargetNotFoundOnMountUserSaveData)
{
    // 削除後はエラー
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::MountSaveData("save", nnt::fs::util::TestUserId0));
}

TEST_F(MountSaveData, MountSystemSaveData)
{
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSystemSaveData("save", 0x8000000000000001));

    nnt::fs::util::DumpDirectoryRecursive("save:/");
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("save:/test.file", 32));
    nnt::fs::util::DumpDirectoryRecursive("save:/");
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile("save:/test.file"));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));

    nn::fs::Unmount("save");
}

TEST_F(MountSaveData, TargetNotFoundOnMountSystemSaveData)
{
    // 削除後はエラー
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::MountSystemSaveData("save", 0x8000000000000001));
}

TEST_F(MountSaveData, MountOthersSaveData)
{
    // Set up
    nn::ncm::ApplicationId otherApplicationId;
    otherApplicationId.value = 3;

    uint64_t ownerId = otherApplicationId.value;

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSaveData(otherApplicationId, nnt::fs::util::TestUserId0, ownerId, SaveDataSize, SaveDataJournalSize, 0));

    // Enumerate
    nn::util::optional<nn::fs::SaveDataInfo> info;
    nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::User, [&](const nn::fs::SaveDataInfo& i) NN_NOEXCEPT
        {
            return (i.applicationId.value == otherApplicationId.value);
        });

    EXPECT_EQ(nnt::fs::util::TestUserId0, info->saveDataUserId);

    // Mount
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountSaveData("otherSave", otherApplicationId, info->saveDataUserId));

    nnt::fs::util::DumpDirectoryRecursive("otherSave:/");

    nn::fs::Unmount("otherSave");
}

TEST_F(MountSaveData, TargetNotFoundOnMountOthersSaveData)
{
    nn::ncm::ApplicationId otherApplicationId;
    otherApplicationId.value = 3;

    // 削除後はエラー
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::MountSaveDataReadOnly("otherSave", otherApplicationId, nnt::fs::util::TestUserId0));
}

TEST_F(MountSaveData, MountOthersSystemSaveData)
{
    // Set up
    nn::fs::SystemSaveDataId id = 0x8000000000000003;

    nn::ncm::ApplicationId otherApplicationId;
    otherApplicationId.value = 3;

    uint64_t ownerId = otherApplicationId.value;

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(id, ownerId, SaveDataSize, SaveDataJournalSize, 0));

    // Enumerate
    nn::util::optional<nn::fs::SaveDataInfo> info;
    nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::System, [&](const nn::fs::SaveDataInfo& i) NN_NOEXCEPT
        {
            return (i.saveDataId == id);
        });

    // TODO: OwnerId を取得して検証

    // Mount
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSystemSaveData("otherSysSave", id));

    nnt::fs::util::DumpDirectoryRecursive("otherSysSave:/");

    nn::fs::Unmount("otherSysSave");
}

TEST_F(MountSaveData, MountBcatSaveData)
{
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountBcatSaveData("save", nnt::fs::util::TestApplicationId0));

    nnt::fs::util::DumpDirectoryRecursive("save:/");
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("save:/test.file", 32));
    nnt::fs::util::DumpDirectoryRecursive("save:/");
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile("save:/test.file"));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));

    nn::fs::Unmount("save");
}

TEST_F(MountSaveData, MountDeviceSaveData)
{
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountDeviceSaveData("save"));

    nnt::fs::util::DumpDirectoryRecursive("save:/");
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("save:/test.file", 32));
    nnt::fs::util::DumpDirectoryRecursive("save:/");
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile("save:/test.file"));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));

    nn::fs::Unmount("save");
}

TEST_F(MountSaveData, MountOthersDeviceSaveData)
{
    // Set up
    nn::ncm::ApplicationId otherApplicationId;
    otherApplicationId.value = 3;

    uint64_t ownerId = otherApplicationId.value;
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDeviceSaveData(otherApplicationId, ownerId, SaveDataSize, SaveDataJournalSize, 0));

    // Enumerate
    nn::util::optional<nn::fs::SaveDataInfo> info;
    nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::User, [&](const nn::fs::SaveDataInfo& i) NN_NOEXCEPT
        {
            return (i.applicationId.value == otherApplicationId.value) && (i.saveDataType == nn::fs::SaveDataType::Device);
        });

    ASSERT_NE(info, nullptr);

    // Mount
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountDeviceSaveData("otherSave", otherApplicationId));
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("otherSave:/test.file", 32));
        nnt::fs::util::DumpDirectoryRecursive("otherSave:/");
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("otherSave"));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile("otherSave:/test.file"));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("otherSave"));
    }
    nn::fs::Unmount("otherSave");
}

TEST_F(MountSaveData, MountUserSystemSaveData)
{
    nn::fs::SystemSaveDataId id = 0x8000000000000003;

    nn::fs::UserId userId[2] = { { { 0 } },{ { 1 } } };
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSystemSaveData("save0", id, userId[0]));
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSystemSaveData("save1", id, userId[1]));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("save0:/0", 0));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("save1:/1", 1));

    nn::fs::DirectoryEntryType type;
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetEntryType(&type, "save0:/0"));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetEntryType(&type, "save1:/1"));

    // 違うセーブデータであること
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, nn::fs::GetEntryType(&type, "save0:/1"));
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, nn::fs::GetEntryType(&type, "save1:/0"));

    nn::fs::Unmount("save0");
    nn::fs::Unmount("save1");
}

TEST_F(MountSaveData, MountSystemBcatSaveData)
{
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSystemBcatSaveData("save", 0x8000000000000005));

    nnt::fs::util::DumpDirectoryRecursive("save:/");
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("save:/test.file", 32));
    nnt::fs::util::DumpDirectoryRecursive("save:/");
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile("save:/test.file"));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData("save"));

    nn::fs::Unmount("save");
}

TEST_F(MountSaveData, TargetNotFoundOnMountSystemBcatSaveData)
{
    // 削除後はエラー
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::MountSystemBcatSaveData("save", 0x8000000000000005));
}

#if !defined(NN_BUILD_CONFIG_OS_WIN)

// ディレクトリ版セーブがマウントできないこと
TEST_F(MountSaveData, MountDirectorySaveData)
{
    nn::fs::SystemSaveDataId id = 0x8000000000000003;

    // システムセーブ置き場に直接ディレクトリ版セーブ相当を生成
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::System, nullptr));
    nn::fs::CreateDirectory("@System:/save");
    nn::fs::CreateDirectory("@System:/save/8000000000000003");
    nn::fs::CreateDirectory("@System:/save/8000000000000003/0");
    nn::fs::CreateDirectory("@System:/save/8000000000000003/1");
    // NNT_EXPECT_RESULT_SUCCESS(DumpDirectoryRecursiveWithWhiteList("@System:/"));

    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::MountSystemSaveData("save", id));

    nn::fs::DeleteDirectoryRecursively("@System:/save/8000000000000003");

    nn::fs::Unmount("@System");
}
#endif

#if 0 // nn::account::Initialize() による memory leak のため無効化
TEST(Mount, MountAccountSystemSaveData)
{
    nn::account::Initialize();
    SystemSaveDataId id = 0x8000000000000003;
    nn::account::Uid dummyUid;
    EXPECT_DEATH_IF_SUPPORTED(nnt::fs::util::CreateAndMountSystemSaveData("save", id, dummyUid), "");
}


TEST(Mount, CreateAccountSystemSaveData)
{
    nn::account::Initialize();
    SystemSaveDataId id = 0x8000000000000003;
    nn::account::Uid dummyUid;
    EXPECT_DEATH_IF_SUPPORTED(CreateSystemSaveData(id, dummyUid, SaveDataSize, SaveDataJournalSize, 0), "");
}

#endif

TEST_F(SystemSaveData, CreateWithoutOwnerId)
{
    // Set up
    nn::fs::SystemSaveDataId id = 0x8000000000000003;

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(id, SaveDataSize, SaveDataJournalSize, 0));

    // Enumerate
    nn::util::optional<nn::fs::SaveDataInfo> info;
    nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::System, [&](const nn::fs::SaveDataInfo& i) NN_NOEXCEPT
        {
            return (i.saveDataId == id);
        });

    // TODO: OwnerId を取得して検証
}


TEST_F(SystemSaveData, CreateUserSystemSaveData)
{
    nn::fs::SystemSaveDataId id = 0x8000000000000003;

    nn::fs::UserId userId[2] = { nnt::fs::util::TestUserId0, nnt::fs::util::TestUserId1 };
    for( int i = 0; i < 2; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(id, userId[i], SaveDataSize, SaveDataJournalSize, 0));
    }

    nn::util::optional<nn::fs::SaveDataInfo> info[2];
    for( int i = 0; i < 2; ++i )
    {
        nnt::fs::util::FindSaveData(&info[i], nn::fs::SaveDataSpaceId::System, [&](const nn::fs::SaveDataInfo& currentInfo) NN_NOEXCEPT
            {
                return (currentInfo.saveDataType == nn::fs::SaveDataType::System && currentInfo.saveDataUserId == userId[i]);
            });
    }

    EXPECT_NE(info[0]->saveDataId, id);
    EXPECT_NE(info[0]->saveDataId, info[1]->saveDataId);

    for( int i = 0; i < 2; ++i )
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountSystemSaveData("usave", nn::fs::SaveDataSpaceId::System, id, userId[i]));
        nn::fs::Unmount("usave");
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteSystemSaveData(nn::fs::SaveDataSpaceId::System, id, userId[i]));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::MountSystemSaveData("usave", nn::fs::SaveDataSpaceId::System, id, userId[i]));
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultTargetNotFound, nn::fs::DeleteSystemSaveData(nn::fs::SaveDataSpaceId::System, id, userId[i]));
    }
}

namespace
{
    const nn::fs::SystemSaveDataId TestSystemSaveDataId = 0x8000000000000001;

    void SetUpAndTearDown() NN_NOEXCEPT
    {
        nnt::fs::util::DeleteAllTestSaveData();
    }

    const nnt::fs::util::MountTestAttribute GetAttribute() NN_NOEXCEPT
    {
        nnt::fs::util::MountTestAttribute attribute = {};
        attribute.setUp = SetUpAndTearDown;
        attribute.tearDowm = SetUpAndTearDown;
#if defined(NN_BUILD_CONFIG_OS_WIN)
        attribute.isGlobalNewCallExpected = true;
#endif // defined(NN_BUILD_CONFIG_OS_WIN)
        return attribute;
    }

    nn::Result MountBcatSaveDataForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountBcatSaveData(mountName, nnt::fs::util::TestApplicationId0));
        NN_RESULT_SUCCESS;
    }

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    // Result MountCacheStorage(const char* name)
    nn::Result MountCacheStorageType1ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountCacheStorage(mountName));
        NN_RESULT_SUCCESS;
    }

    // Result MountCacheStorage(const char* name, int index)
    nn::Result MountCacheStorageType2ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountCacheStorage(mountName, 0));
        NN_RESULT_SUCCESS;
    }
#endif // !defined(NN_BUILD_CONFIG_OS_WIN)

    // Result MountCacheStorage(const char* name, const nn::ncm::ApplicationId applicationId)
    nn::Result MountCacheStorageType3ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountCacheStorage(mountName, nnt::fs::util::TestApplicationId0));
        NN_RESULT_SUCCESS;
    }

    // Result MountCacheStorage(const char* name, const nn::ncm::ApplicationId applicationId, int index)
    nn::Result MountCacheStorageType4ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountCacheStorage(mountName, nnt::fs::util::TestApplicationId0, 0));
        NN_RESULT_SUCCESS;
    }

    // Result MountDeviceSaveData(const char* name)
    nn::Result MountDeviceSaveDataType1ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountDeviceSaveData(mountName));
        NN_RESULT_SUCCESS;
    }

    // Result MountDeviceSaveData(const char* name, const nn::ncm::ApplicationId applicationId)
    nn::Result MountDeviceSaveDataType2ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountDeviceSaveData(mountName, nnt::fs::util::TestApplicationId0));
        NN_RESULT_SUCCESS;
    }

    // Result MountSaveData(const char* name, UserId userId)
    nn::Result MountSaveDataType1ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountSaveData(mountName, nnt::fs::util::TestUserId0));
        NN_RESULT_SUCCESS;
    }

    // Result MountSaveData(const char* name, const nn::ncm::ApplicationId applicationId, UserId userId)
    nn::Result MountSaveDataType2ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountSaveData(mountName, nnt::fs::util::TestApplicationId0, nnt::fs::util::TestUserId0));
        NN_RESULT_SUCCESS;
    }

    // Result MountSaveData(const char* name, const nn::account::Uid& user)
    // → testFs_FsLib_MountSaveData

    // Result MountSaveData(const char* name, nn::ApplicationId applicationId, const nn::account::Uid& user)
    // → testFs_FsLib_MountSaveData

    // Result MountSaveDataReadOnly(const char* name, const nn::ncm::ApplicationId applicationId, UserId userId)
    nn::Result MountSaveDataReadOnlyType1ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountSaveDataReadOnly(mountName, nnt::fs::util::TestApplicationId0, nnt::fs::util::TestUserId0));
        NN_RESULT_SUCCESS;
    }

    // Result MountSaveDataReadOnly(const char* name, const nn::ApplicationId applicationId, const nn::account::Uid& user)
    // → testFs_FsLib_MountSaveData

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    // Result MountSaveDataInternalStorage(const char* name, SaveDataSpaceId spaceId, SaveDataId saveDataId)
    nn::Result MountSaveDataInternalStorageForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountSaveDataInternalStorage(mountName, TestSystemSaveDataId));
        NN_RESULT_SUCCESS;
    }
#endif // !defined(NN_BUILD_CONFIG_OS_WIN)

    // Result MountSystemBcatSaveData(const char* name, SystemBcatSaveDataId id)
    nn::Result MountSystemBcatSaveDataForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountSystemBcatSaveData(mountName, TestSystemSaveDataId));
        NN_RESULT_SUCCESS;
    }

    // Result MountSystemSaveData(const char* name, SystemSaveDataId id)
    nn::Result MountSystemSaveDataType1ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountSystemSaveData(mountName, TestSystemSaveDataId));
        NN_RESULT_SUCCESS;
    }

    // Result MountSystemSaveData(const char* name, SaveDataSpaceId saveDataSpaceId, SystemSaveDataId id)
    nn::Result MountSystemSaveDataType2ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountSystemSaveData(mountName, nn::fs::SaveDataSpaceId::System, TestSystemSaveDataId));
        NN_RESULT_SUCCESS;
    }

    // Result MountSystemSaveData(const char* name, SystemSaveDataId id, UserId userId)
    nn::Result MountSystemSaveDataType3ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountSystemSaveData(mountName, TestSystemSaveDataId, nnt::fs::util::TestUserId0));
        NN_RESULT_SUCCESS;
    }

    // Result MountSystemSaveData(const char* name, SaveDataSpaceId saveDataSpaceId, SystemSaveDataId id, UserId userId)
    nn::Result MountSystemSaveDataType4ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountSystemSaveData(mountName, nn::fs::SaveDataSpaceId::System, TestSystemSaveDataId, nnt::fs::util::TestUserId0));
        NN_RESULT_SUCCESS;
    }

    // Result MountSystemSaveData(const char* name, SystemSaveDataId id, const nn::account::Uid& user)
    // → testFs_FsLib_MountSaveData

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    // Result MountTemporaryStorage(const char* name)
    nn::Result MountTemporaryStorageForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nnt::fs::util::CreateAndMountTemporaryStorage(mountName));
        NN_RESULT_SUCCESS;
    }
#endif // !defined(NN_BUILD_CONFIG_OS_WIN)

    nn::Result MountSaveDataForDebugForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountSaveDataForDebug(mountName));
        NN_RESULT_SUCCESS;
    }

    const nnt::fs::util::MountTestParameter MountTestParameters[] = {
        { "MountBcatSaveData", MountBcatSaveDataForMountNameTest, nullptr, GetAttribute },
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        { "MountCacheStorageType1", MountCacheStorageType1ForMountNameTest, nullptr, GetAttribute },
        { "MountCacheStorageType2", MountCacheStorageType2ForMountNameTest, nullptr, GetAttribute },
#endif // !defined(NN_BUILD_CONFIG_OS_WIN)
        { "MountCacheStorageType3", MountCacheStorageType3ForMountNameTest, nullptr, GetAttribute },
        { "MountCacheStorageType4", MountCacheStorageType4ForMountNameTest, nullptr, GetAttribute },
        { "MountDeviceSaveDataType1", MountDeviceSaveDataType1ForMountNameTest, nullptr, GetAttribute },
        { "MountDeviceSaveDataType2", MountDeviceSaveDataType2ForMountNameTest, nullptr, GetAttribute },
        { "MountSaveDataType1", MountSaveDataType1ForMountNameTest, nullptr, GetAttribute },
        { "MountSaveDataType2", MountSaveDataType2ForMountNameTest, nullptr, GetAttribute },
        { "MountSaveDataReadOnlyType1", MountSaveDataReadOnlyType1ForMountNameTest, nullptr, GetAttribute },
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        { "MountSaveDataInternalStorage", MountSaveDataInternalStorageForMountNameTest, nullptr, GetAttribute },
#endif // !defined(NN_BUILD_CONFIG_OS_WIN)
        { "MountSystemBcatSaveData", MountSystemBcatSaveDataForMountNameTest, nullptr, GetAttribute },
        { "MountSystemSaveDataType1", MountSystemSaveDataType1ForMountNameTest, nullptr, GetAttribute },
        { "MountSystemSaveDataType2", MountSystemSaveDataType2ForMountNameTest, nullptr, GetAttribute },
        { "MountSystemSaveDataType3", MountSystemSaveDataType3ForMountNameTest, nullptr, GetAttribute },
        { "MountSystemSaveDataType4", MountSystemSaveDataType4ForMountNameTest, nullptr, GetAttribute },
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        { "MountTemporaryStorage", MountTemporaryStorageForMountNameTest, nullptr, GetAttribute },
#endif // !defined(NN_BUILD_CONFIG_OS_WIN)
        { "MountSaveDataForDebug", MountSaveDataForDebugForMountNameTest, nullptr, GetAttribute },
    };
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
NNT_FS_INSTANTIATE_TEST_CASE_MOUNT(WithMountSaveData1_Heavy, ::testing::ValuesIn(MountTestParameters));
#else
NNT_FS_INSTANTIATE_TEST_CASE_MOUNT(WithMountSaveData1, ::testing::ValuesIn(MountTestParameters));
#endif // defined(NN_BUILD_CONFIG_OS_WIN)
