﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_DebugPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

namespace {

#if !defined(NN_BUILD_CONFIG_OS_WIN)

TEST(PaddingFile, CreateAndDelete)
{
    NNT_ASSERT_RESULT_SUCCESS(MountBis("bis", BisPartitionId::User));

    int64_t initialFreeSize;
    NNT_ASSERT_RESULT_SUCCESS(GetFreeSpaceSize(&initialFreeSize, "bis:/"));
    ASSERT_GE(initialFreeSize, 1024 * 1024 * 1024);
    NN_LOG("FreeSpaceSize: %lld\n", initialFreeSize);

    NNT_EXPECT_RESULT_SUCCESS(CreatePaddingFile(initialFreeSize / 2));
    NNT_EXPECT_RESULT_SUCCESS(CreatePaddingFile(initialFreeSize / 4));
    NNT_EXPECT_RESULT_SUCCESS(CreatePaddingFile(initialFreeSize / 8));

    int64_t freeSize;
    NNT_EXPECT_RESULT_SUCCESS(GetFreeSpaceSize(&freeSize, "bis:/"));
    EXPECT_LE(freeSize, initialFreeSize / 8);
    NN_LOG("FreeSpaceSize: %lld\n", freeSize);

    NNT_EXPECT_RESULT_SUCCESS(CreatePaddingFile(freeSize - 64 * 1024)); // メタデータ占有があるため freeSize そのままのサイズは作成できない

    NNT_EXPECT_RESULT_SUCCESS(GetFreeSpaceSize(&freeSize, "bis:/"));
    EXPECT_LE(freeSize, 64 * 1024);
    NN_LOG("FreeSpaceSize: %lld\n", freeSize);

    NNT_EXPECT_RESULT_SUCCESS(DeleteAllPaddingFiles());

    NNT_EXPECT_RESULT_SUCCESS(GetFreeSpaceSize(&freeSize, "bis:/"));
    EXPECT_LE(freeSize, initialFreeSize);
    NN_LOG("FreeSpaceSize: %lld\n", freeSize);

    Unmount("bis");
}

#endif

TEST(PaddingFile, DeleteTwice)
{
    NNT_EXPECT_RESULT_SUCCESS(DeleteAllPaddingFiles());
    NNT_EXPECT_RESULT_SUCCESS(DeleteAllPaddingFiles());
}

TEST(PaddingFile, ZeroPadding)
{
    NNT_EXPECT_RESULT_SUCCESS(CreatePaddingFile(0));
    NNT_EXPECT_RESULT_SUCCESS(DeleteAllPaddingFiles());
}

}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
