﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account.h>
#include <nn/fs.h>
#include <nn/oe.h>

#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_DebugPrivate.h>
#include "testFs_FsLib_AccessLog.h"

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
namespace {
void EmptyNandSpace(void* pArgument) NN_NOEXCEPT
{
    NN_UNUSED(pArgument);

    // 3秒待ってからNAND 埋めを解除
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteAllPaddingFiles());
}}

// 初回容量不足警告、2度目の容量不足警告も続行して容量不足を示す Result が返ってくる
TEST(ExtendSaveDataWithPaddingFile, Failure)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00003E02, handle: 0x0000000000000000, priority: Normal, function: "ExtendSaveData", userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_size: 33554432, save_data_journal_size: 33554432 }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: System }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: 4 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: SdSystem }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: 3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountBis", name: "@User", bispartitionid: User, path: "(null)" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "@User" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x1, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00003E02, handle: 0x0000000000000000, priority: Normal, function: "ExtendSaveData", userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_size: 33554432, save_data_journal_size: 33554432 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteSaveData", savedataid: 0x1234 }
*/
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::oe::Initialize();
    nn::account::Initialize();
    nn::account::Uid user = nnt::fs::GetDefaultUid();

    // NAND 埋め
    {
        int64_t freeSpaceSize;
        const int64_t space = 16 * 1024 * 1024;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::User, nullptr));
        nn::fs::GetFreeSpaceSize(&freeSpaceSize, "@User:/");
        nn::fs::Unmount("@User");

        NNT_EXPECT_RESULT_SUCCESS(
            nn::fs::CreatePaddingFile(freeSpaceSize - space)
        );
    }
    NN_UTIL_SCOPE_EXIT
    {
        // NAND 埋めを解除
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteAllPaddingFiles());
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::ApplicationId, nnt::fs::GetDefaultFsUid(), 1, 0x10000, 0x10000, 0));
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(nnt::fs::GetSaveDataId(nnt::fs::GetDefaultFsUid())));
    };

    NN_LOG("press [Next]\n");
    int64_t size = 32 * 1024 * 1024;
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultUsableSpaceNotEnough,
        nn::fs::ExtendSaveData(user, size, size)
    );
}

// 初回容量不足警告後、容量不足を解消すると成功を示す Result が返ってくる
TEST(ExtendSaveDataWithPaddingFile, Success)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ExtendSaveData", userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_size: 33554432, save_data_journal_size: 33554432 }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: System }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: 4 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: SdSystem }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: 3 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountBis", name: "@User", bispartitionid: User, path: "(null)" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "@User" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x1, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ExtendSaveData", userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_size: 33554432, save_data_journal_size: 33554432 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteSaveData", savedataid: 0x1234 }
*/
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::oe::Initialize();
    nn::account::Initialize();
    nn::account::Uid user = nnt::fs::GetDefaultUid();

    // NAND 埋め
    {
        int64_t freeSpaceSize;
        const int64_t space = 16 * 1024 * 1024;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::BisPartitionId::User, nullptr));
        nn::fs::GetFreeSpaceSize(&freeSpaceSize, "@User:/");
        nn::fs::Unmount("@User");

        NNT_EXPECT_RESULT_SUCCESS(
            nn::fs::CreatePaddingFile(freeSpaceSize - space)
        );
    }
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::ApplicationId, nnt::fs::GetDefaultFsUid(), 1, 0x10000, 0x10000, 0));
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(nnt::fs::GetSaveDataId(nnt::fs::GetDefaultFsUid())));
    };

    // NAND 埋め解除スレッド起動
    nn::os::ThreadType thread;
    static NN_OS_ALIGNAS_THREAD_STACK char threadBuffer[16 * 1024];
    NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
        &thread,
        EmptyNandSpace,
        nullptr,
        threadBuffer,
        sizeof(threadBuffer),
        nn::os::DefaultThreadPriority
    ));
    nn::os::StartThread(&thread);

    NN_LOG("After 3 second, press [Next]\n");
    int64_t size = 32 * 1024 * 1024;
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::ExtendSaveData(user, size, size));

    nn::os::WaitThread(&thread);
    nn::os::DestroyThread(&thread);
}

#endif
