﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>

#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>

#include "testFs_FsLib_AccessLog.h"

namespace{
void DeleteSystemSaveData(const nn::fs::SystemSaveDataId systemSaveDataId) NN_NOEXCEPT
{
    nn::util::optional<nn::fs::SaveDataInfo> saveDataInfo;
    nnt::fs::util::FindSaveData(
        &saveDataInfo,
        nn::fs::SaveDataSpaceId::System,
        [&](const nn::fs::SaveDataInfo info)
        {
            return info.systemSaveDataId == systemSaveDataId;
        });
    if (saveDataInfo != nn::util::nullopt)
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(saveDataInfo->saveDataId));
    }
}
}

TEST(EnsureSaveData, Success)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnsureSaveData", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "EnsureSaveData", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteSaveData", savedataid: 0x1234 }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnsureSaveData", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::EnsureSaveData", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteSaveData", savedataid: 0x1234 }
*/
{
    nn::account::Uid uid = nnt::fs::GetDefaultUid();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::EnsureSaveData(uid));

    nn::fs::SaveDataId saveDataId;
#if defined(NN_BUILD_CONFIG_OS_WIN)
    // Win版では、EnsureSaveData で作成したセーブデータは、ApplicationId を用いてセーブデータ ID を取得する必要がある
    nn::util::optional<nn::fs::SaveDataInfo> saveDataInfo;
    nnt::fs::util::FindSaveData(
        &saveDataInfo,
        nn::fs::SaveDataSpaceId::User,
        [&](const nn::fs::SaveDataInfo info)
        {
            return info.applicationId == nnt::fs::ApplicationId;
        });
    ASSERT_TRUE(saveDataInfo != nn::util::nullopt);
    saveDataId = saveDataInfo->saveDataId;
#else
    saveDataId = nnt::fs::GetSaveDataId(uid);
#endif
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(saveDataId));
}

TEST(EnsureSaveData, DISABLED_Failure)
{}

TEST(CreateDeleteSaveData, Success)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x1, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteSaveData", savedataid: 0x1234 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x1, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteSaveData", savedataid: 0x1234 }
*/
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::ApplicationId, nnt::fs::GetDefaultFsUid(), 1, 0x10000, 0x10000, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(nnt::fs::GetSaveDataId(nnt::fs::GetDefaultFsUid())));
}

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
TEST(CreateDeleteSaveData, Failure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00345802, handle: 0x0000000000000000, priority: Normal, function: "CreateSaveData", applicationid: 0x9876543210987654, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x1, save_data_size: 4096, save_data_journal_size: 4096, save_data_flags: 0x00000002 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "DeleteSaveData", savedataid: 0x1234 }
*/
{
    ASSERT_TRUE(nn::fs::CreateSaveData(nnt::fs::InvalidApplicationId, nnt::fs::GetDefaultFsUid(), 1, 0x1000, 0x1000, 0x2).IsFailure());
    ASSERT_TRUE(nn::fs::DeleteSaveData(0x1000).IsFailure());
}
#endif

TEST(CreateDeleteSystemSaveData, Success)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateSystemSaveData", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x0, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteSystemSaveData", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateSystemSaveData", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x0, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteSystemSaveData", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
*/
{
    const nn::fs::SystemSaveDataId systemSaveDataId = 0x8000000000004000;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(nn::fs::SaveDataSpaceId::System, systemSaveDataId, nnt::fs::GetDefaultFsUid(), 0, 0x10000, 0x10000, 0));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSystemSaveData(nn::fs::SaveDataSpaceId::System, systemSaveDataId, nnt::fs::GetDefaultFsUid()));
}

TEST(CreateDeleteSystemSaveData, Failure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "CreateSystemSaveData", savedataspaceid: User, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x0, save_data_size: 4096, save_data_journal_size: 4096, save_data_flags: 0x00000002 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "DeleteSystemSaveData", savedataspaceid: User, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateSystemSaveData", savedataspaceid: User, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x0, save_data_size: 4096, save_data_journal_size: 4096, save_data_flags: 0x00000002 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteSystemSaveData", savedataspaceid: User, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
*/
{
    ASSERT_TRUE(nn::fs::CreateSystemSaveData(nn::fs::SaveDataSpaceId::User, 1, nnt::fs::GetDefaultFsUid(), 0, 0x1000, 0x1000, 0x2).IsFailure());
    ASSERT_TRUE(nn::fs::DeleteSystemSaveData(nn::fs::SaveDataSpaceId::User, 1, nnt::fs::GetDefaultFsUid()).IsFailure());
}

TEST(MountSaveData, Success)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x1, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteSaveData", savedataid: 0x1234 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateSaveData", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x1, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: User }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteSaveData", savedataid: 0x1234 }
*/
{
    nn::fs::UserId userId = nnt::fs::GetDefaultFsUid();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSaveData(nnt::fs::ApplicationId, userId, 1, 0x10000, 0x10000, 0));
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(nnt::fs::GetSaveDataId(userId)));
    };

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(nnt::fs::MountName, userId));
    nn::fs::Unmount(nnt::fs::MountName);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSaveData(nnt::fs::MountName, nnt::fs::ApplicationId, userId));
    nn::fs::Unmount(nnt::fs::MountName);
#endif
}

TEST(MountSaveData, Failure)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", applicationid: 0x9876543210987654, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "MountSaveData", name: "test", applicationid: 0x9876543210987654, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", applicationid: 0x9876543210987654, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveData", name: "test", applicationid: 0x9876543210987654, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
*/
{
    nn::fs::UserId userId = nnt::fs::GetDefaultFsUid();
    ASSERT_TRUE(nn::fs::MountSaveData(nnt::fs::MountName, userId).IsFailure());
    ASSERT_TRUE(nn::fs::MountSaveData(nnt::fs::MountName, nnt::fs::InvalidApplicationId, userId).IsFailure());
}

TEST(MountSaveDataReadOnly, Success)
{}

TEST(MountSaveDataReadOnly, Failure)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "MountSaveDataReadOnly", name: "test", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "MountSaveDataReadOnly", name: "test", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
AccessLogTest: Generic
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveDataReadOnly", name: "test", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x0007D402, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSaveDataReadOnly", name: "test", applicationid: 0x100000000003006, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
*/
{
    ASSERT_TRUE(nn::fs::MountSaveDataReadOnly(nnt::fs::MountName, nnt::fs::ApplicationId, nnt::fs::GetDefaultFsUid()).IsFailure());
}

TEST(SystemMountSystemSaveData, Success)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: System }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "CreateSystemSaveData", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x0, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSystemSaveData", name: "test", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSystemSaveData", name: "test", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSystemSaveData", name: "test", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "MountSystemSaveData", name: "test", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: System }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "DeleteSaveData", savedataid: 0x1234 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: System }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::CreateSystemSaveData", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234, save_data_owner_id: 0x0, save_data_size: 65536, save_data_journal_size: 65536, save_data_flags: 0x00000000 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSystemSaveData", name: "test", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSystemSaveData", name: "test", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSystemSaveData", name: "test", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSystemSaveData", name: "test", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::Unmount", name: "test" }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: System }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::DeleteSaveData", savedataid: 0x1234 }
*/
{
    const nn::fs::SystemSaveDataId systemSaveDataId = 0x8000000000004000;
    DeleteSystemSaveData(systemSaveDataId);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(systemSaveDataId, 0x10000, 0x10000, 0));
    NN_UTIL_SCOPE_EXIT
    {
        DeleteSystemSaveData(systemSaveDataId);
    };
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSystemSaveData(nnt::fs::MountName, systemSaveDataId));
    nn::fs::Unmount(nnt::fs::MountName);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSystemSaveData(nnt::fs::MountName, systemSaveDataId, nn::fs::InvalidUserId));
    nn::fs::Unmount(nnt::fs::MountName);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSystemSaveData(nnt::fs::MountName, nn::fs::SaveDataSpaceId::System, systemSaveDataId));
    nn::fs::Unmount(nnt::fs::MountName);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSystemSaveData(nnt::fs::MountName, nn::fs::SaveDataSpaceId::System, systemSaveDataId, nn::fs::InvalidUserId));
    nn::fs::Unmount(nnt::fs::MountName);
}

TEST(SystemMountSystemSaveData, Failure)
/*
AccessLogTest: NX
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "OpenSaveDataIterator", savedataspaceid: System }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "MountSystemSaveData", name: "test", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "MountSystemSaveData", name: "test", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "MountSystemSaveData", name: "test", savedataspaceid: SdSystem, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "MountSystemSaveData", name: "test", savedataspaceid: User, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
AccessLogTest: Generic
AccessLogTest: Generic-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::OpenSaveDataIterator", savedataspaceid: System }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::SaveDataIterator::ReadSaveDataInfo", size: 1 }
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSystemSaveData", name: "test", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSystemSaveData", name: "test", savedataspaceid: System, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSystemSaveData", name: "test", savedataspaceid: SdSystem, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x002EE202, handle: 0x0000000000000000, priority: Normal, function: "nn::fs::MountSystemSaveData", name: "test", savedataspaceid: User, savedataid: 0x1234, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
*/
{
    const nn::fs::SystemSaveDataId systemSaveDataId = 1;
    DeleteSystemSaveData(systemSaveDataId);
    ASSERT_TRUE(nn::fs::MountSystemSaveData(nnt::fs::MountName, systemSaveDataId).IsFailure());
    ASSERT_TRUE(nn::fs::MountSystemSaveData(nnt::fs::MountName, systemSaveDataId, nnt::fs::GetDefaultFsUid()).IsFailure());
    ASSERT_TRUE(nn::fs::MountSystemSaveData(nnt::fs::MountName, nn::fs::SaveDataSpaceId::SdSystem, systemSaveDataId).IsFailure());
    ASSERT_TRUE(nn::fs::MountSystemSaveData(nnt::fs::MountName, nn::fs::SaveDataSpaceId::User, systemSaveDataId, nnt::fs::GetDefaultFsUid()).IsFailure());
}
