﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <algorithm>
#include <nn/nn_Allocator.h>
#include <nn/util/util_IntUtil.h>
#include <nn/fs/fs_QueryRange.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util_VirtualMemoryStorage.h>

namespace {
    class StandardHeapAllocator : public nn::MemoryResource
    {
    private:
        void* do_allocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT
        {
            NN_UNUSED(alignment);
            return new(std::nothrow) char[bytes];
        }

        void do_deallocate(void* p, std::size_t bytes, std::size_t alignment) NN_NOEXCEPT
        {
            NN_UNUSED(bytes);
            NN_UNUSED(alignment);
            delete[] reinterpret_cast<char*>(p);
        }

        virtual bool do_is_equal(const MemoryResource& other) const NN_NOEXCEPT
        {
            return this == &other;
        }
    };

    StandardHeapAllocator g_StandardAllocator;
}

namespace nnt { namespace fs { namespace util {

NN_DEFINE_STATIC_CONSTANT(const size_t VirtualMemoryStorage::BlockSize);

nn::MemoryResource* VirtualMemoryStorage::g_pAllocator = &g_StandardAllocator;

void VirtualMemoryStorage::SetAllocator(nn::MemoryResource* pAllocator) NN_NOEXCEPT
{
    if( pAllocator == nullptr )
    {
        g_pAllocator = &g_StandardAllocator;
    }
    else
    {
        g_pAllocator = pAllocator;
    }
}

void VirtualMemoryStorage::DeleteBuffer(void* buffer) NN_NOEXCEPT
{
    g_pAllocator->deallocate(buffer, 0);
}

/**
 * @brief   コンストラクタです。
 */
VirtualMemoryStorage::VirtualMemoryStorage() NN_NOEXCEPT
    : m_Size(0)
    , m_Mutex(true)
    , m_List()
    , m_IsInitialized(false)
{
}

/**
 * @brief   コンストラクタです。
 *
 * @param[in]   storageSize ストレージのサイズ
 */
VirtualMemoryStorage::VirtualMemoryStorage(int64_t storageSize) NN_NOEXCEPT
    : m_Size(0)
    , m_Mutex(true)
    , m_List()
    , m_IsInitialized(false)
{
    Initialize(storageSize);
}

/**
 * @brief   初期化をします。
 *
 * @param[in]   storageSize ストレージのサイズ
 */
void VirtualMemoryStorage::Initialize(int64_t storageSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(storageSize, 0);

    std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

    NN_SDK_REQUIRES(!m_IsInitialized);

    m_Size = storageSize;
    m_IsInitialized = true;
}

/**
 * @brief   終了処理をします。
 */
void VirtualMemoryStorage::Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

    if( m_IsInitialized )
    {
        while( !m_List.empty() )
        {
            auto iter = m_List.begin();
            auto* ptr = &(*iter);

            m_List.erase(iter);
            g_pAllocator->deallocate(const_cast<Block*>(ptr), sizeof(Block));
        }

        m_Size = 0;
        m_IsInitialized = false;
    }
}

/**
 * @brief   ストレージの内容をバッファに読み込みます。
 *
 * @param[in]   offset  読み込み開始位置
 * @param[out]  buffer  読み込んだ内容をコピーするバッファ
 * @param[in]   size    読み込むデータサイズ
 *
 * @return  関数の処理結果を返します。
 */
nn::Result VirtualMemoryStorage::Read(
    int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    if( buffer == nullptr )
    {
        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        // バッファの指定が不正
        return nn::fs::ResultInvalidArgument();
    }

    std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

    if( offset < 0 || m_Size < offset )
    {
        // オフセットの指定が不正
        return nn::fs::ResultInvalidOffset();
    }

    // NOTE: offset + size をするとオーバーフローする可能性があるので uint64_t にキャストしておく
    if( !nn::util::IsIntValueRepresentable<int64_t>(size) ||
        static_cast<uint64_t>(m_Size) < static_cast<uint64_t>(offset) + size )
    {
        // サイズが大きすぎる
        return nn::fs::ResultInvalidSize();
    }

    if( 0 < size )
    {
        const int64_t firstBlock = offset / BlockSize;
        const int64_t lastBlock = (offset + size - 1) / BlockSize;

        for( auto i = firstBlock; i <= lastBlock; ++i )
        {
            const int64_t blockOffset = i * BlockSize;

            auto pos = std::find_if(m_List.begin(), m_List.end(), BlockFinder(blockOffset));

            // 読み込むデータなし
            if( pos == m_List.end() )
            {
                ClearBuffer(buffer, offset, size, blockOffset);
            }
            else
            {
                ReadBuffer(buffer, offset, size, *pos);
            }
        }
    }

    NN_RESULT_SUCCESS;
}

/**
 * @brief   バッファの内容をストレージに書き込みます。
 *
 * @param[in]   offset  書き込み開始位置
 * @param[in]   buffer  書き込むデータ
 * @param[in]   size    書き込むデータサイズ
 *
 * @return  関数の処理結果を返します。
 */
nn::Result VirtualMemoryStorage::Write(
    int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
{
    if( buffer == nullptr )
    {
        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        // バッファの指定が不正
        return nn::fs::ResultInvalidArgument();
    }

    std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

    if( offset < 0 || m_Size < offset )
    {
        // オフセットの指定が不正
        return nn::fs::ResultInvalidOffset();
    }

    if( !nn::util::IsIntValueRepresentable<int64_t>(size) ||
        static_cast<uint64_t>(m_Size) < static_cast<uint64_t>(offset) + size )
    {
        // サイズが大きすぎる
        return nn::fs::ResultInvalidSize();
    }

    if( 0 < size )
    {
        const int64_t firstBlock = offset / BlockSize;
        const int64_t lastBlock = (offset + size - 1) / BlockSize;

        for( auto i = firstBlock; i <= lastBlock; ++i )
        {
            const int64_t blockOffset = i * BlockSize;

            auto pos = std::find_if(m_List.begin(), m_List.end(), BlockFinder(blockOffset));

            // 書き込み先のブロックがなければ新しくブロックを作る
            if( pos == m_List.end() )
            {
                pos = CreateBlock(blockOffset);
                if( pos == m_List.end() )
                {
                    return nn::fs::ResultAllocationMemoryFailed();
                }
            }

            WriteBuffer(buffer, offset, size, *pos);
        }
    }

    NN_RESULT_SUCCESS;
}

/**
 * @brief   指定したバッファをクリア（0 埋め）します。
 *
 * @param[in]   buffer      クリアするバッファ
 * @param[in]   offset      ストレージの先頭からの開始位置
 * @param[in]   size        クリアするサイズ
 * @param[in]   blockOffset 指定した領域を含むブロックのオフセット
 */
void VirtualMemoryStorage::ClearBuffer(
    void* buffer, int64_t offset, size_t size, int64_t blockOffset) NN_NOEXCEPT
{
    const auto stride = blockOffset - offset;

    // BlockSize アライメントを上回る部分をクリア
    if( 0 <= stride )
    {
        const auto clearSize = std::min(BlockSize, static_cast<size_t>(size - stride));

        std::memset(reinterpret_cast<char*>(buffer) + stride, 0, clearSize);
    }
    // BlockSize アライメントを下回る部分をクリア
    else
    {
        const auto clearSize = std::min(size, static_cast<size_t>(BlockSize + stride));

        std::memset(buffer, 0, clearSize);
    }
}

/**
 * @brief   指定したバッファにストレージからデータをコピーします。
 *
 * @param[in]   buffer  コピー先のバッファ
 * @param[in]   offset  ストレージの先頭からの開始位置
 * @param[in]   size    コピーするサイズ
 * @param[in]   block   指定した領域を含むブロック
 */
void VirtualMemoryStorage::ReadBuffer(
    void* buffer, int64_t offset, size_t size, const Block& block) NN_NOEXCEPT
{
    const char* const src = block.GetBuffer().get();
    char* const dst = static_cast<char*>(buffer);
    const auto stride = block.GetOffset() - offset;

    // BlockSize アライメントを上回る部分からコピー
    if( 0 <= stride )
    {
        const auto copySize = std::min(BlockSize, static_cast<size_t>(size - stride));

        std::memcpy(dst + stride, src, copySize);
    }
    // BlockSize アライメントを下回る部分からコピー
    else
    {
        auto copySize = std::min(size, static_cast<size_t>(BlockSize + stride));

        std::memcpy(dst, src - stride, copySize);
    }
}

/**
 * @brief   指定したバッファからストレージにデータをコピーします。
 *
 * @param[in]   buffer  コピー先元のバッファ
 * @param[in]   offset  ストレージの先頭からの開始位置
 * @param[in]   size    コピーするサイズ
 * @param[in]   block   指定した領域を含むブロック
 */
void VirtualMemoryStorage::WriteBuffer(
    const void* buffer, int64_t offset, size_t size, const Block& block) NN_NOEXCEPT
{
    const char* const src = static_cast<const char*>(buffer);
    char* const dst = block.GetBuffer().get();
    const auto stride = block.GetOffset() - offset;

    // BlockSize アライメントを上回る部分からコピー
    if( 0 <= stride )
    {
        const auto copySize = std::min(BlockSize, static_cast<size_t>(size - stride));

        std::memcpy(dst, src + stride, copySize);
    }
    // BlockSize アライメントを下回る部分からコピー
    else
    {
        const auto copySize = std::min(size, static_cast<size_t>(BlockSize + stride));

        std::memcpy(dst - stride, src, copySize);
    }
}

/**
 * @brief   ブロックを作成します。
 *
 * @param[in]   offset  ストレージの先頭からの開始位置
 *
 * @return  ブロックを表すイテレータ
 */
VirtualMemoryStorage::BlockList::iterator VirtualMemoryStorage::CreateBlock(
    int64_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(nn::util::is_aligned(offset, BlockSize));

    BlockBuffer buffer = BlockBuffer(reinterpret_cast<char*>(g_pAllocator->allocate(BlockSize)), DeleteBuffer);
    if( buffer.get() != nullptr )
    {
        Block* pBlock = new(g_pAllocator->allocate(sizeof(Block))) Block(offset, std::move(buffer));
        if( pBlock != nullptr )
        {
            // 確保したバッファは 0 クリアしておく
            std::memset(pBlock->GetBuffer().get(), 0, BlockSize);

            m_List.push_back(*pBlock);

            return --m_List.end();
        }
    }
    return m_List.end();
}

/**
 * @brief   特定のメモリ領域に既定の操作を行います。
 *
 * @param[out]  outBuffer       範囲指定処理の結果を格納するバッファ
 * @param[in]   outBufferSize   範囲指定処理の結果を格納するバッファのサイズ
 * @param[in]   operationId     実行する操作
 * @param[in]   offset          メモリ領域の開始位置
 * @param[in]   size            メモリ領域のサイズ
 * @param[in]   inBuffer        範囲指定処理に渡すバッファ
 * @param[in]   inBufferSize    範囲指定処理に渡すバッファのサイズ
 *
 * @return  関数の処理結果を返します。
 */
nn::Result VirtualMemoryStorage::OperateRange(
    void* outBuffer,
    size_t outBufferSize,
    nn::fs::OperationId operationId,
    int64_t offset,
    int64_t size,
    const void* inBuffer,
    size_t inBufferSize) NN_NOEXCEPT
{
    NN_UNUSED(outBuffer);
    NN_UNUSED(outBufferSize);
    NN_UNUSED(inBuffer);
    NN_UNUSED(inBufferSize);

    if( operationId == nn::fs::OperationId::QueryRange )
    {
        // QueryRange テストのために成功するように
        NN_RESULT_THROW_UNLESS(outBuffer != nullptr, nn::fs::ResultNullptrArgument());
        NN_RESULT_THROW_UNLESS(outBufferSize == sizeof(nn::fs::QueryRangeInfo), nn::fs::ResultInvalidSize());

        reinterpret_cast<nn::fs::QueryRangeInfo*>(outBuffer)->Clear();
        NN_RESULT_SUCCESS;
    }

    if( operationId != nn::fs::OperationId::FillZero )
    {
        // 0 埋め以外はサポート対象外
        return nn::fs::ResultUnsupportedOperation();
    }

    std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

    if( offset < 0 || m_Size < offset )
    {
        // オフセットの指定が不正
        return nn::fs::ResultInvalidOffset();
    }

    if( static_cast<uint64_t>(m_Size) < static_cast<uint64_t>(offset) + size )
    {
        // サイズが大きすぎる
        return nn::fs::ResultInvalidSize();
    }

    if( 0 < size )
    {
        const int64_t firstBlock = offset / BlockSize;
        const int64_t lastBlock = (offset + size - 1) / BlockSize;

        for( auto i = firstBlock; i <= lastBlock; ++i )
        {
            const int64_t blockOffset = i * BlockSize;

            auto pos = std::find_if(
                m_List.begin(), m_List.end(), BlockFinder(blockOffset));

            // クリアするブロックあり
            if( pos != m_List.end() )
            {
                const auto stride = blockOffset - offset;
                const auto& block = *pos;

                // BlockSize アライメントを上回る部分のクリア or 削除
                if( 0 <= stride )
                {
                    // ブロックが 0 埋めする範囲内に納まるので削除する
                    if( block.GetOffset() + static_cast<int64_t>(BlockSize) <= offset + size )
                    {
                        m_List.erase(pos);

                        g_pAllocator->deallocate(const_cast<Block*>(&block), sizeof(Block));
                    }
                    // ブロックの内部に 0 埋めする範囲があるので 0 埋めする
                    else
                    {
                        const auto clearSize = static_cast<size_t>(size - stride);

                        std::memset(block.GetBuffer().get(), 0, clearSize);
                    }
                }
                // BlockSize アライメントを下回る部分から 0 埋め（ブロックは削除できない）
                else
                {
                    const auto clearSize = std::min(
                        static_cast<size_t>(size), static_cast<size_t>(BlockSize + stride));

                    std::memset(block.GetBuffer().get() - stride, 0, clearSize);
                }
            }
        }
    }

    NN_RESULT_SUCCESS;
}

/**
 * @brief   ストレージのサイズを設定します。
 *
 * @param[in]   size    新しいサイズ
 *
 * @return  関数の処理結果を返します。
 */
nn::Result VirtualMemoryStorage::SetSize(int64_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(size, 0);

    std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

    if( size < m_Size )
    {
        NN_RESULT_DO(OperateRange(nn::fs::OperationId::FillZero, size, m_Size - size));
    }

    m_Size = size;

    NN_RESULT_SUCCESS;
}

}}}
