﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <numeric>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nn/util/util_FormatString.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

namespace nnt { namespace fs { namespace api {
    void LoadSizeArgumentRoTests()
    {
        return;
    }

namespace{

    struct ParamsStructure{
        int64_t fileSize;
        int64_t offset;
        int64_t size;
    };

    void MakeExpectImage(char* pBuffer, const size_t bufferSize) NN_NOEXCEPT
    {
        for (unsigned int cnt = 0; cnt < bufferSize; cnt++)
        {
            pBuffer[cnt] = cnt & 0xffU;
        }
    }

    enum class AccessRange
    {
        InRange,
        OutOfRange
    };

    typedef nn::Result(*ReadFunction)(size_t* outValue, nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, AccessRange accessRange, size_t expectedSize);

    nn::Result ReadFunction0(size_t* outValue, nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, AccessRange accessRange, size_t expectedSize) NN_NOEXCEPT
    {
        NN_UNUSED(size);

        if( (accessRange == AccessRange::InRange) && (size != expectedSize) )
        {
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultOutOfRange,
                pTestFile->Read(offset, buffer, size)
            );
        }

        auto accessSize = ((accessRange == AccessRange::OutOfRange && size > 0) ? size : expectedSize);
        NN_RESULT_DO(pTestFile->Read(offset, buffer, accessSize, nn::fs::ReadOption()));

        *outValue = expectedSize;

        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction1(size_t* outValue, nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, AccessRange accessRange, size_t expectedSize) NN_NOEXCEPT
    {
        NN_UNUSED(size);

        if( (accessRange == AccessRange::InRange) && (size != expectedSize) )
        {
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultOutOfRange,
                pTestFile->Read(offset, buffer, size)
            );
        }

        auto accessSize = ((accessRange == AccessRange::OutOfRange && size > 0) ? size : expectedSize);
        NN_RESULT_DO(pTestFile->Read(offset, buffer, accessSize));

        *outValue = expectedSize;

        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction2(size_t* outValue, nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, AccessRange accessRange, size_t expectedSize) NN_NOEXCEPT
    {
        NN_UNUSED(accessRange);

        util::InvalidateVariable(outValue);
        NN_RESULT_DO(pTestFile->Read(outValue, offset, buffer, size, nn::fs::ReadOption()));
        EXPECT_EQ(expectedSize, *outValue);
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction3(size_t* outValue, nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, AccessRange accessRange, size_t expectedSize) NN_NOEXCEPT
    {
        NN_UNUSED(accessRange);

        util::InvalidateVariable(outValue);
        NN_RESULT_DO(pTestFile->Read(outValue, offset, buffer, size));
        EXPECT_EQ(expectedSize, *outValue);
        NN_RESULT_SUCCESS;
    }

    bool IsReadOverload(ReadFunction read) NN_NOEXCEPT
    {
        return read != ReadFunction2;
    }

    const ReadFunction ReadFunctions[] = {
        ReadFunction0,
        ReadFunction1,
        ReadFunction2,
        ReadFunction3
    };
}

    class SizeArgumentRoFileSize : public nnt::fs::api::GetFileSystemTestFixture,
        public ::testing::WithParamInterface<int64_t>
    {
    };

    class SizeArgumentRoFileReadWrite : public nnt::fs::api::GetFileSystemTestFixture,
        public ::testing::WithParamInterface<ParamsStructure>
    {
        enum ReadWriteType {
            ReadWriteType_Read,
            ReadWriteType_WriteRead
        };

    protected:
        void ReadTest(int64_t fileSize, int64_t offset, size_t size, ReadFunction read) NN_NOEXCEPT
        {
            ReadTestProcess(fileSize, offset, size, ReadWriteType_Read, read);
            FsApiTestDebugPrint("\n");
        }

        // ファイルオープン・Read・クローズの実施及び結果の照合
        void ReadTestProcess(int64_t fileSize, int64_t offset, size_t size, int type, ReadFunction read) NN_NOEXCEPT
        {
            NNT_FS_UTIL_SKIP_TEST_UNLESS(fileSize < GetFsAttribute()->storageSize &&
                offset < GetFsAttribute()->storageSize &&
                static_cast<int64_t>(size) < GetFsAttribute()->storageSize);

            char name[16];
            nn::util::SNPrintf(name, sizeof(name), "%lld.file", fileSize);
            String fileName = GetTestRootPath().append("/SizeArgumentRo/").append(name);

            // 大容量バッファ動的確保時のサイズ調整
            size_t       workSize = size;                           // バッファサイズ
            int64_t      remainSize = size;                         // 残数値
            const size_t ThresholdValue = 256 * 1024 * 1024;        // 大容量確認の閾値
            if(size > ThresholdValue){
                workSize = ThresholdValue;
                FsApiTestDebugPrint("  size changed : 0x%x to 0x%x \n",
                    static_cast<uint32_t>(size),
                    static_cast<uint32_t>(workSize));
            }

            const auto allocateSize = (workSize == 0 ? 1 : workSize);
            auto readBuffer(nnt::fs::util::AllocateBuffer(allocateSize));
            if(readBuffer.get() == nullptr)
            {
                ADD_FAILURE();
                NN_LOG("Buffer Area Out Of Memory Allocate Size 0x%x\n",
                    static_cast<uint32_t>(workSize));
                return;
            }

            // ファイルオープン
            std::unique_ptr<ITestFile> file;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), OpenMode_Read));

            // 大容量バッファ対応（バッファサイズを丸めた際、繰り返すためのループ処理）
            do
            {
                size_t readSize = 0;
                util::InvalidateVariable(readBuffer.get(), static_cast<int>(workSize));
                // read確認（fileSize < (offset + size)
                if(type == ReadWriteType_Read)
                {
                    if(fileSize < offset)
                    {
                        // （無効なファイルポジション時）fileSize < offset の場合は、常に ResultOutOfRange が返ること。
                        FsApiTestDebugPrint(
                            "  fileSize = 0x%llx \toffset = 0x%llx \tsize = 0x%x ",
                            fileSize,
                            offset,
                            static_cast<uint32_t>(workSize)
                        );
                        readSize = 0;
                        const auto result = read(
                            &readSize, file.get(), offset, readBuffer.get(), workSize, AccessRange::OutOfRange, 0);
                        if( workSize == 0 )
                        {
                            NNT_EXPECT_RESULT_SUCCESS(result);
                        }
                        else
                        {
                            NNT_EXPECT_RESULT_FAILURE(ResultOutOfRange, result);
                        }
                        FsApiTestDebugPrint(
                            "(read(0x%x)) %s \n",
                            static_cast<uint32_t>(readSize),
                            size == workSize ? "" : "(loop)"
                        );
                    }
                    else
                    {   // （正常なファイルポジション）fileSize >= offset の場合
                        FsApiTestDebugPrint(
                            "  fileSize = 0x%llx \toffset = 0x%llx \tsize = 0x%x ",
                            fileSize,
                            offset,
                            static_cast<uint32_t>(workSize)
                        );
                        size_t expectedSize = 0;
                        if( fileSize >= static_cast<int64_t>(offset + workSize) )
                        {
                            // （実行後ファイルサイズに収まるケース）fileSize >= (offset + workSize) の場合は、workSize と readSize が一致すること。
                            expectedSize = workSize;
                        }
                        else
                        {
                            // （実行後ファイルサイズを超えるケース）fileSize < (offset + workSize) の場合で、
                            //   offset > fileSize の場合は、0、
                            //   offset <= fileSize の場合は、(fileSize - offset) 分が read されること。
                            expectedSize = fileSize <= offset ? 0 : static_cast<size_t>(fileSize - offset);
                        }
                        readSize = 0;
                        NNT_EXPECT_RESULT_SUCCESS(
                            read(&readSize, file.get(), offset, readBuffer.get(), workSize, AccessRange::InRange, expectedSize)
                        );
                        FsApiTestDebugPrint(
                            "(read(0x%x)) %s \n",
                            static_cast<uint32_t>(readSize),
                            size == workSize ? "" : "(loop)"
                        );
                    }
                }
                // 残数値更新
                remainSize = remainSize - workSize;
                offset = offset + workSize;
                workSize = static_cast<size_t>(remainSize) < workSize ?
                               static_cast<size_t>(remainSize) : workSize;

            } while (remainSize > 0);  // end of while

            file.reset(nullptr);
        }
    };

    class SizeArgumentRoVariousAccess : public nnt::fs::api::GetFileSystemTestFixture, public ::testing::WithParamInterface<ReadFunction>
    {
    public:
        SizeArgumentRoVariousAccess() NN_NOEXCEPT
            : m_File()
            , m_FilePath()
            , m_FileSize(0)
            , m_WorkBuffer(nullptr, nnt::fs::util::DeleterBuffer)
            , m_WorkBufferSize(0)
        {
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::SetUp();

            // 一番サイズが大きいファイルを探す
            nnt::fs::util::String directoryPath = GetTestRootPath().append("/SizeArgumentRo/");
            std::unique_ptr<ITestDirectory> directory;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenDirectory(
                &directory,
                directoryPath.c_str(),
                nn::fs::OpenDirectoryMode_File)); // 子ディレクトリは無い想定

            int64_t entryCount = 0;
            NNT_ASSERT_RESULT_SUCCESS(directory->GetEntryCount(&entryCount));
            auto entriesBuffer = nnt::fs::util::AllocateBuffer(static_cast<size_t>(sizeof(nn::fs::DirectoryEntry) * entryCount));
            auto entries = reinterpret_cast<nn::fs::DirectoryEntry*>(entriesBuffer.get());
            int64_t readCount = 0;
            NNT_ASSERT_RESULT_SUCCESS(directory->Read(&readCount, entries, entryCount));
            ASSERT_EQ(entryCount, readCount);

            int64_t maxFileSize = 0;
            const char* maxFileSizeName = nullptr;
            for( int i = 0; i < entryCount; ++i )
            {
                if( entries[i].fileSize > maxFileSize )
                {
                    maxFileSize = entries[i].fileSize;
                    maxFileSizeName = entries[i].name;
                }
            }
            ASSERT_NE(nullptr, maxFileSizeName);

            m_FileSize = maxFileSize;
            m_FilePath = directoryPath.append(maxFileSizeName);
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&m_File, m_FilePath.c_str(), nn::fs::OpenMode_Read));

            m_WorkBufferSize = static_cast<size_t>(std::min(static_cast<int64_t>(BufferSizeMax), m_FileSize));
            m_WorkBuffer = nnt::fs::util::AllocateBuffer(m_WorkBufferSize);
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            m_File.reset();
            m_FilePath = "";
            m_FileSize = 0;
            GetFileSystemTestFixture::TearDown();
        }

        ITestFile* GetFile() NN_NOEXCEPT
        {
            return m_File.get();
        }

        int64_t GetFileSize() const NN_NOEXCEPT
        {
            return m_FileSize;
        }

        // バッファが先頭から dataOffset + 0x00, 0x01, 0x02, ... で埋められていることを確認する
        void VerifyFilledData(const char* buffer, size_t size, int64_t dataOffset) NN_NOEXCEPT
        {
            // workBufferSize が 0x100 の倍数ならループごとに FillBuffer() する必要がない
            // workBufferSize >= size の場合はループしないので 0x100 の倍数でなくて良い
            ASSERT_TRUE(m_WorkBufferSize >= size || m_WorkBufferSize % 0x100 == 0);
            ASSERT_NE(nullptr, m_WorkBuffer);

            const size_t bufferSize = std::min(size, m_WorkBufferSize);
            FillBuffer(m_WorkBuffer.get(), bufferSize, dataOffset);

            for( size_t compareOffset = 0; compareOffset < size; compareOffset += bufferSize )
            {
                const size_t compareSize = std::min(bufferSize, static_cast<size_t>(size - compareOffset));
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(m_WorkBuffer.get(), buffer + compareOffset, compareSize);
            }
        }

    private:
        const size_t BufferSizeMax = 1 * 1024 * 1024;

    private:
        // バッファの内容を先頭から 0x00, 0x01, 0x02, ... で埋める
        void FillBuffer(char* buffer, size_t size, int64_t offset) NN_NOEXCEPT
        {
            std::iota(reinterpret_cast<uint8_t*>(buffer), reinterpret_cast<uint8_t*>(buffer + size), static_cast<uint8_t>(offset));
        }

    private:
        std::unique_ptr<ITestFile> m_File;
        nnt::fs::util::String m_FilePath;
        int64_t m_FileSize;
        std::unique_ptr<char, decltype(&nnt::fs::util::DeleterBuffer)> m_WorkBuffer;
        size_t m_WorkBufferSize;
    };

    typedef SizeArgumentRoFileReadWrite SizeArgumentRoOffsetAndSize;
    typedef SizeArgumentRoFileSize SizeArgumentRoOffsetAndSizeFull;
    typedef SizeArgumentRoFileSize SizeArgumentRoOffsetAndSizeIllegal;
    typedef SizeArgumentRoFileSize SizeArgumentRoEmpty;
    typedef nnt::fs::api::GetFileSystemTestFixture SizeArgumentRo;

    // TODO: ファイル作成部分の共通化

    //! @brief GetParam() サイズのファイルを、GetSize() でサイズを取得できること
    TEST_P(SizeArgumentRoFileSize, GetSize)
    {
        int64_t fileSize = GetParam();
        char name[16];
        nn::util::SNPrintf(name, sizeof(name), "%lld.file", fileSize);
        String fileName = GetTestRootPath().append("/SizeArgumentRo/").append(name);
        NNT_FS_UTIL_SKIP_TEST_UNLESS( fileSize < GetFsAttribute()->storageSize );

        std::unique_ptr<ITestFile> file;
        int64_t actualSize;
        NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), OpenMode_Read));
        NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&actualSize));
        FsApiTestDebugPrint("  fileSize = 0x%llx actualSize = 0x%llx \n", fileSize, actualSize);
        EXPECT_EQ(fileSize, actualSize);
        file.reset(nullptr);
    }

    //! @brief GetParam() サイズのファイルを、offset、size の組合せでReadが正しく動作すること
    TEST_P(SizeArgumentRoOffsetAndSize, ReadFile)
    {
        ParamsStructure fileParam = GetParam();

        for( auto read : ReadFunctions )
        {
            if( IsReadOverload(read) && !GetFsAttribute()->isReadOverloadsSupported )
            {
                continue;
            }

            ReadTest(fileParam.fileSize, fileParam.offset, static_cast<size_t>(fileParam.size), read);
        }
    }

    //! @brief GetParam() サイズのファイルを、1 バイト単位でファイル末尾まで正しく読み込めること
    TEST_P(SizeArgumentRoOffsetAndSizeFull, ReadFile)
    {
        const int BufferSize = 16 * 1024;

        int64_t fileSize = GetParam();
        NNT_FS_UTIL_SKIP_TEST_UNLESS( fileSize + BufferSize < GetFsAttribute()->fileSizeMax );
        NNT_FS_UTIL_SKIP_TEST_UNLESS( fileSize < GetFsAttribute()->storageSize );

        char name[16];
        nn::util::SNPrintf(name, sizeof(name), "%lld.file", fileSize);
        String fileName = GetTestRootPath().append("/SizeArgumentRo/").append(name);

        for( auto readFunction : ReadFunctions )
        {
            if( IsReadOverload(readFunction) && !GetFsAttribute()->isReadOverloadsSupported )
            {
                continue;
            }

            std::unique_ptr<ITestFile> file;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), OpenMode_Read));
            static char writeBuffer[BufferSize];
            MakeExpectImage(writeBuffer, sizeof(writeBuffer));

            size_t accessSizeArray[] = { 1 };
            for( auto accessSize : accessSizeArray )
            {
                static char readBuffer[BufferSize] = { 0 };
                size_t read = 0;
                while( read != static_cast<size_t>(fileSize) )
                {
                    size_t readSize = 0;
                    NNT_EXPECT_RESULT_SUCCESS(
                        readFunction(&readSize, file.get(), read, readBuffer + read, accessSize, AccessRange::InRange, accessSize)
                    );
                    read += readSize;
                }
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer, readBuffer, static_cast<size_t>(fileSize));
            }
        }
    }

    //! @brief 負のオフセットやファイルサイズより大きいオフセット指定で Read() を実行すると OutOfRange が返ること
    TEST_P(SizeArgumentRoOffsetAndSizeIllegal, ReadFile)
    {
        int64_t fileSize = GetParam();
        const int BufferSize = 32;
        char name[16];
        nn::util::SNPrintf(name, sizeof(name), "%lld.file", fileSize);
        String fileName = GetTestRootPath().append("/SizeArgumentRo/").append(name);

        for( auto read : ReadFunctions )
        {
            if( IsReadOverload(read) && !GetFsAttribute()->isReadOverloadsSupported )
            {
                continue;
            }

            std::unique_ptr<ITestFile> file;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), OpenMode_Read));
            char readBuffer[BufferSize] = { 0 };

            // 負のオフセットで Read
            size_t readSize = 0;
            NNT_EXPECT_RESULT_FAILURE(
                ResultOutOfRange,
                read(&readSize, file.get(), -1, readBuffer, sizeof(readBuffer), AccessRange::OutOfRange, static_cast<size_t>(0))
            );

            // 実際のファイルサイズを取得
            int64_t actualFileSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&actualFileSize));

            // ファイルサイズより大きいオフセットで Read
            readSize = 0;
            NNT_EXPECT_RESULT_FAILURE(
                ResultOutOfRange,
                read(&readSize, file.get(), actualFileSize + 1, readBuffer, sizeof(readBuffer), AccessRange::OutOfRange, static_cast<size_t>(0))
            );
        }
    }

    //! @brief サイズ 0 の読み込みが成功すること
    TEST_P(SizeArgumentRoEmpty, ReadFile)
    {
        const int64_t fileSize = GetParam();
        NNT_FS_UTIL_SKIP_TEST_UNLESS(fileSize < GetFsAttribute()->storageSize);

        char name[16];
        nn::util::SNPrintf(name, sizeof(name), "%lld.file", fileSize);
        const String fileName = GetTestRootPath().append("/SizeArgumentRo/").append(name);

        for( auto read : ReadFunctions )
        {
            if( IsReadOverload(read) && !GetFsAttribute()->isReadOverloadsSupported )
            {
                continue;
            }

            std::unique_ptr<ITestFile> file;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), OpenMode_Read));

            // サイズ 0 の場合はバッファが nullptr でも成功する
            size_t readSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(read(&readSize, file.get(), 0, nullptr, 0, AccessRange::InRange, static_cast<size_t>(0)));

            readSize = 0;
            auto buffer = AllocateBuffer(std::max<size_t>(static_cast<size_t>(fileSize), 16));
            NNT_EXPECT_RESULT_SUCCESS(read(&readSize, file.get(), 0, buffer.get(), 0, AccessRange::InRange, static_cast<size_t>(0)));
        }
    }

    int64_t param1[] = { 0, 32, 4096, 16384 };
    // 重いテストパラメータは無効化します。
#if 0
    int64_t param2[] = { 0xFFFFFFFF };
#endif
    int64_t param4[] = { 32 };
    int64_t param5[] = { 16384 };

    ParamsStructure offsetAndSizeParam[] = {
      // | fileSize  |   offset    |  size |
      //  -----------------------------------------
        { 0x0,        0x0,          0x0     },
        { 0x0,        0x0,          0x20    },
        { 0x0,        0x20,         0x0     },
        { 0x0,        0x20,         0x20    },
        { 0x20,       0x0,          0x0     },
        { 0x20,       0x0,          0x20    },
        { 0x20,       0x0,          0x10    },
        { 0x20,       0x0,          0x21    },
        { 0x20,       0x20,         0x0     },
        { 0x20,       0x20,         0x10    },
        { 0x20,       0x10,         0x0     },
        { 0x20,       0x10,         0x20    },
        { 0x20,       0x10,         0x10    },
        { 0x20,       0x21,         0x0     },
        { 0x20,       0x21,         0x20    },
        { 0x20,       0x40,         0x0     },
        { 0x20,       0x40,         0x20    },
        { 0x1000,     0x0,          0x0     },
        { 0x1000,     0x0,          0x1000  },
        { 0x1000,     0x0,          0x800   },
        { 0x1000,     0x0,          0x1001  },
        { 0x1000,     0x1000,       0x0     },
        { 0x1000,     0x1000,       0x800   },
        { 0x1000,     0x800,        0x0     },
        { 0x1000,     0x800,        0x20    },
        { 0x1000,     0x800,        0x1000  },
        { 0x1000,     0x800,        0x800   },
        { 0x1000,     0x1001,       0x0     },
        { 0x1000,     0x1001,       0x20    },
        { 0x4000,     0x0,          0x0     },
        { 0x4000,     0x0,          0x4000  },
        { 0x4000,     0x0,          0x2000  },
        { 0x4000,     0x0,          0x4001  },
        { 0x4000,     0x4000,       0x0     },
        { 0x4000,     0x4000,       0x2000  },
        { 0x4000,     0x2000,       0x0     },
        { 0x4000,     0x2000,       0x20    },
        { 0x4000,     0x2000,       0x4000  },
        { 0x4000,     0x2000,       0x2000  },
        { 0x4000,     0x4001,       0x0     },
        { 0x4000,     0x4001,       0x20    }
    };

#if 0
    ParamsStructure offsetAndSizeHeavyParam[] = {
      // | fileSize  |   offset    |   size   |
      //  -----------------------------------------
        { 0xFFFFFFFF, 0x0,          0x0        },
        { 0xFFFFFFFF, 0x0,          0xFFFFFFFF },
        { 0xFFFFFFFF, 0x0,          0x7FFFFFFF },
        { 0xFFFFFFFF, 0xFFFFFFFF,   0x0        },
        { 0xFFFFFFFF, 0xFFFFFFFF,   0x20       },
        { 0xFFFFFFFF, 0x100000000,  0x0        },
        { 0xFFFFFFFF, 0x100000000,  0x20       },
        { 0xFFFFFFFF, 0x7FFFFFFF,   0x0        },
        { 0xFFFFFFFF, 0x7FFFFFFF,   0x20       },
        { 0xFFFFFFFF, 0x7FFFFFFF,   0xFFFFFFFF },
        { 0xFFFFFFFF, 0x7FFFFFFF,   0x7FFFFFFF },
        { 0x0,        0x0,          0xFFFFFFFF },
        { 0x0200000000, 0x0100000000, 0x20 },
        { 0x0400000000, 0x0200000000, 0x20 },
        { 0x0800000000, 0x0400000000, 0x20 },
        { 0x1000000000, 0x0800000000, 0x20 }
    };
#endif

    INSTANTIATE_TEST_CASE_P(WithVariousSize,
                            SizeArgumentRoFileSize,
                            ::testing::ValuesIn(param1));
    INSTANTIATE_TEST_CASE_P(WithVariousSize,
                            SizeArgumentRoOffsetAndSize,
                            ::testing::ValuesIn(offsetAndSizeParam));

#if 0
    INSTANTIATE_TEST_CASE_P(WithVariousSizeHeavy,
                            SizeArgumentRoOffsetAndSize,
                            ::testing::ValuesIn(offsetAndSizeHeavyParam));
#endif

    INSTANTIATE_TEST_CASE_P(WithVariousSize,
                            SizeArgumentRoOffsetAndSizeFull,
                            ::testing::ValuesIn(param5));

    INSTANTIATE_TEST_CASE_P(WithVariousSize,
                            SizeArgumentRoOffsetAndSizeIllegal,
                            ::testing::ValuesIn(param4));
#if 0
    INSTANTIATE_TEST_CASE_P(WithVariousSizeHeavy,
                            SizeArgumentRoFileSize,
                            ::testing::ValuesIn(param2));
#endif
    INSTANTIATE_TEST_CASE_P(WithVariousSize,
                            SizeArgumentRoEmpty,
                            ::testing::ValuesIn(param1));


    class SizeArgumentRoTransferSize : public nnt::fs::api::GetFileSystemTestFixture, public ::testing::WithParamInterface<ReadFunction>
    {
    };

    //! @brief 0 ～ FileSizeMax のサイズでファイルの読み込みが出来ること
    TEST_P(SizeArgumentRoTransferSize, DISABLED_ReadFile_Heavy) // 非常に重いので基本無効
    {
        auto read = GetParam();
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        String fileName = GetTestRootPath().append("SizeArgumentRo/4294967295.file");

        const int FileSizeMax = 256 * 1024;
        const int Step = 1;

        for(int fileSize = 0; fileSize <= FileSizeMax; fileSize += Step)
        {
            std::unique_ptr<ITestFile> file;
            auto writeBuffer(nnt::fs::util::AllocateBuffer(fileSize));
            MakeExpectImage(writeBuffer.get(), sizeof(fileSize));

            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().OpenFile(&file, fileName.c_str(), static_cast<OpenMode>(OpenMode_Read))
            );
            auto readBuffer(nnt::fs::util::AllocateBuffer(fileSize));
            size_t readSize = 0;

            NNT_EXPECT_RESULT_SUCCESS(
                read(&readSize, file.get(), 0, readBuffer.get(), fileSize, AccessRange::InRange, fileSize)
            );
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), fileSize);
        }
    }

    INSTANTIATE_TEST_CASE_P(WithReadOverloads,
        SizeArgumentRoTransferSize,
        ::testing::ValuesIn(ReadFunctions));

    //! @brief 様々なパターンのオフセットとサイズで読み込む
    TEST_P(SizeArgumentRoVariousAccess, ReadFileWithPattern)
    {
        auto read = GetParam();
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        // 1 分以内でテストが終わるような大きさのパターンを用意
        // パターンはオフセット、サイズの領域を 32 等分したものと、それを + 1 したもの
        static const int OffsetPatternLength = 64;
        NN_STATIC_ASSERT(OffsetPatternLength % 2 == 0);
        int64_t offsetPatterns[OffsetPatternLength];
        for( int i = 0; i < OffsetPatternLength - 1; i += 2 )
        {
            offsetPatterns[i] = GetFileSize() / OffsetPatternLength * i;
            offsetPatterns[i + 1] = offsetPatterns[i] + 1;
        }

        const size_t bufferSize = static_cast<size_t>(std::min<int64_t>(2 * 1024 * 1024, GetFileSize()));
        auto buffer = nnt::fs::util::AllocateBuffer(bufferSize);

        static const int SizePatternLength = 64;
        NN_STATIC_ASSERT(SizePatternLength % 2 == 0);
        size_t sizePatterns[SizePatternLength];
        for( int i = 0; i < SizePatternLength - 1; i += 2 )
        {
            sizePatterns[i] = bufferSize / SizePatternLength * i;
            sizePatterns[i + 1] = sizePatterns[i] + 1;
        }

        for( auto size : sizePatterns )
        {
            for( auto offset : offsetPatterns )
            {
                const size_t accessSize = static_cast<size_t>(std::min(static_cast<int64_t>(size), GetFileSize() - offset));
                size_t readSize = 0;
                NNT_EXPECT_RESULT_SUCCESS(read(&readSize, GetFile(), offset, buffer.get(), accessSize, AccessRange::InRange, accessSize));
                VerifyFilledData(buffer.get(), readSize, offset);
            }
        }
    }

    //! @brief オフセット: 0 ～ 16 KB, サイズ 2 MB + 1 で読み込む
    TEST_P(SizeArgumentRoVariousAccess, ReadFileOffsetCoverage)
    {
        auto read = GetParam();
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        const size_t bufferSize = static_cast<size_t>(std::min<int64_t>(2 * 1024 * 1024 + 1, GetFileSize()));
        auto buffer = nnt::fs::util::AllocateBuffer(bufferSize);

        const int64_t offsetEnd = std::min<int64_t>(16 * 1024, static_cast<int64_t>(GetFileSize() - bufferSize));
        for( int64_t offset = 0; offset < offsetEnd; ++offset )
        {
            NNT_FS_SCOPED_TRACE("offset: %llx size: %llx\n", offset, static_cast<uint64_t>(bufferSize));
            size_t readSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(read(&readSize, GetFile(), offset, buffer.get(), bufferSize, AccessRange::InRange, bufferSize));
            VerifyFilledData(buffer.get(), bufferSize, offset);
        }
    }

    //! @brief オフセット: 16 KB + 1, サイズ 0 ～ 16 KB で読み込む
    TEST_P(SizeArgumentRoVariousAccess, ReadFileSizeCoverage)
    {
        auto read = GetParam();
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        const size_t bufferSize = static_cast<size_t>(std::min<int64_t>(16 * 1024, GetFileSize()));
        auto buffer = nnt::fs::util::AllocateBuffer(bufferSize);

        for( size_t size = 0; size < bufferSize; ++size )
        {
            const int64_t offset = std::min<int64_t>(16 * 1024 + 1, static_cast<int64_t>(GetFileSize() - size));
            NNT_FS_SCOPED_TRACE("offset: %llx size: %llx\n", offset, static_cast<uint64_t>(size));
            size_t readSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(read(&readSize, GetFile(), offset, buffer.get(), size, AccessRange::InRange, size));
            VerifyFilledData(buffer.get(), size, offset);
        }
    }

    INSTANTIATE_TEST_CASE_P(WithReadOverloads,
        SizeArgumentRoVariousAccess,
        ::testing::ValuesIn(ReadFunctions));

    //!< @brief OperateRange の QueryRange で不正なサイズを渡すと失敗することを確認する
    TEST_F(SizeArgumentRo, InvalidSize_OperateRange_QueryRange)
    {
        auto fileName = GetTestRootPath().append("/OperateRangeRo/test.file");

        // ファイルオープン
        std::unique_ptr<ITestFile> file;

        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().OpenFile(
                &file,
                fileName.c_str(),
                nn::fs::OpenMode::OpenMode_Read
            )
        );

        int64_t size;
        NNT_ASSERT_RESULT_SUCCESS(file->GetSize(&size));

        nn::fs::QueryRangeInfo info;

        if( GetFsAttribute()->isSupportedQueryRange )
        {
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize,
                file->OperateRange(&info, 0, nn::fs::OperationId::QueryRange, 0, size, nullptr, 0));
        }
        else
        {
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultUnsupportedOperation,
                file->OperateRange(&info, 0, nn::fs::OperationId::QueryRange, 0, size, nullptr, 0));
        }
    }
}}}
