﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/nn_Result.h>
#include <nnt/nntest.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nn/util/util_FormatString.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

/**
* @file
* @brief    ファイルシステム関数、リソースオープン観点を定義します。
*/
namespace {

    // テストファイル拡張子の定義
    const char* g_testFileExt       = ".file";

    // ディレクトリのエントリ名定義
    const char* g_testEntryFile     = "test.file";

    typedef nn::Result(*ReadFunction)(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size);

    nn::Result ReadFunction0(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(pTestFile->Read(offset, buffer, size, nn::fs::ReadOption()));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction1(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(pTestFile->Read(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction2(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        size_t sizeRead;
        nnt::fs::util::InvalidateVariable(&sizeRead);
        NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size, nn::fs::ReadOption()));
        EXPECT_EQ(size, sizeRead);
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction3(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        size_t sizeRead;
        nnt::fs::util::InvalidateVariable(&sizeRead);
        NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size));
        EXPECT_EQ(size, sizeRead);
        NN_RESULT_SUCCESS;
    }

    bool IsReadOverload(ReadFunction read) NN_NOEXCEPT
    {
        return read != ReadFunction2;
    }

    const ReadFunction ReadFunctions[] =
    {
        ReadFunction0,
        ReadFunction1,
        ReadFunction2,
        ReadFunction3
    };
}

namespace nnt { namespace fs { namespace api {
    void LoadResourceOpenCountRoTests()
    {
        return;
    }

    /**
     * @brief   ResourceOpenCountRo で利用するテストフィクスチャです。
     */
    class ResourceOpenCountRo : public GetFileSystemTestFixture, public ::testing::WithParamInterface<ReadFunction>
    {
    protected:

        // 各リソースサイズ定数定義
        static const int32_t nameMaxLength      = 260;
        static const int32_t openMaxfile        = 1024;
        static const int32_t openMaxDir         = 64;
        static const int32_t openMaxBufferSize  = 1024;
        static const int32_t openMaxEntrySize   = 2;

        // ファイル比較構造体
        struct ResourceFileInfo {
            char name[nameMaxLength];
        };

        // フォルダ比較構造体
        struct ResourceDirInfo {
            char   name[nameMaxLength];
        };

        ResourceFileInfo fileInfo[openMaxfile];
        ResourceDirInfo dirInfo[openMaxDir];

        // ファイルの書き込み／読み込みバッファ定義
        uint8_t writeBuffer[openMaxBufferSize];
        uint8_t readBuffer[openMaxBufferSize];

        // エントリバッファ定義
        DirectoryEntry readEntry[openMaxEntrySize];

        // ファイル／フォルダ作成の管理配列
        std::unique_ptr<ITestFile> file[openMaxfile];
        std::unique_ptr<ITestDirectory> dir[openMaxDir];

        String rootTestdir;

        /**
         * @brief テスト開始時に毎回呼び出される関数です。
         */
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::SetUp();

            // ルートディレクトリ設定・作成
            this->rootTestdir = GetTestRootPath();
            this->rootTestdir.append("/ResourceOpenCountRo/Test/");
        }

        /**
         * @brief テスト終了時に毎回呼び出される関数です。
         */
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            // ルートディレクトリ削除
            this->rootTestdir = "";

            GetFileSystemTestFixture::TearDown();
        }

        void MakeExpectImage(uint8_t* pBuffer, const size_t bufferSize) NN_NOEXCEPT
        {
            for (unsigned int cnt = 0; cnt < bufferSize; cnt++)
            {
                pBuffer[cnt] = cnt & 0xffU;
            }
        }

        /**
         * @brief テストファイルを作成する関数です
         */
        void CreateTestFiles()  NN_NOEXCEPT
        {
            char* pFilePath = nullptr;
            int32_t openMax = GetFsAttribute()->fileOpenMax;

            NN_ASSERT(this->openMaxfile >= openMax);

            for (int32_t i = 0; i < openMax; i++)
            {
                pFilePath = this->fileInfo[i].name;

                nn::util::SNPrintf(pFilePath, this->nameMaxLength, "%s%04d%s", this->rootTestdir.c_str(), i, g_testFileExt);
            }
        }

        /**
         * @brief テストファイルをオープンする関数です
         */
        void OpenTestFiles()  NN_NOEXCEPT
        {
            int32_t openMax     = 100;//GetFsAttribute()->fileOpenMax;

            NN_ASSERT(this->openMaxfile >= openMax);

            // 指定ファイル数までファイルをオープンします
            for (int32_t i = 0; i < openMax; i++)
            {
                NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&this->file[i],
                    this->fileInfo[i].name, OpenMode_Read));
            }
        }

        /**
         * @brief 開かれたファイルに対してRead()操作検証する関数です
         */
        void OpenedFilesOperateVerifyTest(ReadFunction read)  NN_NOEXCEPT
        {
            int32_t openMax     = 100;//GetFsAttribute()->fileOpenMax;

            NN_ASSERT(this->openMaxfile >= openMax);

            MakeExpectImage(this->writeBuffer, sizeof(this->writeBuffer));

            // 指定ファイル数までオープンしたファイルの読み込みデータが一致しているか確認します
            for (int32_t i = 0; i < openMax; i++)
            {
                EXPECT_TRUE(IsSpecifiedEntryTypeExist(this->fileInfo[i].name,
                    DirectoryEntryType::DirectoryEntryType_File) == true);

                util::InvalidateVariable(this->readBuffer, this->openMaxBufferSize);

                NNT_EXPECT_RESULT_SUCCESS(read(this->file[i].get(), 0, this->readBuffer,
                    this->openMaxBufferSize));

                NNT_FS_UTIL_EXPECT_MEMCMPEQ(this->writeBuffer, this->readBuffer, this->openMaxBufferSize);
            }
        }

        /**
         * @brief テストファイルをクローズする関数です
         */
        void CloseTestFiles()  NN_NOEXCEPT
        {
            int32_t openMax     = 100;//GetFsAttribute()->fileOpenMax;

            NN_ASSERT(this->openMaxfile >= openMax);

            // 指定ファイル数までオープンしたファイルを閉じます
            for (int32_t i = 0; i < openMax; i++)
            {
                this->file[i].reset(nullptr);
            }
        }

        /**
         * @brief 指定エントリタイプが存在するか確認する関数です
         *   @param[in]  name                   確認するパスを指定します
         *   @param[in]  entryType              確認するファイル／ディレクトリのエントリタイプを指定します
         *
         *   @return     指定エントリタイプの結果が返ります。
         *   @retval     true                   指定のエントリタイプで存在した場合に返ります。
         *   @retval     false                  指定のエントリタイプで存在しない場合に返ります。
         */
        bool IsSpecifiedEntryTypeExist(const String name, const DirectoryEntryType entryType)   NN_NOEXCEPT
        {
            DirectoryEntryType entry;

            // 指定パスのエントリタイプを取得します
            if (GetFs().GetEntryType(&entry, name.c_str()).IsSuccess() != true)
            {
                return false;
            }

            // 取得したエントリタイプの識別を判定します
            return ((entry & entryType) == entryType);
        }

        /**
         * @brief テストディレクトリを作成する関数です
         */
        void CreateTestDirectories()    NN_NOEXCEPT
        {
            int32_t openMax     = GetFsAttribute()->directoryOpenMax;
            String entryFile    = "";
            char* pFilePath     = nullptr;

            NN_ASSERT(this->openMaxDir >= openMax);

            // 最大オープン数まで重複しないユニークな名称でディレクトリを作成します
            for (int32_t i = 0; i < openMax; i++)
            {
                pFilePath = this->dirInfo[i].name;

                nn::util::SNPrintf(pFilePath, this->nameMaxLength, "%s%04d", this->rootTestdir.c_str(), i);

                // 作成したディレクトリ内にエントリ取得用ファイルのパスを作成します
                entryFile = pFilePath;
                entryFile.append("/");
                entryFile.append(g_testEntryFile);
            }
        }

        /**
         * @brief テストディレクトリをオープンする関数です
         */
        void OpenTestDirectories()    NN_NOEXCEPT
        {
            int32_t openMax     = GetFsAttribute()->directoryOpenMax;

            NN_ASSERT(this->openMaxDir >= openMax);

            // 指定ディレクトリ数までディレクトリをオープンします
            for (int32_t i = 0; i < openMax; i++)
            {
                NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&this->dir[i],
                    this->dirInfo[i].name, static_cast<OpenDirectoryMode>(OpenDirectoryMode_All)));
            }
        }

        /**
         * @brief 開かれたディレクトリのGetEntryCount()／Read()操作検証する関数です
         */
        void OpenedDirectoriesOperateVerifyTest()    NN_NOEXCEPT
        {
            int32_t openMax     = GetFsAttribute()->directoryOpenMax;
            int64_t resultCount = 0;

            NN_ASSERT(this->openMaxDir >= openMax);

            // 指定ディレクトリ数までオープンしたディレクトリで、エントリ情報がディレクトリ作成時のエントリ数、ファイル名が一致するか確認します
            for (int32_t i = 0; i < openMax; i++)
            {
                NNT_EXPECT_RESULT_SUCCESS(this->dir[i]->GetEntryCount(&resultCount));

                EXPECT_TRUE(IsSpecifiedEntryTypeExist(this->dirInfo[i].name,
                    DirectoryEntryType::DirectoryEntryType_Directory) == true);

                EXPECT_TRUE(resultCount == 1);

                NNT_EXPECT_RESULT_SUCCESS(this->dir[i]->Read(&resultCount, this->readEntry, this->openMaxEntrySize));

                EXPECT_TRUE((resultCount == 1 && std::strcmp(g_testEntryFile, this->readEntry[0].name) == 0));
            }
        }

        /**
         * @brief テストディレクトリをクローズする関数です
         */
        void CloseTestDirectories()    NN_NOEXCEPT
        {
            int32_t openMax     = GetFsAttribute()->directoryOpenMax;

            NN_ASSERT(this->openMaxDir >= openMax);

            // 指定ディレクトリ数までオープンしたディレクトリを閉じます
            for (int32_t i = 0; i < openMax; i++)
            {
                this->dir[i].reset(nullptr);
            }
        }

    };

    const int32_t ResourceOpenCountRo::nameMaxLength;
    const int32_t ResourceOpenCountRo::openMaxfile;
    const int32_t ResourceOpenCountRo::openMaxDir;
    const int32_t ResourceOpenCountRo::openMaxBufferSize;
    const int32_t ResourceOpenCountRo::openMaxEntrySize;

    //!< @brief 指定された数だけ同時にファイルオープンでき、読み込みのデータが一致すること
    TEST_P(ResourceOpenCountRo, FilesOpenHeavy)
    {
        auto read = GetParam();
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        CreateTestFiles();

        // 指定のファイル数までファイルの操作検証を実行します
        OpenTestFiles();
        OpenedFilesOperateVerifyTest(read);
        CloseTestFiles();
    }

    //!< @brief 指定された数だけ同時にディレクトリオープンでき、取得したエントリ情報が作成したファイルと一致すること
    TEST_F(ResourceOpenCountRo, DirectoriesOpen)
    {
        CreateTestDirectories();

        // 指定のディレクトリ数までディレクトリの操作検証を実行します
        OpenTestDirectories();
        OpenedDirectoriesOperateVerifyTest();
        CloseTestDirectories();
    }

    //!< @brief 指定された数だけのファイルオープンテストとディレクトリテストを同時に実行でき、比較結果に差異がないこと
    TEST_P(ResourceOpenCountRo, FilesAndDirectoriesOpenHeavy)
    {
        CreateTestFiles();
        CreateTestDirectories();

        // 指定数までファイル／ディレクトリをオープンします
        OpenTestFiles();
        OpenTestDirectories();

        // 指定数までファイル／ディレクトリの操作テストを実行します
        OpenedFilesOperateVerifyTest(GetParam());
        OpenedDirectoriesOperateVerifyTest();

        // 指定数までファイル／ディレクトリをクローズします
        CloseTestFiles();
        CloseTestDirectories();
    }

    INSTANTIATE_TEST_CASE_P(WithReadOverloads,
        ResourceOpenCountRo,
        ::testing::ValuesIn(ReadFunctions));
}}}
