﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/nn_Result.h>
#include <nnt/nntest.h>
#include <nnt/fsApiRo/testFs_ApiRo.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

namespace {
    typedef nn::Result(*ReadFunction)(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size);

    nn::Result ReadFunction0(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(pTestFile->Read(offset, buffer, size, nn::fs::ReadOption()));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction1(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(pTestFile->Read(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction2(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        size_t sizeRead;
        nnt::fs::util::InvalidateVariable(&sizeRead);
        NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size, nn::fs::ReadOption()));
        EXPECT_EQ(size, sizeRead);
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction3(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        size_t sizeRead;
        nnt::fs::util::InvalidateVariable(&sizeRead);
        NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size));
        EXPECT_EQ(size, sizeRead);
        NN_RESULT_SUCCESS;
    }

    bool IsReadOverload(ReadFunction read) NN_NOEXCEPT
    {
        return read != ReadFunction2;
    }

    const ReadFunction ReadFunctions[] =
    {
        ReadFunction0,
        ReadFunction1,
        ReadFunction2,
        ReadFunction3
    };
}

namespace nnt { namespace fs { namespace api {
    // FIXME: libnnt_fsApi と同様、適宜観点を分けてコードを分ける

    void LoadReadOnlyTests()
    {
        return;
    }

    class ReadOnlyBasic : public GetFileSystemTestFixture, public ::testing::WithParamInterface<ReadFunction>
    {
    };

    typedef ReadOnlyBasic ReadOnlyWrite;

    //! @brief リソースの全列挙ができること
    TEST_F(ReadOnlyBasic, ReadDirectoryRecursive)
    {
        // TODO: nact で自動生成時の正解リストをコードにも書いて全データベリファイ
        std::unique_ptr<ITestDirectory> dir;
        auto dirName = GetTestRootPath().append("/dir0");
        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().OpenDirectory(&dir, dirName.c_str(), OpenDirectoryMode_All)
        );

        int64_t entryCount;
        NNT_EXPECT_RESULT_SUCCESS(dir->GetEntryCount(&entryCount));
        EXPECT_EQ(1, entryCount);

        DirectoryEntry directoryEntry;
        NNT_EXPECT_RESULT_SUCCESS(dir->Read(&entryCount, &directoryEntry, 1));
        EXPECT_EQ(1, entryCount);
        EXPECT_TRUE(strncmp(directoryEntry.name, "file1.dat", sizeof(directoryEntry.name)) == 0);
        EXPECT_EQ(DirectoryEntryType_File, directoryEntry.directoryEntryType);
        size_t fileSize = 0x100;
        EXPECT_EQ(fileSize, directoryEntry.fileSize);
    }

    //! @brief OpenMode_Read でのみオープンできること
    TEST_F(ReadOnlyBasic, OpenFileMode)
    {
        std::unique_ptr<ITestFile> file;
        auto fileName = GetTestRootPath().append("/file0.dat");
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidArgument,
            GetFs().OpenFile(&file, fileName.c_str(), OpenMode_Write)
        );
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidArgument,
            GetFs().OpenFile(
                &file,
                fileName.c_str(),
                static_cast<OpenMode>(OpenMode_Write | OpenMode_Read)
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidArgument,
            GetFs().OpenFile(
                &file,
                fileName.c_str(),
                static_cast<OpenMode>(OpenMode_Write | OpenMode_AllowAppend)
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidArgument,
            GetFs().OpenFile(
                &file,
                fileName.c_str(),
                static_cast<OpenMode>(OpenMode_Write | OpenMode_Read | OpenMode_AllowAppend)
            )
        );
    }

    //! @brief 32 バイトのファイルのデータを正しくリードできること
    TEST_P(ReadOnlyBasic, ReadFile)
    {
        auto read = GetParam();
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        std::unique_ptr<ITestFile> file;
        auto fileName = GetTestRootPath().append("/file0.dat");
        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().OpenFile(&file, fileName.c_str(), static_cast<OpenMode>(OpenMode_Read))
        );

        size_t fileSize = 0x100;
        auto readBuffer = nnt::fs::util::AllocateBuffer(fileSize);
        NNT_EXPECT_RESULT_SUCCESS(
            read(file.get(), 0, readBuffer.get(), fileSize)
        );

        auto verifyBuffer = nnt::fs::util::AllocateBuffer(fileSize);
        std::memset(verifyBuffer.get(), 0x0, fileSize);
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(verifyBuffer.get(), readBuffer.get(), fileSize);
    }

    //! @brief ファイルリードの読み込みサイズが 0 でも成功すること
    TEST_P(ReadOnlyBasic, ReadFileSizeZero)
    {
        auto read = GetParam();
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        std::unique_ptr<ITestFile> file;
        auto fileName = GetTestRootPath().append("/file0.dat");
        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().OpenFile(&file, fileName.c_str(), static_cast<OpenMode>(OpenMode_Read))
        );

        NNT_EXPECT_RESULT_SUCCESS(
            read(file.get(), 0, nullptr, 0)
        );

        size_t fileSize = 0x100;
        auto readBuffer(nnt::fs::util::AllocateBuffer(fileSize));
        NNT_EXPECT_RESULT_SUCCESS(
            read(file.get(), 0, readBuffer.get(), 0)
        );
    }

    //! @brief ファイルへの書き込みが失敗すること
    TEST_F(ReadOnlyWrite, WriteFile)
    {
        std::unique_ptr<ITestFile> file;
        auto fileName = GetTestRootPath().append("/file0.dat");
        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().OpenFile(&file, fileName.c_str(), static_cast<OpenMode>(OpenMode_Read))
        );

        // 書き込みできるサイズがあるか
        int64_t fileSize = 0;
        NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&fileSize));
        char writeData = 123;
        EXPECT_GE(fileSize, static_cast<int64_t>(sizeof(writeData)));

        // 書き込もうとして失敗するか
        nn::fs::WriteOption writeOption = {};
        auto result = file->Write(0, &writeData, sizeof(writeData), writeOption);
        NN_ASSERT(ResultUnsupportedOperation::Includes(result) || ResultWriteUnpermitted::Includes(result));
    }

    //! @brief ファイルのフラッシュが成功すること
    TEST_F(ReadOnlyWrite, FlushFile)
    {
        std::unique_ptr<ITestFile> file;
        auto fileName = GetTestRootPath().append("/file0.dat");
        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().OpenFile(&file, fileName.c_str(), static_cast<OpenMode>(OpenMode_Read))
        );
        NNT_EXPECT_RESULT_SUCCESS(file->Flush());
    }

    //! @brief サイズの変更が失敗すること
    TEST_F(ReadOnlyWrite, SetFileSize)
    {
        std::unique_ptr<ITestFile> file;
        auto fileName = GetTestRootPath().append("/file0.dat");
        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().OpenFile(&file, fileName.c_str(), static_cast<OpenMode>(OpenMode_Read))
        );

        // 元のサイズを取得
        int64_t fileSize = 0;
        NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&fileSize));

        // サイズ変更しようとして失敗するか
        NNT_EXPECT_RESULT_FAILURE(ResultUnsupportedOperation, file->SetSize(fileSize + 1));
    }

    //! @brief 変更系のファイルシステム操作が失敗すること
    TEST_F(ReadOnlyWrite, WriteOperation)
    {
        {
            auto existFileName = GetTestRootPath().append("/file0.dat");
            auto newFileName = GetTestRootPath().append("/newfile.dat");

            NNT_EXPECT_RESULT_FAILURE(
                ResultUnsupportedOperation,
                GetFs().CreateFile(newFileName.c_str(), 100, 0));
            NNT_EXPECT_RESULT_FAILURE(
                ResultUnsupportedOperation,
                GetFs().DeleteFile(existFileName.c_str()));
            NNT_EXPECT_RESULT_FAILURE(
                ResultUnsupportedOperation,
                GetFs().RenameFile(existFileName.c_str(), newFileName.c_str())
            );
        }
        {
            auto existDirectoryName = GetTestRootPath().append("/dir0");
            auto newDirectoryName = GetTestRootPath().append("/newdir");

            NNT_EXPECT_RESULT_FAILURE(
                ResultUnsupportedOperation,
                GetFs().CreateDirectory(newDirectoryName.c_str()));
            NNT_EXPECT_RESULT_FAILURE(
                ResultUnsupportedOperation,
                GetFs().DeleteDirectory(existDirectoryName.c_str()));
            NNT_EXPECT_RESULT_FAILURE(
                ResultUnsupportedOperation,
                GetFs().DeleteDirectoryRecursively(existDirectoryName.c_str())
            );
            NNT_EXPECT_RESULT_FAILURE(
                ResultUnsupportedOperation,
                GetFs().RenameDirectory(existDirectoryName.c_str(), newDirectoryName.c_str())
            );
        }

        int64_t size = 0;
        if( GetFsType() == nnt::fs::api::FileSystemType_HtmlDocumentFileSystem )
        {
            NNT_EXPECT_RESULT_FAILURE(
                ResultNotImplemented,
                GetFs().GetFreeSpaceSize(&size, GetTestRootPath().append("/").c_str()));
            NNT_EXPECT_RESULT_FAILURE(
                ResultNotImplemented,
                GetFs().GetTotalSpaceSize(&size, GetTestRootPath().append("/").c_str()));
        }
        else
        {
            NNT_EXPECT_RESULT_FAILURE(
                ResultUnsupportedOperation,
                GetFs().GetFreeSpaceSize(&size, GetTestRootPath().append("/").c_str()));
            NNT_EXPECT_RESULT_FAILURE(
                ResultUnsupportedOperation,
                GetFs().GetTotalSpaceSize(&size, GetTestRootPath().append("/").c_str()));
        }
    }

    INSTANTIATE_TEST_CASE_P(WithReadOverloads,
        ReadOnlyBasic,
        ::testing::ValuesIn(ReadFunctions));
}}}
