﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/fsApi/testFs_Api.h>
#include <nn/util/util_FormatString.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

/**
* @file
* @brief    ファイルシステム関数、リソースオープン観点を定義します。
*/
namespace {

    // テストファイル拡張子の定義
//    const char* TestFileExt       = ".file";

    // ディレクトリのエントリ名定義
    const char* TestEntryFile     = "test.file";

    // テストファイルサイズ定義
//    const size_t TestFileLength   = 10;

    // テストディレクトリサイズ定義
//    const size_t TestDirLength    = 10;

    typedef nn::Result(*ReadFunction)(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size);

    nn::Result ReadFunction0(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(pTestFile->Read(offset, buffer, size, nn::fs::ReadOption()));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction1(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(pTestFile->Read(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction2(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        size_t sizeRead;
        NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size, nn::fs::ReadOption()));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction3(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        size_t sizeRead;
        NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    bool IsReadOverload(ReadFunction read) NN_NOEXCEPT
    {
        return read != ReadFunction2;
    }

    const ReadFunction ReadFunctions[] =
    {
        ReadFunction0,
        ReadFunction1,
        ReadFunction2,
        ReadFunction3
    };
}

namespace nnt { namespace fs { namespace api {
    void LoadResourceOpenCountTests() NN_NOEXCEPT
    {
        return;
    }

    const int CallOpen = 0x01;
    const int CallRead = 0x02;
    const int CallClose = 0x04;

    struct ParamsStructure {
        int position1;
        int position2;
        int position3;
    };
    /**
     * @brief   ResourceOpenCount で利用するテストフィクスチャです。
     */
    class ResourceOpenCount : public CleanupFileSystemTestFixture, public :: testing::WithParamInterface<ParamsStructure>
    {
    protected:

        // 各リソースサイズ定数定義

        static const int32_t OpenMaxfile        = 1024;
        static const int32_t OpenMaxDir         = 64;
        static const int32_t OpenMaxBufferSize  = 1024;
        static const int32_t OpenMaxEntrySize   = 2;

        char fileName[OpenMaxfile][NameMaxLength];
        char dirName[OpenMaxDir][NameMaxLength];

        // ファイルの書き込み／読み込みバッファ定義
        uint8_t writeBuffer[OpenMaxBufferSize];

        // エントリバッファ定義
        DirectoryEntry readEntry[OpenMaxEntrySize];

        // ファイル／フォルダ作成の管理配列
        std::unique_ptr<ITestFile> file[OpenMaxfile];
        std::unique_ptr<ITestDirectory> dir[OpenMaxDir];

        String rootTestdir;

        /**
         * @brief テスト開始時に毎回呼び出される関数です。
         */
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();

            // ルートディレクトリ設定・作成
            this->rootTestdir = GetTestRootPath();
            this->rootTestdir.append("/Test/");
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(this->rootTestdir.c_str()));
        }

        /**
         * @brief テスト終了時に毎回呼び出される関数です。
         */
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            // ルートディレクトリ削除
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectoryRecursively(this->rootTestdir.c_str()));
            this->rootTestdir = "";

            CleanupFileSystemTestFixture::TearDown();
        }

        /**
         * @brief テストファイルをオープンする関数です
         */
        void OpenTestFiles()  NN_NOEXCEPT
        {
            int32_t openMax     = GetFsAttribute()->fileOpenMax;

            NN_ASSERT(this->OpenMaxfile >= openMax);

            // 指定ファイル数までファイルをオープンします
            for (int32_t i = 0; i < openMax; i++)
            {
                NNT_FS_SCOPED_TRACE_SAFE("loop count: %d, file path: %s\n", i, this->fileName[i]);
                NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&this->file[i],
                    this->fileName[i], static_cast<OpenMode>(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)));
            }
        }

        /**
         * @brief 開かれたファイルに対してWrite()／Read()操作検証する関数です
         */
        void OpenedFilesOperateVerifyTest()  NN_NOEXCEPT
        {
            int32_t openMax     = GetFsAttribute()->fileOpenMax;

            NN_ASSERT(this->OpenMaxfile >= openMax);

            // 指定ファイル数までオープンしたファイルの書き込みデータ／読み込みデータが一致しているか確認します
            for (int32_t i = 0; i < openMax; i++)
            {
                NNT_FS_SCOPED_TRACE_SAFE("loop count: %d, file path: %s\n", i, this->fileName[i]);
                EXPECT_TRUE(IsEntryExist(&GetFs(), this->fileName[i], DirectoryEntryType::DirectoryEntryType_File) == true);

                ConfirmWriteRead(this->file[i].get(), this->writeBuffer, static_cast<size_t>(this->OpenMaxBufferSize));
            }
        }

        /**
         * @brief テストファイルをクローズする関数です
         */
        void CloseTestFiles()  NN_NOEXCEPT
        {
            int32_t openMax     = GetFsAttribute()->fileOpenMax;

            NN_ASSERT(this->OpenMaxfile >= openMax);

            // 指定ファイル数までオープンしたファイルを閉じます
            for (int32_t i = 0; i < openMax; i++)
            {
                this->file[i].reset(nullptr);
            }
        }

        /**
         * @brief テストディレクトリをオープンする関数です
         */
        void OpenTestDirectories()    NN_NOEXCEPT
        {
            int32_t openMax     = GetFsAttribute()->directoryOpenMax;

            NN_ASSERT(this->OpenMaxDir >= openMax);

            // 指定ディレクトリ数までディレクトリをオープンします
            for (int32_t i = 0; i < openMax; i++)
            {
                NNT_FS_SCOPED_TRACE_SAFE(
                    "loop count: %d, directory path: %s\n", i, this->dirName[i]);
                NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenDirectory(&this->dir[i],
                    this->dirName[i], static_cast<OpenDirectoryMode>(OpenDirectoryMode_All)));
            }
        }

        /**
         * @brief 開かれたディレクトリのGetEntryCount()／Read()操作検証する関数です
         */
        void OpenedDirectoriesOperateVerifyTest()    NN_NOEXCEPT
        {
            int32_t openMax     = GetFsAttribute()->directoryOpenMax;
            int64_t resultCount = 0;

            NN_ASSERT(this->OpenMaxDir >= openMax);

            // 指定ディレクトリ数までオープンしたディレクトリで、エントリ情報がディレクトリ作成時のエントリ数、ファイル名が一致するか確認します
            for (int32_t i = 0; i < openMax; i++)
            {
                NNT_FS_SCOPED_TRACE_SAFE(
                    "loop count: %d, directory path: %s\n", i, this->dirName[i]);
                NNT_EXPECT_RESULT_SUCCESS(this->dir[i]->GetEntryCount(&resultCount));

                EXPECT_TRUE(IsEntryExist(&GetFs(), this->dirName[i],
                    DirectoryEntryType::DirectoryEntryType_Directory) == true);

                EXPECT_TRUE(resultCount == 1);

                NNT_EXPECT_RESULT_SUCCESS(this->dir[i]->Read(&resultCount, this->readEntry, this->OpenMaxEntrySize));

                EXPECT_TRUE((resultCount == 1 && std::strcmp(TestEntryFile, this->readEntry[0].name) == 0));
            }
        }

        /**
         * @brief テストディレクトリをクローズする関数です
         */
        void CloseTestDirectories()    NN_NOEXCEPT
        {
            int32_t openMax     = GetFsAttribute()->directoryOpenMax;

            NN_ASSERT(this->OpenMaxDir >= openMax);

            // 指定ディレクトリ数までオープンしたディレクトリを閉じます
            for (int32_t i = 0; i < openMax; i++)
            {
                this->dir[i].reset(nullptr);
            }
        }

    };

    const int32_t ResourceOpenCount::OpenMaxfile;
    const int32_t ResourceOpenCount::OpenMaxDir;
    const int32_t ResourceOpenCount::OpenMaxBufferSize;
    const int32_t ResourceOpenCount::OpenMaxEntrySize;

    //!< @brief 指定された数だけ同時にファイルオープンでき、書き込み／読み込みのデータが一致すること
    TEST_F(ResourceOpenCount, FilesOpenHeavy)
    {

        // 事前に指定のファイル数まで空のファイルを作成します
        NN_ASSERT(this->OpenMaxfile >= GetFsAttribute()->fileOpenMax);
        NNT_ASSERT_RESULT_SUCCESS(CreateTestFile(&GetFs(), fileName, this->rootTestdir.c_str(), GetFsAttribute()->fileOpenMax, 0));

        // 指定のファイル数までファイルの操作検証を実行します
        OpenTestFiles();
        OpenedFilesOperateVerifyTest();
        CloseTestFiles();
    }

    //!< @brief 指定された数だけ同時にディレクトリオープンでき、取得したエントリ情報が作成したファイルと一致すること
    TEST_F(ResourceOpenCount, DirectoriesOpen)
    {
        // 事前にディレクトリとエントリ用ファイルを作成します
        NNT_ASSERT_RESULT_SUCCESS(CreateTestDirectory(&GetFs(), dirName, this->rootTestdir.c_str(), GetFsAttribute()->directoryOpenMax, DIRTYPE_FLAT, true, 0));

        // 指定のディレクトリ数までディレクトリの操作検証を実行します
        OpenTestDirectories();
        OpenedDirectoriesOperateVerifyTest();
        CloseTestDirectories();
    }

    //!< @brief 指定された数だけのファイルオープンテストとディレクトリテストを同時に実行でき、比較結果に差異がないこと
    TEST_F(ResourceOpenCount, FilesAndDirectoriesOpenHeavy)
    {
        // 事前に指定の数までファイル、ディレクトリを作成します
        NN_ASSERT(this->OpenMaxfile >= GetFsAttribute()->fileOpenMax);
        NNT_ASSERT_RESULT_SUCCESS(CreateTestFile(&GetFs(), fileName, this->rootTestdir.c_str(), GetFsAttribute()->fileOpenMax, 0));
        NNT_ASSERT_RESULT_SUCCESS(CreateTestDirectory(&GetFs(), dirName, this->rootTestdir.c_str(), GetFsAttribute()->directoryOpenMax, DIRTYPE_FLAT, true, 0));

        // 指定数までファイル／ディレクトリをオープンします
        OpenTestFiles();
        OpenTestDirectories();

        // 指定数までファイル／ディレクトリの操作テストを実行します
        OpenedFilesOperateVerifyTest();
        OpenedDirectoriesOperateVerifyTest();

        // 指定数までファイル／ディレクトリをクローズします
        CloseTestFiles();
        CloseTestDirectories();
    }

    ParamsStructure patternParam[] = {
        {CallOpen|CallRead|CallClose, 0,                           0},
        {CallOpen|CallRead,           CallClose,                   0},
        {CallOpen|CallRead,           0,                           CallClose},
        {CallOpen,                    CallRead|CallClose,          0},
        {CallOpen,                    CallRead,                    CallClose},
        {CallOpen,                    0,                           CallRead|CallClose},
        {0,                           CallOpen|CallRead|CallClose, 0},
        {0,                           CallOpen|CallRead,           CallClose},
        {0,                           CallOpen,                    CallRead|CallClose},
        {0,                           0,                           CallOpen|CallRead|CallClose},
    };

    INSTANTIATE_TEST_CASE_P(WithVariousPattern, ResourceOpenCount, ::testing::ValuesIn(patternParam));

    void FilesOpenMultipleTestOperate(int position, ITestFileSystem* fs, std::unique_ptr<ITestFile>* file, char* fileName, int fileSize, int readSize, char* buf, ReadFunction read)
    {
        if ((position & CallOpen) != 0)
        {
            NN_LOG(" -> Open(2)");
            NNT_ASSERT_RESULT_SUCCESS(fs->OpenFile(file, fileName, OpenMode_Read));
        }
        if ((position & CallRead) != 0)
        {
            NN_LOG(" -> Read(2)");
            NNT_ASSERT_RESULT_SUCCESS(read(file->get(), fileSize - readSize, buf, readSize));
        }
        if ((position & CallClose) != 0)
        {
            NN_LOG(" -> Close(2)");
            (*file)->Close();
        }
    }

    //!< @brief 同一ファイルに対する二つのOpen/Read/Closeが順序に関係なく実行できること
    TEST_P(ResourceOpenCount, FilesOpenMultiple)
    {
        const int fileSize = (GetFsAttribute()->storageSize >= 0x10000000/*256MB*/) ? 0x8000000/*128MB*/ : 0x100000/*1MB*/;
        const int readSize = 256;
        char buf[readSize];
        std::unique_ptr<ITestFile> file1;
        std::unique_ptr<ITestFile> file2;
        ParamsStructure param = GetParam();

        NNT_ASSERT_RESULT_SUCCESS(CreateTestFile(&GetFs(), fileName, this->rootTestdir.c_str(), 1, fileSize));

        for( auto read : ReadFunctions )
        {
            if( IsReadOverload(read) && !GetFsAttribute()->isReadOverloadsSupported )
            {
                continue;
            }

            NN_LOG("Open(1)");
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file1, fileName[0], OpenMode_Read));
            FilesOpenMultipleTestOperate(param.position1, &GetFs(), &file2, fileName[0], fileSize, readSize, buf, read);

            NN_LOG(" -> Read(1)");
            NNT_ASSERT_RESULT_SUCCESS(read(file1.get(), fileSize - readSize, buf, readSize));
            FilesOpenMultipleTestOperate(param.position2, &GetFs(), &file2, fileName[0], fileSize, readSize, buf, read);

            NN_LOG(" -> Close(1)");
            file1.reset(nullptr);
            FilesOpenMultipleTestOperate(param.position3, &GetFs(), &file2, fileName[0], fileSize, readSize, buf, read);

            NN_LOG("\n");
        }
    }

    //!< @brief 指定された数だけの同一ファイルオープンテストと、比較結果に差異がないこと
    TEST_F(ResourceOpenCount, SameFileOpenHeavy)
    {
        // 事前に１つファイルを作成します
        NN_ASSERT(this->OpenMaxfile >= GetFsAttribute()->fileOpenMax);
        NNT_ASSERT_RESULT_SUCCESS(CreateTestFile(&GetFs(), fileName, this->rootTestdir.c_str(), 1, 0));
        NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&this->file[0],
            this->fileName[0], static_cast<OpenMode>(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)));
        ConfirmWriteRead(this->file[0].get(), this->writeBuffer, static_cast<size_t>(this->OpenMaxBufferSize));
        CloseTestFiles();

        // 指定のファイル数までファイルの操作検証を実行します
        auto OpenTestSameFile = [=]() NN_NOEXCEPT
        {
            int32_t openMax = GetFsAttribute()->fileOpenMax;

            NN_ASSERT(this->OpenMaxfile >= openMax);

            const size_t size = static_cast<size_t>(this->OpenMaxBufferSize);
            auto readBuffer = AllocateBuffer(size);

            // 指定ファイル数までファイルをオープンします
            for( int32_t i = 0; i < openMax; i++ )
            {
                auto read = ReadFunctions[i % NN_ARRAY_SIZE(ReadFunctions)];

                NNT_FS_SCOPED_TRACE_SAFE("loop count: %d, file path: %s\n", i, this->fileName[i]);
                NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&this->file[i],
                    this->fileName[0], OpenMode_Read));

                // Read したバッファが一致するか
                InvalidateVariable(readBuffer.get(), static_cast<int>(size));

                NNT_EXPECT_RESULT_SUCCESS(read(this->file[i].get(), 0, readBuffer.get(), size));

                NNT_FS_UTIL_EXPECT_MEMCMPEQ(this->writeBuffer, readBuffer.get(), size);
            }
        };

        OpenTestSameFile();
        CloseTestFiles();

        // 同一テストケース内で2回試行する
        OpenTestSameFile();
        CloseTestFiles();
    }

}}}
