﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_ResultPrivate.h>
#include <nnt/fsApi/testFs_Api.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

namespace {
    const int FileSize = 32;
}

namespace nnt { namespace fs { namespace api {
    void LoadPreConditionTests() NN_NOEXCEPT
    {
        return;
    }

    namespace {
        const OpenMode OpenFileModesRead[] =
        {
            OpenMode_Read,
            static_cast<OpenMode>(OpenMode_Read  | OpenMode_AllowAppend),
        };

        const OpenMode OpenFileModesWrite[] =
        {
            OpenMode_Write,
            static_cast<OpenMode>(OpenMode_Write | OpenMode_AllowAppend),
        };

        typedef nn::Result(*ReadFunction)(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size);

        nn::Result ReadFunction0(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
        {
            NN_RESULT_DO(pTestFile->Read(offset, buffer, size, nn::fs::ReadOption()));
            NN_RESULT_SUCCESS;
        }

        nn::Result ReadFunction1(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
        {
            NN_RESULT_DO(pTestFile->Read(offset, buffer, size));
            NN_RESULT_SUCCESS;
        }

        nn::Result ReadFunction2(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
        {
            size_t sizeRead;
            NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size, nn::fs::ReadOption()));
            NN_RESULT_SUCCESS;
        }

        nn::Result ReadFunction3(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
        {
            size_t sizeRead;
            NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size));
            NN_RESULT_SUCCESS;
        }

        bool IsReadOverload(ReadFunction read) NN_NOEXCEPT
        {
            return read != ReadFunction2;
        }

        const ReadFunction ReadFunctions[] =
        {
            ReadFunction0,
            ReadFunction1,
            ReadFunction2,
            ReadFunction3
        };
    }

    class PreConditionInvalidOpenMode : public CleanupFileSystemTestFixture, public ::testing::WithParamInterface<ReadFunction>
    {
    protected:
        PreConditionInvalidOpenMode() NN_NOEXCEPT
        {
            CheckGlobalNewDeleteFlagTestFixture::DisableCheck();
            CheckMemoryLeakFixture::DisableCheck();
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();
            m_FileName = GetTestRootPath().append("/test.file");
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(m_FileName.c_str(), FileSize));
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(m_FileName.c_str()));
            CleanupFileSystemTestFixture::TearDown();
        }

        const String& GetFileName() NN_NOEXCEPT
        {
            return m_FileName;
        }

    private:
        String m_FileName;
    };

    class PreConditionNotAllowAppend : public PreConditionInvalidOpenMode
    {
    };

    class PreConditionInvalidOpenDirectoryMode : public PreConditionInvalidOpenMode
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            PreConditionInvalidOpenMode::SetUp();
            m_DirectoryName = GetTestRootPath().append("/dir");
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(m_DirectoryName.c_str()));
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectory(m_DirectoryName.c_str()));
            PreConditionInvalidOpenMode::TearDown();
        }

        const String& GetDirectoryName() NN_NOEXCEPT
        {
            return m_DirectoryName;
        }

    private:
        String m_DirectoryName;
    };

    typedef PreConditionInvalidOpenDirectoryMode PreConditionInvalidArgument;
    typedef PreConditionInvalidOpenMode PreConditionViolationWithFile;


    //! @brief OpenMode_AllowAppend / OpenMode なしでファイルを開き、ResultInvalidArgument が返却されることを確認する
    TEST_F(PreConditionInvalidOpenMode, OpenFile)
    {
        std::unique_ptr<ITestFile> file;
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidArgument,
            GetFs().OpenFile(&file, GetFileName().c_str(), OpenMode_AllowAppend));
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidArgument,
            GetFs().OpenFile(&file, GetFileName().c_str(), static_cast<OpenMode>(0)));
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidArgument,
            GetFs().OpenFile(&file, GetFileName().c_str(), static_cast<OpenMode>(15)));
    }

    //! @brief 書き込みモードで開いたファイルを読み込み、ResultInvalidOperationForOpenMode が返却されることを確認する
    TEST_P(PreConditionInvalidOpenMode, ReadFile)
    {
        auto read = GetParam();
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        for( auto openMode : OpenFileModesWrite )
        {
            std::unique_ptr<ITestFile> file;
            char buffer[FileSize + 1] = { 0 };

            NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&file, GetFileName().c_str(), openMode));
            NNT_EXPECT_RESULT_FAILURE(
                ResultInvalidOperationForOpenMode,
                read(file.get(), 0, buffer, FileSize));
        }
    }

    //! @brief 読み込みモードで開いたファイルに書き込み、ResultInvalidOperationForOpenMode が返却されることを確認する
    TEST_F(PreConditionInvalidOpenMode, WriteFile)
    {
        for( auto openMode : OpenFileModesRead )
        {
            std::unique_ptr<ITestFile> file;
            char buffer[FileSize + 1] = {0};

            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, GetFileName().c_str(), openMode));
            NNT_EXPECT_RESULT_FAILURE(
                ResultInvalidOperationForOpenMode,
                file->Write(0, buffer, FileSize, WriteOption()));
        }
    }

    //! @brief 読み込みモードで開いたファイルを SetSize し、ResultInvalidOperationForOpenMode が返却されることを確認する
    TEST_F(PreConditionInvalidOpenMode, SetFileSize)
    {
        for( auto openMode : OpenFileModesRead )
        {
            std::unique_ptr<ITestFile> file;

            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, GetFileName().c_str(), openMode));
            NNT_EXPECT_RESULT_FAILURE(ResultInvalidOperationForOpenMode, file->SetSize(0));
            NNT_EXPECT_RESULT_FAILURE(
                ResultInvalidOperationForOpenMode, file->SetSize(FileSize + 1));
        }
    }

    //! @brief OpenMode_Write モードで開いたファイルに追記書き込みし、ResultFileExtensionWithoutOpenModeAllowAppend が返却されることを確認する
    TEST_F(PreConditionNotAllowAppend, WriteFile)
    {
        std::unique_ptr<ITestFile> file;
        char buffer[FileSize + 1] = {0};

        NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, GetFileName().c_str(), OpenMode_Write));
        NNT_EXPECT_RESULT_FAILURE(
            ResultFileExtensionWithoutOpenModeAllowAppend,
            file->Write(0, buffer, FileSize + 1, WriteOption()));
        NNT_EXPECT_RESULT_FAILURE(
            ResultFileExtensionWithoutOpenModeAllowAppend,
            file->Write(FileSize, buffer, 1, WriteOption()));
    }

    //! @brief OpenDirectoryMode が不正な場合にディレクトリを開けないことを確かめる
    TEST_F(PreConditionInvalidOpenDirectoryMode, OpenDirectory)
    {
        std::unique_ptr<ITestDirectory> pDirectory;
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidArgument,
            GetFs().OpenDirectory(
                &pDirectory, GetDirectoryName().c_str(), static_cast<OpenDirectoryMode>(0)));
        NNT_EXPECT_RESULT_FAILURE(
            ResultInvalidArgument,
            GetFs().OpenDirectory(
                &pDirectory, GetDirectoryName().c_str(), static_cast<OpenDirectoryMode>(7)));
    }

    //! @brief ディレクトリの読み込みで読み込むエントリ数が負数なら ResultInvalidArgument が返却されることを確認する
    TEST_F(PreConditionInvalidArgument, ReadDirectory)
    {
        const OpenDirectoryMode openDirectoryModes[] =
        {
            OpenDirectoryMode_Directory,
            OpenDirectoryMode_File,
            OpenDirectoryMode_All
        };
        for( auto openDirecotryMode : openDirectoryModes )
        {
            // ITestDirectory のテストをするためにディレクトリを開く
            std::unique_ptr<ITestDirectory> directory;
            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().OpenDirectory(&directory, GetDirectoryName().c_str(), openDirecotryMode)
            );

            // ITestDirectory::Read
            int64_t readCount = 0;
            DirectoryEntry entry = {};
            NNT_EXPECT_RESULT_FAILURE(
                ResultInvalidArgument,
                directory->Read(&readCount, &entry, -1)
            );
        }
    }

    //! @brief OpenMode_Write でファイルオープンしたまま Commit した場合はエラー
    //         todo: テストケース分け
    TEST_F(PreConditionViolationWithFile, CommitWithOpenFile)
    {
        NNT_FS_UTIL_SKIP_TEST_UNLESS(GetFsAttribute()->isSaveFileSystem && !GetFsAttribute()->isSupportedFileOpenCheckOnCommit);

        std::unique_ptr<ITestFile> pFile;
        NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&pFile, GetFileName().c_str(), nn::fs::OpenMode_Write));
        NNT_EXPECT_RESULT_FAILURE(ResultPreconditionViolation, GetFs().Commit());
    }

    INSTANTIATE_TEST_CASE_P(WithReadOverloads,
        PreConditionInvalidOpenMode,
        ::testing::ValuesIn(ReadFunctions));

}}}
