﻿# coding: UTF-8

# Copyright (C)Nintendo All rights reserved.
#
# These coded instructions, statements, and computer programs contain proprietary
# information of Nintendo and/or its licensed developers and are protected by
# national and international copyright laws. They may not be disclosed to third
# parties or copied or duplicated in any form, in whole or in part, without the
# prior written consent of Nintendo.
#
# The content herein is highly confidential and should be handled accordingly.

# 詳細は、output_testFs_util_storagePerformance.bat を見て下さい

import sys
import zipfile
from os import path
from datetime import datetime

class PerformanceTestResult:
    def __init__(self):
        self.name = ""
        self.data = {
            "seq": {str(pow(2,i)):0.0 for i in range(10,23)},
            "rnd": {str(pow(2,i)):0.0 for i in range(10,23)}}

    def ReadData(self, line):
        key = line.split("'")[1]
        # _ で区切ったとき、2 番目か 3 番目にアクセスサイズが現れる
        if key.split("_")[1].isdigit():
            accessSize = key.split("_")[1]
            self.name = key.split("_")[0]
        else:
            accessSize = key.split("_")[2]
            self.name = key.split("_")[0] + "_" + key.split("_")[1]
        # ' で区切ったとき、4 番目に速度が現れる
        value = line.split("'")[3]
        self.data[self.GetType(line)][accessSize] = float(value)

    def GetType(self, line):
        # Velocity の前の 3 文字を返す
        return line.split("Velocity")[0][-3:]

    def WriteData(self, file):
        file.write("""    {{
        "{}",
        {{
            // Seq
            {{{:7.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f}}},
            // Rand
            {{{:7.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f},{:8.3f}}},
        }}
    }},
""".format( self.name, 
        self.data["seq"]["1024"], self.data["seq"]["2048"], self.data["seq"]["4096"], self.data["seq"]["8192"], self.data["seq"]["16384"], self.data["seq"]["32768"], self.data["seq"]["65536"], self.data["seq"]["131072"], self.data["seq"]["262144"], self.data["seq"]["524288"], self.data["seq"]["1048576"], self.data["seq"]["2097152"], self.data["seq"]["4194304"],
        self.data["rnd"]["1024"], self.data["rnd"]["2048"], self.data["rnd"]["4096"], self.data["rnd"]["8192"], self.data["rnd"]["16384"], self.data["rnd"]["32768"], self.data["rnd"]["65536"], self.data["rnd"]["131072"], self.data["rnd"]["262144"], self.data["rnd"]["524288"], self.data["rnd"]["1048576"], self.data["rnd"]["2097152"], self.data["rnd"]["4194304"]))

class PerformanceData:
    def __init__(self, zipfile):
        self.zip = zipfile
        self.test_time = "----/--/-- --:--:--"
        self.commit_hash = "----------"
        for line in self.LoadTestInfo():
            if "Test date" in line:
                self.test_time = line.split(" : ")[1].strip()
            if "Commit hash" in line:
                self.commit_hash = line.split(" : ")[1].strip()
        self.storage_speed_table = []

    def LoadTestInfo(self):
        for name in self.zip.namelist():
            if "Summary.txt" in name:
                with self.zip.open(name) as f:
                    return str(f.read().decode('utf-8-sig')).split("List: ")[0].split("\n")
        print("[ERROR] Invalid input file.")


    def LoadStorageSpeed(self, logname):
        self.storage_speed_table += self.ReadFileDataFromName(logname + ".log.txt")

    def ReadFileDataFromName(self, filename):
        for name in self.zip.namelist():
            if filename in name:
                return self.ReadFileDataFromPath(name)
        print("[ERROR] Not found: %s." % filename)
        return []

    def ReadFileDataFromPath(self, path):
        aligned_data = PerformanceTestResult()
        unaligned_data = PerformanceTestResult()
        with self.zip.open(path) as f:
            for l in f:
                line = str(l)
                if ("##teamcity" in line) and ("Velocity" in line) and ("Read" in line):
                    if "Unaligned" in line:
                        unaligned_data.ReadData(line)
                    else:
                        aligned_data.ReadData(line)
        if len(unaligned_data.name) == 0:
            return [aligned_data]
        else:
            return [aligned_data, unaligned_data]

def ReadDataFromZipfile(zipfile_path, output_directory):
    # zip ファイルでないなら終了
    if not zipfile.is_zipfile(zipfile_path):
        print("[ERROR] %s is not zip format." % zipfile_path)
        return

    # zip から必要なデータを取り込む
    with zipfile.ZipFile(zipfile_path, "r") as zip:
        outputdata = PerformanceData(zip)
        print("Date        {}".format(outputdata.test_time))

        outputdata.LoadStorageSpeed("FsPerformanceReadRomFsTest_NANDFaster")
        outputdata.LoadStorageSpeed("FsPerformanceReadRomFsTest_NANDSlower")
        outputdata.LoadStorageSpeed("FsPerformanceReadRomFsTest_NANDOff")
        #outputdata.LoadStorageSpeed("FsPerformanceReadRomFsTestWithEticket")
        outputdata.LoadStorageSpeed("FsPerformanceReadRomFsTest_GC50")
        outputdata.LoadStorageSpeed("FsPerformanceReadRomFsTest_GC25")
        outputdata.LoadStorageSpeed("FsPerformanceReadRomFsTest_SdCard")
        outputdata.LoadStorageSpeed("FsPerformanceReadWriteTest_SaveData_Off")
        outputdata.LoadStorageSpeed("FsPerformanceReadWriteTest_BisFs")
        outputdata.LoadStorageSpeed("FsPerformanceReadWriteTest_SdCard")
        outputdata.LoadStorageSpeed("FsPerformanceReadWriteTest_SaveData_Slower")
        outputdata.LoadStorageSpeed("FsPerformanceReadWriteTest_SaveData_Faster")
        outputdata.LoadStorageSpeed("FsPerformanceReadWriteTest_CacheStorage_Off")
        outputdata.LoadStorageSpeed("FsPerformanceReadWriteTest_CacheStorage_Slower")
        outputdata.LoadStorageSpeed("FsPerformanceReadWriteTest_TemporaryStorage_Off")
        outputdata.LoadStorageSpeed("FsPerformanceReadWriteTest_TemporaryStorage_Slower")
        outputdata.LoadStorageSpeed("FsPerformanceReadRomFsTest_NANDOff_PriorityRealtime")
        outputdata.LoadStorageSpeed("FsPerformanceReadRomFsTest_NANDOff_PriorityLow")
        outputdata.LoadStorageSpeed("FsPerformanceReadRomFsTest_NANDOff_PriorityBackground")
        outputdata.LoadStorageSpeed("FsPerformanceReadWriteTest_SaveData_Off_PriorityRealtime")
        outputdata.LoadStorageSpeed("FsPerformanceReadWriteTest_SaveData_Off_PriorityLow")
        outputdata.LoadStorageSpeed("FsPerformanceReadWriteTest_SaveData_Off_PriorityBackground")

        return outputdata


def OutputHeaderFile(header_data, output_path):
    with open(output_path, "w", encoding='utf8') as f:
        f.write('\ufeff')
        f.write("""/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nnt { namespace fs { namespace util {
/**
* 以下の成果物を元に Tests/Fs/Scripts/Performance/output_testFs_util_storagePerformance.bat が自動生成しました。
*
*   https://devsedci01.nintendo.co.jp/TeamCity/viewType.html?buildTypeId=SigloSdk_Nx_Public_TestModuleSdevFs&tab=buildTypeHistoryList
""")
        f.write("*   {}".format(header_data.test_time))
        f.write("""
*/

static const size_t AccessPatternNum   = 2;
static const size_t ChunkSizeNum       = 13;

struct StorageSpeedSample
{
    const char* storageTypeName;
    double data[AccessPatternNum][ChunkSizeNum];
};

/**
* @brief ストレージ・チャンクサイズ・アクセスパターン別の Release ビルドでのサンプル速度
*/
static const StorageSpeedSample StorageSpeedSampleTable[] =
{
""");
        for storage_speed in header_data.storage_speed_table:
            storage_speed.WriteData(f)
        f.write("""};
}}}
""")

def main():
    args = sys.argv
    if len(args) < 2:
        print("[ERROR] Please Drag & Drop zip file to output_PerformanceThreadshould.bat.")
        sys.exit(1)

    current_directory = path.dirname(args[0])

    # 各 zip ファイルを出力
    zipfile_path = args[1]
    print("Load '{}'...".format(zipfile_path))
    header_data = ReadDataFromZipfile(zipfile_path, current_directory)
    print("'{}' is loaded.".format(zipfile_path))

    output_path = path.join(current_directory, "testFs_util_storagePerformance.autogen.h")
    print("Output to '{}'...".format(output_path))
    OutputHeaderFile(header_data, output_path)
    print("Please copy '{}' to '%NINTENDO_SDK_ROOT%\\Tests\\Include\\nnt\\fsUtil\\testFs_util_storagePerformance.h'".format(output_path))

if __name__ == '__main__':
  main()
