﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testFriends_Common.h"

namespace
{
    nn::account::Uid s_Users[nn::account::UserCountMax] = {};
    nn::account::NetworkServiceAccountId s_AccountIds[nn::account::UserCountMax] = {};
    int s_UserCount = 0;
}

class FriendsProfile : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());

        nnt::friends::LoadAccounts(&s_UserCount, s_Users, s_AccountIds, NN_ARRAY_SIZE(s_Users));
        NN_ABORT_UNLESS_GREATER_EQUAL(s_UserCount, 3);

        nn::friends::SetOption(nn::friends::OptionAdmin_CheckUserStatus, 0);
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }
};

TEST_F(FriendsProfile, CleanupRelationship)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    for (int i = 0; i < 3; i++)
    {
        nnt::friends::CleanupRelationship(s_Users[i]);
    }
}

TEST_F(FriendsProfile, SyncUserSetting)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::account::UserHandle handles[3] = {};

    // プレイログを生成する。
    for (int i = 0; i < 3; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handles[i], s_Users[i]));
    }

    NN_UTIL_SCOPE_EXIT
    {
        for (int i = 0; i < 3; i++)
        {
            nn::account::CloseUser(handles[i]);
        }
    };

    // システムにイベントが通知されるまで少し待機する。
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    for (int i = 0; i < s_UserCount; i++)
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SyncUserSetting(&context, s_Users[i]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
}

TEST_F(FriendsProfile, ChangePlayLogPermission)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::UserSetting setting;
    NNT_ASSERT_RESULT_SUCCESS(setting.Initialize(s_Users[0]));

    nn::friends::AsyncContext context;
    NNT_ASSERT_RESULT_SUCCESS(setting.ChangePlayLogPermission(&context, nn::friends::ExtraPermission_Friends));

    nn::os::SystemEvent completionEvent;
    NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

    completionEvent.Wait();

    NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
}

TEST_F(FriendsProfile, CreateFriendship)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[0], s_AccountIds[2],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[2], s_AccountIds[0],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SyncFriendList(&context, s_Users[0]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
}

TEST_F(FriendsProfile, Get)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    static nn::friends::ProfileExtra s_Profiles[nn::account::UserCountMax];

    nn::friends::AsyncContext context;
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetProfileExtraList(&context, s_Profiles, s_Users[0], s_AccountIds, NN_ARRAY_SIZE(s_Profiles)));

    nn::os::SystemEvent completionEvent;
    NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

    completionEvent.Wait();

    NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

    EXPECT_NE(s_Profiles[0].GetPlayLog(0).appInfo.appId, nn::ApplicationId::GetInvalidId());
    // 0 と 1 はフレンドではないので、プレイログは見えない。
    EXPECT_EQ(s_Profiles[1].GetPlayLog(0).appInfo.appId, nn::ApplicationId::GetInvalidId());
    EXPECT_NE(s_Profiles[2].GetPlayLog(0).appInfo.appId, nn::ApplicationId::GetInvalidId());

    for (int i = 0; i < s_UserCount; i++)
    {
        ASSERT_TRUE(s_Profiles[i].IsValid());

        EXPECT_EQ(s_Profiles[i].GetAccountId(), s_AccountIds[i]);
    }
    for (int i = s_UserCount; i < NN_ARRAY_SIZE(s_Profiles); i++)
    {
        ASSERT_FALSE(s_Profiles[i].IsValid());
    }

    nnt::friends::Dump(s_Profiles, NN_ARRAY_SIZE(s_Profiles));
}

TEST_F(FriendsProfile, GetFromFriendCode)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::ProfileExtra profile;

    for (int i = 0; i < 3; i++)
    {
        nn::friends::UserSetting setting;

        NNT_ASSERT_RESULT_SUCCESS(setting.Initialize(s_Users[i]));

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetProfileExtra(&context, &profile, s_Users[i], setting.GetFriendCode()));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        EXPECT_EQ(profile.GetAccountId(), s_AccountIds[i]);

        nnt::friends::Dump(&profile, 1);
    }
    // 不明なユーザー
    {
        nn::friends::FriendCode friendCode = {{"0000-0000-0000"}};

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetProfileExtra(&context, &profile, s_Users[0], friendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNetworkServiceAccountNotExistsGeneral, context.GetResult());
    }
    // 無効なフレンドコード
    {
        nn::friends::FriendCode friendCode = {{"ABCDEFG"}};

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetProfileExtra(&context, &profile, s_Users[0], friendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultInvalidFriendCodeFormat, context.GetResult());
    }
}

TEST_F(FriendsProfile, LargeRequest)
{
    static nn::account::NetworkServiceAccountId s_TestAccountIds[nn::friends::GetProfileCountMax] = {};

    {
        static nn::friends::Profile s_Profiles[nn::friends::GetProfileCountMax];

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetProfileList(&context, s_Profiles, s_Users[0],
            s_TestAccountIds, NN_ARRAY_SIZE(s_TestAccountIds)));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        for (int i = 0; i < NN_ARRAY_SIZE(s_TestAccountIds); i++)
        {
            ASSERT_FALSE(s_Profiles[i].IsValid());
        }
    }
    {
        static nn::friends::ProfileExtra s_Profiles[nn::friends::GetProfileCountMax];

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetProfileExtraList(&context, s_Profiles, s_Users[0],
            s_TestAccountIds, NN_ARRAY_SIZE(s_TestAccountIds)));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        for (int i = 0; i < NN_ARRAY_SIZE(s_TestAccountIds); i++)
        {
            ASSERT_FALSE(s_Profiles[i].IsValid());
        }
    }
}
