﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testFriends_Common.h"

#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiPrivate.h>
#include <nn/ns/ns_UserResourceManagementApi.h>

namespace
{
    nn::account::Uid s_Users[nn::account::UserCountMax] = {};
    int s_UserCount = 0;
}

namespace
{
    void DeleteUserAccount(const nn::account::Uid& user) NN_NOEXCEPT
    {
        nn::friends::DaemonSuspension suspension;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SuspendDaemon(&suspension));

        nn::ns::ProgressMonitorForDeleteUserSaveDataAll progress;
        NNT_ASSERT_RESULT_SUCCESS(nn::ns::DeleteUserSaveDataAll(&progress, user));

        nn::os::SystemEvent deletedEvent;
        progress.GetSystemEvent(&deletedEvent);

        deletedEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(progress.GetResult());

        NNT_ASSERT_RESULT_SUCCESS(nn::account::DeleteUser(user));
    }
}

class FriendsDaemonStatus : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();

        nnt::friends::LoadAccounts(&s_UserCount, s_Users, NN_ARRAY_SIZE(s_Users));
        NN_ABORT_UNLESS_GREATER_EQUAL(s_UserCount, 1);

        nn::friends::SetOption(nn::friends::OptionAdmin_CheckUserStatus, 0);
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }
};

TEST_F(FriendsDaemonStatus, Basic)
{
    nn::friends::DaemonSuspension suspension;
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::SuspendDaemon(&suspension));

    nn::account::UserHandle handle = {};
    NNT_ASSERT_RESULT_SUCCESS(nn::account::OpenUser(&handle, s_Users[0]));

    NN_LOG("User account opened.\n");

    NN_UTIL_SCOPE_EXIT
    {
        nn::account::CloseUser(handle);

        NN_LOG("User account closed.\n");
    };

    NN_LOG("  Sleep 3 seconds ...\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    NN_LOG("  Sleep 3 seconds done!\n");

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::RequestSyncFriendList(s_Users[0]));
    NN_LOG("nn::friends::RequestSyncFriendList()\n");

    NN_LOG("  Sleep 3 seconds ...\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    NN_LOG("  Sleep 3 seconds done!\n");
}

TEST_F(FriendsDaemonStatus, DeletionAging)
{
    nn::account::Uid users[nn::account::UserCountMax] = {};
    int count;

    NNT_ASSERT_RESULT_SUCCESS(nn::account::ListAllUsers(&count, users, NN_ARRAY_SIZE(users)));

    nn::account::Finalize();
    nn::account::InitializeForAdministrator();

    if (count == nn::account::UserCountMax)
    {
        ASSERT_NO_FATAL_FAILURE(DeleteUserAccount(users[count - 1]));
    }

    for (int i = 0; i < 100; i++)
    {
        nn::account::Uid user;

        NNT_ASSERT_RESULT_SUCCESS(nn::account::BeginUserRegistration(&user));
        NNT_ASSERT_RESULT_SUCCESS(nn::account::CompleteUserRegistration(user));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(rand() % 500));

        ASSERT_NO_FATAL_FAILURE(DeleteUserAccount(user));
    }
}
