﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testFriends_Common.h"

namespace
{
    nn::account::Uid s_Users[nn::account::UserCountMax] = {};
    nn::account::NetworkServiceAccountId s_AccountIds[nn::account::UserCountMax] = {};
    int s_UserCount = 0;

    nn::account::UserHandle s_UserHandle = {};
}

namespace
{
    void* MallocFunction(size_t size) NN_NOEXCEPT
    {
        void* p = std::malloc(size);

        NN_LOG("Malloc p=%p, size=%zu\n", p, size);
        return p;
    }

    void FreeFunction(void* p, size_t) NN_NOEXCEPT
    {
        NN_LOG("Free   p=%p\n", p);
        std::free(p);
    }
}

class FriendsBasic : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());

        nn::friends::SetAllocator(MallocFunction, FreeFunction);
        nn::friends::Initialize();

        nnt::friends::LoadAccounts(&s_UserCount, s_Users, s_AccountIds, NN_ARRAY_SIZE(s_Users));
        NN_ABORT_UNLESS_GREATER_EQUAL(s_UserCount, 1);
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }
};

TEST_F(FriendsBasic, NotOpened)
{
    static nn::account::NetworkServiceAccountId s_Friends[nn::friends::FriendCountMax] = {};
    int count = 0;
    nn::friends::FriendFilter filter = {};

    NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultUserNotOpened,
        nn::friends::GetFriendList(&count, s_Friends, s_Users[0], 0, NN_ARRAY_SIZE(s_Friends), filter));
}

TEST_F(FriendsBasic, OpenUser)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&s_UserHandle, s_Users[0]));
}

TEST_F(FriendsBasic, FriendList)
{
    {
        static nn::account::NetworkServiceAccountId s_Friends[nn::friends::FriendCountMax] = {};
        int count = 0;
        nn::friends::FriendFilter filter = {};

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendList(&count, s_Friends, 0, NN_ARRAY_SIZE(s_Friends), filter));
    }
    {
        static nn::friends::Friend s_Friends[nn::friends::FriendCountMax] = {};
        int count = 0;
        nn::friends::FriendFilter filter = {};

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendList(&count, s_Friends, 0, NN_ARRAY_SIZE(s_Friends), filter));

        nnt::friends::Dump(s_Friends, count);
    }
}

TEST_F(FriendsBasic, BlockedUserList)
{
    static nn::account::NetworkServiceAccountId s_BlockedUsers[nn::friends::BlockedUserCountMax] = {};
    int count = 0;

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetBlockedUserList(&count, s_BlockedUsers, 0, NN_ARRAY_SIZE(s_BlockedUsers)));
}

TEST_F(FriendsBasic, InternetRequestNotAccepted)
{
    nn::friends::Profile profile;

    nn::friends::AsyncContext context;
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetProfileList(&context, &profile, s_Users[0], s_AccountIds, 1));

    nn::os::SystemEvent completionEvent;
    NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

    completionEvent.Wait();

    NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultInternetRequestNotAccepted, context.GetResult());
}

TEST_F(FriendsBasic, ProfileList)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    static nn::friends::Profile s_Profiles[nn::account::UserCountMax];

    nn::friends::AsyncContext context;
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetProfileList(&context, s_Profiles, s_AccountIds, s_UserCount));

    nn::os::SystemEvent completionEvent;
    NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

    completionEvent.Wait();

    NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

    for (int i = 0; i < s_UserCount; i++)
    {
        ASSERT_TRUE(s_Profiles[i].IsValid());

        EXPECT_EQ(s_Profiles[i].GetAccountId(), s_AccountIds[i]);
    }
    for (int i = s_UserCount; i < NN_ARRAY_SIZE(s_Profiles); i++)
    {
        ASSERT_FALSE(s_Profiles[i].IsValid());
    }

    nnt::friends::Dump(s_Profiles, s_UserCount);
}

TEST_F(FriendsBasic, AsyncContext)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::AsyncContext context;

    // HasDone を使用した待機
    {
        nn::friends::Profile profile;

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetProfileList(&context, &profile, s_AccountIds, 1));

        bool done = false;

        do
        {
            NNT_ASSERT_RESULT_SUCCESS(context.HasDone(&done));

            if (!done)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
            }
        }
        while (!done);

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    // AsyncContext 使い回し
    {
        nn::friends::Profile profile;

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetProfileList(&context, &profile, s_AccountIds, 1));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    // 即時キャンセル
    {
        nn::friends::Profile profile;

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetProfileList(&context, &profile, s_AccountIds, 1));
        NNT_ASSERT_RESULT_SUCCESS(context.Cancel());

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_EXPECT_RESULT_FAILURE(nn::friends::ResultCanceled, context.GetResult());
    }
    // 数ミリ秒後キャンセル
    {
        nn::friends::Profile profile;

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetProfileList(&context, &profile, s_AccountIds, 1));
        NNT_ASSERT_RESULT_SUCCESS(context.Cancel());

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.TimedWait(nn::TimeSpan::FromMilliSeconds(10));
        context.Cancel();
        completionEvent.Wait();

        if (context.GetResult().IsFailure())
        {
            NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultCanceled, context.GetResult());
        }
    }
    // 即時破棄
    for (int i = 0; i < 100; i++)
    {
        nn::friends::Profile profile;

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetProfileList(&context, &profile, s_AccountIds, 1));

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(i));
    }
}

TEST_F(FriendsBasic, CloseUser)
{
    nn::account::CloseUser(s_UserHandle);
}
