﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/erpt.h>
#include <nn/erpt/erpt_Manager.h>
#include <nn/eupld.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nnt.h>

using namespace nn;

bool IsNetworkAvailable()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nifm::Initialize());
    nifm::SubmitNetworkRequestAndWait();
    NN_UTIL_SCOPE_EXIT{ nifm::CancelNetworkRequest(); };
    return nifm::IsNetworkAvailable();
}

nn::Result IsOptedInAccount(bool* pOutvalue, const nn::account::Uid& uid)
{
    static Bit8 buffer[nn::account::RequiredBufferSizeForCachedNintendoAccountInfo];
    nn::account::NetworkServiceAccountManager accountManager;
    NN_RESULT_DO(nn::account::GetNetworkServiceAccountManager(&accountManager, uid));

    bool isCachRefreshStarted = false;
    nn::account::AsyncContext accountContext;
    NN_RESULT_DO(accountManager.RefreshCachedNintendoAccountInfoAsyncIfTimeElapsed(&isCachRefreshStarted, &accountContext, nn::TimeSpan::FromMinutes(1)));

    if( isCachRefreshStarted )
    {
        nn::os::SystemEvent accountContextEvent;
        NN_ABORT_UNLESS_RESULT_SUCCESS(accountContext.GetSystemEvent(&accountContextEvent));
        if( !accountContextEvent.TimedWait(nn::TimeSpan::FromSeconds(120)) )
        {
            NN_LOG("Failed to refresh cached nintendo account info : timeout (120s)\n");
            accountContext.Cancel();
            accountContextEvent.Wait();
            NN_RESULT_THROW(nn::os::ResultTimedout());
        }
        NN_RESULT_DO(accountContext.GetResult());
    }

    nn::account::CachedNintendoAccountInfoForSystemService accountInfo;
    NN_RESULT_DO(accountManager.LoadCachedNintendoAccountInfo(&accountInfo, buffer, sizeof(buffer)));

    *pOutvalue = accountInfo.GetAnalyticsOptedInFlag();
    NN_RESULT_SUCCESS;
}


bool IsAnyOptedInNetworkAccountLinked()
{
    account::InitializeForSystemService();

    nn::nifm::NetworkConnection nc;
    nc.SubmitRequestAndWait();
    NN_ABORT_UNLESS(nc.IsAvailable());

    int                userIdCount;
    nn::account::Uid   userIds[nn::account::UserCountMax] = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&userIdCount, userIds, NN_ARRAY_SIZE(userIds)));

    for( int i = 0; i < userIdCount; i++ )
    {
        bool isOptedIn = false;
        auto result = IsOptedInAccount(&isOptedIn, userIds[i]);
        if( result.IsFailure() )
        {
            NN_LOG("Failed to check opted-in flag : %03d-%04d (0x%08x)\n", result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
            continue;
        }
        if( isOptedIn )
        {
            return true;
        }
    }
    return false;
}

void CreateErrorReport()
{
    const char errorCode[] = "1234-1234";
    erpt::Context context(nn::erpt::ErrorInfo);
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::ErrorCode, errorCode, sizeof(errorCode)));
    NNT_ASSERT_RESULT_SUCCESS(context.CreateReport(erpt::ReportType_Invisible));
}

void UploadErrorReport()
{
    nn::nifm::NetworkConnection nc;
    nc.SubmitRequestAndWait();
    ASSERT_TRUE(nc.IsAvailable());

    erpt::Manager manager;
    NNT_ASSERT_RESULT_SUCCESS(manager.Initialize());
    erpt::ReportList reportList;
    NNT_ASSERT_RESULT_SUCCESS(manager.GetReportList(reportList));
    erpt::ReportId reportIds[erpt::NumberOfReports];
    for( uint32_t i = 0; i < reportList.reportCount; i++ )
    {
        reportIds[i] = reportList.Report[i].reportId;
    }
    eupld::Request uploadRequest;
    NNT_ASSERT_RESULT_SUCCESS(uploadRequest.Initialize());
    NNT_ASSERT_RESULT_SUCCESS(uploadRequest.UploadSelected(reportIds, reportList.reportCount));
    ASSERT_TRUE(uploadRequest.GetEventPointer()->TimedWait(nn::TimeSpan::FromSeconds(60)));
    NNT_ASSERT_RESULT_SUCCESS(uploadRequest.GetResult());

    eupld::ReportUploadList reportStatus;
    NNT_ASSERT_RESULT_SUCCESS(uploadRequest.GetUploadStatus(reportStatus));
    NN_LOG("Manual Upload Result\n");
    for( uint32_t i = 0; i < reportList.reportCount; i++ )
    {
        char idStr[64];
        reportStatus.Report[i].reportId.u.uuidRFC4122.ToString(idStr, sizeof(idStr));
        NN_LOG("[%02u] %s : result = 0x%08x\n", i, idStr, reportStatus.Report[i].result.GetInnerValueForDebug());
        NNT_EXPECT_RESULT_SUCCESS(reportStatus.Report[i].result);
    }
}

// The tests here are intented to be run by the following command.
// NNTest.cmd -Platform NXFP2-a64 -File Tests/Eupld/EupldAuto/ErrorUploadAuto-spec.NX.testlist.yml

// To successfully run this test, error report auto upload must be ENABLED.
TEST(EupldAuto, CheckAutoReportUploadSuccess)
{
    ASSERT_TRUE(IsNetworkAvailable()) << "Network is not available. Test settings might be wrong.\n";
    ASSERT_TRUE(IsAnyOptedInNetworkAccountLinked()) << "No Opted-In Nintendo Account is linked.\n";

    CreateErrorReport();

    // Wait for the created report to be uploaded.
    erpt::Manager manager;
    erpt::ReportList reportList;
    NNT_ASSERT_RESULT_SUCCESS(manager.Initialize());
    const int RetryCountMax = 10;
    for( int i = 0; i < RetryCountMax; i++ )
    {
        NNT_ASSERT_RESULT_SUCCESS(manager.GetReportList(reportList));
        ASSERT_GE(reportList.reportCount, 0u);
        if( reportList.Report[0].reportFlags.Test(erpt::ReportFlag::Transmitted::Index) )
        {
            NN_LOG("[%d/%d] Uploaded.\n", i + 1, RetryCountMax);
            SUCCEED();
            return;
        }
        NN_LOG("[%d/%d] Not uploaded yet ...\n", i + 1, RetryCountMax);
        os::SleepThread(TimeSpan::FromSeconds(5));
    }
    // If upload failed, try manual upload to see the result.
    UploadErrorReport();
    FAIL();
}

// To successfully run this test, error report auto upload must be DISABLED.
TEST(EupldAuto, CheckAutoReportUploadFailure)
{
    ASSERT_TRUE(IsNetworkAvailable()) << "Network is not available. Test settings might be wrong.\n";
    ASSERT_FALSE(IsAnyOptedInNetworkAccountLinked()) << "Opted-In Nintendo Account is linked.\n";

    CreateErrorReport();

    erpt::Manager manager;
    erpt::ReportList reportList;
    NNT_ASSERT_RESULT_SUCCESS(manager.Initialize());
    const int RetryCountMax = 10;
    for( int i = 0; i < RetryCountMax; i++ )
    {
        NNT_ASSERT_RESULT_SUCCESS(manager.GetReportList(reportList));
        ASSERT_GE(reportList.reportCount, 0u);
        if( reportList.Report[0].reportFlags.Test(erpt::ReportFlag::Transmitted::Index) )
        {
            NN_LOG("[%d/%d] Uploaded, but it shoudn't.\n", i + 1, RetryCountMax);
            FAIL();
            return;
        }
        NN_LOG("[%d/%d] Not uploaded yet (expected)...\n", i + 1, RetryCountMax);
        os::SleepThread(TimeSpan::FromSeconds(5));
    }
    SUCCEED();
}
