﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/*
 *               Copyright (C) 2008, BroadOn Communications Corp.
 *
 *  These coded instructions, statements, and computer programs contain
 *  unpublished  proprietary information of BroadOn Communications Corp.,
 *  and are protected by Federal copyright law. They may not be disclosed
 *  to third  parties or copied or duplicated in any form, in whole or in
 *  part, without the prior written consent of BroadOn Communications Corp.
 *
 */
#include <nnt/nntest.h>
#include <nn/nn_Macro.h>

#define TESTNAME "Hash2test"
#define NEED_NUMTICKS

#include "testEs_Testcommon.h"

#include <nn/csl/sha1.h>
#include <nn/csl/sha256.h>

const int IOSC_SHA1_BLOCKSIZE = 64;

NN_ALIGNAS( ALIGN_SIZE ) u8 hashdata[( 64 * 3 )];
NN_ALIGNAS( ALIGN_SIZE ) u8 testdata[( 64 * 1024 )];

static unsigned long seed = 1;

unsigned long testRand()
{
    seed = 0x343fd * seed + 0x269ec3;
    return ( (seed)&0xffffffff );
}

int doSha1Tests()
{
    SHA1Context sha;
    u8 hash[20];

    IOSCHash hashresult;
    IOSCHashContext ioshash;
    int iter;
    u32 multiplesize;

    IOSCError err;
    int i;
    int hashsize;
    int hashsizemult;
#ifdef PERF
    u32 startTime, endTime;
#endif

    /*
     * Test of IOSC_GenerateHash in SHA1 mode
     */
    for( i = 0; i < sizeof( hashdata ); i++ )
    {
        hashdata[i] = static_cast<u8>( i & 0xff );
    }

    /* pick sizes to trigger padding of one and two blocks */
    for( hashsize = 1; hashsize < 128 + 56 + 8; hashsize++ )
    {
        /* trigger two block padding */
        SHA1Reset( &sha );
        SHA1Input( &sha, hashdata, hashsize );
        SHA1Result( &sha, hash );

        err = IOSC_GenerateHash( ioshash, 0, 0, IOSC_HASH_FIRST, NULL );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash first call" );

        err = IOSC_GenerateHash( ioshash, hashdata, hashsize, IOSC_HASH_LAST, hashresult );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash HASH_LAST call" );

        if( memcmp( hash, hashresult, sizeof( hash ) ) == 0 )
        {
            LOG_PASS( "Generate hash 0 PASS, hashsize = %d", hashsize );
        }
        else
        {
            LOG_FAIL( "Generate hash 0 FAIL, hashsize = %d", hashsize );
            EXIT( -1 );
        }

        /*
         * Now use HASH_MIDDLE one byte at time combined with HASH_RESTART
         * just to be pathological about it
         */
        memset( hashresult, 0, sizeof( hashresult ) );

        err = IOSC_GenerateHash( ioshash, hashdata, 1, IOSC_HASH_FIRST, 0 );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash first call" );

        for( i = 1; i < hashsize; i++ )
        {
            err = IOSC_GenerateHash( ioshash, NULL, 0, IOSC_HASH_RESTART, NULL );
            CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash HASH_RESTART" );
            err = IOSC_GenerateHash( ioshash, &hashdata[i], 1, IOSC_HASH_MIDDLE, NULL );
            CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash HASH_MIDDLE" );
        }

        err = IOSC_GenerateHash( ioshash, NULL, 0, IOSC_HASH_LAST, hashresult );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash HASH_LAST call" );

        if( memcmp( hash, hashresult, sizeof( hash ) ) == 0 )
        {
            LOG_PASS( "Generate hash byte-at-a-time 0 PASS, hashsize = %d", hashsize );
        }
        else
        {
            LOG_FAIL( "Generate hash byte-at-a-time 0 FAIL, hashsize = %d", hashsize );
            EXIT( -1 );
        }
    }

    hashsizemult = 128;
    SHA1Reset( &sha );
    SHA1Input( &sha, hashdata, hashsizemult );
    SHA1Result( &sha, hash );

    err = IOSC_GenerateHash( ioshash, hashdata, hashsizemult, IOSC_HASH_FIRST, 0 );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash first call" );

    err = IOSC_GenerateHash( ioshash, 0, 0, IOSC_HASH_LAST, hashresult );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash second call" );
    if( memcmp( hash, hashresult, sizeof( hash ) ) == 0 )
    {
        LOG_PASS( "Generate hash 1" );
    }
    else
    {
        LOG_FAIL( "Generate hash 1" );
        EXIT( -1 );
    }

    SHA1Reset( &sha );
    SHA1Input( &sha, hashdata, hashsize );
    SHA1Result( &sha, hash );

    /* another usage */
    memset( hashresult, 0, sizeof( hashresult ) );
    err = IOSC_GenerateHash( ioshash, 0, 0, IOSC_HASH_FIRST, 0 );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash first call" );
    err = IOSC_GenerateHash( ioshash, hashdata, hashsize, IOSC_HASH_LAST, hashresult );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash second call" );

    if( memcmp( hash, hashresult, sizeof( hash ) ) == 0 )
    {
        LOG_PASS( "Generate hash 2" );
    }
    else
    {
        LOG_FAIL( "Generate hash 2" );
        EXIT( -1 );
    }

    /* three calls */
    for( iter = 0; iter < 3; iter++ )
    {
        /* change value in input */
        for( i = 0; i < hashsize; i++ )
        {
            hashdata[i] = static_cast<u8>( testRand() & 0xff );
        }
        SHA1Reset( &sha );
        SHA1Input( &sha, hashdata, hashsize );
        SHA1Result( &sha, hash );

        memset( hashresult, 0, sizeof( hashresult ) );
        multiplesize = ( hashsize & ~( IOSC_SHA1_BLOCKSIZE - 1 ) );
        err = IOSC_GenerateHash( ioshash, 0, 0, IOSC_HASH_FIRST, 0 );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash first call" );
        err = IOSC_GenerateHash( ioshash, hashdata, multiplesize, IOSC_HASH_MIDDLE, 0 );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash second call" );

        err = IOSC_GenerateHash( ioshash, hashdata + multiplesize, ( hashsize - multiplesize ), IOSC_HASH_LAST, hashresult );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash last call" );

        if( memcmp( hash, hashresult, sizeof( hash ) ) == 0 )
        {
            LOG_PASS( "Generate hash 3" );
        }
        else
        {
            LOG_FAIL( "Generate hash 3" );
            EXIT( -1 );
        }
    }

/* check alignment */
#if defined( RVL )
    err = IOSC_GenerateHash( ioshash, 0, 0, IOSC_HASH_FIRST, 0 );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash first call" );

    err = IOSC_GenerateHash( ioshash, hashdata + 16, 30, IOSC_HASH_LAST, hashresult );
    if( err != IOSC_ERROR_INVALID_ALIGN )
    {
        LOG_FAIL( "Error Check for RVL Alignment" );
        EXIT( -1 );
    }
    else
    {
        LOG_PASS( "Error Check for RVL Alignment PASS" );
    }
#elif defined( NET_CARD )
    err = IOSC_GenerateHash( ioshash, hashdata + 8, 30, IOSC_HASH_LAST, hashresult );
    if( err != IOSC_ERROR_INVALID_ALIGN )
    {
        LOG_FAIL( "Error Check for NC Alignment FAIL err = %d", err );
        EXIT( -1 );
    }
    else
    {
        LOG_PASS( "Error Check for NC Alignment" );
    }
#endif

    memset( testdata, 0, sizeof( testdata ) );
    /* Specific test : generalise later with randoms */
    testdata[0] = 0x6c;
    testdata[1] = 0xfa;
    testdata[2] = 0x7a;
    testdata[3] = 0x54;
    testdata[4] = 0xe4;
    testdata[5] = 0x26;
    testdata[6] = 0x79;
    testdata[7] = 0xa0;
    testdata[8] = 0x4e;
    testdata[9] = 0x7f;
    testdata[10] = 0xfe;
    testdata[11] = 0x23;
    testdata[12] = 0x4d;
    testdata[13] = 0x5f;
    testdata[14] = 0xb9;
    testdata[15] = 0xd3;
    testdata[16] = 0x59;
    testdata[17] = 0xb7;
    testdata[18] = 0xcc;
    testdata[19] = 0xf2;

#ifdef USE_OBSOLETE_CODE
#if !defined( LINUX ) && !defined( SDK_TWL ) && !defined( NN_PLATFORM_CTR )
    /* Negative test, exceed size */
    err = IOSC_GenerateHash( ioshash, testdata, ( 64 * 1024 ) + 64, IOSC_HASH_FIRST, 0 );
    if( err != IOSC_ERROR_INVALID_SIZE )
    {
        LOG_FAIL( "Max Size Error Test FAIL err = %d", err );
        EXIT( -1 );
    }
    else
    {
        LOG_PASS( "Max Size Error Test" );
    }
    err = IOSC_GenerateHash( ioshash, testdata, ( 64 * 1024 ) - 16, IOSC_HASH_FIRST, 0 );
    if( err != IOSC_ERROR_INVALID_SIZE )
    {
        LOG_FAIL( "Multiple Size Error Test FAIL err = %d", err );
        EXIT( -1 );
    }
    else
    {
        LOG_PASS( "Multiple Size Error Test" );
    }
#endif
#endif

    /* Largest Block with chaining test */
    err = IOSC_GenerateHash( ioshash, testdata, ( 64 * 1024 ), IOSC_HASH_FIRST, 0 );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash first call" );

    /* Compute in software */
    SHA1Reset( &sha );
    SHA1Input( &sha, testdata, ( 64 * 1024 ) );
    memset( testdata, 0, sizeof( testdata ) );

    err = IOSC_GenerateHash( ioshash, testdata, ( 32 * 1024 ), IOSC_HASH_LAST, hashresult );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash last call" );

    /* Compute in software */
    SHA1Input( &sha, testdata, ( 32 * 1024 ) );
    SHA1Result( &sha, hash );

    /* Compare */

    if( memcmp( hash, hashresult, sizeof( hash ) ) == 0 )
    {
        LOG_PASS( "Generate hash LARGE (with chaining) TEST" );
    }
    else
    {
        LOG_FAIL( "Generate hash LARGE (with chaining) TEST" );
        EXIT( -1 );
    }

    /* Blocks Size test */
    for( i = 0; i < sizeof( testdata ); i++ )
    {
        testdata[i] = i & 0xff;
    }
    for( hashsize = 64; hashsize <= ( 64 * 1024 ); hashsize = hashsize * 2 )
    {
        err = IOSC_GenerateHash( ioshash, testdata, hashsize, IOSC_HASH_FIRST, 0 );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash first call" );
        err = IOSC_GenerateHash( ioshash, 0, 0, IOSC_HASH_LAST, hashresult );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash last call" );

        /* Compute in software */
        SHA1Reset( &sha );
        SHA1Input( &sha, testdata, hashsize );
        SHA1Result( &sha, hash );

        /* Compare */

        if( memcmp( hash, hashresult, sizeof( hash ) ) == 0 )
        {
            LOG_PASS( "Generate hash LARGE TEST, size = %d", hashsize );
        }
        else
        {
            LOG_FAIL( "Generate hash LARGE TEST, size = %d", hashsize );
            EXIT( -1 );
        }
    }
#ifdef PERF
    hashsize = 32 * 1024;
    startTime = IO_READ( SYS_TIMER_REG );
    err = IOSC_GenerateHash( ioshash, testdata, hashsize, IOSC_HASH_FIRST, 0 );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash first call" );
    err = IOSC_GenerateHash( ioshash, 0, 0, IOSC_HASH_LAST, hashresult );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "GenerateHash last call" );
    endTime = IO_READ( SYS_TIMER_REG );
    NN_LOG( "Number of Ticks for IOSC_GenerateHash = %d, Time = %d us per %d bytes\n", getNumTicks( startTime, endTime ),
            getNumTicks( startTime, endTime ) * NUM_TICKS / CLOCK_FREQ, hashsize );
#endif
    return 0;
} // NOLINT (readability/fn_size)

int doSha256Tests()
{
    SHA256Context sha;
    u8 hash[32];

    IOSCHash256 hashresult;
    IOSCHashContext ioshash;
    int iter;
    u32 multiplesize;

    IOSCError err;
    int i;
    int hashsize;
    int hashsizemult;
#ifdef PERF
    u32 startTime, endTime;
#endif

    /*
     * Test of IOSC_GenerateHash in SHA256 mode
     */
    for( i = 0; i < sizeof( hashdata ); i++ )
    {
        hashdata[i] = static_cast<u8>( i & 0xff );
    }

    /* pick sizes to trigger padding of one and two blocks */
    for( hashsize = 1; hashsize < 128 + 56 + 8; hashsize++ )
    {
        /* Use software implementation as reference */
        SHA256Reset( &sha );
        SHA256Input( &sha, hashdata, hashsize );
        SHA256Result( &sha, hash );

        err = IOSC_GenerateHash( ioshash, 0, 0, IOSC_SHA256_INIT, NULL );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 first call" );

        err = IOSC_GenerateHash( ioshash, hashdata, hashsize, IOSC_SHA256_FINAL, hashresult );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 HASH_LAST call" );

        if( memcmp( hash, hashresult, sizeof( hash ) ) == 0 )
        {
            LOG_PASS( "Generate SHA-256 0 PASS, hashsize = %d, digestsize = %d", hashsize, sizeof( hash ) );
        }
        else
        {
            LOG_FAIL( "Generate SHA-256 0 FAIL, hashsize = %d", hashsize );
            EXIT( -1 );
        }

        /*
         * Now use HASH_MIDDLE one byte at time combined with HASH_RESTART
         * just to be pathological about it
         */
        memset( hashresult, 0, sizeof( hashresult ) );

        err = IOSC_GenerateHash( ioshash, hashdata, 1, IOSC_SHA256_INIT, 0 );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 first call" );

        for( i = 1; i < hashsize; i++ )
        {
            err = IOSC_GenerateHash( ioshash, NULL, 0, IOSC_SHA256_RESTART, NULL );
            CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 HASH_RESTART" );
            err = IOSC_GenerateHash( ioshash, &hashdata[i], 1, IOSC_SHA256_UPDATE, NULL );
            CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 HASH_MIDDLE" );
        }

        err = IOSC_GenerateHash( ioshash, NULL, 0, IOSC_SHA256_FINAL, hashresult );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 HASH_LAST call" );

        if( memcmp( hash, hashresult, sizeof( hash ) ) == 0 )
        {
            LOG_PASS( "Generate SHA-256 byte-at-a-time 0 PASS, hashsize = %d", hashsize );
        }
        else
        {
            LOG_FAIL( "Generate SHA-256 byte-at-a-time 0 FAIL, hashsize = %d", hashsize );
            EXIT( -1 );
        }
    }

    hashsizemult = 128;
    SHA256Reset( &sha );
    SHA256Input( &sha, hashdata, hashsizemult );
    SHA256Result( &sha, hash );

    err = IOSC_GenerateHash( ioshash, hashdata, hashsizemult, IOSC_HASH_FIRST | IOSC_HASH_SHA256, 0 );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 first call" );

    err = IOSC_GenerateHash( ioshash, 0, 0, IOSC_HASH_LAST | IOSC_HASH_SHA256, hashresult );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 second call" );
    if( memcmp( hash, hashresult, sizeof( hash ) ) == 0 )
    {
        LOG_PASS( "Generate SHA-256 1" );
    }
    else
    {
        LOG_FAIL( "Generate SHA-256 1" );
        EXIT( -1 );
    }

    SHA256Reset( &sha );
    SHA256Input( &sha, hashdata, hashsize );
    SHA256Result( &sha, hash );

    /* another usage */
    memset( hashresult, 0, sizeof( hashresult ) );
    err = IOSC_GenerateHash( ioshash, 0, 0, IOSC_HASH_FIRST | IOSC_HASH_SHA256, 0 );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 first call" );
    err = IOSC_GenerateHash( ioshash, hashdata, hashsize, IOSC_HASH_LAST | IOSC_HASH_SHA256, hashresult );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 second call" );

    if( memcmp( hash, hashresult, sizeof( hash ) ) == 0 )
    {
        LOG_PASS( "Generate SHA-256 2" );
    }
    else
    {
        LOG_FAIL( "Generate SHA-256 2" );
        EXIT( -1 );
    }

    /* three calls */
    for( iter = 0; iter < 3; iter++ )
    {
        /* change value in input */
        for( i = 0; i < hashsize; i++ )
        {
            hashdata[i] = static_cast<u8>( testRand() & 0xff );
        }
        SHA256Reset( &sha );
        SHA256Input( &sha, hashdata, hashsize );
        SHA256Result( &sha, hash );

        memset( hashresult, 0, sizeof( hashresult ) );
        multiplesize = ( hashsize & ~( IOSC_SHA1_BLOCKSIZE - 1 ) );
        err = IOSC_GenerateHash( ioshash, 0, 0, IOSC_HASH_FIRST | IOSC_HASH_SHA256, 0 );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 first call" );
        err = IOSC_GenerateHash( ioshash, hashdata, multiplesize, IOSC_HASH_MIDDLE | IOSC_HASH_SHA256, 0 );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 second call" );

        err = IOSC_GenerateHash( ioshash, hashdata + multiplesize, ( hashsize - multiplesize ), IOSC_HASH_LAST | IOSC_HASH_SHA256, hashresult );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 last call" );

        if( memcmp( hash, hashresult, sizeof( hash ) ) == 0 )
        {
            LOG_PASS( "Generate SHA-256 3" );
        }
        else
        {
            LOG_FAIL( "Generate SHA-256 3" );
            EXIT( -1 );
        }
    }

/* check alignment */
#if defined( RVL )
    err = IOSC_GenerateHash( ioshash, 0, 0, IOSC_HASH_FIRST | IOSC_HASH_SHA256, 0 );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 first call" );

    err = IOSC_GenerateHash( ioshash, hashdata + 16, 30, IOSC_HASH_LAST | IOSC_HASH_SHA256, hashresult );
    if( err != IOSC_ERROR_INVALID_ALIGN )
    {
        LOG_FAIL( "Error Check for RVL Alignment" );
        EXIT( -1 );
    }
    else
    {
        LOG_PASS( "Error Check for RVL Alignment PASS" );
    }
#elif defined( NET_CARD )
    err = IOSC_GenerateHash( ioshash, hashdata + 8, 30, IOSC_HASH_LAST | IOSC_HASH_SHA256, hashresult );
    if( err != IOSC_ERROR_INVALID_ALIGN )
    {
        LOG_FAIL( "Error Check for NC Alignment FAIL err = %d", err );
        EXIT( -1 );
    }
    else
    {
        LOG_PASS( "Error Check for NC Alignment" );
    }
#endif

    memset( testdata, 0, sizeof( testdata ) );
    /* Specific test : generalise later with randoms */
    testdata[0] = 0x6c;
    testdata[1] = 0xfa;
    testdata[2] = 0x7a;
    testdata[3] = 0x54;
    testdata[4] = 0xe4;
    testdata[5] = 0x26;
    testdata[6] = 0x79;
    testdata[7] = 0xa0;
    testdata[8] = 0x4e;
    testdata[9] = 0x7f;
    testdata[10] = 0xfe;
    testdata[11] = 0x23;
    testdata[12] = 0x4d;
    testdata[13] = 0x5f;
    testdata[14] = 0xb9;
    testdata[15] = 0xd3;
    testdata[16] = 0x59;
    testdata[17] = 0xb7;
    testdata[18] = 0xcc;
    testdata[19] = 0xf2;

#ifdef USE_OBSOLETE_CODE
#if !defined( LINUX ) && !defined( SDK_TWL ) && !defined( NN_PLATFORM_CTR )
    /* Negative test, exceed size */
    err = IOSC_GenerateHash( ioshash, testdata, ( 64 * 1024 ) + 64, IOSC_HASH_FIRST | IOSC_HASH_SHA256, 0 );
    if( err != IOSC_ERROR_INVALID_SIZE )
    {
        LOG_FAIL( "Max Size Error Test FAIL err = %d", err );
        EXIT( -1 );
    }
    else
    {
        LOG_PASS( "Max Size Error Test" );
    }
    err = IOSC_GenerateHash( ioshash, testdata, ( 64 * 1024 ) - 16, IOSC_HASH_FIRST | IOSC_HASH_SHA256, 0 );
    if( err != IOSC_ERROR_INVALID_SIZE )
    {
        LOG_FAIL( "Multiple Size Error Test FAIL err = %d", err );
        EXIT( -1 );
    }
    else
    {
        LOG_PASS( "Multiple Size Error Test" );
    }
#endif
#endif

    /* Largest Block with chaining test */
    err = IOSC_GenerateHash( ioshash, testdata, ( 64 * 1024 ), IOSC_HASH_FIRST | IOSC_HASH_SHA256, 0 );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 first call" );

    /* Compute in software */
    SHA256Reset( &sha );
    SHA256Input( &sha, testdata, ( 64 * 1024 ) );
    memset( testdata, 0, sizeof( testdata ) );

    err = IOSC_GenerateHash( ioshash, testdata, ( 32 * 1024 ), IOSC_HASH_LAST | IOSC_HASH_SHA256, hashresult );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 last call" );

    /* Compute in software */
    SHA256Input( &sha, testdata, ( 32 * 1024 ) );
    SHA256Result( &sha, hash );

    /* Compare */

    if( memcmp( hash, hashresult, sizeof( hash ) ) == 0 )
    {
        LOG_PASS( "Generate SHA-256 LARGE (with chaining) TEST" );
    }
    else
    {
        LOG_FAIL( "Generate SHA-256 LARGE (with chaining) TEST" );
        EXIT( -1 );
    }

    /* Blocks Size test */
    for( i = 0; i < sizeof( testdata ); i++ )
    {
        testdata[i] = i & 0xff;
    }
    for( hashsize = 64; hashsize <= ( 64 * 1024 ); hashsize = hashsize * 2 )
    {
        err = IOSC_GenerateHash( ioshash, testdata, hashsize, IOSC_HASH_FIRST | IOSC_HASH_SHA256, 0 );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 first call" );
        err = IOSC_GenerateHash( ioshash, 0, 0, IOSC_HASH_LAST | IOSC_HASH_SHA256, hashresult );
        CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 last call" );

        /* Compute in software */
        SHA256Reset( &sha );
        SHA256Input( &sha, testdata, hashsize );
        SHA256Result( &sha, hash );

        /* Compare */

        if( memcmp( hash, hashresult, sizeof( hash ) ) == 0 )
        {
            LOG_PASS( "Generate SHA-256 LARGE TEST, size = %d", hashsize );
        }
        else
        {
            LOG_FAIL( "Generate SHA-256 LARGE TEST, size = %d", hashsize );
            EXIT( -1 );
        }
    }
#ifdef PERF
    hashsize = 32 * 1024;
    startTime = IO_READ( SYS_TIMER_REG );
    err = IOSC_GenerateHash( ioshash, testdata, hashsize, IOSC_HASH_FIRST | IOSC_HASH_SHA256, 0 );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 first call" );
    err = IOSC_GenerateHash( ioshash, 0, 0, IOSC_HASH_LAST | IOSC_HASH_SHA256, hashresult );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "Generate SHA-256 last call" );
    endTime = IO_READ( SYS_TIMER_REG );
    NN_LOG( "Number of Ticks for IOSC_GenerateHash = %d, Time = %d us per %d bytes\n", getNumTicks( startTime, endTime ),
            getNumTicks( startTime, endTime ) * NUM_TICKS / CLOCK_FREQ, hashsize );
#endif
    return 0;
}  // NOLINT (readability/fn_size)

#if defined( SEPARATE_MAIN_PROGRAM )

int main( int argc, char **argv )
{
    IOSC_Initialize();
    doSha1Tests();
    doSha256Tests();
    EXIT( 0 );
    return 0;
}
#endif
