﻿/*
 *  Copyright 2005-2014 Acer Cloud Technology, Inc.
 *  All Rights Reserved.
 *
 *  This software contains confidential information and
 *  trade secrets of Acer Cloud Technology, Inc.
 *  Use, disclosure or reproduction is prohibited without
 *  the prior express written permission of Acer Cloud
 *  Technology, Inc.
 */

/*
 *               Copyright (C) 2008, BroadOn Communications Corp.
 *
 *  These coded instructions, statements, and computer programs contain
 *  unpublished  proprietary information of BroadOn Communications Corp.,
 *  and  are protected by Federal copyright law. They may not be disclosed
 *  to  third  parties or copied or duplicated in any form, in whole or in
 *  part, without the prior written consent of BroadOn Communications Corp.
 *
 */
#include <cstring>
#include <iostream>

#include <nnt/nntest.h>
#include <nn/nn_Macro.h>
#include <nn/util/util_FormatString.h>
#include <nn/fs.h>
#include <nn/nn_Result.h>

#define TESTNAME "CertTest"
#define NEED_NUMTICKS

#include "testEs_Testcommon.h"

#include <nn/ioscrypto/iosccert.h>

#ifdef RVL
extern IOSError createEncryptedRandoms( IOSCSecretKeyHandle handle, u8 *iv, u8 *rand, u32 size );
#endif

NN_ALIGNAS( ALIGN_SIZE ) static IOSCRsa2048RsaCert xscert;
NN_ALIGNAS( ALIGN_SIZE ) static IOSCRsa4096RsaCert cacert;
NN_ALIGNAS( ALIGN_SIZE ) static IOSCRsa2048EccCert mscert;
NN_ALIGNAS( ALIGN_SIZE ) static IOSCEccSignedCert returnedcert;
#ifdef NEED_CHANGE_DEV_CERT
NN_ALIGNAS( ALIGN_SIZE ) static IOSCEccEccCert devcert;
NN_ALIGNAS( ALIGN_SIZE ) static IOSCEccSignedCert appcert;
#endif

static char *makePath( const char *relPath )
{
    static char pathName[256];
    static const char *basePath = NULL;

    if( basePath == NULL )
    {
        if( ( ( basePath = getenv( "ROOT" ) ) == NULL ) && ( ( basePath = getenv( "ES_HOME" ) ) == NULL ) )
        {
            basePath = "./";
        }
    }
    nn::util::SNPrintf( pathName, sizeof pathName, "%s%s", basePath, relPath );

    return pathName;
}

static void readFile( const char *fileName, void* buffer, size_t size)
{
    std::string path;
    //#define NNT_ES_SIGLO_ROOT set by nact script or vcxporj to the root path of Siglo.
    const char* kResourcePath = "/Tests/Escore/Sources/Tests/CryptoRegression/testEs_CryptoRegression/";
    path = std::string(NNT_ES_SIGLO_ROOT) + std::string(kResourcePath) + fileName;

    nn::fs::FileHandle file;
    nn::Result result;
    result = nn::fs::OpenFile(&file, path.c_str(), nn::fs::OpenMode_Read);
    if ( result.IsFailure() )
    {
        LOG_MESSAGE( "open failed for %s", fileName );
        GTEST_FATAL_FAILURE_(fileName);
    }

    size_t readSize = 0;
    result = nn::fs::ReadFile(&readSize, file, 0, buffer, size, nn::fs::ReadOption());
    if (readSize == 0)
    {
        LOG_MESSAGE( "read failed for %s", fileName );
        GTEST_FATAL_FAILURE_(fileName);
    }
    nn::fs::CloseFile(file);
}

static void writeFile( const char *fileName, const void* buffer, size_t size)
{
    std::string path;
    //#define NNT_ES_SIGLO_ROOT set by nact script or vcxporj to the root path of Siglo.
    const char* kResourcePath = "/Tests/Escore/Sources/Tests/CryptoRegression/testEs_CryptoRegression/";
    path = std::string(NNT_ES_SIGLO_ROOT) + std::string(kResourcePath) + fileName;

    nn::fs::FileHandle file;
    nn::Result result;
    nn::fs::DeleteFile(path.c_str());
    result = nn::fs::CreateFile(path.c_str(), size);
    if ( result.IsFailure() )
    {
        LOG_MESSAGE( "create failed for %s", fileName );
        GTEST_FATAL_FAILURE_(fileName);
    }

    result = nn::fs::OpenFile(&file, path.c_str(), nn::fs::OpenMode_Write);
    if ( result.IsFailure() )
    {
        LOG_MESSAGE( "open failed for %s", fileName );
        GTEST_FATAL_FAILURE_(fileName);
    }

    result = nn::fs::WriteFile(file, 0, buffer, size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
    if (result.IsFailure())
    {
        LOG_MESSAGE( "write failed for %s", fileName );
        GTEST_FATAL_FAILURE_(fileName);
    }
    nn::fs::CloseFile(file);
}

int doCertTest()
{
    IOSCError err;
#ifdef PERF
    u32 startTime, endTime;
#endif
    IOSCRsaExponent exponent;
    IOSCRsa4096RsaCert *prsa4rsa;
    IOSCRsa2048RsaCert *prsa2rsa;
    IOSCPublicKeyHandle mspublickeyh, xspublickeyh, capublickeyh;
    IOSCSecretKeyHandle inputprivkeyh;
    u64 devId;

    IOSCRsaPublicKey2048 returnedpubkey;

#ifdef NEED_CHANGE_DEV_CERT
    IOSCEccPrivateKey inputprivate;
    IOSCCertName appname;
    int i;
    IOSCPublicKeyHandle devpublickeyh;
    IOSCPublicKeyHandle apppublickeyh;
#endif

    /*
     * Check that structure definitions are correctly packed
     */
    if( sizeof( IOSCRsa4096RsaCert ) != 1024 )
    {
        LOG_FAIL( "RSA 4096 RSA Cert size incorrect!" );
        EXIT( -1 );
    }
    if( sizeof( IOSCRsa2048RsaCert ) != 768 )
    {
        LOG_FAIL( "RSA 2048 RSA Cert size incorrect!" );
        EXIT( -1 );
    }
    if( sizeof( IOSCRsa2048EccCert ) != 576 )
    {
        LOG_FAIL( "RSA 2048 ECC Cert size incorrect!" );
        EXIT( -1 );
    }
    if( sizeof( IOSCEccEccCert ) != 384 )
    {
        LOG_FAIL( "ECC ECC Cert size incorrect!" );
        EXIT( -1 );
    }

    /*
     * Print out device ID
     */
    err = IOSC_GetData( IOSC_DEV_ID_HANDLE, &devId );
    CHECK_EQUAL( err, IOSC_ERROR_OK, "Get Device ID error" );
    LOG_MESSAGE( "Device ID = %08x", devId );

    /*
     * Test of CA import
     */
    readFile(makePath( "pki_data/ca_dpki.cert" ), &cacert, sizeof( IOSCRsa4096RsaCert ));

    IOSC_CreateObject( &capublickeyh, IOSC_PUBLICKEY_TYPE, IOSC_RSA2048_SUBTYPE );
#ifdef PERF
    startTime = IO_READ( SYS_TIMER_REG );
#endif
    err = IOSC_ImportCertificate( (u8 *)&cacert, IOSC_ROOT_KEY_HANDLE, capublickeyh );
#ifdef PERF
    endTime = IO_READ( SYS_TIMER_REG );
    NN_LOG( "root handle = %08x\n", IOSC_ROOT_KEY_HANDLE );
    NN_LOG( "Number of Ticks for IOSC_ImportCertificate = %d, Time = %d  \n", getNumTicks( startTime, endTime ),
            getNumTicks( startTime, endTime ) * 128 / 243 );
#endif
    if( err == IOSC_ERROR_OK )
    {
        LOG_PASS( "CA Import" );
    }
    else
    {
        LOG_FAIL( "CA Import err %d", err );
        EXIT( -1 );
    }

    /*
     * Neg test for bad key type
     */
    IOSC_CreateObject( &inputprivkeyh, IOSC_SECRETKEY_TYPE, IOSC_ECC233_SUBTYPE );
    err = IOSC_ImportCertificate( (u8 *)&cacert, IOSC_ROOT_KEY_HANDLE, inputprivkeyh );
    if( err == IOSC_ERROR_INVALID_OBJTYPE )
    {
        LOG_PASS( "CA Import PASS NEG KEY TYPE TEST" );
    }
    else
    {
        LOG_FAIL( "CA Import FAIL NEG KEY TYPE TEST" );
        EXIT( -1 );
    }
    IOSC_DeleteObject( inputprivkeyh );

    /*
     * Neg test for bad sign type
     */
    prsa4rsa = static_cast<IOSCRsa4096RsaCert *>( &cacert );

    /* make bad sig */
    ( prsa4rsa->sig ).sigType += 1;
    err = IOSC_ImportCertificate( (u8 *)&cacert, IOSC_ROOT_KEY_HANDLE, capublickeyh );
    if( err != IOSC_ERROR_OK )
    {
        LOG_PASS( "CA Import PASS NEG SIGN TYPE TEST" );
    }
    else
    {
        LOG_FAIL( "CA Import FAIL NEG SIGN TYPE TEST" );
        EXIT( -1 );
    }

    /*
     * Test of XS import
     */
    readFile(makePath( "pki_data/xs_dpki.cert" ), &xscert, sizeof( IOSCRsa2048RsaCert ));

    IOSC_CreateObject( &xspublickeyh, IOSC_PUBLICKEY_TYPE, IOSC_RSA2048_SUBTYPE );
#ifdef PERF
    startTime = IO_READ( SYS_TIMER_REG );
#endif
    err = IOSC_ImportCertificate( (u8 *)&xscert, capublickeyh, xspublickeyh );
#ifdef PERF
    endTime = IO_READ( SYS_TIMER_REG );
    NN_LOG( "Number of Ticks for IOSC_ImportCertificate = %d, Time = %d  \n", getNumTicks( startTime, endTime ),
            getNumTicks( startTime, endTime ) * 128 / 243 );
#endif
    if( err == IOSC_ERROR_OK )
    {
        LOG_PASS( "XS Import PASS" );
    }
    else
    {
        LOG_FAIL( "XS Import FAIL" );
        EXIT( -1 );
    }

    /*
     * Neg test for bad sign type
     */
    prsa2rsa = static_cast<IOSCRsa2048RsaCert *>( &xscert );

    /* make bad sig */
    ( prsa2rsa->sig ).sigType += 1;
    err = IOSC_ImportCertificate( (u8 *)&xscert, capublickeyh, xspublickeyh );
    if( err != IOSC_ERROR_OK )
    {
        LOG_PASS( "XS Import PASS NEG SIGN TYPE TEST" );
    }
    else
    {
        LOG_FAIL( "XS Import FAIL NEG SIGN TYPE TEST" );
        EXIT( -1 );
    }

    /*
     * Neg Test for public key type
     */
    ( prsa2rsa->sig ).sigType -= 1;              /* restore */
    ( ( prsa2rsa->head ).pubKeyType ) += 0xffff; /* bad pub key type */
    err = IOSC_ImportCertificate( (u8 *)&xscert, capublickeyh, xspublickeyh );
    if( err != IOSC_ERROR_OK )
    {
        LOG_PASS( "XS Import PASS NEG PUBKEY TYPE TEST" );
    }
    else
    {
        LOG_FAIL( "XS Import FAIL NEG PUBKEY TYPE TEST err = %d", err );
    }

    /*
     * Check if pub key is imported properly
     */
    IOSC_ExportPublicKey( returnedpubkey, exponent, xspublickeyh );

    if( memcmp( returnedpubkey, ( (IOSCRsa2048RsaCert)xscert ).pubKey, sizeof( IOSCRsaPublicKey2048 ) ) == 0 )
    {
        LOG_PASS( "Imported XS key" );
    }
    else
    {
        LOG_FAIL( "Imported XS key" );
        EXIT( -1 );
    }

    IOSC_CreateObject( &mspublickeyh, IOSC_PUBLICKEY_TYPE, IOSC_ECC233_SUBTYPE );
    /* read in ms cert to get public key */
    readFile(makePath( "pki_data/ms_dpki.cert" ), &mscert, sizeof( IOSCRsa2048EccCert ));

    /* import the MS cert */
    err = IOSC_ImportCertificate( (u8 *)&mscert, capublickeyh, mspublickeyh );
    if( err != IOSC_ERROR_OK )
    {
        LOG_FAIL( "MS IOSC_ImportCertificate, err = %d", err );
    }
    else
    {
        LOG_PASS( "MS Import PASS" );
    }

    IOSC_DeleteObject( capublickeyh );
    IOSC_DeleteObject( xspublickeyh );

    /*
     * Test of GetDeviceCertificate
     */
    err = IOSC_GetDeviceCertificate( (IOSCEccSignedCert *)&returnedcert );
    if( err != IOSC_ERROR_OK )
    {
        LOG_FAIL( "error in IOSC_GetDeviceCertificate" );
    }
    writeFile("dev.cert", &returnedcert, sizeof( IOSCEccEccCert ));

#ifdef NEED_CHANGE_DEV_CERT
    /*
     * Read the one generated by the tools
     */
    readFile("pki_data/dev_ng_dpki.cert", &devcert, sizeof( IOSCEccEccCert ));

    if( memcmp( returnedcert, &devcert, sizeof( IOSCEccEccCert ) ) == 0 )
    {
        LOG_PASS( "Get Device Cert" );
    }
    else
    {
        LOG_FAIL( "Get Device Cert FAIL: may be OK, this is comparing" );
    }

    /*
     * Test of ImportCertificate
     */

    /* export to check */
    IOSC_ExportPublicKey( returnedpubkey, 0, mspublickeyh );

    if( memcmp( returnedpubkey, mscert.pubKey, sizeof( IOSCEccPublicKey ) ) == 0 )
    {
        LOG_PASS( "MS public key export" );
    }
    else
    {
        LOG_FAIL( "MS public key export" );
        EXIT( -1 );
    }

    IOSC_CreateObject( &devpublickeyh, IOSC_PUBLICKEY_TYPE, IOSC_ECC233_SUBTYPE );

    err = IOSC_ImportCertificate( (u8 *)returnedcert, mspublickeyh, devpublickeyh );
    if( err == IOSC_ERROR_OK )
    {
        LOG_PASS( "Device Certificate import" );
    }
    else
    {
        LOG_FAIL( "Device Certificate import FAIL, err = %d", err );
        EXIT( -1 );
    }

    IOSC_DeleteObject( mspublickeyh );

    /*
     * Test of GenerateCertificate for input secret key
     */
    for( i = 0; i < sizeof( IOSCEccPrivateKey ); i++ )
    {
        inputprivate[i] = static_cast<u8>( i );
    }

    IOSC_CreateObject( &inputprivkeyh, IOSC_SECRETKEY_TYPE, IOSC_ECC233_SUBTYPE );
    IOSC_ImportSecretKey( inputprivkeyh, 0, 0, IOSC_NOSIGN_NOENC, 0, 0, inputprivate );
    /* generate device signed cert for this key */
    memset( appname, 0x0, sizeof( IOSCCertName ) );
    memcpy( appname, "APP00000001", sizeof( "APP00000001" ) );

    err = IOSC_GenerateCertificate( inputprivkeyh, appname, &appcert );
    if( err == IOSC_ERROR_OK )
    {
        LOG_PASS( "Generate App Certificate" );
    }
    else
    {
        LOG_FAIL( "error in IOSC_GenerateCertificate, err %d", err );
    }
    writeFile("app.cert", &appcert, sizeof( IOSCEccEccCert ));

    /* check with respect to imported device public key */

    IOSC_CreateObject( &apppublickeyh, IOSC_PUBLICKEY_TYPE, IOSC_ECC233_SUBTYPE );
    err = IOSC_ImportCertificate( appcert, devpublickeyh, apppublickeyh );
    if( err == IOSC_ERROR_OK )
    {
        LOG_PASS( "App cert import" );
    }
    else
    {
        LOG_FAIL( "App cert import, err %d", err );
        EXIT( -1 );
    }

    IOSC_DeleteObject( devpublickeyh );
    IOSC_DeleteObject( inputprivkeyh );
    IOSC_DeleteObject( apppublickeyh );
#endif
    IOSC_DeleteObject( mspublickeyh );

    return 0;
}  // NOLINT (readability/fn_size)


#if defined( SEPARATE_MAIN_PROGRAM )

int main( int argc, char **argv )
{
    IOSC_Initialize();
    doCertTest();
    EXIT( 0 );
    return 0;
}
#endif
