﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/crypto.h>
#include <nn/es.h>
#include <nn/nifm.h>
#include <nn/socket.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_SdkLog.h>
#include <nn/crypto/crypto_RsaOaepEncryptor.h>
#include <nn/crypto/crypto_RsaOaepDecryptor.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/nim/srv/nim_EciAccessor.h>
#include <nn/settings/factory/settings_DeviceCertificate.h>
#include <nn/settings/factory/settings_DeviceKey.h>
#include <nn/spl/spl_Api.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>
#include <curl/curl.h>

namespace {
    class PrivateKeyVerification : public ::testing::Test
    {
    protected:
        virtual void SetUp()
        {
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            nn::es::Initialize();
            nn::spl::InitializeForEs();
        }

        static void TearDownTestCase()
        {
            nn::es::Finalize();
            nn::spl::Finalize();
        }
    };

    class DownloadTicketAndDecryptTitleKey : public testing::Test
    {
    protected:
        nn::nim::srv::HttpConnection connection;
        nn::nim::srv::EciAccessor eciAccessor;
        nn::nim::srv::DeviceContext deviceContext;

        // デバイスの DeviceAccountId と DeviceAccountToken をセットする
        nn::nim::DeviceAccountId deviceAccountId = { "" };
        nn::nim::DeviceAccountToken deviceAccountToken = { "" };

        DownloadTicketAndDecryptTitleKey()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(deviceContext.Initialize());
            NN_ABORT_UNLESS_RESULT_SUCCESS(connection.Initialize(&deviceContext));
            NN_ABORT_UNLESS_RESULT_SUCCESS(eciAccessor.Initialize(&deviceContext, &connection));
        }

        virtual void SetUp()
        {
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
            nn::nifm::SubmitNetworkRequestAndWait();

            static NN_ALIGNAS(4096) uint8_t s_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(reinterpret_cast<void*>(s_SocketMemoryPoolBuffer),
                nn::socket::DefaultSocketMemoryPoolSize,
                nn::socket::DefaultSocketAllocatorSize,
                nn::socket::DefaultConcurrencyLimit));
            auto result = curl_global_init(CURL_GLOBAL_ALL);
            NN_ASSERT_EQUAL(CURLE_OK, result);

            nn::spl::InitializeForEs();
        }

        static void TearDownTestCase()
        {
            curl_global_cleanup();
            nn::socket::Finalize();

            nn::nifm::CancelNetworkRequest();

            nn::spl::Finalize();
        }
    };
}

TEST_F(PrivateKeyVerification, VerifyKeyOnLocal)
{
    const int KeySize = 256;

    const int DeviceCertificateModulusOffset = 0x108;
    const int DeviceCertificatePublicExponentOffset = 0x208;
    const int ModulusSize = 256;
    const int PublicExponentSize = 4;

    // 暗号化されていない生の TitleKey
    const uint8_t titleKey[] =
    {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    // 開発版の CommonKey
    const uint8_t commonKey[] =
    {
        0xC4, 0x67, 0xD5, 0x3F, 0x4F, 0xDF, 0xEB, 0x19, 0x3F, 0x61, 0xBF, 0xDA, 0x57, 0xF4, 0x89, 0xD0,
    };

    const uint8_t iv[] =
    {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    // TitleKey を CommonKey で AES 暗号化した鍵
    uint8_t commonTitleKey[16];
    nn::crypto::EncryptAes128Cbc(commonTitleKey, sizeof(commonTitleKey), commonKey, sizeof(commonKey), iv, sizeof(iv), titleKey, sizeof(titleKey));

    // 0 バイトに SHA256 をかけたダイジェスト値
    const uint8_t labelDigest[32] =
    {
        0xE3, 0xB0, 0xC4, 0x42, 0x98, 0xFC, 0x1C, 0x14, 0x9A, 0xFB, 0xF4, 0xC8, 0x99, 0x6F, 0xB9, 0x24,
        0x27, 0xAE, 0x41, 0xE4, 0x64, 0x9B, 0x93, 0x4C, 0xA4, 0x95, 0x99, 0x1B, 0x78, 0x52, 0xB8, 0x55,
    };

    nn::settings::factory::Rsa2048DeviceKey rsa2048DeviceKey;
    nn::settings::factory::GetEticketDeviceKey(&rsa2048DeviceKey);

    nn::settings::factory::Rsa2048DeviceCertificate rsa2048DeviceCertificate;
    nn::settings::factory::GetEticketDeviceCertificate(&rsa2048DeviceCertificate);

    // commonKey で AES 暗号化した TitleKey を更に RSA で暗号化した TitleKey
    // 暗号化は本体の ETicket 用デバイス証明書から、公開鍵を取り出しで使用する
    uint8_t personalizedTitleKey[256];

    // nn::crypto によって RSA 暗号化を行う
    uint8_t seed[nn::crypto::Sha256Generator::HashSize];
    nn::crypto::GenerateCryptographicallyRandomBytes(seed, sizeof(seed));

    nn::crypto::RsaOaepEncryptor<KeySize, nn::crypto::Sha256Generator>::Encrypt(
        personalizedTitleKey, sizeof(personalizedTitleKey),
        &rsa2048DeviceCertificate.data[DeviceCertificateModulusOffset], ModulusSize,
        &rsa2048DeviceCertificate.data[DeviceCertificatePublicExponentOffset], PublicExponentSize,
        commonTitleKey, sizeof(commonTitleKey),
        seed, sizeof(seed),
        "", 0
    );

    // セキュアモニタによる復号
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::LoadEsDeviceKey(rsa2048DeviceKey.data, rsa2048DeviceKey.size));

    nn::spl::AccessKey accessKey;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::PrepareEsTitleKey(&accessKey, &personalizedTitleKey, sizeof(personalizedTitleKey), &rsa2048DeviceCertificate.data[DeviceCertificateModulusOffset], ModulusSize, labelDigest, sizeof(labelDigest), 0));

    int slotIndex;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::AllocateAesKeySlot(&slotIndex));
    NN_UTIL_SCOPE_EXIT{ nn::spl::DeallocateAesKeySlot(slotIndex); };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::LoadPreparedAesKey(slotIndex, accessKey));

    // セキュアモニタで復号した鍵と生の TitleKey で別のデータを暗号化して、同じ暗号文が生成されるか確認
    const uint8_t plainData[] =
    {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    const uint8_t initialCounter[] =
    {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    uint8_t encryptedByTitleKey[16];
    nn::crypto::EncryptAes128Ctr(encryptedByTitleKey, sizeof(encryptedByTitleKey), titleKey, sizeof(titleKey), initialCounter, sizeof(initialCounter), plainData, sizeof(plainData));

    uint8_t encryptedBySecureMoniter[16];
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::ComputeCtr(encryptedBySecureMoniter, sizeof(encryptedBySecureMoniter), slotIndex, plainData, sizeof(plainData), initialCounter, sizeof(initialCounter)));

    EXPECT_TRUE(memcmp(encryptedBySecureMoniter, encryptedByTitleKey, sizeof(encryptedBySecureMoniter)) == 0);
}

//TEST_F(PrivateKeyVerification, DecryptCommonTitleKey)
//{
//  // チケットの RightsId
//  nn::es::RightsIdIncludingKeyId rightsId =
//  {
//      { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }
//  };
//
//  // RightsId に対応する生の TitleKey
//  const uint8_t titleKey[] =
//  {
//      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
//  };
//
//  const uint8_t plain[] =
//  {
//      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
//  };
//
//  const uint8_t initialCounter[] =
//  {
//      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
//  };
//
//  uint8_t encrypted[16];
//  nn::crypto::EncryptAes128Ctr(encrypted, sizeof(encrypted), titleKey, sizeof(titleKey), initialCounter, sizeof(initialCounter), plain, sizeof(plain));
//
//  uint8_t decryptedBySecureMoniter[16];
//  uint8_t decryptedByTitleKey[16];
//
//  // セキュアモニターに設定された TitleKey で復号
//  {
//      nn::es::AesKey aesKey;
//      NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::GetTitleKey(&aesKey, rightsId));
//
//      int slotIndex = 0;
//      NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::LoadPreparedAesKey(slotIndex, *reinterpret_cast<nn::spl::AccessKey*>(&aesKey._data)));
//      NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::ComputeCtr(decryptedBySecureMoniter, sizeof(decryptedBySecureMoniter), slotIndex, encrypted, sizeof(encrypted), initialCounter, sizeof(initialCounter)));
//  }
//
//  // 生の TitleKey で復号
//  {
//      nn::crypto::DecryptAes128Ctr(decryptedByTitleKey, sizeof(decryptedByTitleKey), titleKey, sizeof(titleKey), initialCounter, sizeof(initialCounter), encrypted, sizeof(encrypted));
//  }
//
//  EXPECT_TRUE(memcmp(decryptedBySecureMoniter, plain, sizeof(decryptedBySecureMoniter)) == 0);
//  EXPECT_TRUE(memcmp(decryptedByTitleKey, plain, sizeof(decryptedBySecureMoniter)) == 0);
//}

//TEST_F(DownloadTicketAndDecryptTitleKey, DecryptPersonalizedTitleKey)
//{
//    // チケットのダウンロード
//    {
//        // ダウンロードするチケットの RightsId
//        nn::nim::RightsId nimRightsId = { "01000000000000000000000000000000" };
//
//        int outTicketIdCount;
//        nn::Bit64 ticketIdList[1];
//
//        NN_ABORT_UNLESS_RESULT_SUCCESS(eciAccessor.AccountListETicketIdsByRightsId(&outTicketIdCount, ticketIdList, sizeof(ticketIdList) / sizeof(ticketIdList[0]), deviceAccountId, deviceAccountToken, nimRightsId));
//        NN_ABORT_UNLESS_RESULT_SUCCESS(eciAccessor.AccountGetETickets(deviceAccountId, deviceAccountToken, ticketIdList, sizeof(ticketIdList) / sizeof(ticketIdList[0])));
//    }
//
//    // 鍵の検証
//    {
//        // チケットの RightsId
//        nn::es::RightsIdIncludingKeyId rightsId =
//        {
//            { 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }
//        };
//
//        // RightsId に対応する生の TitleKey
//        const uint8_t titleKey[] =
//        {
//          0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
//        };
//
//        const uint8_t plain[] =
//        {
//            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
//        };
//
//        const uint8_t initialCounter[] =
//        {
//            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
//        };
//
//        uint8_t encrypted[16];
//        nn::crypto::EncryptAes128Ctr(encrypted, sizeof(encrypted), titleKey, sizeof(titleKey), initialCounter, sizeof(initialCounter), plain, sizeof(plain));
//
//        uint8_t decryptedBySecureMoniter[16];
//        uint8_t decryptedByTitleKey[16];
//
//        // セキュアモニターに設定された TitleKey で復号
//        {
//            nn::es::AesKey aesKey;
//
//            nn::es::Initialize();
//            NN_UTIL_SCOPE_EXIT{ nn::es::Finalize(); };
//
//            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::es::GetTitleKey(&aesKey, rightsId));
//
//            int slotIndex = 0;
//            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::LoadPreparedAesKey(slotIndex, *reinterpret_cast<nn::spl::AccessKey*>(&aesKey._data)));
//            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::ComputeCtr(decryptedBySecureMoniter, sizeof(decryptedBySecureMoniter), slotIndex, encrypted, sizeof(encrypted), initialCounter, sizeof(initialCounter)));
//        }
//
//        // 生の TitleKey で復号
//        {
//            nn::crypto::DecryptAes128Ctr(decryptedByTitleKey, sizeof(decryptedByTitleKey), titleKey, sizeof(titleKey), initialCounter, sizeof(initialCounter), encrypted, sizeof(encrypted));
//        }
//
//        EXPECT_TRUE(memcmp(decryptedBySecureMoniter, plain, sizeof(decryptedBySecureMoniter)) == 0);
//        EXPECT_TRUE(memcmp(decryptedByTitleKey, plain, sizeof(decryptedBySecureMoniter)) == 0);
//    }
//}


