﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/erpt.h>
#include <nn/erpt/erpt_Manager.h>
#include <nn/erpt/erpt_MultipleCategoryContext.h>
#include <nn/erpt/erpt_Report.h>
#include <nn/erpt/server/erpt_ServerTypes.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nnt.h>

namespace nnt  {
namespace erpt {

uint8_t                  g_Buffer[nn::erpt::MaxArrayBufferLength];
nn::Bit8                 g_LargeFieldBuffer[nn::erpt::MaxArrayFieldSize + 32]; // + 32 for the tests which tries to submit a too large field.
nn::erpt::Report         g_Reports[nn::erpt::NumberOfReports];
nn::erpt::ReportMetaData g_Meta;
nn::erpt::ReportList     g_ReportList;

nn::Result MakeMinimumErrorReport()
{
    nn::erpt::Context context(nn::erpt::ErrorInfo);
    NN_RESULT_DO(context.Add(nn::erpt::ErrorCode, "1234-5678", static_cast<uint32_t>(std::strlen("1234-5678"))));
    NN_RESULT_DO(context.CreateReport(nn::erpt::ReportType_Invisible));
    NN_RESULT_SUCCESS;
}

nn::Result ClearAllContext()
{
    for( int i = 0; i < static_cast<int>(nn::erpt::CategoryId::CategoryId_Last); i++ )
    {
        auto category = static_cast<nn::erpt::CategoryId>(i);
        if( category == nn::erpt::CategoryId::ErrorInfo )
        {
            continue;
        }
        nn::erpt::Context context(category);
        NN_RESULT_DO(context.SubmitContext());
    }
    NN_RESULT_SUCCESS;
}

TEST(erpt, ListExistingReports)
{
    nn::Result        result;
    nn::erpt::Manager reportManager;

    result = reportManager.Initialize();
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = reportManager.GetReportList(g_ReportList);
    NNT_EXPECT_RESULT_SUCCESS(result);

    NN_LOG("Found %u stored reports\n", g_ReportList.reportCount);

    for (uint32_t i = 0; i < g_ReportList.reportCount; i++)
    {
        char idStr[64];
        g_ReportList.Report[i].reportId.u.uuidRFC4122.ToString(idStr, sizeof(idStr));

        NN_LOG("id: %s\n",
            idStr);

        NN_LOG("time: %08x%08x\n",
              static_cast<uint32_t>(g_ReportList.Report[i].reportTimeStamp.value >> 32),
              static_cast<uint32_t>(g_ReportList.Report[i].reportTimeStamp.value));

        NN_LOG("type: %d\n", g_ReportList.Report[i].reportType);
        NN_LOG("flag: %u\n", g_ReportList.Report[i].reportFlags.Test<nn::erpt::ReportFlag::Transmitted>());
    }

    reportManager.Finalize();
}

TEST(erpt, UnalignedArray)
{
    nn::Result result;
    char       charBuffer[]   = "12";
    uint64_t   uint64Buffer[] = {0x5, 0x6, 0x7, 0x8};

    nn::erpt::Context context(nn::erpt::Test);

    if ((result = context.Add(nn::erpt::TestString, charBuffer, sizeof(charBuffer))).IsFailure() ||
        (result = context.Add(nn::erpt::TestU64Array, uint64Buffer, sizeof(uint64Buffer) / sizeof(uint64Buffer[0]))).IsFailure() ||
        (result = context.SubmitContext()).IsFailure())
    {
        NNT_EXPECT_RESULT_SUCCESS(result);
    }
}

TEST(erpt, ContextOutOfStringBufferSpace)
{
    nn::Result result;
    uint8_t    buf[128];
    uint32_t   count = nn::erpt::ArrayBufferLength / sizeof(buf);

    nn::erpt::Context context(nn::erpt::Test);

    for (uint32_t i = 0; i < count; i++)
    {
        result = context.Add(nn::erpt::TestU8Array, buf, sizeof(buf));
        NNT_EXPECT_RESULT_SUCCESS(result);
    }

    result = context.Add(nn::erpt::TestU8Array, buf, 1);
    if (!(result <= nn::erpt::ResultOutOfArraySpace()))
    {
        ADD_FAILURE();
    }

    result = context.SubmitContext();
    NNT_EXPECT_RESULT_SUCCESS(result);
}

TEST(erpt, ContextUserProvidedStringBuffer)
{
    nn::Result result;
    uint8_t    userBuf[nn::erpt::ArrayBufferLength + 1];
    uint8_t    buf[128];
    uint32_t   count = nn::erpt::ArrayBufferLength / sizeof(buf);

    nn::erpt::Context context(nn::erpt::Test, userBuf, sizeof(userBuf));

    for (uint32_t i = 0; i < count; i++)
    {
        result = context.Add(nn::erpt::TestU8Array, buf, sizeof(buf));
        NNT_EXPECT_RESULT_SUCCESS(result);
    }

    result = context.Add(nn::erpt::TestU8Array, buf, 1);
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = context.SubmitContext();
    NNT_EXPECT_RESULT_SUCCESS(result);
}

TEST(erpt, ContextTooManyEntries)
{
    nn::Result result;

    nn::erpt::Context context(nn::erpt::Test);

    for (int i = 0; i < nn::erpt::FieldsPerContext; i++)
    {
        result = context.Add(nn::erpt::TestU32, (uint32_t)0x1234);
        NNT_EXPECT_RESULT_SUCCESS(result);
    }

    result = context.Add(nn::erpt::TestU32, (uint32_t)0x1234);
    if (!(result <= nn::erpt::ResultOutOfFieldSpace()))
    {
        ADD_FAILURE();
    }

    result = context.SubmitContext();
    NNT_EXPECT_RESULT_SUCCESS(result);
}

TEST(erpt, ContextWrongField)
{
    nn::Result result;
    const char serialNumber[] = "123456789";

    nn::erpt::Context context(nn::erpt::Test);

    result = context.Add(nn::erpt::SerialNumber, serialNumber, sizeof(serialNumber));
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = context.SubmitContext();
    if (!(result <= nn::erpt::ResultFieldCategoryMismatch()))
    {
        ADD_FAILURE();
    }
}

TEST(erpt, ContextWrongFieldType)
{
    nn::Result result;

    nn::erpt::Context context(nn::erpt::Test);

    result = context.Add(nn::erpt::TestU32, (uint64_t)0x1234);
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = context.SubmitContext();
    if (!(result <= nn::erpt::ResultFieldTypeMismatch()))
    {
        ADD_FAILURE();
    }
}

TEST(erpt, ContextReportWrongCategoryOrField)
{
    nn::Result result;

    nn::erpt::Context context(nn::erpt::NetworkInfo);

    result = context.CreateReport(nn::erpt::ReportType_Visible, g_Meta);
    if (!(result <= nn::erpt::ResultRequiredContextMissing()))
    {
        ADD_FAILURE();
    }

    context.ResetContext(nn::erpt::ErrorInfo);

    result = context.CreateReport(nn::erpt::ReportType_Visible, g_Meta);
    if (!(result <= nn::erpt::ResultRequiredFieldMissing()))
    {
        ADD_FAILURE();
    }
}

TEST(erpt, GenerateManyReportWithAllFields)
{
    nn::erpt::Context    context(nn::erpt::ErrorInfo, g_Buffer, sizeof(g_Buffer));
    nn::erpt::CategoryId categoryId;
    nn::erpt::FieldId    fieldId;
    nn::Result           result;

    const uint8_t data[257]   = {0};
    const char    string[257] = "abcdefghijklmnopqrstuvwxyz0123456789";
    const char    errorCode[] = "1234-5678";

    for (int i = 0;
             i < nn::erpt::CategoryId_Last;
             i++)
    {
        categoryId = static_cast<nn::erpt::CategoryId>(i);

        if (categoryId == nn::erpt::ErrorInfo)
        {
            continue;
        }

        context.ResetContext(categoryId);

        for (int j = 0;
                 j < nn::erpt::FieldId_Last;
                 j++)
        {
            fieldId = static_cast<nn::erpt::FieldId>(j);

            // Deprecated fields. Theses fields must not be used.
            if (fieldId == nn::erpt::FieldId::DEPRECATED_ServerFQDN)
            {
                continue;
            }

            if (nn::erpt::srv::FieldToCategoryMap[fieldId] == categoryId)
            {
                switch (nn::erpt::srv::FieldToTypeMap[fieldId])
                {
                case nn::erpt::FieldType_Bool:
                    result = context.Add(fieldId, true);
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                case nn::erpt::FieldType_NumericU64:
                    result = context.Add(fieldId, static_cast<uint64_t>(0xe234567812345678));
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                case nn::erpt::FieldType_NumericU32:
                    result = context.Add(fieldId, static_cast<uint32_t>(0xe2345678));
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                case nn::erpt::FieldType_NumericU16:
                    result = context.Add(fieldId, static_cast<uint16_t>(0xe234));
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                case nn::erpt::FieldType_NumericU8:
                    result = context.Add(fieldId, static_cast<uint8_t>(0xe2));
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                case nn::erpt::FieldType_NumericI64:
                    result = context.Add(fieldId, static_cast<int64_t>(0xe234567812345678));
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                case nn::erpt::FieldType_NumericI32:
                    result = context.Add(fieldId, static_cast<int32_t>(0xe2345678));
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                case nn::erpt::FieldType_NumericI16:
                    result = context.Add(fieldId, static_cast<int16_t>(0xe23));
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                case nn::erpt::FieldType_NumericI8:
                    result = context.Add(fieldId, static_cast<int8_t>(0xe));
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                case nn::erpt::FieldType_String:
                    result = context.Add(fieldId, string, sizeof(string));
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                case nn::erpt::FieldType_U8Array:
                    result = context.Add(fieldId, reinterpret_cast<const uint8_t*>(data), sizeof(data) / sizeof(uint8_t));
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                case nn::erpt::FieldType_U32Array:
                    result = context.Add(fieldId, reinterpret_cast<const uint32_t*>(data), sizeof(data) / sizeof(uint32_t));
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                case nn::erpt::FieldType_U64Array:
                    result = context.Add(fieldId, reinterpret_cast<const uint64_t*>(data), sizeof(data) / sizeof(uint64_t));
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                case nn::erpt::FieldType_I8Array:
                    result = context.Add(fieldId, reinterpret_cast<const int8_t*>(data), sizeof(data) / sizeof(int8_t));
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                case nn::erpt::FieldType_I32Array:
                    result = context.Add(fieldId, reinterpret_cast<const int32_t*>(data), sizeof(data) / sizeof(int32_t));
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                case nn::erpt::FieldType_I64Array:
                    result = context.Add(fieldId, reinterpret_cast<const int64_t*>(data), sizeof(data) / sizeof(int64_t));
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    break;
                default:
                    ADD_FAILURE();
                    break;
                }
            }
        }

        NNT_EXPECT_RESULT_SUCCESS(result);

        result = context.SubmitContext();
        NNT_EXPECT_RESULT_SUCCESS(result);
    }

    context.ResetContext(nn::erpt::ErrorInfo);

    if ((result = context.Add(nn::erpt::ErrorCode, errorCode, sizeof(errorCode))).IsFailure())
    {
        NNT_EXPECT_RESULT_SUCCESS(result);
    }

    for (int type = nn::erpt::ReportType_First;
             type < nn::erpt::ReportType_Last;
             type++)
    {
        for (int i = 0; i < nn::erpt::NumberOfReports; i++)
        {
            result = context.CreateReport(static_cast<nn::erpt::ReportType>(type), g_Meta);
            NNT_EXPECT_RESULT_SUCCESS(result);

            NN_LOG(".");
        }
    }

    NN_LOG("\n");
} // NOLINT(impl/function_size)

TEST(erpt, ReportReplacementPolicy)
{
    nn::Result        result;
    uint32_t          reportCount[nn::erpt::ReportType_Last] = {0};
    nn::erpt::Manager reportManager;

    // after the previous test, reports of all types should have the same
    // number of stored reports.

    result = reportManager.Initialize();
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = reportManager.GetReportList(g_ReportList);
    NNT_EXPECT_RESULT_SUCCESS(result);

    if (g_ReportList.reportCount != nn::erpt::NumberOfReports)
    {
        ADD_FAILURE();
    }

    for (uint32_t i = 0; i < g_ReportList.reportCount; i++)
    {
        if (g_ReportList.Report[i].reportType > nn::erpt::ReportType_Last)
        {
            ADD_FAILURE();
        }
        reportCount[g_ReportList.Report[i].reportType]++;
    }

    for (int i = 1; i < nn::erpt::ReportType_Last; i++)
    {
        if (reportCount[i] != reportCount[0])
        {
            NN_LOG("number of reports is not the same: (%u) != (%u)\n", reportCount[i], reportCount[0]);
            ADD_FAILURE();
            break;
        }
    }

    reportManager.Finalize();
}

TEST(erpt, ReportEvent)
{
    nn::Result           result;
    nn::os::SystemEvent* pEvent;
    nn::erpt::Manager    reportManager;
    const char errorCode[] = "1234-5678";

    result = reportManager.Initialize();
    NNT_EXPECT_RESULT_SUCCESS(result);

    pEvent = reportManager.GetEventPointer();
    pEvent->Clear();

    if (pEvent->TimedWait(nn::TimeSpan::FromMicroSeconds(100)))
    {
        ADD_FAILURE();
    }

    nn::erpt::Context context(nn::erpt::ErrorInfo);

    if ((result = context.Add(nn::erpt::ErrorCode, errorCode, sizeof(errorCode))).IsFailure() ||
        (result = context.CreateReport(nn::erpt::ReportType_Invisible, g_Meta)).IsFailure())
    {
        NNT_EXPECT_RESULT_SUCCESS(result);
    }

    if (!pEvent->TimedWait(nn::TimeSpan::FromMicroSeconds(100)))
    {
        ADD_FAILURE();
    }

    reportManager.Finalize();
}

TEST(erpt, ReportRetrieval)
{
    nn::Result result;
    int64_t    reportSize;

    nn::erpt::Manager reportManager;
    nn::erpt::Report  reportStream;

    result = reportManager.Initialize();
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = reportManager.GetReportList(g_ReportList);
    NNT_EXPECT_RESULT_SUCCESS(result);

    if (g_ReportList.reportCount == 0)
    {
        ADD_FAILURE();
    }

    for (uint32_t i = 0; i < g_ReportList.reportCount; i++)
    {
        char idStr[64];
        g_ReportList.Report[i].reportId.u.uuidRFC4122.ToString(idStr, sizeof(idStr));

        NN_LOG("id: %s\n",
            idStr);

        NN_LOG("time: %08x%08x\n",
              static_cast<uint32_t>(g_ReportList.Report[i].reportTimeStamp.value >> 32),
              static_cast<uint32_t>(g_ReportList.Report[i].reportTimeStamp.value));

        NN_LOG("type: %d\n", g_ReportList.Report[i].reportType);
        NN_LOG("flag: %u\n", g_ReportList.Report[i].reportFlags.Test<nn::erpt::ReportFlag::Transmitted>());

        result = reportStream.Open(g_ReportList.Report[i].reportId);
        NNT_EXPECT_RESULT_SUCCESS(result);

        result = reportStream.GetSize(&reportSize);
        NNT_EXPECT_RESULT_SUCCESS(result);

        NN_LOG("size: %u\n", static_cast<uint32_t>(reportSize));

        result = reportStream.Close();
        NNT_EXPECT_RESULT_SUCCESS(result);
    }

    reportManager.Finalize();
}

TEST(erpt, ReportFileWithoutMeta)
{
    nn::Result result;
    const char errorCode[] = "1234-5678";
    nn::erpt::Context context(nn::erpt::ErrorInfo);

    if ((result = context.Add(nn::erpt::ErrorCode, errorCode, sizeof(errorCode))).IsFailure() ||
        (result = context.CreateReport(nn::erpt::ReportType_Invisible)).IsFailure())
    {
        NNT_EXPECT_RESULT_SUCCESS(result);
    }
}

TEST(erpt, FilteredSummary)
{
    nn::Result        result;
    nn::erpt::Manager reportManager;

    result = reportManager.Initialize();
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = reportManager.GetReportList(g_ReportList, nn::erpt::ReportType_Visible);
    NNT_EXPECT_RESULT_SUCCESS(result);

    if (g_ReportList.reportCount == 0)
    {
        ADD_FAILURE();
    }

    for (uint32_t i = 0; i < g_ReportList.reportCount; i++)
    {
        if (g_ReportList.Report[i].reportType != nn::erpt::ReportType_Visible)
        {
            ADD_FAILURE();
        }
    }

    result = reportManager.GetReportList(g_ReportList, nn::erpt::ReportType_Invisible);
    NNT_EXPECT_RESULT_SUCCESS(result);

    if (g_ReportList.reportCount == 0)
    {
        ADD_FAILURE();
    }

    for (uint32_t i = 0; i < g_ReportList.reportCount; i++)
    {
        if (g_ReportList.Report[i].reportType != nn::erpt::ReportType_Invisible)
        {
            ADD_FAILURE();
        }
    }

    reportManager.Finalize();
}

TEST(erpt, UninitializedManager)
{
    nn::Result        result;
    nn::erpt::Manager reportManager;

    result = reportManager.GetReportList(g_ReportList);
    if (!(result <= nn::erpt::ResultNotInitialized()))
    {
        ADD_FAILURE();
    }

    reportManager.Finalize();
}

TEST(erpt, DoubleInitializedManager)
{
    nn::Result        result;
    nn::erpt::Manager reportManager;

    result = reportManager.Initialize();
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = reportManager.Initialize();
    if (!(result <= nn::erpt::ResultAlreadyInitialized()))
    {
        ADD_FAILURE();
    }

    reportManager.Finalize();
}

TEST(erpt, UnopenedReport)
{
    nn::Result       result;
    int64_t          reportSize;
    nn::erpt::Report reportStream;

    result = reportStream.GetSize(&reportSize);
    if (!(result <= nn::erpt::ResultNotInitialized()))
    {
        ADD_FAILURE();
    }
}

TEST(erpt, DoubleOpenedReport)
{
    nn::Result        result;
    nn::erpt::Manager reportManager;
    nn::erpt::Report  reportStream;

    result = reportManager.Initialize();
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = reportManager.GetReportList(g_ReportList, nn::erpt::ReportType_Visible);
    NNT_EXPECT_RESULT_SUCCESS(result);

    if (g_ReportList.reportCount == 0)
    {
        ADD_FAILURE();
    }

    result = reportStream.Open(g_ReportList.Report[0].reportId);
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = reportStream.Open(g_ReportList.Report[0].reportId);
    if (!(result <= nn::erpt::ResultAlreadyInitialized()))
    {
        ADD_FAILURE();
    }

    reportManager.Finalize();
}

TEST(erpt, MultipleOpenReports)
{
    nn::Result        result;
    uint8_t           reportBuffer[128];
    uint32_t          reportReadCount;
    nn::erpt::Manager reportManager;

    result = reportManager.Initialize();
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = reportManager.GetReportList(g_ReportList);
    NNT_EXPECT_RESULT_SUCCESS(result);

    if (g_ReportList.reportCount == 0)
    {
        ADD_FAILURE();
    }

    for (uint32_t i = 0; i < g_ReportList.reportCount; i++)
    {
        result = g_Reports[i].Open(g_ReportList.Report[i].reportId);
        NNT_EXPECT_RESULT_SUCCESS(result);

        result = g_Reports[i].Read(&reportReadCount, reportBuffer, sizeof(reportBuffer));
        NNT_EXPECT_RESULT_SUCCESS(result);
    }

    for (uint32_t i = 0; i < g_ReportList.reportCount; i++)
    {
        result = g_Reports[i].Close();
        NNT_EXPECT_RESULT_SUCCESS(result);
    }

    reportManager.Finalize();
}

TEST(erpt, InvalidReportId)
{
    nn::Result         result;
    nn::erpt::Report   reportStream;
    nn::erpt::ReportId reportId;

    memset(&reportId, 0x0, sizeof(reportId));

    result = reportStream.Open(reportId);
    if (!(result <= nn::erpt::ResultNotFound()))
    {
        ADD_FAILURE();
    }
}

TEST(erpt, TransmittedFlagSetGet)
{
    nn::Result              result;
    nn::erpt::Manager       reportManager;
    nn::erpt::Report        reportStream;
    nn::erpt::ReportFlagSet flags;

    result = reportManager.Initialize();
    NNT_EXPECT_RESULT_SUCCESS(result);

    result = reportManager.GetReportList(g_ReportList);
    NNT_EXPECT_RESULT_SUCCESS(result);

    if (g_ReportList.reportCount == 0)
    {
        ADD_FAILURE();
    }

    for (uint32_t i = 0; i < g_ReportList.reportCount; i++)
    {
        result = reportStream.Open(g_ReportList.Report[i].reportId);
        NNT_EXPECT_RESULT_SUCCESS(result);

        flags.Set<nn::erpt::ReportFlag::Transmitted>(1);

        result = reportStream.SetFlags(flags);
        NNT_EXPECT_RESULT_SUCCESS(result);

        result = reportStream.GetFlags(&flags);
        NNT_EXPECT_RESULT_SUCCESS(result);

        if (!(flags.Test<nn::erpt::ReportFlag::Transmitted>()))
        {
            ADD_FAILURE();
        }

        flags.Set<nn::erpt::ReportFlag::Transmitted>(0);

        result = reportStream.SetFlags(flags);
        NNT_EXPECT_RESULT_SUCCESS(result);

        result = reportStream.GetFlags(&flags);
        NNT_EXPECT_RESULT_SUCCESS(result);

        // transmit flag is "set only", it can't be cleared
        if (!(flags.Test<nn::erpt::ReportFlag::Transmitted>()))
        {
            ADD_FAILURE();
        }

        reportStream.Close();

        NN_LOG(".");
    }

    NN_LOG("\n");

    reportManager.Finalize();
}

void FillWithSequencialAlphabets(char* buffer, size_t bufferLength)
{
    for( size_t i = 0; i < bufferLength; i++ )
    {
        buffer[i] = i % ('z' - 'a' + 1) + 'a';
    }
}

TEST(erpt, ErrorContext)
{
    static uint8_t buffer[1024];
    const char* errorCodeStr = "1234-5678";
    {
        nn::err::ErrorContext errorContext;
        errorContext.type = nn::err::ErrorContextType::Http;
        FillWithSequencialAlphabets(errorContext.http.fqdn, sizeof(errorContext.http.fqdn));
        FillWithSequencialAlphabets(errorContext.http.ip, sizeof(errorContext.http.ip));

        nn::erpt::Context erptContext(nn::erpt::CategoryId::ErrorInfo, buffer, sizeof(buffer));
        NNT_EXPECT_RESULT_SUCCESS(erptContext.AddErrorContext(errorContext));
        NNT_EXPECT_RESULT_SUCCESS(erptContext.Add(nn::erpt::FieldId::ErrorCode, errorCodeStr, static_cast<uint32_t>(std::strlen(errorCodeStr))));
        NNT_EXPECT_RESULT_SUCCESS(erptContext.CreateReport(nn::erpt::ReportType_Invisible));
    }
    {
        nn::err::ErrorContext errorContext;
        errorContext.type = nn::err::ErrorContextType::FileSystem;
        FillWithSequencialAlphabets(errorContext.fileSystem.path, sizeof(errorContext.fileSystem.path));
        errorContext.fileSystem.fsResultValue = 1234u;

        nn::erpt::Context erptContext(nn::erpt::CategoryId::ErrorInfo, buffer, sizeof(buffer));
        NNT_EXPECT_RESULT_SUCCESS(erptContext.AddErrorContext(errorContext));
        NNT_EXPECT_RESULT_SUCCESS(erptContext.Add(nn::erpt::FieldId::ErrorCode, errorCodeStr, static_cast<uint32_t>(std::strlen(errorCodeStr))));
        NNT_EXPECT_RESULT_SUCCESS(erptContext.CreateReport(nn::erpt::ReportType_Invisible));
    }
    {
        nn::err::ErrorContext errorContext;
        errorContext.type = nn::err::ErrorContextType::WebMediaPlayer;
        FillWithSequencialAlphabets(errorContext.webMediaPlayer.openUrl, sizeof(errorContext.webMediaPlayer.openUrl));
        for( int i = 0; i < NN_ARRAY_SIZE(errorContext.webMediaPlayer.lastSocketErrors); i++ )
        {
            errorContext.webMediaPlayer.lastSocketErrors[i] = i;
        }

        nn::erpt::Context erptContext(nn::erpt::CategoryId::ErrorInfo, buffer, sizeof(buffer));
        NNT_EXPECT_RESULT_SUCCESS(erptContext.AddErrorContext(errorContext));
        NNT_EXPECT_RESULT_SUCCESS(erptContext.Add(nn::erpt::FieldId::ErrorCode, errorCodeStr, static_cast<uint32_t>(std::strlen(errorCodeStr))));
        NNT_EXPECT_RESULT_SUCCESS(erptContext.CreateReport(nn::erpt::ReportType_Invisible));
    }
}

TEST(erpt, MultipleCategoryContextFill)
{
    nn::erpt::MultipleCategoryContext context;

    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::ConnectionStatusInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ConnectionStatus, "ConnectionStatus", static_cast<uint32_t>(sizeof("ConnectionStatus"))));

    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::AccessPointInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::AccessPointSSID, "AccessPointSSID", static_cast<uint32_t>(sizeof("AccessPointSSID"))));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::AccessPointSecurityType, "AccessPointSecurityType", static_cast<uint32_t>(sizeof("AccessPointSecurityType"))));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::AccessPointChannel, static_cast<uint16_t>(1)));

    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::RadioStrengthInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::RadioStrength, static_cast<uint32_t>(1)));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::AccessPointRssi, static_cast<int32_t>(1)));

    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::WirelessAPMacAddressInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::WirelessAPMacAddress, "Wi-re-le-ss-AP-Ma", static_cast<uint32_t>(sizeof("AA-AA-AA-AA-AA-AA"))));

    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::StealthNetworkInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::UseStealthNetworkFlag, true));

    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::NXMacAddressInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::NXMacAddress, "NX-Ma-cA-dd-re-ss", static_cast<uint32_t>(sizeof("AA-AA-AA-AA-AA-AA"))));

    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::LANAdapterMacAddressInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::LANAdapterMacAddress, "LA-NA-da-pt-er-Ma", static_cast<uint32_t>(sizeof("AA-AA-AA-AA-AA-AA"))));

    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::NetworkInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::IPAddressAcquisitionMethod, static_cast<uint32_t>(0)));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::DNSType, static_cast<uint32_t>(0)));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ConnectAutomaticallyFlag, true));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::MTU, static_cast<uint32_t>(1500)));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::UseProxyFlag, true));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ProxyIPAddress, "000.000.000.000", static_cast<uint32_t>(sizeof("000.000.000.000"))));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ProxyPort, static_cast<uint32_t>(8888)));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ProxyAutoAuthenticateFlag, true));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::CurrentIPAddress, "100.000.000.000", static_cast<uint32_t>(sizeof("000.000.000.000"))));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::SubnetMask, "200.000.000.000", static_cast<uint32_t>(sizeof("000.000.000.000"))));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::GatewayIPAddress, "300.000.000.000", static_cast<uint32_t>(sizeof("000.000.000.000"))));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::PriorityDNSIPAddress, "400.000.000.000", static_cast<uint32_t>(sizeof("000.000.000.000"))));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::AlternateDNSIPAddress, "500.000.000.000", static_cast<uint32_t>(sizeof("000.000.000.000"))));

    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::LimitHighCapacityInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::LimitHighCapacityFlag, false));

    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::NintendoZoneSSIDListVersionInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::NintendoZoneSSIDListVersion, "NintendoZoneSSIDListVersion", static_cast<uint32_t>(sizeof("NintendoZoneSSIDListVersion"))));

    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::NintendoZoneConnectedInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::NintendoZoneConnectedFlag, false));

    NNT_EXPECT_RESULT_SUCCESS(context.SubmitContext());

    const char errorCode[] = "1234-5678";
    nn::erpt::Context reportContext(nn::erpt::ErrorInfo);
    NNT_EXPECT_RESULT_SUCCESS(reportContext.Add(nn::erpt::ErrorCode, errorCode, sizeof(errorCode)));
    NNT_EXPECT_RESULT_SUCCESS(reportContext.CreateReport(nn::erpt::ReportType_Invisible));
}

TEST(erpt, MultipleCategoryContextClear)
{
    nn::erpt::MultipleCategoryContext context;
    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::ConnectionStatusInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::AccessPointInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::RadioStrengthInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::WirelessAPMacAddressInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::StealthNetworkInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::NXMacAddressInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::LANAdapterMacAddressInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::NetworkInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::LimitHighCapacityInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::NintendoZoneSSIDListVersionInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.MoveToNextCategory(nn::erpt::CategoryId::NintendoZoneConnectedInfo));
    NNT_EXPECT_RESULT_SUCCESS(context.SubmitContext());

    const char errorCode[] = "1234-5678";
    nn::erpt::Context reportContext(nn::erpt::ErrorInfo);
    NNT_EXPECT_RESULT_SUCCESS(reportContext.Add(nn::erpt::ErrorCode, errorCode, sizeof(errorCode)));
    NNT_EXPECT_RESULT_SUCCESS(reportContext.CreateReport(nn::erpt::ReportType_Invisible));
}

TEST(erpt, DeleteReport)
{
    nn::erpt::Manager reportManager;
    NNT_ASSERT_RESULT_SUCCESS(reportManager.Initialize());
    NN_UTIL_SCOPE_EXIT{ reportManager.Finalize(); };
    reportManager.CleanupReports();

    for( int i = 0; i < 4; i++ )
    {
        const char errorCode[] = "1234-5678";
        nn::erpt::Context reportContext(nn::erpt::ErrorInfo);
        NNT_EXPECT_RESULT_SUCCESS(reportContext.Add(nn::erpt::ErrorCode, errorCode, sizeof(errorCode)));
        NNT_EXPECT_RESULT_SUCCESS(reportContext.CreateReport(nn::erpt::ReportType_Invisible));
    }
    // Delete the middle report
    {
        NNT_ASSERT_RESULT_SUCCESS(reportManager.GetReportList(g_ReportList));
        ASSERT_EQ(g_ReportList.reportCount, 4u);
        auto reportId = g_ReportList.Report[1].reportId;
        NNT_EXPECT_RESULT_SUCCESS(reportManager.DeleteReport(reportId));
        NNT_EXPECT_RESULT_FAILURE(nn::erpt::ResultInvalidArgument, reportManager.DeleteReport(reportId));

        NNT_ASSERT_RESULT_SUCCESS(reportManager.GetReportList(g_ReportList));
        ASSERT_EQ(g_ReportList.reportCount, 3u);
        for( uint32_t i = 0; i < g_ReportList.reportCount; i++ )
        {
            EXPECT_NE(g_ReportList.Report[i].reportId.u.uuidRFC4122.data, reportId.u.uuidRFC4122.data);
        }
    }
    // Delete the last report
    {
        NNT_ASSERT_RESULT_SUCCESS(reportManager.GetReportList(g_ReportList));
        ASSERT_EQ(g_ReportList.reportCount, 3u);
        auto reportId = g_ReportList.Report[2].reportId;
        NNT_EXPECT_RESULT_SUCCESS(reportManager.DeleteReport(reportId));
        NNT_EXPECT_RESULT_FAILURE(nn::erpt::ResultInvalidArgument, reportManager.DeleteReport(reportId));

        NNT_ASSERT_RESULT_SUCCESS(reportManager.GetReportList(g_ReportList));
        ASSERT_EQ(g_ReportList.reportCount, 2u);
        for( uint32_t i = 0; i < g_ReportList.reportCount; i++ )
        {
            EXPECT_NE(g_ReportList.Report[i].reportId.u.uuidRFC4122.data, reportId.u.uuidRFC4122.data);
        }
    }
    // Delete the first report (one remains after delete)
    {
        NNT_ASSERT_RESULT_SUCCESS(reportManager.GetReportList(g_ReportList));
        ASSERT_EQ(g_ReportList.reportCount, 2u);
        auto reportId = g_ReportList.Report[0].reportId;
        NNT_EXPECT_RESULT_SUCCESS(reportManager.DeleteReport(reportId));
        NNT_EXPECT_RESULT_FAILURE(nn::erpt::ResultInvalidArgument, reportManager.DeleteReport(reportId));

        NNT_ASSERT_RESULT_SUCCESS(reportManager.GetReportList(g_ReportList));
        ASSERT_EQ(g_ReportList.reportCount, 1u);
        for( uint32_t i = 0; i < g_ReportList.reportCount; i++ )
        {
            EXPECT_NE(g_ReportList.Report[i].reportId.u.uuidRFC4122.data, reportId.u.uuidRFC4122.data);
        }
    }
    // Delete the only remaining report (and then there were none)
    {
        NNT_ASSERT_RESULT_SUCCESS(reportManager.GetReportList(g_ReportList));
        ASSERT_EQ(g_ReportList.reportCount, 1u);
        auto reportId = g_ReportList.Report[0].reportId;
        NNT_EXPECT_RESULT_SUCCESS(reportManager.DeleteReport(reportId));
        NNT_EXPECT_RESULT_FAILURE(nn::erpt::ResultInvalidArgument, reportManager.DeleteReport(reportId));

        NNT_ASSERT_RESULT_SUCCESS(reportManager.GetReportList(g_ReportList));
        ASSERT_EQ(g_ReportList.reportCount, 0u);
    }
}

TEST(erpt, LargeStringField)
{
    nn::erpt::Context context(nn::erpt::CategoryId::Test, g_Buffer, sizeof(g_Buffer));

    std::memset(g_LargeFieldBuffer, 'A', sizeof(g_LargeFieldBuffer));

    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestString, reinterpret_cast<char*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize));
    NNT_ASSERT_RESULT_SUCCESS(context.SubmitContext());

    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestString, reinterpret_cast<char*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize + 1));
    NNT_EXPECT_RESULT_FAILURE(nn::erpt::ResultArrayFieldSizeTooLarge, context.SubmitContext());

    context.ResetContext(nn::erpt::CategoryId::ErrorInfo);
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ErrorCode, "1234-5678", sizeof("1234-5678")));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ErrorDescription, reinterpret_cast<char*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize));
    NNT_ASSERT_RESULT_SUCCESS(context.CreateReport(nn::erpt::ReportType_Invisible));

    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ErrorDescription, reinterpret_cast<char*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize + 1));
    NNT_EXPECT_RESULT_FAILURE(nn::erpt::ResultArrayFieldSizeTooLarge, context.CreateReport(nn::erpt::ReportType_Invisible));
}

TEST(erpt, LargeArrayField)
{
    nn::erpt::Context context(nn::erpt::CategoryId::Test, g_Buffer, sizeof(g_Buffer));

    std::memset(g_LargeFieldBuffer, 'A', sizeof(g_LargeFieldBuffer));

    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestU32Array, reinterpret_cast<uint32_t*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize / sizeof(uint32_t)));
    NNT_ASSERT_RESULT_SUCCESS(context.SubmitContext());

    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestU32Array, reinterpret_cast<uint32_t*>(g_LargeFieldBuffer), (nn::erpt::MaxArrayFieldSize / sizeof(uint32_t)) + 1));
    NNT_EXPECT_RESULT_FAILURE(nn::erpt::ResultArrayFieldSizeTooLarge, context.SubmitContext());

    context.ResetContext(nn::erpt::CategoryId::ErrorInfo);
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ErrorCode, "1234-5678", sizeof("1234-5678")));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ResultBacktrace, reinterpret_cast<uint32_t*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize / sizeof(uint32_t)));
    NNT_ASSERT_RESULT_SUCCESS(context.CreateReport(nn::erpt::ReportType_Invisible));

    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ResultBacktrace, reinterpret_cast<uint32_t*>(g_LargeFieldBuffer), (nn::erpt::MaxArrayFieldSize / sizeof(uint32_t)) + 1));
    NNT_EXPECT_RESULT_FAILURE(nn::erpt::ResultArrayFieldSizeTooLarge, context.CreateReport(nn::erpt::ReportType_Invisible));
}

TEST(erpt, LargeBuffer)
{
    static const size_t LargeBufferSize = nn::erpt::MaxArrayFieldSize +                       // TestString
                        nn::erpt::MaxArrayFieldSize +                                         // TestStringEnxrypt
                        nn::erpt::MaxArrayFieldSize +                                         // TestU8Array
                        sizeof(uint32_t) * (nn::erpt::MaxArrayFieldSize / sizeof(uint32_t)) + // TestU32Array
                        sizeof(uint64_t) * (nn::erpt::MaxArrayFieldSize / sizeof(uint64_t)) + // TestU32Array
                        sizeof(int32_t)  * (nn::erpt::MaxArrayFieldSize / sizeof(int32_t))  + // TestI32Array
                        sizeof(int64_t)  * (nn::erpt::MaxArrayFieldSize / sizeof(int64_t));   // TestI64Array

    // Ensure that the sum of the fields' size exceeds nn::erpt::MaxArrayBufferLength.
    NN_STATIC_ASSERT(LargeBufferSize > nn::erpt::MaxArrayBufferLength);

    static uint8_t largeBuffer[LargeBufferSize];
    nn::erpt::Context context(nn::erpt::CategoryId::Test, largeBuffer, sizeof(largeBuffer));

    std::memset(g_LargeFieldBuffer, 'A', sizeof(g_LargeFieldBuffer));

    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestString, reinterpret_cast<char*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestStringEncrypt, reinterpret_cast<char*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestU8Array, g_LargeFieldBuffer, nn::erpt::MaxArrayFieldSize));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestU32Array, reinterpret_cast<uint32_t*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize / sizeof(uint32_t)));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestU64Array, reinterpret_cast<uint64_t*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize / sizeof(uint64_t)));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestI32Array, reinterpret_cast<int32_t*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize / sizeof(int32_t)));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestI64Array, reinterpret_cast<int64_t*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize / sizeof(int64_t)));
    NNT_EXPECT_RESULT_FAILURE(nn::erpt::ResultInvalidArgument, context.SubmitContext());
}

TEST(erpt, OutOfMemory)
{
    NNT_ASSERT_RESULT_SUCCESS(ClearAllContext());
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(ClearAllContext());
    };

    nn::erpt::Context context(nn::erpt::CategoryId::Test, g_Buffer, sizeof(g_Buffer));
    // About 96 KiB (16 * 6)
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestString, reinterpret_cast<char*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestU8Array, g_LargeFieldBuffer, nn::erpt::MaxArrayFieldSize));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestU32Array, reinterpret_cast<uint32_t*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize / sizeof(uint32_t)));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestU64Array, reinterpret_cast<uint64_t*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize / sizeof(uint64_t)));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestI32Array, reinterpret_cast<int32_t*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize / sizeof(int32_t)));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::TestI64Array, reinterpret_cast<int64_t*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize / sizeof(int64_t)));
    auto result = context.SubmitContext();
    if ( result.IsSuccess() || nn::erpt::ResultOutOfMemory::Includes(result) )
    {
        // The result of this submit is unstable. It may fail due to memory fragmentation.
    }
    else
    {
        // Failures other than ResultOutOfMemory are unacceptable
        NNT_ASSERT_RESULT_SUCCESS(result);
    }

    context.ResetContext(nn::erpt::CategoryId::ErrorInfo);
    // About 96 KiB (16 * 6)
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ErrorDescription, reinterpret_cast<char*>(g_LargeFieldBuffer), nn::erpt::MaxArrayFieldSize - 9 /* for ErrorCode */));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::EncryptedDyingMessage, g_LargeFieldBuffer, nn::erpt::MaxArrayFieldSize));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::EncryptedExceptionInfo, g_LargeFieldBuffer, nn::erpt::MaxArrayFieldSize));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::EncryptedExceptionInfo1, g_LargeFieldBuffer, nn::erpt::MaxArrayFieldSize));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::EncryptedExceptionInfo2, g_LargeFieldBuffer, nn::erpt::MaxArrayFieldSize));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::EncryptedExceptionInfo3, g_LargeFieldBuffer, nn::erpt::MaxArrayFieldSize));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ErrorCode, "1234-5678", static_cast<uint32_t>(std::strlen("1234-5678"))));

    // Programs/Eris/Sources/Processes/erpt/erpt_main.cpp, ReportMemoryPoolSize = 192 * 1024.
    // With ErrorInfoAuto category and others, the above context would be enough to cause ResultOutOfMemory.
    NNT_EXPECT_RESULT_FAILURE(nn::erpt::ResultOutOfMemory, context.CreateReport(nn::erpt::ReportType_Invisible));

    // Check that making an error report succeeds after ResultOufOfMemory.
    NNT_ASSERT_RESULT_SUCCESS(MakeMinimumErrorReport());
}

TEST(erpt, GenerateManyLargeReports)
{
    NNT_ASSERT_RESULT_SUCCESS(ClearAllContext());
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(ClearAllContext());
    };

    // Similar to the creport's max report size (without other context info).
    nn::erpt::Context context(nn::erpt::CategoryId::ErrorInfo, g_Buffer, sizeof(g_Buffer));
    // One thread's information size is up to 550 bytes, and an application can create 96 threads (http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-77668)
    auto exceptionInfoSizeMax = 550 * 96;
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::EncryptedExceptionInfo, g_LargeFieldBuffer, nn::erpt::MaxArrayFieldSize));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::EncryptedExceptionInfo1, g_LargeFieldBuffer, nn::erpt::MaxArrayFieldSize));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::EncryptedExceptionInfo2, g_LargeFieldBuffer, nn::erpt::MaxArrayFieldSize));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::EncryptedExceptionInfo3, g_LargeFieldBuffer, static_cast<uint32_t>(exceptionInfoSizeMax - nn::erpt::MaxArrayFieldSize * 3)));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::EncryptedDyingMessage, g_LargeFieldBuffer, 4 * 1024));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::EncryptionKey, g_LargeFieldBuffer, 256));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::CrashReportHash, g_LargeFieldBuffer, 32));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ProgramId, "0123456789abcdef", static_cast<uint32_t>(std::strlen("0123456789abcdef"))));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::AbortFlag, true));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ErrorCode, "1234-5678", static_cast<uint32_t>(std::strlen("1234-5678"))));

    for( int type = nn::erpt::ReportType_First; type < nn::erpt::ReportType_Last; type++ )
    {
        for( int i = 0; i <= nn::erpt::NumberOfReports; i++ )
        {
            auto result = context.CreateReport(static_cast<nn::erpt::ReportType>(type));
            if( result.IsSuccess() )
            {
                NN_LOG("o");
            }
            else if( nn::erpt::ResultOutOfMemory::Includes(result) )
            {
                // Creating large error reports in a row sometimes fails with ResultOutOfMemory.
                NN_LOG("x");
                // Workaround : 'reset' ErrorInfo context by creating a small error report.
                NNT_ASSERT_RESULT_SUCCESS(MakeMinimumErrorReport());
            }
            else
            {
                NNT_ASSERT_RESULT_SUCCESS(result) << "ReporType = " << type << ", ReportNumber = " << i;
            }
        }
    }
    NN_LOG("\n");
}


TEST(erpt, GenerateManyTypicalCrashReports)
{
    // Similar to the creport's typical report size.
    nn::erpt::Context context(nn::erpt::CategoryId::ErrorInfo, g_Buffer, sizeof(g_Buffer));
    // 400 bytes/thread * 40 threads.
    const auto exceptionInfoSize = 400 * 40;
    NN_STATIC_ASSERT(exceptionInfoSize <= nn::erpt::MaxArrayFieldSize);
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::EncryptedExceptionInfo, g_LargeFieldBuffer, exceptionInfoSize));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::EncryptedDyingMessage, g_LargeFieldBuffer, 4 * 1024));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::EncryptionKey, g_LargeFieldBuffer, 256));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::CrashReportHash, g_LargeFieldBuffer, 32));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ProgramId, "0123456789abcdef", static_cast<uint32_t>(std::strlen("0123456789abcdef"))));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::AbortFlag, true));
    NNT_ASSERT_RESULT_SUCCESS(context.Add(nn::erpt::FieldId::ErrorCode, "1234-5678", static_cast<uint32_t>(std::strlen("1234-5678"))));

    for( int type = nn::erpt::ReportType_First; type < nn::erpt::ReportType_Last; type++ )
    {
        for( int i = 0; i <= nn::erpt::NumberOfReports; i++ )
        {
            NNT_EXPECT_RESULT_SUCCESS(context.CreateReport(static_cast<nn::erpt::ReportType>(type))) << "ReporType = " << type << ", ReportNumber = " << i;
            NN_LOG(".");
        }
    }
    NN_LOG("\n");
}

}}
