﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ens/detail/util/ens_MessagePackInputStream.h>
#include <nn/fs.h>

namespace nnt { namespace ens {

/**
 * @brief   MessagePack 読み込み用の入力ストリーム（ファイル）
 *
 * @details
 */
class MessagePackFileInputStream : public nn::ens::detail::util::MessagePackInputStream
{
public:
    /**
     * @brief   コンストラクタ
     *
     * @details
     *  本コンストラクタは、読み込みキャッシュを使用しません。
     */
    MessagePackFileInputStream() NN_NOEXCEPT;

    /**
     * @brief   コンストラクタ
     *
     * @param[in]   pBuffer 読み込みキャッシュバッファ
     * @param[in]   size    読み込みキャッシュバッファのサイズ
     *
     * @pre
     *  - pBuffer != nullptr
     *  - size > 0
     *
     * @details
     */
    MessagePackFileInputStream(void* pBuffer, size_t size) NN_NOEXCEPT;

    /**
     * @brief   デストラクタ
     *
     * @details
     */
    virtual ~MessagePackFileInputStream() NN_NOEXCEPT;

    /**
     * @brief   入力ストリームを開きます。
     *
     * @param[in]   pPath   ファイルパス
     *
     * @return  処理結果
     *
     * @pre
     *  - pPath != nullptr
     *
     * @details
     */
    nn::Result Open(const char* pPath) NN_NOEXCEPT;

    /**
     * @brief   入力ストリームを閉じます。
     *
     * @details
     */
    void Close() NN_NOEXCEPT;

    /**
     * @brief   最後に発生したファイル読み込みエラーを取得します。
     *
     * @return  最後に発生したファイル読み込みエラー
     *
     * @details
     */
    nn::Result GetLastReadError() const NN_NOEXCEPT;

    /**
     * @brief   データを読み込みます。
     *
     * @param[in]   pBuffer バッファ
     * @param[in]   size    バッファサイズ
     *
     * @return  成功したかどうか
     *
     * @pre
     *  - pBuffer != nullptr
     *  - size > 0
     *
     * @details
     */
    virtual bool Read(void* pBuffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   メモリ上に展開されているデータのポインタを取得します。
     *
     * @param[out]  ppOutData   データのポインタ
     * @param[in]   size        読み込みたいサイズ
     *
     * @return  読み込みたいサイズのデータがメモリ上に展開されているかどうか
     *
     * @pre
     *  - ppOutData != nullptr
     *  - size > 0
     *
     * @details
     */
    virtual bool GetDataPointerFromMemoryCache(const void** ppOutData, size_t size) NN_NOEXCEPT NN_OVERRIDE;

private:
    //
    nn::fs::FileHandle m_Handle;
    bool m_IsOpened;
    nn::Result m_LastError;
    //
    int64_t m_FilePosition;
    //
    nn::Bit8* m_pCacheBuffer;
    size_t m_CacheBufferSize;
    size_t m_ReadSize;
    size_t m_ReadPosition;

private:
    //
    bool ReadWithMemoryCache(void* pBuffer, size_t size) NN_NOEXCEPT;
    bool ReadWithoutMemoryCache(void* pBuffer, size_t size) NN_NOEXCEPT;
    //
    size_t ReadFromFile(void* pBuffer, size_t size) NN_NOEXCEPT;
    size_t ReadFromMemoryCache(void* pBuffer, size_t size) NN_NOEXCEPT;
};

}}
