﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testEns_Common.h"

#include <nn/ens/detail/core/ens_HttpRequestHeader.h>

namespace
{
    // このサイズでアロケートしようとすると失敗する。
    const size_t BadAllocateSize = 666;
}

namespace
{
    void* TestMalloc(size_t size) NN_NOEXCEPT
    {
        if (size == BadAllocateSize)
        {
            return nullptr;
        }

        return std::malloc(size);
    }

    void TestFree(void* ptr) NN_NOEXCEPT
    {
        std::free(ptr);
    }

    void* TestRealloc(void* ptr, size_t size) NN_NOEXCEPT
    {
        return std::realloc(ptr, size);
    }

    char* TestStrdup(const char* str) NN_NOEXCEPT
    {
        char* p = static_cast<char*>(TestMalloc(std::strlen(str) + 1));

        if (!p)
        {
            return nullptr;
        }

        std::strcpy(p, str);

        return p;
    }

    void* TestCalloc(size_t nmemb, size_t size) NN_NOEXCEPT
    {
        return std::calloc(nmemb, size);
    }

    class EnsHttpRequestHeader : public testing::Test
    {
    protected:
        static void SetUpTestCase() NN_NOEXCEPT
        {
            // 実際の通信は行わないのでオプションは不要。
            curl_global_init_mem(CURL_GLOBAL_NOTHING,
                TestMalloc, TestFree, TestRealloc, TestStrdup, TestCalloc);
        }

        static void TearDownTestCase() NN_NOEXCEPT
        {
            curl_global_cleanup();
        }

        virtual void SetUp() NN_NOEXCEPT
        {
        }

        virtual void TearDown() NN_NOEXCEPT
        {
        }
    };
}

TEST_F(EnsHttpRequestHeader, Case1)
{
    nn::ens::detail::core::HttpRequestHeader header;

    NNT_ASSERT_RESULT_SUCCESS(header.Add("Name1", "Value1"));
    NNT_ASSERT_RESULT_SUCCESS(header.Add("Name2", "Value2"));
    NNT_ASSERT_RESULT_SUCCESS(header.Add("Name3", "Value3"));

    curl_slist* pList = header.Get();

    EXPECT_STREQ(pList->data, "Name1: Value1");
    pList = pList->next;

    EXPECT_STREQ(pList->data, "Name2: Value2");
    pList = pList->next;

    EXPECT_STREQ(pList->data, "Name3: Value3");
}

TEST_F(EnsHttpRequestHeader, Case2)
{
    nn::ens::detail::core::HttpRequestHeader header;

    NNT_ASSERT_RESULT_SUCCESS(header.Add("Name", ""));

    curl_slist* pList = header.Get();

    EXPECT_STREQ(pList->data, "Name: ");
}

TEST_F(EnsHttpRequestHeader, Case3)
{
    nn::ens::detail::core::HttpRequestHeader header;

    NNT_ASSERT_RESULT_SUCCESS(header.Add("Authorization", "Bearer %.*s", 5, "1234567"));

    curl_slist* pList = header.Get();

    EXPECT_STREQ(pList->data, "Authorization: Bearer 12345");
}

TEST_F(EnsHttpRequestHeader, TooBig)
{
    nn::ens::detail::core::HttpRequestHeader header;

    // FieldSize - ": "
    char name[nn::ens::detail::core::HttpRequestHeader::FieldSize - 2] = {};

    for (size_t i = 0; i < sizeof (name) - 1; i++)
    {
        name[i] = 'x';
    }

    NNT_ASSERT_RESULT_SUCCESS(header.Add(name, ""));

    header.Clear();

    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultInvalidArgument, header.Add(name, "x"));
}

TEST_F(EnsHttpRequestHeader, OutOfMemory)
{
    nn::ens::detail::core::HttpRequestHeader header;

    NNT_ASSERT_RESULT_SUCCESS(header.Add("Name", "Value"));

    // BadAllocateSize - ": "
    char name[BadAllocateSize - 2] = {};

    for (size_t i = 0; i < sizeof (name) - 1; i++)
    {
        name[i] = 'x';
    }

    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultHttpErrorOutOfMemory, header.Add(name, ""));

    curl_slist* pList = header.Get();

    EXPECT_STREQ(pList->data, "Name: Value");
}
