﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testEns_Common.h"
#include "../../Common/testEns_AccountUtility.h"
#include "../../Common/testEns_NetworkUtility.h"

#include <nn/ens/detail/core/task/ens_TaskReport.h>

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStack[64 * 1024];

    NN_ALIGNAS(4096) nn::Bit8 g_ServiceWorkMemory[nn::ens::RequiredMemorySizeMin + 4 * 1024 * 1024];

    nn::ens::Credential g_Credential = nn::ens::InvalidCredential;
    bool g_IsGenerated = false;
}

namespace
{
    void WorkerThread(void*) NN_NOEXCEPT
    {
        nn::ens::StartServiceLoop("acbaa", g_ServiceWorkMemory, sizeof (g_ServiceWorkMemory));
    }

    // TORIAEZU: 検証がすんだら公開 API 化する。
    void Report(nn::ens::AsyncContext* pOutContext,
        const char* pType,
        nn::account::NetworkServiceAccountId nsaId,
        nn::ens::detail::ReportCategory category,
        const char* pReason,
        const nn::ens::SendBuffer& image,
        const nn::ens::detail::ReportContentInfo& contentInfo,
        const nn::ens::Credential& credential, const char* pToken) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutContext);
        NN_SDK_REQUIRES_NOT_NULL(pType);
        NN_SDK_REQUIRES_NOT_NULL(pReason);
        NN_SDK_REQUIRES(image.IsValid());
        NN_SDK_REQUIRES(credential.IsValid());
        NN_SDK_REQUIRES_NOT_NULL(pToken);

#if defined(NN_DETAIL_ENABLE_SDK_ASSERT)

        {
            size_t length = static_cast<size_t>(nn::util::Strnlen(pType, INT_MAX));

            if (length > 0)
            {
                NN_SDK_REQUIRES(nn::util::VerifyUtf8String(pType, length));
            }
        }
        {
            size_t length = static_cast<size_t>(nn::util::Strnlen(pReason, INT_MAX));

            if (length > 0)
            {
                NN_SDK_REQUIRES(nn::util::VerifyUtf8String(pReason, length));
            }
        }

#endif

        NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, nn::ens::detail::core::task::TaskReport);

        pTask->SetCredential(credential);
        pTask->SetNetworkServiceAccountIdToken(pToken);

        pTask->SetParameter(pType, nsaId, category, pReason, image, contentInfo);

        nn::ens::detail::core::RegisterTask(pTask, pOutContext);
    }
}

class EnsReport : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfig));

        nnt::ens::ReserveSockets();
        nnt::ens::InitializeLibcurl();
        nnt::ens::EnableCommunicationLogDump();
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
        nnt::ens::FinalizeLibcurl();
        nnt::ens::CancelSocketsReservation();

        nn::socket::Finalize();
    }

    virtual void SetUp() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
            g_ThreadStack, sizeof (g_ThreadStack), nn::os::DefaultThreadPriority + 1));

        nn::os::StartThread(&g_Thread);

        nn::nifm::SubmitNetworkRequestAndWait();
    }

    virtual void TearDown() NN_NOEXCEPT
    {
        nn::nifm::CancelNetworkRequest();

        nn::ens::StopServiceLoop();

        nn::os::DestroyThread(&g_Thread);
    }
};

TEST_F(EnsReport, GenerateCredencial)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    nn::ens::AsyncContext context;

    nn::ens::GenerateCredential(&context, &g_Credential, token.Get());

    context.GetEvent().Wait();

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

    NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

#else

    // Generic ではローカルサーバ起動が必要なので、接続タイムアウトのみ除外する。
    if (context.GetResult().IsFailure() &&
        !nn::ens::ResultHttpErrorOperationTimedout::Includes(context.GetResult()))
    {
        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }

#endif

    if (context.GetResult().IsSuccess())
    {
        NN_LOG("UserId = %llu\n", g_Credential.userId.value);
        g_IsGenerated = true;
    }
}

TEST_F(EnsReport, Report)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsGenerated)
    {
        NN_LOG("Credentials was not generated. This test is skipped.\n");
        return;
    }

#if 1 // サーバー側の通報周りのタスクが一通り解決するまでは CI 上でテストしない。

    NN_UNUSED(&Report);

#else

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    nn::account::NetworkServiceAccountId nsaId = {};
    ASSERT_NO_FATAL_FAILURE(nnt::ens::GetNsaId(&nsaId));

    nn::ens::detail::ReportContentInfo contentInfo = {};

    contentInfo.id = 1000;
    contentInfo.language = {"ja-JP"};

    // 2018-04-25 17:00:00
    contentInfo.createdAt.value = 1524675600ll;

    {
        nn::ens::AsyncContext context;

        Report(&context, "messagecard", nsaId, nn::ens::detail::ReportCategory_Harassment, u8"悪口が書かれている。",
            nn::ens::NullSendBuffer, contentInfo, g_Credential, token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::ens::SendBuffer image = {"image", 5};

        nn::ens::AsyncContext context;

        Report(&context, "messagecard", nsaId, nn::ens::detail::ReportCategory_Harassment, u8"悪口が書かれている。",
            image, contentInfo, g_Credential, token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }

#endif
}
