﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testEns_Common.h"
#include "../../Common/testEns_AccountUtility.h"
#include "../../Common/testEns_NetworkUtility.h"

#include <nn/ens/detail/core/task/ens_TaskGetAuthToken.h>

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStack[64 * 1024];

    NN_ALIGNAS(4096) nn::Bit8 g_ServiceWorkMemory[nn::ens::RequiredMemorySizeMin + 4 * 1024 * 1024];
}

namespace
{
    void WorkerThread(void*) NN_NOEXCEPT
    {
        nn::ens::StartServiceLoop("acbaa", g_ServiceWorkMemory, sizeof (g_ServiceWorkMemory));
    }

    void CreateGetAuthTokenTask(nn::ens::AsyncContext* pOutContext, nn::ens::detail::AuthToken* pOutAuthToken,
        const nn::ens::Credential& credential, const char* pToken) NN_NOEXCEPT
    {
        NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, nn::ens::detail::core::task::TaskGetAuthToken);

        pTask->SetParameter(pOutAuthToken, credential);
        pTask->SetNetworkServiceAccountIdToken(pToken);

        nn::ens::detail::core::RegisterTask(pTask, pOutContext);
    }
}

class EnsAuthentication : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfig));

        nnt::ens::ReserveSockets();
        nnt::ens::InitializeLibcurl();
        nnt::ens::EnableCommunicationLogDump();
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
        nnt::ens::FinalizeLibcurl();
        nnt::ens::CancelSocketsReservation();

        nn::socket::Finalize();
    }

    virtual void SetUp() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
            g_ThreadStack, sizeof (g_ThreadStack), nn::os::DefaultThreadPriority + 1));

        nn::os::StartThread(&g_Thread);

        nn::nifm::SubmitNetworkRequestAndWait();
    }

    virtual void TearDown() NN_NOEXCEPT
    {
        nn::nifm::CancelNetworkRequest();

        nn::ens::StopServiceLoop();

        nn::os::DestroyThread(&g_Thread);
    }
};

TEST_F(EnsAuthentication, InvalidNsaIdToken)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    nn::ens::Credential credential = {};

    {
        nn::ens::AsyncContext context;

        nn::ens::GenerateCredential(&context, &credential, "invalid");

        context.GetEvent().Wait();

#if defined(NN_BUILD_CONFIG_OS_WIN)

        // Generic ではローカルサーバ起動が必要なので、接続タイムアウトのみ除外する。
        if (nn::ens::ResultHttpErrorOperationTimedout::Includes(context.GetResult()))
        {
            return;
        }

#endif

        NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultNetworkServiceAccountIdTokenVerificationFailed, context.GetResult());
    }
}

TEST_F(EnsAuthentication, Authenticate)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    nn::ens::Credential credential = {};

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    {
        nn::ens::AsyncContext context;

        nn::ens::GenerateCredential(&context, &credential, token.Get());

        context.GetEvent().Wait();

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

#else

        // Generic ではローカルサーバ起動が必要なので、接続タイムアウトのみ除外する。
        if (context.GetResult().IsFailure() &&
            !nn::ens::ResultHttpErrorOperationTimedout::Includes(context.GetResult()))
        {
            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }

#endif

        if (context.GetResult().IsFailure())
        {
            NN_LOG("Credentials was not generated. This test is skipped.\n");
            return;
        }
    }

    {
        nn::ens::AsyncContext context;
        nn::ens::detail::AuthToken authToken;

        CreateGetAuthTokenTask(&context, &authToken, credential, token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::ens::Credential credential2 = credential;
        nn::ens::AsyncContext context;
        nn::ens::detail::AuthToken authToken;

        credential2.userId.value++;

        CreateGetAuthTokenTask(&context, &authToken, credential2, token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultNotFound, context.GetResult());
    }
    {
        nn::ens::Credential credential2 = credential;
        nn::ens::AsyncContext context;
        nn::ens::detail::AuthToken authToken;

        nn::util::Strlcpy(credential2._password, "invalid password", sizeof (credential2._password));

        CreateGetAuthTokenTask(&context, &authToken, credential2, token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultWrongCredentials, context.GetResult());
    }
}
