﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testEns_TaskGetMyDesignMetadataFixture.h"

#include <nn/ens/detail/core/ens_HttpRequestHeader.h>
#include <nn/ens/detail/util/ens_MessagePackReader.h>

namespace nnt { namespace ens {

TaskGetMyDesignMetadataFixture::TaskGetMyDesignMetadataFixture() NN_NOEXCEPT
    : m_pOutMetadata(nullptr)
    , m_pDesignName(nullptr)
{
}

TaskGetMyDesignMetadataFixture::~TaskGetMyDesignMetadataFixture() NN_NOEXCEPT
{
}

void TaskGetMyDesignMetadataFixture::SetParameter(nn::ens::ReceiveBuffer* pOutMetadata,
    const char* pDesignName) NN_NOEXCEPT
{
    m_pOutMetadata = pOutMetadata;

    m_pDesignName = pDesignName;
}

nn::Result TaskGetMyDesignMetadataFixture::Main() NN_NOEXCEPT
{
    NN_RESULT_DO(StepGet());

    NN_RESULT_SUCCESS;
}

nn::Result TaskGetMyDesignMetadataFixture::StepGet() NN_NOEXCEPT
{
    NN_DETAIL_ENS_SCOPED_ACQUIRE_HTTP_CONNECTION(pCurl);

    char query[1536] = {};
    size_t offset = 0;

    // 入力パラメータへのアクセス
    {
        NN_UTIL_LOCK_GUARD(GetParameterLock());

        NN_RESULT_THROW_UNLESS(IsAsyncContextRegistered(), nn::ens::ResultInvalidState());

        if (m_pDesignName)
        {
            offset += static_cast<size_t>(nn::util::SNPrintf(query, sizeof (query), "mMtDNm="));

            size_t escapedLength = 0;

            NN_RESULT_THROW_UNLESS(nn::ens::detail::util::EscapeUriDataString(&escapedLength, &query[offset], sizeof (query) - offset,
                m_pDesignName, static_cast<size_t>(nn::util::Strnlen(m_pDesignName, INT_MAX))), nn::ens::ResultInvalidArgument());

            offset += escapedLength;
        }
    }

    NN_SDK_ASSERT_LESS(offset, sizeof (query));

    char url[2048] = {};
    nn::ens::detail::core::MakeUrl(url, sizeof (url), "api", "/api/v1/fixtures/designs%s%s", query[0] ? "?" : "", query);

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_CUSTOMREQUEST, "GET"));
    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_URL, url));

    nn::ens::detail::core::HttpRequestHeader header;

    NN_RESULT_DO(header.Add("Content-Type", "application/x-msgpack"));

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_HTTPHEADER, header.Get()));

    NN_RESULT_DO(Perform());

    NN_DETAIL_ENS_DUMP_RESPONSE(nn::ens::detail::util::MessagePackReader, GetResponse(), GetResponseSize());

    // 出力パラメータへのアクセス
    {
        NN_UTIL_LOCK_GUARD(GetParameterLock());

        if (IsAsyncContextRegistered())
        {
            NN_RESULT_THROW_UNLESS(m_pOutMetadata->bufferSize >= GetResponseSize(), nn::ens::ResultInvalidResponse());

            std::memcpy(m_pOutMetadata->pBuffer, GetResponse(), GetResponseSize());
            m_pOutMetadata->receivedSize = GetResponseSize();
        }
    }

    NN_RESULT_SUCCESS;
}

}}
